/*
 *  Copyright 2010 BetaSteward_at_googlemail.com. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are
 *  permitted provided that the following conditions are met:
 *
 *     1. Redistributions of source code must retain the above copyright notice, this list of
 *        conditions and the following disclaimer.
 *
 *     2. Redistributions in binary form must reproduce the above copyright notice, this list
 *        of conditions and the following disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY BetaSteward_at_googlemail.com ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BetaSteward_at_googlemail.com OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  The views and conclusions contained in the software and documentation are those of the
 *  authors and should not be interpreted as representing official policies, either expressed
 *  or implied, of BetaSteward_at_googlemail.com.
 */
package mage.cards.k;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import mage.abilities.Ability;
import mage.abilities.effects.OneShotEffect;
import mage.cards.CardImpl;
import mage.cards.CardSetInfo;
import mage.choices.ChooseFriendsAndFoes;
import mage.constants.CardType;
import mage.constants.Outcome;
import mage.game.Game;
import mage.players.Player;

/**
 *
 * @author TheElk801
 */
public final class KhorvathsFury extends CardImpl {

    public KhorvathsFury(UUID ownerId, CardSetInfo setInfo) {
        super(ownerId, setInfo, new CardType[]{CardType.SORCERY}, "{4}{R}");

        // For each player, choose friend or foe. Each friend discards all cards from their hand, then draws that many cards plus one. Khorvath's Fury deals damage to each foe equal to the number of cards in their hand.
        this.getSpellAbility().addEffect(new KhorvathsFuryEffect());
    }

    public KhorvathsFury(final KhorvathsFury card) {
        super(card);
    }

    @Override
    public KhorvathsFury copy() {
        return new KhorvathsFury(this);
    }
}

class KhorvathsFuryEffect extends OneShotEffect {

    KhorvathsFuryEffect() {
        super(Outcome.Benefit);
        this.staticText = "For each player, choose friend or foe. "
                + "Each friend discards all cards from their hand, "
                + "then draws that many cards plus one."
                + " {this} deals damage to each foe equal to the number of cards in their hand";
    }

    KhorvathsFuryEffect(final KhorvathsFuryEffect effect) {
        super(effect);
    }

    @Override
    public KhorvathsFuryEffect copy() {
        return new KhorvathsFuryEffect(this);
    }

    @Override
    public boolean apply(Game game, Ability source) {
        Player controller = game.getPlayer(source.getControllerId());
        if (controller == null) {
            return false;
        }
        ChooseFriendsAndFoes choice = new ChooseFriendsAndFoes();
        choice.chooseFriendOrFoe(controller, source, game);
        Map<UUID, Integer> cardsToDraw = new HashMap<>();
        for (Player player : choice.getFriends()) {
            if (player != null) {
                int cardsInHand = player.getHand().size();
                player.discard(cardsInHand, false, source, game);
                if (cardsInHand > 0) {
                    cardsToDraw.put(player.getId(), cardsInHand);
                }
            }
        }
        for (Player player : choice.getFriends()) {
            if (player != null) {
                player.drawCards(cardsToDraw.get(player.getId()) + 1, game);
            }
        }
        for (Player player : choice.getFoes()) {
            if (player != null) {
                player.damage(player.getHand().size(), source.getSourceId(), game, false, true);
            }
        }
        return true;
    }
}
