/*
 *  Copyright 2010 BetaSteward_at_googlemail.com. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are
 *  permitted provided that the following conditions are met:
 *
 *     1. Redistributions of source code must retain the above copyright notice, this list of
 *        conditions and the following disclaimer.
 *
 *     2. Redistributions in binary form must reproduce the above copyright notice, this list
 *        of conditions and the following disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY BetaSteward_at_googlemail.com ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BetaSteward_at_googlemail.com OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  The views and conclusions contained in the software and documentation are those of the
 *  authors and should not be interpreted as representing official policies, either expressed
 *  or implied, of BetaSteward_at_googlemail.com.
 */
package mage.cards.s;

import java.util.Collection;
import java.util.TreeSet;
import java.util.UUID;
import mage.MageInt;
import mage.abilities.Ability;
import mage.abilities.common.DealsCombatDamageToAPlayerTriggeredAbility;
import mage.abilities.effects.OneShotEffect;
import mage.abilities.keyword.FlyingAbility;
import mage.cards.Card;
import mage.cards.CardImpl;
import mage.cards.CardSetInfo;
import mage.choices.Choice;
import mage.choices.ChoiceImpl;
import mage.constants.CardType;
import mage.constants.Outcome;
import mage.constants.SubType;
import mage.constants.Zone;
import mage.game.Game;
import mage.game.permanent.Permanent;
import mage.players.Player;
import mage.target.common.TargetCardInLibrary;

/**
 *
 * @author North
 */
public final class SphinxAmbassador extends CardImpl {

    public SphinxAmbassador(UUID ownerId, CardSetInfo setInfo) {
        super(ownerId, setInfo, new CardType[]{CardType.CREATURE}, "{5}{U}{U}");
        this.subtype.add(SubType.SPHINX);

        this.power = new MageInt(5);
        this.toughness = new MageInt(5);

        this.addAbility(FlyingAbility.getInstance());
        // Whenever Sphinx Ambassador deals combat damage to a player, search that player's library for a card, then that player names a card. If you searched for a creature card that isn't the named card, you may put it onto the battlefield under your control. Then that player shuffles their library.
        this.addAbility(new DealsCombatDamageToAPlayerTriggeredAbility(new SphinxAmbassadorEffect(), false, true));
    }

    public SphinxAmbassador(final SphinxAmbassador card) {
        super(card);
    }

    @Override
    public SphinxAmbassador copy() {
        return new SphinxAmbassador(this);
    }
}

class SphinxAmbassadorEffect extends OneShotEffect {

    public SphinxAmbassadorEffect() {
        super(Outcome.PutCreatureInPlay);
        this.staticText = "search that player's library for a card, then that player names a card. If you searched for a creature card that isn't the named card, you may put it onto the battlefield under your control. Then that player shuffles their library";
    }

    public SphinxAmbassadorEffect(final SphinxAmbassadorEffect effect) {
        super(effect);
    }

    @Override
    public SphinxAmbassadorEffect copy() {
        return new SphinxAmbassadorEffect(this);
    }

    @Override
    public boolean apply(Game game, Ability source) {
        Player controller = game.getPlayer(source.getControllerId());
        Player targetPlayer = game.getPlayer(targetPointer.getFirst(game, source));
        Permanent sourcePermanent = game.getPermanentOrLKIBattlefield(source.getSourceId());
        if (controller != null && targetPlayer != null && sourcePermanent != null) {
            TargetCardInLibrary target = new TargetCardInLibrary();
            controller.searchLibrary(target, game, targetPlayer.getId());

            Card card = game.getCard(target.getFirstTarget());
            if (card != null) {
                TreeSet<String> choices = new TreeSet<>();
                Collection<Card> cards = game.getCards();
                for (Card gameCard : cards) {
                    if (gameCard.getOwnerId().equals(targetPlayer.getId())) {
                        choices.add(gameCard.getName());
                    }
                }

                Choice cardChoice = new ChoiceImpl();
                cardChoice.setChoices(choices);
                cardChoice.clearChoice();
                if (!targetPlayer.choose(Outcome.Benefit, cardChoice, game)) {
                    return false;
                }
                String cardName = cardChoice.getChoice();

                game.informPlayers(sourcePermanent.getName() + ", named card: [" + cardName + ']');
                if (!card.getName().equals(cardName) && card.isCreature()) {
                    if (controller.chooseUse(outcome, "Put " + card.getName() + " onto the battlefield?", source, game)) {
                        controller.moveCards(card, Zone.BATTLEFIELD, source, game);
                    }
                }
            }

            targetPlayer.shuffleLibrary(source, game);
            return true;
        }
        return false;
    }
}
