/*
 *  Copyright 2010 BetaSteward_at_googlemail.com. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are
 *  permitted provided that the following conditions are met:
 *
 *     1. Redistributions of source code must retain the above copyright notice, this list of
 *        conditions and the following disclaimer.
 *
 *     2. Redistributions in binary form must reproduce the above copyright notice, this list
 *        of conditions and the following disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY BetaSteward_at_googlemail.com ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BetaSteward_at_googlemail.com OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  The views and conclusions contained in the software and documentation are those of the
 *  authors and should not be interpreted as representing official policies, either expressed
 *  or implied, of BetaSteward_at_googlemail.com.
 */
package mage.filter.predicate.mageobject;

import java.util.UUID;
import mage.MageItem;
import mage.abilities.Mode;
import mage.filter.predicate.ObjectSourcePlayer;
import mage.filter.predicate.ObjectSourcePlayerPredicate;
import mage.game.Game;
import mage.game.stack.StackObject;
import mage.target.Target;

/**
 * All targets that are already selected in other target definitions of the
 * source are omitted To use this predicate you have to set the targetTag of all
 * targets involved in the card constructor to a unique value (e.g. using 1,2,3
 * for three targets)
 *
 * @author LevelX2
 */
public class AnotherTargetPredicate implements ObjectSourcePlayerPredicate<ObjectSourcePlayer<MageItem>> {

    private final int targetTag;
    private final boolean crossModalCheck;

    /**
     *
     * @param targetTag tag of the target the filter belongs to
     */
    public AnotherTargetPredicate(int targetTag) {
        this(targetTag, false);
    }

    public AnotherTargetPredicate(int targetTag, boolean crossModalCheck) {
        this.targetTag = targetTag;
        this.crossModalCheck = crossModalCheck;
    }

    @Override
    public boolean apply(ObjectSourcePlayer<MageItem> input, Game game) {
        StackObject source = game.getStack().getStackObject(input.getSourceId());
        if (source != null && source.getStackAbility().getTargets() != null) {
            if (crossModalCheck) {
                for (UUID modeId : source.getStackAbility().getModes().getSelectedModes()) {
                    Mode mode = source.getStackAbility().getModes().get(modeId);
                    for (Target target : mode.getTargets()) {
                        if (target.getTargetTag() > 0 // target is included in the target group to check
                                && target.getTargetTag() != targetTag // it's not the target of this predicate
                                && target.getTargets().contains(input.getObject().getId())) { // if the uuid already is used for another target in the group it's not allowed here
                            return false;
                        }
                    }
                }
            } else {
                for (Target target : source.getStackAbility().getTargets()) {
                    if (target.getTargetTag() > 0 // target is included in the target group to check
                            && target.getTargetTag() != targetTag // it's not the target of this predicate
                            && target.getTargets().contains(input.getObject().getId())) { // if the uuid already is used for another target in the group it's not allowed here
                        return false;
                    }
                }
            }
        }
        return true;
    }

    @Override
    public String toString() {
        return "Another target";
    }
}
