/*
 *  Copyright 2011 BetaSteward_at_googlemail.com. All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification, are
 *  permitted provided that the following conditions are met:
 * 
 *     1. Redistributions of source code must retain the above copyright notice, this list of
 *        conditions and the following disclaimer.
 * 
 *     2. Redistributions in binary form must reproduce the above copyright notice, this list
 *        of conditions and the following disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 * 
 *  THIS SOFTWARE IS PROVIDED BY BetaSteward_at_googlemail.com ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BetaSteward_at_googlemail.com OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 *  The views and conclusions contained in the software and documentation are those of the
 *  authors and should not be interpreted as representing official policies, either expressed
 *  or implied, of BetaSteward_at_googlemail.com.
 */
package mage.watchers.common;

import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import mage.MageObject;
import mage.MageObjectReference;
import mage.constants.WatcherScope;
import mage.game.Game;
import mage.game.events.GameEvent;
import mage.game.events.GameEvent.EventType;
import mage.game.permanent.Permanent;
import mage.watchers.Watcher;

/**
 * @author BetaSteward_at_googlemail.com
 */
public class DamagedByWatcher extends Watcher {

    public final Set<MageObjectReference> damagedBySource = new HashSet<>();

    private final boolean watchPlaneswalkers;

    public DamagedByWatcher() {
        this(false);
    }

    public DamagedByWatcher(boolean watchPlaneswalkers) {
        super(DamagedByWatcher.class.getSimpleName(), WatcherScope.CARD);
        this.watchPlaneswalkers = watchPlaneswalkers;
    }

    public DamagedByWatcher(final DamagedByWatcher watcher) {
        super(watcher);
        this.damagedBySource.addAll(watcher.damagedBySource);
        this.watchPlaneswalkers = watcher.watchPlaneswalkers;
    }

    @Override
    public DamagedByWatcher copy() {
        return new DamagedByWatcher(this);
    }

    @Override
    public void watch(GameEvent event, Game game) {
        boolean eventHasAppropriateType = (event.getType() == EventType.DAMAGED_CREATURE) ||
                (watchPlaneswalkers && event.getType() == EventType.DAMAGED_PLANESWALKER);
        if (eventHasAppropriateType && sourceId.equals(event.getSourceId())) {
            MageObjectReference mor = new MageObjectReference(event.getTargetId(), game);
            damagedBySource.add(mor);

        }
    }

    @Override
    public void reset() {
        super.reset();
        damagedBySource.clear();
    }

    public boolean wasDamaged(UUID sourceId, Game game) {
        MageObject mageObject = game.getObject(sourceId);
        if (mageObject instanceof Permanent) {
            return wasDamaged((Permanent) mageObject, game);
        }
        return false;
    }

    public boolean wasDamaged(Permanent permanent, Game game) {
        return damagedBySource.contains(new MageObjectReference(permanent, game));
    }
}
