/* Keyleds -- Gaming keyboard tool
 * Copyright (C) 2017 Julien Hartmann, juli1.hartmann@gmail.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "keyledsd/LayoutDescription.h"

#include "config.h"
#include "logging.h"
#include "tools/Paths.h"
#include <algorithm>
#include <cstddef>
#include <cstring>
#include <fstream>
#include <istream>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <limits>
#include <memory>
#include <sstream>

LOGGING("layout");

using keyleds::LayoutDescription;

/****************************************************************************/

using xmlString = std::unique_ptr<xmlChar[], void(*)(void *)>;

static constexpr xmlChar SPURIOUS_TAG[] = "spurious";
static constexpr xmlChar KEYBOARD_TAG[] = "keyboard";
static constexpr xmlChar ROW_TAG[] = "row";
static constexpr xmlChar KEY_TAG[] = "key";

static constexpr xmlChar ROOT_ATTR_NAME[] = "layout";
static constexpr xmlChar KEYBOARD_ATTR_X[] = "x";
static constexpr xmlChar KEYBOARD_ATTR_Y[] = "y";
static constexpr xmlChar KEYBOARD_ATTR_WIDTH[] = "width";
static constexpr xmlChar KEYBOARD_ATTR_HEIGHT[] = "height";
static constexpr xmlChar KEYBOARD_ATTR_ZONE[] = "zone";
static constexpr xmlChar KEY_ATTR_CODE[] = "code";
static constexpr xmlChar KEY_ATTR_GLYPH[] = "glyph";
static constexpr xmlChar KEY_ATTR_WIDTH[] = "width";

/****************************************************************************/

static void hideErrorFunc(void *, xmlErrorPtr) {}

static unsigned parseUInt(xmlNode * node, const xmlChar * name, int base)
{
    xmlString valueStr(xmlGetProp(node, name), xmlFree);
    if (valueStr == nullptr) {
        std::ostringstream errMsg;
        errMsg <<"Element '" <<node->name <<"' misses a '" <<name <<"' attribute";
        throw LayoutDescription::ParseError(errMsg.str(), xmlGetLineNo(node));
    }
    char * strEnd;
    auto valueUInt = std::strtoul(reinterpret_cast<char*>(valueStr.get()), &strEnd, base);
    if (*strEnd != '\0') {
        std::ostringstream errMsg;
        errMsg <<"Value '" <<valueStr.get() <<"' in attribute '" <<name <<"' of '"
               <<node->name <<"' element cannot be parsed as an integer";
        throw LayoutDescription::ParseError(errMsg.str(), xmlGetLineNo(node));
    }
    if (valueUInt > std::numeric_limits<unsigned>::max()) {
        std::ostringstream errMsg;
        errMsg <<"Value '" <<valueStr.get() <<"' in attribute '" <<name <<"' of '"
               <<node->name <<"' is too large";
        throw LayoutDescription::ParseError(errMsg.str(), xmlGetLineNo(node));
    }
    return static_cast<unsigned>(valueUInt);
}

/****************************************************************************/

static void parseKeyboard(xmlNode * keyboard, LayoutDescription::key_list & keys)
{
    char * strEnd;
    unsigned kbX = parseUInt(keyboard, KEYBOARD_ATTR_X, 10);
    unsigned kbY = parseUInt(keyboard, KEYBOARD_ATTR_Y, 10);
    unsigned kbWidth = parseUInt(keyboard, KEYBOARD_ATTR_WIDTH, 10);
    unsigned kbHeight = parseUInt(keyboard, KEYBOARD_ATTR_HEIGHT, 10);
    unsigned kbZone = parseUInt(keyboard, KEYBOARD_ATTR_ZONE, 0);

    unsigned nbRows = 0;
    for (const xmlNode * row = keyboard->children; row != nullptr; row = row->next) {
        if (row->type == XML_ELEMENT_NODE || xmlStrcmp(row->name, ROW_TAG) == 0) { nbRows += 1; }
    }

    unsigned rowIdx = 0;
    for (const xmlNode * row = keyboard->children; row != nullptr; row = row->next) {
        if (row->type != XML_ELEMENT_NODE || xmlStrcmp(row->name, ROW_TAG) != 0) { continue; }

        unsigned totalWidth = 0;
        for (xmlNode * key = row->children; key != nullptr; key = key->next) {
            if (key->type != XML_ELEMENT_NODE || xmlStrcmp(key->name, KEY_TAG) != 0) { continue; }
            xmlString keyWidthStr(xmlGetProp(key, KEY_ATTR_WIDTH), xmlFree);
            if (keyWidthStr != nullptr) {
                auto keyWidthFloat = ::strtof(reinterpret_cast<char *>(keyWidthStr.get()), &strEnd);
                if (*strEnd != '\0') {
                    std::ostringstream errMsg;
                    errMsg <<"Value '" <<keyWidthStr.get() <<"' in attribute '"
                           <<KEY_ATTR_WIDTH <<"' of '" <<KEY_TAG
                           <<"' element cannot be parsed as a float";
                    throw LayoutDescription::ParseError(errMsg.str(), xmlGetLineNo(key));
                }
                totalWidth += static_cast<unsigned int>(keyWidthFloat * 1000);
            } else {
                totalWidth += 1000u;
            }
        }

        unsigned xOffset = 0;
        for (xmlNode * key = row->children; key != nullptr; key = key->next) {
            if (key->type != XML_ELEMENT_NODE || xmlStrcmp(key->name, KEY_TAG) != 0) { continue; }
            xmlString code(xmlGetProp(key, KEY_ATTR_CODE), xmlFree);
            xmlString glyph(xmlGetProp(key, KEY_ATTR_GLYPH), xmlFree);
            xmlString keyWidthStr(xmlGetProp(key, KEY_ATTR_WIDTH), xmlFree);

            unsigned keyWidth;
            if (keyWidthStr != nullptr) {
                auto keyWidthFloat = ::strtof(reinterpret_cast<char*>(keyWidthStr.get()), &strEnd);
                keyWidth = kbWidth
                         * static_cast<unsigned int>(keyWidthFloat * 1000)
                         / totalWidth;
            } else {
                keyWidth = kbWidth * 1000 / totalWidth;
            }

            if (code != nullptr) {
                unsigned codeVal = parseUInt(key, KEY_ATTR_CODE, 0);
                auto codeNameStr = glyph != nullptr ? std::string(reinterpret_cast<char *>(glyph.get()))
                                                    : std::string();
                std::transform(codeNameStr.begin(), codeNameStr.end(), codeNameStr.begin(), ::toupper);

                keys.push_back({
                    kbZone,
                    codeVal,
                    LayoutDescription::Rect {
                        kbX + xOffset,
                        kbY + rowIdx * (kbHeight / nbRows),
                        kbX + xOffset + keyWidth - 1,
                        kbY + (rowIdx + 1) * (kbHeight / nbRows) - 1
                    },
                    codeNameStr
                });
            }
            xOffset += keyWidth;
        }
        rowIdx += 1;
    }
}

/****************************************************************************/

LayoutDescription LayoutDescription::parse(std::istream & stream)
{
    // Parser context
    std::unique_ptr<xmlParserCtxt, void(*)(xmlParserCtxtPtr)> context(
        xmlNewParserCtxt(), xmlFreeParserCtxt
    );
    if (context == nullptr) {
        throw std::runtime_error("Failed to initialize libxml");
    }
    xmlSetStructuredErrorFunc(context.get(), hideErrorFunc);

    // Document
    std::ostringstream bufferStream;
    bufferStream << stream.rdbuf();

    auto buffer = bufferStream.str();
    if (buffer.size() > std::numeric_limits<int>::max()) {
        throw std::runtime_error("Description file too large");
    }
    std::unique_ptr<xmlDoc, void(*)(xmlDocPtr)> document(
        xmlCtxtReadMemory(context.get(), buffer.data(), static_cast<int>(buffer.size()),
                          nullptr, nullptr, XML_PARSE_NOWARNING | XML_PARSE_NONET),
        xmlFreeDoc
    );
    if (document == nullptr) {
        auto error = xmlCtxtGetLastError(context.get());
        if (error == nullptr) { throw ParseError("empty file", 1); }
        std::string errMsg(error->message);
        errMsg.erase(errMsg.find_last_not_of(" \r\n") + 1);
        throw ParseError(errMsg, error->line);
    }

    // Scan top-level nodes
    xmlNode * const root = xmlDocGetRootElement(document.get());
    auto name = xmlString(xmlGetProp(root, ROOT_ATTR_NAME), xmlFree);

    key_list keys;
    pos_list spurious;
    for (xmlNode * node = root->children; node != nullptr; node = node->next) {
        if (node->type == XML_ELEMENT_NODE && xmlStrcmp(node->name, KEYBOARD_TAG) == 0) {
            parseKeyboard(node, keys);
        } else if (node->type == XML_ELEMENT_NODE && xmlStrcmp(node->name, SPURIOUS_TAG) == 0) {
            unsigned kbZone = parseUInt(node, KEYBOARD_ATTR_ZONE, 0);
            unsigned codeVal = parseUInt(node, KEY_ATTR_CODE, 0);
            if (codeVal > 0) {
                spurious.push_back({kbZone, codeVal});
            }
        }
    }

    // Finalize
    return LayoutDescription{
        std::string(reinterpret_cast<std::string::const_pointer>(name.get())),
        std::move(keys),
        std::move(spurious)
    };
}

LayoutDescription LayoutDescription::loadFile(const std::string & path)
{
    const auto & xdgPaths = tools::paths::getPaths(tools::paths::XDG::Data, true);

    std::vector<std::string> candidates(xdgPaths.size());
    std::transform(xdgPaths.begin(), xdgPaths.end(), candidates.begin(),
                   [](const auto & prefix) { return prefix + "/" KEYLEDSD_DATA_PREFIX "/layouts"; });

    for (const auto & candidate : candidates) {
        auto fullName = std::string();
        fullName.reserve(candidate.size() + 1 + path.size());
        fullName += candidate;
        if (candidate.back() != '/' && path.front() != '/') { fullName += '/'; }
        fullName += path;
        std::ifstream file(fullName);
        if (!file) { continue; }
        try {
            auto result = parse(file);
            INFO("loaded layout ", fullName);
            return result;
        } catch (ParseError & error) {
            ERROR("layout ", fullName, " line ", error.line(), ": ", error.what());
        } catch (std::exception & error) {
            ERROR("layout ", fullName, ": ", error.what());
        }
    }
    return {};
}

/****************************************************************************/

LayoutDescription::ParseError::ParseError(const std::string & what, long line)
 : std::runtime_error(what), m_line(line)
{}

LayoutDescription::ParseError::~ParseError() = default;
