﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

#if DEBUG
#endif
using System.Drawing;

internal static partial class Interop
{
    internal static partial class Gdi32
    {
        /// <summary>
        ///  Helper to scope selecting a given foreground text color into a HDC. Restores the original text color into
        ///  into the HDC when disposed.
        /// </summary>
        /// <remarks>
        ///  Use in a <see langword="using" /> statement. If you must pass this around, always pass by
        ///  <see langword="ref" /> to avoid duplicating the handle and resetting multiple times.
        /// </remarks>
#if DEBUG
        internal class SetTextColorScope : DisposalTracking.Tracker, IDisposable
#else
        internal readonly ref struct SetTextColorScope
#endif
        {
            private readonly int _previousColor;
            private readonly HDC _hdc;

            /// <summary>
            ///  Sets text color <paramref name="color"/> in the given <paramref name="hdc"/> using
            ///  <see cref="SetTextColor(HDC, int)"/>.
            /// </summary>
            public SetTextColorScope(HDC hdc, Color color)
            {
                int colorref = ColorTranslator.ToWin32(color);
                _previousColor = SetTextColor(hdc, colorref);

                // If we didn't actually change the color, don't keep the HDC so we skip putting back the same state.
                _hdc = colorref == _previousColor ? default : hdc;
            }

            public void Dispose()
            {
                if (!_hdc.IsNull)
                {
                    SetTextColor(_hdc, _previousColor);
                }

#if DEBUG
                GC.SuppressFinalize(this);
#endif
            }
        }
    }
}
