// https://github.com/tidwall/pogocache
//
// Copyright 2025 Polypoint Labs, LLC. All rights reserved.
// This file is part of the Pogocache project.
// Use of this source code is governed by the AGPL that can be found in
// the LICENSE file.
//
// For alternative licensing options or general questions, please contact
// us at licensing@polypointlabs.com.
//
// Unit memcache.c provides the parser for the Memcache wire protocol.
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include "util.h"
#include "stats.h"
#include "parse.h"

static __thread size_t mc_n = 0;

size_t parse_lastmc_n(void) {
    return mc_n;
}

bool mc_valid_key(struct args *args, int i) {
    const uint8_t *key = (uint8_t*)args->bufs[i].data;
    size_t len = args->bufs[i].len;
    if (len == 0 || len > 250) {
        return false;
    }
    for (size_t i = 0; i < len; i++) {
        if (key[i] <= ' ' || key[i] == 0x7F) {
            return false;
        }
    }
    return true;
}

enum mc_cmd { MC_UNKNOWN, 
    // writers (optional reply)
    MC_SET, MC_ADD, MC_REPLACE, MC_APPEND, MC_PREPEND, MC_CAS, // storage
    MC_INCR, MC_DECR, // increment/decrement
    MC_FLUSH_ALL, MC_DELETE, // deletion
    MC_TOUCH, // touch
    MC_VERBOSITY, // logging
    // readers (always replys)
    MC_GET, MC_GETS, // retreival
    MC_GAT, MC_GATS, // get and touch
    MC_VERSION, MC_STATS, // information
    MC_QUIT, // client
};

static bool is_mc_store_cmd(enum mc_cmd cmd) {
    return cmd >= MC_SET && cmd <= MC_CAS;
}

static bool is_mc_noreplyable(enum mc_cmd cmd) {
    return cmd >= MC_SET && cmd <= MC_VERBOSITY;
}

static ssize_t parse_memcache_telnet(const char *data, size_t len, 
    struct args *args)
{
    const char *p = data;
    const char *end = data+len;
    const char *s = p;
    char last = 0;
    while (p < end) {
        char ch = *(p++);
        if (ch == ' ') {
            size_t wn = p-s-1;
            // if (wn > 0) {
            args_append(args, s, wn, true);
            s = p;
            continue;
        }
        if (ch == '\n') {
            size_t wn = p-s-1;
            if (last == '\r') {
                wn--;
            }
            if (wn > 0) {
                args_append(args, s, wn, true);
            }
            return p-data;
        }
        last = ch;
    }
    return 0;
}

ssize_t parse_memcache(const char *data, size_t len, struct args *args, 
    bool *noreply)
{
    ssize_t n = parse_memcache_telnet(data, len, args);
    if (n <= 0 || args->len == 0) {
        return n;
    }
    // args_print(args);
    mc_n = n;
    enum mc_cmd cmd;
    struct args args2 = { 0 };
    *noreply = false;
    // check for common get-2
    if (args->len == 2 && arg_const_eq(args, 0, "get")) {
        if (!mc_valid_key(args, 1)) {
            if (args->bufs[1].len == 0) {
                return -1;
            }
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        args->bufs[0].data = "mget";
        args->bufs[0].len = 4;
        return n;
    }
    // Check for common set-5 (allows for expiry)
    if (args->len == 5 && arg_const_eq(args, 0, "set")) {
        if (args->bufs[2].len == 1 && args->bufs[2].data[0] == '0') {
            if (!mc_valid_key(args, 1)) {
                if (args->bufs[1].len == 0) {
                    return -1;
                }
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
            bool expset = false;
            int64_t x;
            if (!(args->bufs[3].len == 1 && args->bufs[3].data[0] == '0')) {
                if (!argi64(args, 3, &x)) {
                    parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                    return -1;
                }
                expset = true;
            }
            if (!argi64(args, 4, &x) || x < 0 || x > MAXARGSZ) {
                stat_store_too_large_incr(0);
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
            if (len-n < (size_t)x+2) {
                return 0;
            }
            const char *value = data+n;
            size_t value_len = x;
            n += x+2;
            mc_n = n;
            if (data[n-2] != '\r' || data[n-1] != '\n') {
                parse_seterror(CLIENT_ERROR_BAD_CHUNK);
                return -1;
            }
            // replace the "flags" with a value
            args->bufs[2].len = value_len;
            args->bufs[2].data = (void*)value;
            args->len = 3;
            if (expset) {
                // add the "ex <expiry>" to last two arguments
                args->bufs[4] = args->bufs[3];
                args->bufs[3].data = "ex";
                args->bufs[3].len = 2;
                args->len = 5;
            }
            return n;
        } else {
            // flags was set, use plus branch
            cmd = MC_SET;
            goto set_plus;
        }
    }
    // Otherwise use lookup command table. This could be optimized into a
    // switch table or hash table. See cmds.c for hash table example.
    cmd =
        arg_const_eq(args, 0, "set") ? MC_SET :               // XY
        arg_const_eq(args, 0, "add") ? MC_ADD :               // XY
        arg_const_eq(args, 0, "cas") ? MC_CAS :               // XY
        arg_const_eq(args, 0, "replace") ? MC_REPLACE :       // XY
        arg_const_eq(args, 0, "get") ? MC_GET :               // XY
        arg_const_eq(args, 0, "delete") ? MC_DELETE :         // XY
        arg_const_eq(args, 0, "append") ? MC_APPEND :         // XY
        arg_const_eq(args, 0, "prepend") ? MC_PREPEND :       // XY
        arg_const_eq(args, 0, "gets") ? MC_GETS :             // XY
        arg_const_eq(args, 0, "incr") ? MC_INCR :             // XY
        arg_const_eq(args, 0, "decr") ? MC_DECR:              // XY
        arg_const_eq(args, 0, "touch") ? MC_TOUCH :           // X
        arg_const_eq(args, 0, "gat") ? MC_GAT :               // X
        arg_const_eq(args, 0, "gats") ? MC_GATS :             // X
        arg_const_eq(args, 0, "flush_all") ? MC_FLUSH_ALL :   // X
        arg_const_eq(args, 0, "stats") ? MC_STATS :           // X
        arg_const_eq(args, 0, "version") ? MC_VERSION :       // X
        arg_const_eq(args, 0, "quit") ? MC_QUIT :             // XY
        arg_const_eq(args, 0, "verbosity") ? MC_VERBOSITY :   // X
        MC_UNKNOWN;
    if (cmd == MC_UNKNOWN) {
        parse_seterror("ERROR");
        return -1;
    }
    if (is_mc_noreplyable(cmd)) {
        if (arg_const_eq(args, args->len-1, "noreply")) {
            *noreply = true;
            buf_clear(&args->bufs[args->len-1]);
            args->len--;
        }
    }
    if (is_mc_store_cmd(cmd)) {
        // Store commands include 'set', 'add', 'replace', 'append', 'prepend',
        // and 'cas'.
        if ((cmd == MC_CAS && args->len != 6) && 
            (cmd != MC_CAS && args->len != 5))
        {
            parse_seterror("ERROR");
            return -1;
        }
    set_plus:
        // check all values before continuing
        if (!mc_valid_key(args, 1)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        int64_t x;
        if (!argi64(args, 2, &x) || x < 0) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        if (!argi64(args, 3, &x)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        if (!argi64(args, 4, &x) || x < 0 || x > MAXARGSZ) {
            stat_store_too_large_incr(0);
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        bool hascas = false;
        char cas[24] = "0";
        if (cmd == MC_CAS) {
            hascas = true;
            uint64_t y;
            if (!argu64(args, 5, &y)) {
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
            assert(args->bufs[5].len < sizeof(cas));
            memcpy(cas, args->bufs[5].data, args->bufs[5].len);
            cas[args->bufs[5].len] = '\0';
            buf_clear(&args->bufs[5]);
            args->len--;
        }

        // Storage commands must read a value that follows the first line.
        if (len-n < (size_t)x+2) {
            return 0;
        }
        const char *value = data+n;
        size_t value_len = x;
        n += x+2;
        mc_n = n;
        if (data[n-2] != '\r' || data[n-1] != '\n') {
            parse_seterror(CLIENT_ERROR_BAD_CHUNK);
            return -1;
        }

        // Reconstruct the command into a RESP format. 
        bool is_append_prepend = false;
        switch (cmd) {
        case MC_APPEND:
            args_append(&args2, "append", 6, true);
            is_append_prepend = true;
            break;
        case MC_PREPEND:
            args_append(&args2, "prepend", 7, true);
            is_append_prepend = true;
            break;
        default:
            args_append(&args2, "set", 3, true);
            break;
        }
        // Move key arg to new args
        take_and_append_arg(1);
        // Add value arg
        args_append(&args2, value, value_len, true);
        if (!is_append_prepend) {
            if (!(args->bufs[2].len == 1 && args->bufs[2].data[0] == '0')) {
                args_append(&args2, "flags", 5, true);
                take_and_append_arg(2);
            }
            
            if (!(args->bufs[3].len == 1 && args->bufs[3].data[0] == '0')) {
                args_append(&args2, "ex", 2, true);
                take_and_append_arg(3);
            }
            if (cmd == MC_ADD) {
                args_append(&args2, "nx", 2, true);
            } else if (cmd == MC_REPLACE) {
                args_append(&args2, "xx", 2, true);
            }
            if (hascas) {
                args_append(&args2, "cas", 3, true);
                args_append(&args2, cas, strlen(cas), false);
            }
        }
    } else if (cmd == MC_GET) {
        // Convert 'get <key>* into 'MGET <key>*'
        if (args->len == 1) {
            parse_seterror("ERROR");
            return -1;
        }
        // check all keys
        for (size_t i = 1; i < args->len; i++) {
            if (!mc_valid_key(args, i)) {
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
        }
        args_append(&args2, "mget", 4, true);
        for (size_t i = 1; i < args->len; i++) {
            take_and_append_arg(i);
        }
    } else if (cmd == MC_DELETE) {
        // Convert 'delete <key>' into 'DEL <key>'
        if (args->len == 1) {
            parse_seterror("ERROR");
            return -1;
        }
        if (args->len > 2) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        // check key
        if (!mc_valid_key(args, 1)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        args_append(&args2, "del", 3, true);
        take_and_append_arg(1);
    } else if (cmd == MC_GETS) {
        // Convert 'gets <key>* into 'MGETS <key>*'
        if (args->len == 1) {
            parse_seterror("ERROR");
            return -1;
        }
        // check all keys
        for (size_t i = 1; i < args->len; i++) {
            if (!mc_valid_key(args, i)) {
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
        }
        args_append(&args2, "mgets", 5, true);
        for (size_t i = 1; i < args->len; i++) {
            take_and_append_arg(i);
        }
    } else if (cmd == MC_GAT) {
        // Convert 'gat <exptime> <key>* into 'gat <exptime> <key>*'
        if (args->len <= 2) {
            parse_seterror("ERROR");
            return -1;
        }
        // check exptime
        int64_t x;
        if (!argi64(args, 2, &x)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        // check all keys
        for (size_t i = 2; i < args->len; i++) {
            if (!mc_valid_key(args, i)) {
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
        }
        args_append(&args2, "gat", 3, true);
        for (size_t i = 1; i < args->len; i++) {
            take_and_append_arg(i);
        }
    } else if (cmd == MC_GATS) {
        // Convert 'gats <exptime> <key>* into 'gats <exptime> <key>*'
        if (args->len <= 2) {
            parse_seterror("ERROR");
            return -1;
        }
        // check exptime
        int64_t x;
        if (!argi64(args, 2, &x)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        // check all keys
        for (size_t i = 2; i < args->len; i++) {
            if (!mc_valid_key(args, i)) {
                parse_seterror(CLIENT_ERROR_BAD_FORMAT);
                return -1;
            }
        }
        args_append(&args2, "gats", 4, true);
        for (size_t i = 1; i < args->len; i++) {
            take_and_append_arg(i);
        }
    } else if (cmd == MC_STATS) {
        args_append(&args2, "stats", 5, true);
        for (size_t i = 1; i < args->len; i++) {
            take_and_append_arg(i);
        }
    } else if (cmd == MC_INCR) {
        // Convert 'incr <key> <delta> into 'uincrby <key> <delta>'
        if (args->len != 3) {
            parse_seterror("ERROR");
            return -1;
        }
        // check key
        if (!mc_valid_key(args, 1)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        args_append(&args2, "uincrby", 7, true);
        take_and_append_arg(1);
        take_and_append_arg(2);
    } else if (cmd == MC_DECR) {
        // Convert 'decr <key> <delta> into 'udecrby <key> <delta>'
        if (args->len != 3) {
            parse_seterror("ERROR");
            return -1;
        }
        // check key
        if (!mc_valid_key(args, 1)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        args_append(&args2, "udecrby", 7, true);
        take_and_append_arg(1);
        take_and_append_arg(2);
    } else if (cmd == MC_TOUCH) {
        // Convert 'touch <key> <exptime>' into 'expire <key> <exptime>'
        if (args->len != 3) {
            parse_seterror("ERROR");
            return -1;
        }
        if (!mc_valid_key(args, 1)) {
            parse_seterror(CLIENT_ERROR_BAD_FORMAT);
            return -1;
        }
        args_append(&args2, "expire", 6, true);
        take_and_append_arg(1);
        take_and_append_arg(2);
    } else if (cmd == MC_FLUSH_ALL) {
        // Convert 'flush_all [delay]' into 'FLUSHALL [DELAY seconds]'
        if (args->len > 2) {
            parse_seterror("ERROR");
            return -1;
        }
        args_append(&args2, "flushall", 8, true);
        if (args->len == 2) {
            args_append(&args2, "delay", 5, true);
            take_and_append_arg(1);
        }
    } else if (cmd == MC_QUIT) {
        args_append(&args2, "quit", 4, true);
        *noreply = true;
    } else if (cmd == MC_VERSION) {
        args_append(&args2, "version", 7, true);
        *noreply = false;
    } else if (cmd == MC_VERBOSITY) {
        if (args->len > 2) {
            parse_seterror("ERROR");
            return -1;
        }
        args_append(&args2, "verbosity", 7, true);
        take_and_append_arg(1);
    } else {
        return -1;
    }
    args_free(args);
    *args = args2;
    return n;
}
