# coding=utf-8
import unittest
import numpy

from safe.storage.vector import Vector
from safe.storage.raster import Raster
from safe.storage.geometry import Polygon
from safe.gis.polygon import (
    separate_points_by_polygon,
    is_inside_polygon,
    is_outside_polygon,
    point_on_line,
    outside_polygon,
    inside_polygon,
    clip_lines_by_polygon,
    clip_lines_by_polygons,
    in_and_outside_polygon,
    intersection,
    join_line_segments,
    clip_line_by_polygon,
    clip_grid_by_polygons,
    populate_polygon,
    generate_random_points_in_bbox,
    PolygonInputError,
    line_dictionary_to_geometry)
from safe.gis.numerics import ensure_numeric

# For polygon testing
TEST_LINES = [numpy.array([[122.231021, -8.626557],
                           [122.230563, -8.626194],
                           [122.228790, -8.624855],
                           [122.227536, -8.624059],
                           [122.226648, -8.623494],
                           [122.225775, -8.623022],
                           [122.224872, -8.622444],
                           [122.224230, -8.622100],
                           [122.221931, -8.621082],
                           [122.221700, -8.620980],
                           [122.220577, -8.620555],
                           [122.219580, -8.621030]]),
              numpy.array([[122.231621, -8.626837],
                           [122.231221, -8.628815],
                           [122.230381, -8.633296],
                           [122.229211, -8.639560],
                           [122.228404, -8.643688],
                           [122.228340, -8.643792],
                           [122.228330, -8.644103],
                           [122.226939, -8.651254],
                           [122.226158, -8.655796],
                           [122.226085, -8.656345],
                           [122.226098, -8.658629],
                           [122.226220, -8.659293],
                           [122.226251, -8.659496],
                           [122.226331, -8.659609],
                           [122.226324, -8.659723],
                           [122.226213, -8.659857],
                           [122.226136, -8.660063],
                           [122.226192, -8.660974],
                           [122.226279, -8.661290],
                           [122.226331, -8.661902]]),
              numpy.array([[122.226889, -8.625599],
                           [122.227299, -8.624500],
                           [122.227409, -8.624221],
                           [122.227536, -8.624059]]),
              numpy.array([[122.237129, -8.628637],
                           [122.233170, -8.627332],
                           [122.231621, -8.626837],
                           [122.231021, -8.626557]]),
              numpy.array([[122.237129, -8.628637],
                           [122.237302, -8.628694],
                           [122.238308, -8.629055],
                           [122.239825, -8.629696],
                           [122.241483, -8.630410],
                           [122.244142, -8.631634],
                           [122.244682, -8.631817],
                           [122.245104, -8.631978],
                           [122.245178, -8.631995],
                           [122.245259, -8.632025],
                           [122.245333, -8.632046],
                           [122.245474, -8.632101],
                           [122.246997, -8.632655],
                           [122.247938, -8.632926],
                           [122.248294, -8.633040],
                           [122.251185, -8.633644],
                           [122.255205, -8.634337],
                           [122.257082, -8.634719],
                           [122.258420, -8.634980],
                           [122.262063, -8.636138],
                           [122.263548, -8.636398]]),
              numpy.array([[122.247938, -8.632926],
                           [122.247940, -8.633560],
                           [122.247390, -8.636220]])]


TEST_POLYGON = numpy.array([[122.229086, -8.624406],
                            [122.229165, -8.624428],
                            [122.229339, -8.624562],
                            [122.229358, -8.624588],
                            [122.229329, -8.624656],
                            [122.229447, -8.624836],
                            [122.229646, -8.624878],
                            [122.229722, -8.624769],
                            [122.229872, -8.624858],
                            [122.230035, -8.625277],
                            [122.230333, -8.625382],
                            [122.231016, -8.625562],
                            [122.231332, -8.625655],
                            [122.231515, -8.625699],
                            [122.231695, -8.625837],
                            [122.231878, -8.625971],
                            [122.232058, -8.626109],
                            [122.232221, -8.626160],
                            [122.232834, -8.626485],
                            [122.233179, -8.626606],
                            [122.233422, -8.626652],
                            [122.233715, -8.626755],
                            [122.234381, -8.626836],
                            [122.234646, -8.627018],
                            [122.235011, -8.627153],
                            [122.235323, -8.627281],
                            [122.235964, -8.627383],
                            [122.237428, -8.627456],
                            [122.237692, -8.627611],
                            [122.237859, -8.627791],
                            [122.238087, -8.627801],
                            [122.238252, -8.627715],
                            [122.238644, -8.627886],
                            [122.238918, -8.627930],
                            [122.239374, -8.628171],
                            [122.239722, -8.628293],
                            [122.240053, -8.628339],
                            [122.240434, -8.628505],
                            [122.240858, -8.628607],
                            [122.241096, -8.628704],
                            [122.241549, -8.628792],
                            [122.241604, -8.628851],
                            [122.242095, -8.629023],
                            [122.242490, -8.629086],
                            [122.242734, -8.629202],
                            [122.242857, -8.629334],
                            [122.243557, -8.629650],
                            [122.244487, -8.630012],
                            [122.244684, -8.630157],
                            [122.244764, -8.630243],
                            [122.245091, -8.630386],
                            [122.245411, -8.630520],
                            [122.245551, -8.630617],
                            [122.245957, -8.630746],
                            [122.246591, -8.630968],
                            [122.246978, -8.631042],
                            [122.247375, -8.631182],
                            [122.247724, -8.631273],
                            [122.248289, -8.631364],
                            [122.248534, -8.631442],
                            [122.248998, -8.631703],
                            [122.249020, -8.631727],
                            [122.249043, -8.631784],
                            [122.249065, -8.631727],
                            [122.249315, -8.631930],
                            [122.249687, -8.632251],
                            [122.249938, -8.632380],
                            [122.250183, -8.632477],
                            [122.250599, -8.632758],
                            [122.251161, -8.633002],
                            [122.251689, -8.633155],
                            [122.253039, -8.633205],
                            [122.253888, -8.633189],
                            [122.254651, -8.633301],
                            [122.254932, -8.633296],
                            [122.255220, -8.633344],
                            [122.255778, -8.633469],
                            [122.256583, -8.633527],
                            [122.256945, -8.633574],
                            [122.257386, -8.633628],
                            [122.257629, -8.633712],
                            [122.258273, -8.633700],
                            [122.258945, -8.633803],
                            [122.259535, -8.633895],
                            [122.260058, -8.633964],
                            [122.260098, -8.634105],
                            [122.260751, -8.634187],
                            [122.260807, -8.634119],
                            [122.260941, -8.634198],
                            [122.260967, -8.634056],
                            [122.261675, -8.634076],
                            [122.261784, -8.634103],
                            [122.261752, -8.634246],
                            [122.262216, -8.634304],
                            [122.262752, -8.634239],
                            [122.262953, -8.634316],
                            [122.263125, -8.634295],
                            [122.263331, -8.634321],
                            [122.263486, -8.634262],
                            [122.263854, -8.634222],
                            [122.263975, -8.634152],
                            [122.264441, -8.634066],
                            [122.264822, -8.634019],
                            [122.264985, -8.634115],
                            [122.265569, -8.634016],
                            [122.265655, -8.633909],
                            [122.265900, -8.633819],
                            [122.266047, -8.633733],
                            [122.266205, -8.633714],
                            [122.266412, -8.633773],
                            [122.266486, -8.633867],
                            [122.266602, -8.633841],
                            [122.266624, -8.634173],
                            [122.266746, -8.634375],
                            [122.267067, -8.634570],
                            [122.267100, -8.634611],
                            [122.266940, -8.634589],
                            [122.266506, -8.634472],
                            [122.266434, -8.634552],
                            [122.266141, -8.634737],
                            [122.265576, -8.634900],
                            [122.265293, -8.635074],
                            [122.264771, -8.635224],
                            [122.264533, -8.635353],
                            [122.264439, -8.635396],
                            [122.264305, -8.635488],
                            [122.264075, -8.635531],
                            [122.263850, -8.635623],
                            [122.263695, -8.635671],
                            [122.263532, -8.635758],
                            [122.263428, -8.635805],
                            [122.263349, -8.635800],
                            [122.263225, -8.635807],
                            [122.263081, -8.636119],
                            [122.262723, -8.636165],
                            [122.262667, -8.636344],
                            [122.262500, -8.636420],
                            [122.262606, -8.636129],
                            [122.262302, -8.636072],
                            [122.262123, -8.636024],
                            [122.261711, -8.635936],
                            [122.261578, -8.635887],
                            [122.261394, -8.635844],
                            [122.261142, -8.635708],
                            [122.261079, -8.635570],
                            [122.261027, -8.635474],
                            [122.260890, -8.635482],
                            [122.260704, -8.635189],
                            [122.260769, -8.635009],
                            [122.260636, -8.634939],
                            [122.260575, -8.635134],
                            [122.260584, -8.635292],
                            [122.260432, -8.635524],
                            [122.260446, -8.635300],
                            [122.260349, -8.635214],
                            [122.260352, -8.635163],
                            [122.260342, -8.634988],
                            [122.260398, -8.634938],
                            [122.260484, -8.634843],
                            [122.260360, -8.634712],
                            [122.260125, -8.634893],
                            [122.259944, -8.634844],
                            [122.259594, -8.634801],
                            [122.259395, -8.634939],
                            [122.259223, -8.634928],
                            [122.259126, -8.634981],
                            [122.258842, -8.635067],
                            [122.257969, -8.634928],
                            [122.257808, -8.634840],
                            [122.257503, -8.634769],
                            [122.257399, -8.634653],
                            [122.257144, -8.634668],
                            [122.256916, -8.634586],
                            [122.257000, -8.634819],
                            [122.256986, -8.635038],
                            [122.256734, -8.634947],
                            [122.256032, -8.635039],
                            [122.255897, -8.635111],
                            [122.255625, -8.634879],
                            [122.255549, -8.635121],
                            [122.255489, -8.635566],
                            [122.255353, -8.635695],
                            [122.255294, -8.634918],
                            [122.255026, -8.634592],
                            [122.254568, -8.634429],
                            [122.254445, -8.634566],
                            [122.254260, -8.634312],
                            [122.253832, -8.634387],
                            [122.253717, -8.634380],
                            [122.253618, -8.634433],
                            [122.253522, -8.634641],
                            [122.253321, -8.634325],
                            [122.253059, -8.634340],
                            [122.252711, -8.634319],
                            [122.252361, -8.634561],
                            [122.252303, -8.634558],
                            [122.252220, -8.634377],
                            [122.252168, -8.634281],
                            [122.251893, -8.634196],
                            [122.251768, -8.634057],
                            [122.251675, -8.634107],
                            [122.251622, -8.634104],
                            [122.251522, -8.633887],
                            [122.250855, -8.633925],
                            [122.250689, -8.633865],
                            [122.250538, -8.633970],
                            [122.250438, -8.633916],
                            [122.250358, -8.633740],
                            [122.250274, -8.633697],
                            [122.250176, -8.634014],
                            [122.250018, -8.634367],
                            [122.249832, -8.634532],
                            [122.249790, -8.634090],
                            [122.249897, -8.633898],
                            [122.249582, -8.633923],
                            [122.249446, -8.633826],
                            [122.249403, -8.634058],
                            [122.249312, -8.634100],
                            [122.249086, -8.633692],
                            [122.248960, -8.633650],
                            [122.248858, -8.633876],
                            [122.248809, -8.633963],
                            [122.248788, -8.634032],
                            [122.248763, -8.634012],
                            [122.248722, -8.633724],
                            [122.248583, -8.633741],
                            [122.248405, -8.633600],
                            [122.248082, -8.633559],
                            [122.247800, -8.633478],
                            [122.247336, -8.633419],
                            [122.247225, -8.633235],
                            [122.247038, -8.633195],
                            [122.246992, -8.633086],
                            [122.246902, -8.633014],
                            [122.246862, -8.632873],
                            [122.246776, -8.632804],
                            [122.246701, -8.632980],
                            [122.246503, -8.633289],
                            [122.246444, -8.633491],
                            [122.246463, -8.633643],
                            [122.246444, -8.633792],
                            [122.246227, -8.633882],
                            [122.246170, -8.634078],
                            [122.246181, -8.634167],
                            [122.245977, -8.634444],
                            [122.245717, -8.634589],
                            [122.245479, -8.634788],
                            [122.245379, -8.634749],
                            [122.245398, -8.634678],
                            [122.245468, -8.634568],
                            [122.245633, -8.634505],
                            [122.245754, -8.634337],
                            [122.245808, -8.634136],
                            [122.245861, -8.634098],
                            [122.245899, -8.633774],
                            [122.245953, -8.633646],
                            [122.246014, -8.633346],
                            [122.246329, -8.633236],
                            [122.246401, -8.633052],
                            [122.246508, -8.632913],
                            [122.246491, -8.632828],
                            [122.246620, -8.632626],
                            [122.246773, -8.632517],
                            [122.246811, -8.632434],
                            [122.246675, -8.632336],
                            [122.246637, -8.632270],
                            [122.246837, -8.632194],
                            [122.246950, -8.632037],
                            [122.246994, -8.631904],
                            [122.246876, -8.631820],
                            [122.246695, -8.631934],
                            [122.246330, -8.631846],
                            [122.246098, -8.631891],
                            [122.245996, -8.631826],
                            [122.245956, -8.631925],
                            [122.246183, -8.632056],
                            [122.246202, -8.632128],
                            [122.246096, -8.632167],
                            [122.245757, -8.632132],
                            [122.245793, -8.632489],
                            [122.245494, -8.632599],
                            [122.245368, -8.632671],
                            [122.245314, -8.632546],
                            [122.245131, -8.632467],
                            [122.245037, -8.632354],
                            [122.244688, -8.632389],
                            [122.244454, -8.632556],
                            [122.244147, -8.632597],
                            [122.244082, -8.632231],
                            [122.243829, -8.632148],
                            [122.243583, -8.632288],
                            [122.243409, -8.632188],
                            [122.243267, -8.632202],
                            [122.243281, -8.632068],
                            [122.243087, -8.632011],
                            [122.243041, -8.631904],
                            [122.242918, -8.631802],
                            [122.242563, -8.631723],
                            [122.242394, -8.631740],
                            [122.242272, -8.631885],
                            [122.242221, -8.631653],
                            [122.242094, -8.631466],
                            [122.241680, -8.631641],
                            [122.241366, -8.631585],
                            [122.241457, -8.631510],
                            [122.241498, -8.631230],
                            [122.241386, -8.631241],
                            [122.241273, -8.631377],
                            [122.241144, -8.631222],
                            [122.240957, -8.631328],
                            [122.240887, -8.631349],
                            [122.240864, -8.630971],
                            [122.240319, -8.631014],
                            [122.240194, -8.631004],
                            [122.240073, -8.630733],
                            [122.240000, -8.630602],
                            [122.239959, -8.630418],
                            [122.239851, -8.630370],
                            [122.239692, -8.630383],
                            [122.239550, -8.630281],
                            [122.239383, -8.630294],
                            [122.239251, -8.630221],
                            [122.239386, -8.629779],
                            [122.239163, -8.629839],
                            [122.239024, -8.629999],
                            [122.238960, -8.629774],
                            [122.239059, -8.629690],
                            [122.239187, -8.629700],
                            [122.239230, -8.629557],
                            [122.239318, -8.629401],
                            [122.239231, -8.629244],
                            [122.239210, -8.629175],
                            [122.239317, -8.629143],
                            [122.239413, -8.628840],
                            [122.239152, -8.628472],
                            [122.239023, -8.628361],
                            [122.238880, -8.628526],
                            [122.238826, -8.628701],
                            [122.238723, -8.628742],
                            [122.238560, -8.628553],
                            [122.238340, -8.628570],
                            [122.238182, -8.628512],
                            [122.238054, -8.628522],
                            [122.237967, -8.628421],
                            [122.237778, -8.628337],
                            [122.237559, -8.628166],
                            [122.237122, -8.628085],
                            [122.237106, -8.628030],
                            [122.237066, -8.628070],
                            [122.237122, -8.628085],
                            [122.237157, -8.628180],
                            [122.236987, -8.628375],
                            [122.236899, -8.628382],
                            [122.236759, -8.628159],
                            [122.236625, -8.628314],
                            [122.236551, -8.628377],
                            [122.236571, -8.628625],
                            [122.236507, -8.628698],
                            [122.236372, -8.628602],
                            [122.236239, -8.628612],
                            [122.236173, -8.628370],
                            [122.236279, -8.628084],
                            [122.236212, -8.627972],
                            [122.236075, -8.628144],
                            [122.236105, -8.628527],
                            [122.235835, -8.628676],
                            [122.235764, -8.628516],
                            [122.235689, -8.628510],
                            [122.235461, -8.628528],
                            [122.235349, -8.628398],
                            [122.235375, -8.628379],
                            [122.235470, -8.628221],
                            [122.235457, -8.628134],
                            [122.235309, -8.628155],
                            [122.235349, -8.628398],
                            [122.235299, -8.628357],
                            [122.235197, -8.628233],
                            [122.235288, -8.627912],
                            [122.235275, -8.627788],
                            [122.235293, -8.627619],
                            [122.235235, -8.627518],
                            [122.235192, -8.627420],
                            [122.234999, -8.627730],
                            [122.234871, -8.627785],
                            [122.234830, -8.627834],
                            [122.234721, -8.627924],
                            [122.234587, -8.628229],
                            [122.234602, -8.628375],
                            [122.234518, -8.628596],
                            [122.234458, -8.628645],
                            [122.234524, -8.628700],
                            [122.234460, -8.628869],
                            [122.234417, -8.629013],
                            [122.234340, -8.628920],
                            [122.234374, -8.628600],
                            [122.234361, -8.628474],
                            [122.234465, -8.628330],
                            [122.234426, -8.627936],
                            [122.234512, -8.627788],
                            [122.234553, -8.627580],
                            [122.234436, -8.627517],
                            [122.234375, -8.627788],
                            [122.234238, -8.627976],
                            [122.234113, -8.628080],
                            [122.233961, -8.628145],
                            [122.233875, -8.628194],
                            [122.233751, -8.628235],
                            [122.233694, -8.627782],
                            [122.233353, -8.627741],
                            [122.233239, -8.627876],
                            [122.233151, -8.627917],
                            [122.233103, -8.627646],
                            [122.233033, -8.627587],
                            [122.232463, -8.627649],
                            [122.232422, -8.627528],
                            [122.232606, -8.627238],
                            [122.232441, -8.627190],
                            [122.232261, -8.627208],
                            [122.232088, -8.627190],
                            [122.231978, -8.627058],
                            [122.231809, -8.627014],
                            [122.231642, -8.627109],
                            [122.231548, -8.626995],
                            [122.231568, -8.626798],
                            [122.231486, -8.626655],
                            [122.231374, -8.626606],
                            [122.231333, -8.626557],
                            [122.231229, -8.626515],
                            [122.231246, -8.626689],
                            [122.231191, -8.626784],
                            [122.231243, -8.626967],
                            [122.231001, -8.627010],
                            [122.230815, -8.627232],
                            [122.230564, -8.627339],
                            [122.230402, -8.627533],
                            [122.230327, -8.627595],
                            [122.230340, -8.627727],
                            [122.230306, -8.627755],
                            [122.230235, -8.627642],
                            [122.230344, -8.627391],
                            [122.230423, -8.627328],
                            [122.230464, -8.627188],
                            [122.230651, -8.627148],
                            [122.230781, -8.627072],
                            [122.230828, -8.626962],
                            [122.230980, -8.626707],
                            [122.231106, -8.626606],
                            [122.231147, -8.626443],
                            [122.231035, -8.626253],
                            [122.230662, -8.626172],
                            [122.230600, -8.625887],
                            [122.230492, -8.625856],
                            [122.230513, -8.625832],
                            [122.230602, -8.625754],
                            [122.230650, -8.625666],
                            [122.230299, -8.625608],
                            [122.230243, -8.625796],
                            [122.230384, -8.625921],
                            [122.230312, -8.625985],
                            [122.229977, -8.625965],
                            [122.229763, -8.626021],
                            [122.229254, -8.625990],
                            [122.229020, -8.626073],
                            [122.228952, -8.625924],
                            [122.228423, -8.625831],
                            [122.228063, -8.625511],
                            [122.227938, -8.625454],
                            [122.227588, -8.625474],
                            [122.227472, -8.625603],
                            [122.227371, -8.625690],
                            [122.227285, -8.625594],
                            [122.227303, -8.625312],
                            [122.227000, -8.625093],
                            [122.226743, -8.625013],
                            [122.226655, -8.624965],
                            [122.226516, -8.624922],
                            [122.226346, -8.624830],
                            [122.225835, -8.624785],
                            [122.225702, -8.624692],
                            [122.225346, -8.624625],
                            [122.225247, -8.624586],
                            [122.225475, -8.624423],
                            [122.225654, -8.624285],
                            [122.225762, -8.624236],
                            [122.225836, -8.624145],
                            [122.225655, -8.624061],
                            [122.225546, -8.624002],
                            [122.225635, -8.623839],
                            [122.226210, -8.623786],
                            [122.226316, -8.623793],
                            [122.226537, -8.623780],
                            [122.226610, -8.623699],
                            [122.226665, -8.623702],
                            [122.226772, -8.623821],
                            [122.226842, -8.623880],
                            [122.226837, -8.623790],
                            [122.226911, -8.623702],
                            [122.227027, -8.623955],
                            [122.227299, -8.624018],
                            [122.227425, -8.624158],
                            [122.227643, -8.624146],
                            [122.227745, -8.624201],
                            [122.227929, -8.624244],
                            [122.228110, -8.624293],
                            [122.228274, -8.624205],
                            [122.228337, -8.624341],
                            [122.228441, -8.624335],
                            [122.228606, -8.624153],
                            [122.228702, -8.624067],
                            [122.228723, -8.624044],
                            [122.228769, -8.624044],
                            [122.228769, -8.624134],
                            [122.228749, -8.624160],
                            [122.228697, -8.624199],
                            [122.228567, -8.624366],
                            [122.228681, -8.624554],
                            [122.228881, -8.624709],
                            [122.228947, -8.624550],
                            [122.229167, -8.624583],
                            [122.229086, -8.624406]])


class TestPolygon(unittest.TestCase):
    def setUp(self):
        pass

    def tearDown(self):
        pass

    # Polygon stuff
    def test_point_on_line(self):
        """Points coinciding with lines are correctly detected
        """

        # Endpoints first
        assert point_on_line([0, 0], [[0, 0], [1, 0]])
        assert point_on_line([1, 0], [[0, 0], [1, 0]])

        # Endpoints first - non-simple
        assert point_on_line([-0.1, 0.0], [[-0.1, 0.0], [1.5, 0.6]])
        assert point_on_line([1.5, 0.6], [[-0.1, 0.0], [1.5, 0.6]])

        # Then points on line
        assert point_on_line([0.5, 0], [[0, 0], [1, 0]])
        assert point_on_line([0, 0.5], [[0, 1], [0, 0]])
        assert point_on_line([1, 0.5], [[1, 1], [1, 0]])
        assert point_on_line([0.5, 0.5], [[0, 0], [1, 1]])

        # Then points not on line
        assert not point_on_line([0.5, 0], [[0, 1], [1, 1]])
        assert not point_on_line([0, 0.5], [[0, 0], [1, 1]])

        # From real example that failed
        assert not point_on_line([40, 50], [[40, 20], [40, 40]])

        # From real example that failed
        assert not point_on_line([40, 19], [[40, 20], [40, 40]])

        # Degenerate line
        assert point_on_line([40, 19], [[40, 19], [40, 19]])
        assert not point_on_line([40, 19], [[40, 40], [40, 40]])

    def test_point_on_line_vector(self):
        """Points coinciding with lines are correctly detected (vector version)

        Vectorised version
        """

        # Common variables
        N = 100
        x0 = 0
        x1 = 50
        x0in = x0 + 5
        x1in = x1 - 5
        space = numpy.linspace(x0, x1, num=N, endpoint=True)

        # First a couple where all points are included
        h_points = numpy.zeros((N, 2), numpy.float64)
        h_points[:, 0] = space
        horiz_line = [[x0, 0], [x1, 0]]
        res = point_on_line(h_points, horiz_line)
        assert numpy.alltrue(res)

        v_points = numpy.zeros((N, 2), numpy.float64)
        v_points[:, 1] = space
        vertical_line = [[0, x0], [0, x1]]
        res = point_on_line(v_points, vertical_line)
        assert numpy.alltrue(res)

        # Then some where points are outside
        horiz_line = [[x0in, 0], [x1in, 0]]
        res = point_on_line(h_points, horiz_line)
        assert numpy.sometrue(res)
        ref = (x0in < h_points[:, 0]) * (h_points[:, 0] < x1in)
        assert numpy.alltrue(res == ref)

        vertical_line = [[0, x0in], [0, x1in]]
        res = point_on_line(v_points, vertical_line)
        ref = (x0in < v_points[:, 1]) * (v_points[:, 1] < x1in)
        assert numpy.alltrue(res == ref)

        # Diagonal example - all in
        diagonal_line = [[x0, x0], [x1, x1]]
        d_points = numpy.zeros((N, 2), numpy.float64)
        d_points[:, 0] = space
        d_points[:, 1] = space
        res = point_on_line(d_points, diagonal_line)
        assert numpy.alltrue(res)

        # Example with all out - all False
        res = point_on_line(d_points, vertical_line)
        assert not numpy.sometrue(res)

        # Then a more realistic example, diagonal with points outside
        points = numpy.concatenate((d_points, [[31, 12], [0, 3]]))
        res = point_on_line(points, diagonal_line)
        assert numpy.alltrue(res[:N])
        assert not numpy.sometrue(res[N:])

    def test_is_inside_polygon_main1(self):
        """Points are classified as either inside polygon or not
        """

        # Simplest case: Polygon is the unit square
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]

        assert is_inside_polygon((0.5, 0.5), polygon)
        assert not is_inside_polygon((0.5, 1.5), polygon)
        assert not is_inside_polygon((0.5, -0.5), polygon)
        assert not is_inside_polygon((-0.5, 0.5), polygon)
        assert not is_inside_polygon((1.5, 0.5), polygon)

        # Try point on borders
        assert is_inside_polygon((1., 0.5), polygon, closed=True)
        assert is_inside_polygon((0.5, 1.), polygon, closed=True)
        assert is_inside_polygon((0., 0.5), polygon, closed=True)
        assert is_inside_polygon((0.5, 0.), polygon, closed=True)

        assert not is_inside_polygon((0.5, 1.), polygon, closed=False)
        assert not is_inside_polygon((0., 0.5), polygon, closed=False)
        assert not is_inside_polygon((0.5, 0.), polygon, closed=False)
        assert not is_inside_polygon((1., 0.5), polygon, closed=False)

    def test_inside_polygon_main2(self):
        """Points can be classified as either inside or outside polygon (2)
        """

        # Simplest case: Polygon is the unit square
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # From real example (that failed)
        polygon = [[20, 20], [40, 20], [40, 40], [20, 40]]
        points = [[40, 50]]
        res = inside_polygon(points, polygon)
        assert len(res) == 0

        polygon = [[20, 20], [40, 20], [40, 40], [20, 40]]
        points = [[25, 25], [30, 20], [40, 50], [90, 20], [40, 90]]
        res = inside_polygon(points, polygon)
        assert len(res) == 2
        assert numpy.allclose(res, [0, 1])

        # More convoluted and non convex polygon
        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        assert is_inside_polygon((0.5, 0.5), polygon)
        assert is_inside_polygon((1, -0.5), polygon)
        assert is_inside_polygon((1.5, 0), polygon)

        assert not is_inside_polygon((0.5, 1.5), polygon)
        assert not is_inside_polygon((0.5, -0.5), polygon)

        # Very convoluted polygon
        polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                   [30, 10], [40, -10]]
        assert is_inside_polygon((5, 5), polygon)
        assert is_inside_polygon((17, 7), polygon)
        assert is_inside_polygon((27, 2), polygon)
        assert is_inside_polygon((35, -5), polygon)
        assert not is_inside_polygon((15, 7), polygon)
        assert not is_inside_polygon((24, 3), polygon)
        assert not is_inside_polygon((25, -10), polygon)

        # Another combination (that failed)
        polygon = [[0, 0], [10, 0], [10, 10], [0, 10]]
        assert is_inside_polygon((5, 5), polygon)
        assert is_inside_polygon((7, 7), polygon)
        assert not is_inside_polygon((-17, 7), polygon)
        assert not is_inside_polygon((7, 17), polygon)
        assert not is_inside_polygon((17, 7), polygon)
        assert not is_inside_polygon((27, 8), polygon)
        assert not is_inside_polygon((35, -5), polygon)

        # Multiple polygons
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1], [0, 0], [10, 10],
                   [11, 10], [11, 11], [10, 11], [10, 10]]
        assert is_inside_polygon((0.5, 0.5), polygon)
        assert is_inside_polygon((10.5, 10.5), polygon)
        assert not is_inside_polygon((0, 5.5), polygon)
        assert is_inside_polygon((5.5, 5.5), polygon)

        # Polygon with a hole
        polygon = [[-1, -1], [2, -1], [2, 2], [-1, 2], [-1, -1],
                   [0, 0], [1, 0], [1, 1], [0, 1], [0, 0]]

        assert is_inside_polygon((0, -0.5), polygon)
        assert not is_inside_polygon((0.5, 0.5), polygon)

    def test_duplicate_points_being_ok(self):
        """Polygons can have duplicate points without problems
        """

        polygon = [[0, 0], [1, 0], [1, 0], [1, 0], [1, 1], [0, 1], [0, 0]]

        assert is_inside_polygon((0.5, 0.5), polygon)
        assert not is_inside_polygon((0.5, 1.5), polygon)
        assert not is_inside_polygon((0.5, -0.5), polygon)
        assert not is_inside_polygon((-0.5, 0.5), polygon)
        assert not is_inside_polygon((1.5, 0.5), polygon)

        # Try point on borders
        assert is_inside_polygon((1., 0.5), polygon, closed=True)
        assert is_inside_polygon((0.5, 1), polygon, closed=True)
        assert is_inside_polygon((0., 0.5), polygon, closed=True)
        assert is_inside_polygon((0.5, 0.), polygon, closed=True)

        assert not is_inside_polygon((0.5, 1), polygon, closed=False)
        assert not is_inside_polygon((0., 0.5), polygon, closed=False)
        assert not is_inside_polygon((0.5, 0.), polygon, closed=False)
        assert not is_inside_polygon((1., 0.5), polygon, closed=False)

        # From real example
        polygon = [[20, 20], [40, 20], [40, 40], [20, 40]]
        points = [[40, 50]]
        res = inside_polygon(points, polygon)
        assert len(res) == 0

    def test_inside_polygon_vector_version(self):
        """Indices of points inside polygon are correct
        """

        # Now try the vector formulation returning indices
        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 0.5], [1, -0.5], [1.5, 0], [0.5, 1.5], [0.5, -0.5]]
        res = inside_polygon(points, polygon)
        assert numpy.allclose(res, [0, 1, 2])

    def test_separate_points_by_polygon0(self):
        """Points can be separated by polygon
        """

        # Now try the vector formulation returning indices
        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 0.5], [1, -0.5], [1.5, 0], [0.5, 1.5], [0.5, -0.5]]

        inside, outside = separate_points_by_polygon(points, polygon)

        assert len(inside) + len(outside) == len(points)
        assert numpy.allclose(inside, [0, 1, 2])
        assert numpy.allclose(outside, [3, 4])

    def test_separate_points_by_polygon_empty_points(self):
        """Separate points by polygon ok when no points in bbox
        """

        # This is from a real example that failed
        polygon = numpy.array([[109.82203092, -7.22977256],
                               [109.82224507, -7.22986774],
                               [109.82255440, -7.22974876],
                               [109.82272096, -7.22960600],
                               [109.82283994, -7.22929667],
                               [109.82283994, -7.22884457],
                               [109.82272096, -7.22860662],
                               [109.82248302, -7.22841627],
                               [109.82241163, -7.22822591],
                               [109.82224507, -7.22822591],
                               [109.82210230, -7.22841627],
                               [109.82191195, -7.22860662],
                               [109.82179297, -7.22872560],
                               [109.82169779, -7.22882077],
                               [109.82172159, -7.22917769],
                               [109.82176918, -7.22932046],
                               [109.82188815, -7.22953461],
                               [109.82203092, -7.22977256]])
        points = numpy.zeros((0, 2))
        separate_points_by_polygon(points, polygon)

    def test_outside_polygon(self):
        """Points are classified as either outside polygon or not
        """

        # unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # evaluate to False as the point 0.5, 0.5 is inside the unit square
        assert not is_outside_polygon([0.5, 0.5], U)

        # evaluate to True as the point 1.5, 0.5 is outside the unit square
        assert is_outside_polygon([1.5, 0.5], U)

        indices = outside_polygon([[0.5, 0.5], [1, -0.5], [0.3, 0.2]], U)
        assert numpy.allclose(indices, [1])

        # One more test of vector formulation returning indices
        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 0.5], [1, -0.5], [1.5, 0], [0.5, 1.5], [0.5, -0.5]]
        res = outside_polygon(points, polygon)
        assert numpy.allclose(res, [3, 4])

        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 1.4], [0.5, 0.5], [1, -0.5], [1.5, 0],
                  [0.5, 1.5], [0.5, -0.5]]
        res = outside_polygon(points, polygon)

        assert numpy.allclose(res, [0, 4, 5])

    def test_outside_polygon2(self):
        """Points are classified as either outside polygon or not (2)
        """

        # unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # evaluate to False as the point 0.5, 1.0 is inside the unit square
        assert not outside_polygon([0.5, 1.0], U, closed=True)

        # evaluate to True as the point 0.5, 1.0 is outside the unit square
        assert is_outside_polygon([0.5, 1.0], U, closed=False)

    def test_all_outside_polygon(self):
        """Corner case where all points are outside polygon works"""

        # unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        points = [[2, 2], [1, 3], [-1, 1], [0, 2]]      # All outside

        inside, outside = separate_points_by_polygon(points, U)
        assert len(inside) == 0                           # None inside
        assert numpy.allclose(outside, [0, 1, 2, 3])

        indices = outside_polygon(points, U, closed=True)
        assert numpy.allclose(indices, [0, 1, 2, 3])

        indices = inside_polygon(points, U, closed=True)
        assert numpy.allclose(indices, [])

    def test_all_inside_polygon(self):
        """Corner case where all points are inside polygon works"""

        # unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        points = [[0.5, 0.5], [0.2, 0.3], [0, 0.5]]  # All inside (or on edge)

        inside, outside = separate_points_by_polygon(points, U)
        assert len(outside) == 0       # All inside
        assert numpy.allclose(inside, [0, 1, 2])

        indices = outside_polygon(points, U, closed=True)
        assert numpy.allclose(indices, [])

        indices = inside_polygon(points, U, closed=True)
        assert numpy.allclose(indices, [0, 1, 2])

    def test_separate_points_by_polygon_edge(self):
        """Points on polygon edge correctly detected
        """

        # Unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # Try with boundary (vertical edge) point
        inside, outside = separate_points_by_polygon([[0, 0.5],
                                                      [0.3, 0.3],
                                                      [0.1, 0.6]],
                                                     U,
                                                     closed=True)
        assert len(inside) == 3
        assert numpy.allclose(inside, [0, 1, 2])

        inside, outside = separate_points_by_polygon([[0, 0.5],
                                                      [0.3, 0.3],
                                                      [0.1, 0.6]],
                                                     U,
                                                     closed=False)
        assert len(inside) == 2
        assert numpy.allclose(inside, [1, 2])

        # Try with boundary (horizontal edge) point
        inside, outside = separate_points_by_polygon([[0.5, 0.0],
                                                      [0.3, 0.3],
                                                      [0.1, 0.6]], U)
        assert len(outside) == 0
        assert len(inside) == 3
        assert numpy.allclose(inside, [0, 1, 2])

        # Try with boundary (corner) point
        inside, outside = separate_points_by_polygon([[0.0, 0.0],
                                                      [0.3, 0.3],
                                                      [0.1, 0.6]], U)
        assert len(outside) == 0
        assert len(inside) == 3
        assert numpy.allclose(inside, [0, 1, 2])

        # One outside
        inside, outside = separate_points_by_polygon([[0, 0.5],
                                                      [1.3, 0.3],
                                                      [0.1, 0.6]], U)
        assert len(outside) == 1
        assert len(inside) == 2
        assert numpy.allclose(inside, [0, 2])

    def test_separate_points_by_polygon1(self):
        """Set of points is correctly separated according to polygon
        """

        # Unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        inside, outside = separate_points_by_polygon([[0.5, 0.5]], U)
        assert len(inside) == 1
        assert numpy.allclose(inside, [0])

        inside, outside = separate_points_by_polygon(
            [[0.5, 0.5], [0.3, 0.2]], U)
        assert len(inside) == 2
        assert numpy.allclose(inside, [0, 1])

        inside, outside = separate_points_by_polygon(
            [[0.5, 0.5], [0.3, 0.2], [0.6, 0.7]], U)
        assert len(inside) == 3
        assert numpy.allclose(inside, [0, 1, 2])

        inside, outside = separate_points_by_polygon([[0.3, 0.2]], U)
        assert len(inside) == 1
        assert numpy.allclose(inside, [0])

        inside, outside = separate_points_by_polygon(
            [[0.3, 0.2], [1, -0.5]], U)
        assert len(inside) == 1
        assert numpy.allclose(inside, [0])
        assert numpy.allclose(outside, [1])

        inside, outside = separate_points_by_polygon(
            [[0.5, 0.5], [1, -0.5], [0.3, 0.2]], U)
        assert numpy.allclose(inside, [0, 2])
        assert numpy.allclose(outside, [1])

        inside, outside = separate_points_by_polygon(
            [[0.1, 0.1], [0.5, 0.5], [1, -0.5], [0.3, 0.2]], U)
        assert numpy.allclose(inside, [0, 1, 3])
        assert numpy.allclose(outside, [2])

        # Try with boundary (edge) point
        inside, outside = separate_points_by_polygon(
            [[0, 0.5], [0.1, 0.2]], U)
        assert numpy.allclose(inside, [0, 1])

        # Try with boundary (corner) point
        inside, outside = separate_points_by_polygon(
            [[0, 0], [0.1, 0.2]], U)
        assert numpy.allclose(inside, [0, 1])

        # Try with a range of cases point
        inside, outside = separate_points_by_polygon(
            [[0, 0], [0, 0.5], [0.1, 0.2]], U)
        assert numpy.allclose(inside, [0, 1, 2])

        # One more test of vector formulation returning indices
        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 0.5], [1, -0.5], [1.5, 0], [0.5, 1.5], [0.5, -0.5]]
        inside, outside = separate_points_by_polygon(points, polygon)
        assert numpy.allclose(inside, [0, 1, 2])
        assert numpy.allclose(outside, [3, 4])

        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 1.4], [0.5, 0.5], [1, -0.5], [1.5, 0],
                  [0.5, 1.5], [0.5, -0.5]]
        inside, outside = separate_points_by_polygon(points, polygon)
        assert numpy.allclose(inside, [1, 2, 3])
        assert numpy.allclose(outside, [0, 4, 5])

    def test_separate_points_by_polygon_characterisation(self):
        """Numpy version of polygon clipping agrees with python version
        """

        # Unit square
        U = [[0, 0], [1, 0], [1, 1], [0, 1]]

        inside_r, outside_r = separate_points_by_polygon([[0.5, 0.5],
                                                          [0.3, 0.2]],
                                                         U,
                                                         use_numpy=True)
        inside_p, outside_p = separate_points_by_polygon([[0.5, 0.5],
                                                          [0.3, 0.2]],
                                                         U,
                                                         use_numpy=False)

        assert numpy.allclose(inside_r, inside_p)
        assert numpy.allclose(outside_r, outside_p)

        inside_r, outside_r = separate_points_by_polygon([[0.5, 0.5],
                                                          [0.3, 0.2],
                                                          [0.6, 0.7]],
                                                         U,
                                                         use_numpy=True)
        inside_p, outside_p = separate_points_by_polygon([[0.5, 0.5],
                                                          [0.3, 0.2],
                                                          [0.6, 0.7]],
                                                         U,
                                                         use_numpy=False)

        assert numpy.allclose(inside_r, inside_p)
        assert numpy.allclose(outside_r, outside_p)

        polygon = [[0, 0], [1, 0], [0.5, -1], [2, -1], [2, 1], [0, 1]]
        points = [[0.5, 1.4], [0.5, 0.5], [1, -0.5], [1.5, 0],
                  [0.5, 1.5], [0.5, -0.5]]
        ins_r, out_r = separate_points_by_polygon(points, polygon,
                                                  use_numpy=True)
        ins_p, out_p = separate_points_by_polygon(points, polygon,
                                                  use_numpy=False)
        assert numpy.allclose(ins_r, ins_p)
        assert numpy.allclose(out_r, out_p)

        assert numpy.allclose(ins_p, [1, 2, 3])
        assert numpy.allclose(out_p, [0, 4, 5])

    def test_polygon_clipping_error_handling(self):
        """Polygon clipping checks input as expected"""

        U = [[0, 0], [1, 0], [1, 1], [0, 1]]
        points = [[2, 2], [1, 3], [-1, 1], [0, 2]]

        # Correct call
        separate_points_by_polygon(points, U)

        # Input errors
        try:
            outside_polygon(points, ['what'])
        except PolygonInputError:
            pass
        else:
            msg = 'Should have raised PolygonInputError:'
            raise Exception(msg)

        try:
            outside_polygon('Hmmm', U)
        except PolygonInputError:
            pass
        else:
            msg = 'Should have raised PolygonInputError'
            raise Exception(msg)

        try:
            inside_polygon(points, U, closed=7)
        except PolygonInputError:
            pass
        else:
            msg = 'Should have raised PolygonInputError'
            raise Exception(msg)

    def test_clip_grid_by_polygon(self):
        """Regular grids can be clipped by polygons (with holes)
        """
        # Define an outer ring
        outer_ring = numpy.array([[106.79, -6.233],
                                  [106.80, -6.24],
                                  [106.78, -6.23],
                                  [106.77, -6.21],
                                  [106.79, -6.233]])

        # Define inner rings
        inner_rings = [numpy.array([[106.77827, -6.2252],
                                    [106.77775, -6.22378],
                                    [106.78, -6.22311],
                                    [106.78017, -6.22530],
                                    [106.77827, -6.2252]]),
                       numpy.array([[106.78652, -6.23215],
                                    [106.78642, -6.23075],
                                    [106.78746, -6.23143],
                                    [106.78831, -6.23307],
                                    [106.78652, -6.23215]])]

        # Make a grid
        N = 10
        A = numpy.arange(N * N).reshape((N, N))
        # Longitudes
        minx = min(outer_ring[:, 0])
        maxx = max(outer_ring[:, 0])
        dx = (maxx - minx) / N

        # Latitudes
        miny = min(outer_ring[:, 1])
        maxy = max(outer_ring[:, 1])
        dy = -(maxy - miny) / N

        # top left x, w-e pixel res, rot, top left y, rot, n-s pixel res
        # (lon_ul, dlon, 0, lat_ul, 0, dlat)
        geotransform = (minx, dx, 0, maxy, 0, dy)

        # Create suitable instance on the fly
        # See
        # http://docs.python.org/library/functions.html#type
        # http://jjinux.blogspot.com/2005/03/
        #      python-create-new-class-on-fly.html
        polygon_arg = [type('', (),
                            dict(outer_ring=outer_ring,
                                 inner_rings=inner_rings))()]

        # Call clipping function
        res, _ = clip_grid_by_polygons(A, geotransform, polygon_arg)

        points = res[0][0]
        values = res[0][1]
        values = [{'val': float(x)} for x in values]

        # Check correctness (from QGIS inspection)
        assert numpy.allclose(points[0], [106.7775, -6.2205])
        assert numpy.allclose(points[3], [106.7865, -6.2295])
        assert numpy.allclose(points[9], [106.7925, -6.2355])
        assert values[0]['val'] == 32
        assert values[3]['val'] == 65
        assert values[9]['val'] == 87

        # Optionally store output for inspection with QGIS (this one is nice)
        if False:
            R = Raster(A, geotransform=geotransform)
            R.write_to_file('test_raster.tif')
            P = Vector(geometry=[Polygon(outer_ring=outer_ring,
                                         inner_rings=inner_rings)])
            P.write_to_file('test_polygon.shp')
            Vector(geometry=points,
                   data=values).write_to_file('test_points.shp')

    def test_populate_polygon(self):
        """Polygon can be populated by random points
        """

        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]
        points = populate_polygon(polygon, 5)

        assert len(points) == 5
        for point in points:
            assert is_inside_polygon(point, polygon)

        # Very convoluted polygon
        polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                   [30, 10], [40, -10]]
        points = populate_polygon(polygon, 5)
        assert len(points) == 5
        for point in points:
            assert is_inside_polygon(point, polygon)

    def test_populate_polygon_with_exclude(self):
        """Polygon with hole can be populated by random points
        """

        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]
        ex_poly = [[0, 0], [0.5, 0], [0.5, 0.5], [0, 0.5]]     # SW quarter
        points = populate_polygon(polygon, 5, exclude=[ex_poly])

        assert len(points) == 5
        for point in points:
            assert is_inside_polygon(point, polygon)
            assert not is_inside_polygon(point, ex_poly)

        # Overlap
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]
        ex_poly = [[-1, -1], [0.5, 0], [0.5, 0.5], [-1, 0.5]]
        points = populate_polygon(polygon, 5, exclude=[ex_poly])

        assert len(points) == 5
        for point in points:
            assert is_inside_polygon(point, polygon)
            assert not is_inside_polygon(point, ex_poly)

        # Multiple
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]
        ex_poly1 = [[0, 0], [0.5, 0], [0.5, 0.5], [0, 0.5]]  # SW quarter
        ex_poly2 = [[0.5, 0.5], [0.5, 1], [1, 1], [1, 0.5]]  # NE quarter

        points = populate_polygon(polygon, 20, exclude=[ex_poly1, ex_poly2])

        assert len(points) == 20
        for point in points:
            assert is_inside_polygon(point, polygon)
            assert not is_inside_polygon(point, ex_poly1)
            assert not is_inside_polygon(point, ex_poly2)

        # Very convoluted polygon
        polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                   [30, 10], [40, -10]]
        ex_poly = [[-1, -1], [5, 0], [5, 5], [-1, 5]]
        points = populate_polygon(polygon, 20, exclude=[ex_poly])

        assert len(points) == 20
        for point in points:
            assert is_inside_polygon(point, polygon)
            assert not is_inside_polygon(point, ex_poly), '%s' % str(point)

    test_populate_polygon_with_exclude.slow = True

    def test_populate_polygon_with_exclude2(self):
        """Polygon with hole can be populated by random points (2)
        """

        M = 200  # Number of points
        min_outer = 0
        max_outer = 1000
        polygon_outer = [[min_outer, min_outer], [max_outer, min_outer],
                         [max_outer, max_outer], [min_outer, max_outer]]

        delta = 10
        min_inner1 = min_outer + delta
        max_inner1 = max_outer - delta
        inner1_polygon = [[min_inner1, min_inner1], [max_inner1, min_inner1],
                          [max_inner1, max_inner1], [min_inner1, max_inner1]]

        min_inner2 = min_outer + 2 * delta
        max_inner2 = max_outer - 2 * delta
        inner2_polygon = [[min_inner2, min_inner2], [max_inner2, min_inner2],
                          [max_inner2, max_inner2], [min_inner2, max_inner2]]

        points = populate_polygon(polygon_outer, M,
                                  exclude=[inner1_polygon, inner2_polygon])

        assert len(points) == M
        for point in points:
            assert is_inside_polygon(point, polygon_outer)
            assert not is_inside_polygon(point, inner1_polygon)
            assert not is_inside_polygon(point, inner2_polygon)

        # Very convoluted polygon
        polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                   [30, 10], [40, -10]]
        ex_poly = [[-1, -1], [5, 0], [5, 5], [-1, 5]]
        points = populate_polygon(polygon, M, exclude=[ex_poly])

        assert len(points) == M
        for point in points:
            assert is_inside_polygon(point, polygon)
            assert not is_inside_polygon(point, ex_poly), '%s' % str(point)

    test_populate_polygon_with_exclude2.slow = True

    def test_large_example(self):
        """Large polygon clipping example works
        """

        M = 500  # Number of points inside
        N = 300  # Number of points outside

        # Main box
        min_main = 0
        max_main = 100
        main_polygon = [[min_main, min_main], [max_main, min_main],
                        [max_main, max_main], [min_main, max_main]]

        # Outer box
        delta = 10
        min_outer = min_main - delta
        max_outer = max_main + delta
        outer_polygon = [[min_outer, min_outer], [max_outer, min_outer],
                         [max_outer, max_outer], [min_outer, max_outer]]

        # Create points inside and outside main polygon
        points_inside = populate_polygon(main_polygon, M)
        points_outside = populate_polygon(outer_polygon, N,
                                          exclude=[main_polygon])

        assert len(points_inside) == M
        for point in points_inside:
            assert is_inside_polygon(point, main_polygon)

        assert len(points_outside) == N
        for point in points_outside:
            assert not is_inside_polygon(point, main_polygon)

        # Test separation algorithm
        all_points = numpy.concatenate((points_inside, points_outside))
        assert all_points.shape[0] == M + N

        inside, outside = separate_points_by_polygon(all_points, main_polygon)
        count = len(inside)
        msg = 'Expected %i points inside, got %i' % (M, count)
        assert count == M, msg

        msg = 'Expected %i indices outside, got %i' % (N, len(outside))
        assert len(outside) == N, msg

        for point in all_points[inside]:
            assert is_inside_polygon(point, main_polygon)

        for point in all_points[outside]:
            assert not is_inside_polygon(point, main_polygon)

    test_large_example.slow = True

    def test_large_convoluted_example(self):
        """Large convoluted polygon clipping example works
        """

        M = 500  # Number of points inside
        N = 300  # Number of points outside

        # Main box
        main_polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                        [30, 10], [40, -10]]
        main_polygon = ensure_numeric(main_polygon)

        # Outer box
        delta = 5
        x_min = min(main_polygon[:, 0]) - delta
        x_max = max(main_polygon[:, 0]) + delta
        y_min = min(main_polygon[:, 1]) - delta
        y_max = max(main_polygon[:, 1]) + delta

        outer_polygon = [[x_min, y_min], [x_max, y_min],
                         [x_max, y_max], [x_min, y_max]]

        # Create points inside and outside main polygon
        points_inside = populate_polygon(main_polygon, M)
        points_outside = populate_polygon(outer_polygon, N,
                                          exclude=[main_polygon])

        assert len(points_inside) == M
        for point in points_inside:
            assert is_inside_polygon(point, main_polygon)

        assert len(points_outside) == N
        for point in points_outside:
            assert not is_inside_polygon(point, main_polygon)

        # Test separation algorithm
        all_points = numpy.concatenate((points_inside, points_outside))
        assert all_points.shape[0] == M + N

        inside, outside = separate_points_by_polygon(all_points, main_polygon)
        count = len(inside)
        msg = 'Expected %i points inside, got %i' % (M, count)
        assert count == M, msg

        msg = 'Expected %i indices outside, got %i' % (N, len(outside))
        assert len(outside) == N, msg

        for point in all_points[inside]:
            assert is_inside_polygon(point, main_polygon)

        for point in all_points[outside]:
            assert not is_inside_polygon(point, main_polygon)

    test_large_convoluted_example.slow = True

    def test_large_convoluted_example_random(self):
        """Large convoluted polygon clipping example works (random points)
        """

        M = 1000  # Number of points

        # Main box
        main_polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                        [30, 10], [40, -10]]
        main_polygon = ensure_numeric(main_polygon)

        # Outer box
        delta = 5
        x_min = min(main_polygon[:, 0]) - delta
        x_max = max(main_polygon[:, 0]) + delta
        y_min = min(main_polygon[:, 1]) - delta
        y_max = max(main_polygon[:, 1]) + delta

        outer_polygon = [[x_min, y_min], [x_max, y_min],
                         [x_max, y_max], [x_min, y_max]]

        # Create points inside and outside main polygon
        all_points = populate_polygon(outer_polygon, M, seed=17)
        assert len(all_points) == M
        all_points = ensure_numeric(all_points)

        # Test separation algorithm
        inside, outside = separate_points_by_polygon(all_points, main_polygon)
        count = len(inside)
        msg = 'Expected %i points inside, got %i' % (271, count)
        assert count == 271, msg

        msg = 'Expected %i indices outside, got %i' % (729, len(outside))
        assert len(outside) == 729, msg

        for point in all_points[inside]:
            assert is_inside_polygon(point, main_polygon)

        for point in all_points[outside]:
            assert not is_inside_polygon(point, main_polygon)

    test_large_convoluted_example_random.slow = True

    def test_in_and_outside_polygon_main(self):
        """Set of points is correctly separated according to polygon (2)
        """

        # Simplest case: Polygon is the unit square
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]

        inside, outside = in_and_outside_polygon((0.5, 0.5), polygon)
        assert inside[0] == 0
        assert len(outside) == 0

        inside, outside = in_and_outside_polygon((1., 0.5), polygon,
                                                 closed=True)
        assert inside[0] == 0
        assert len(outside) == 0

        inside, outside = in_and_outside_polygon((1., 0.5), polygon,
                                                 closed=False)
        assert len(inside) == 0
        assert outside[0] == 0

        points = [(1., 0.25), (1., 0.75)]
        inside, outside = in_and_outside_polygon(points, polygon, closed=True)
        assert numpy.alltrue(inside == [0, 1])
        assert len(outside) == 0

        inside, outside = in_and_outside_polygon(points, polygon, closed=False)
        assert len(inside) == 0
        assert numpy.alltrue(outside == [0, 1])

        points = [(100., 0.25), (0.5, 0.5)]
        inside, outside = in_and_outside_polygon(points, polygon)
        assert numpy.alltrue(inside == [1])
        assert outside[0] == 0

        points = [(100., 0.25), (0.5, 0.5), (39, 20),
                  (0.6, 0.7), (56, 43), (67, 90)]
        inside, outside = in_and_outside_polygon(points, polygon)
        assert numpy.alltrue(inside == [1, 3])
        assert numpy.alltrue(outside == [0, 2, 4, 5])

    def test_clip_points_by_polygons_with_holes(self):
        """Points can be separated by polygons with holes
        """

        # Define an outer ring
        outer_ring = numpy.array([[106.79, -6.233],
                                  [106.80, -6.24],
                                  [106.78, -6.23],
                                  [106.77, -6.21],
                                  [106.79, -6.233]])

        # Define inner rings
        inner_rings = [numpy.array([[106.77827, -6.2252],
                                    [106.77775, -6.22378],
                                    [106.78, -6.22311],
                                    [106.78017, -6.22530],
                                    [106.77827, -6.2252]]),
                       numpy.array([[106.78652, -6.23215],
                                    [106.78642, -6.23075],
                                    [106.78746, -6.23143],
                                    [106.78831, -6.23307],
                                    [106.78652, -6.23215]])]

        # Make some test points
        points = generate_random_points_in_bbox(outer_ring, 1000, seed=13)

        # Clip to outer ring, excluding holes
        inside, outside = in_and_outside_polygon(points, outer_ring,
                                                 holes=inner_rings)

        # Check wrapper functions
        assert numpy.all(inside == inside_polygon(points, outer_ring,
                                                  holes=inner_rings))
        assert numpy.all(outside == outside_polygon(points, outer_ring,
                                                    holes=inner_rings))

        # Verify that each point is where it should
        for point in points[inside, :]:
            # Must be inside outer ring
            assert is_inside_polygon(point, outer_ring)

            # But not in any of the inner rings
            assert not is_inside_polygon(point, inner_rings[0])
            assert not is_inside_polygon(point, inner_rings[1])

        for point in points[outside, :]:
            # Must be either outside outer ring or inside a hole
            assert (is_outside_polygon(point, outer_ring) or
                    is_inside_polygon(point, inner_rings[0]) or
                    is_inside_polygon(point, inner_rings[1]))

    test_clip_points_by_polygons_with_holes.slow = True

    def test_intersection1(self):
        """Intersection of two simple lines works
        """

        line0 = [[-1, 0], [1, 0]]
        line1 = [[0, -1], [0, 1]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [0.0, 0.0])

    def test_intersection2(self):
        """Intersection point is independent of direction
        """

        line0 = [[0, 0], [24, 12]]
        line1 = [[0, 12], [24, 0]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [12.0, 6.0])

        # Swap direction of one line
        line1 = [[24, 0], [0, 12]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [12.0, 6.0])

        # Swap order of lines
        value = intersection(line1, line0)
        assert numpy.allclose(value, [12.0, 6.0])

    def test_intersection3(self):
        """Intersection point is independent of direction (2)
        """

        line0 = [[0, 0], [24, 12]]
        line1 = [[0, 17], [24, 0]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [14.068965517, 7.0344827586])

        # Swap direction of one line
        line1 = [[24, 0], [0, 17]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [14.068965517, 7.0344827586])

        # Swap order of lines
        value = intersection(line1, line0)
        assert numpy.allclose(value, [14.068965517, 7.0344827586])

    def test_intersection_endpoints(self):
        """Intersection of lines with coinciding endpoints works

        Test that coinciding endpoints are picked up
        """

        line0 = [[0, 0], [1, 1]]
        line1 = [[1, 1], [2, 1]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [1.0, 1.0])

        line0 = [[1, 1], [2, 0]]
        line1 = [[1, 1], [2, 1]]

        value = intersection(line0, line1)
        assert numpy.allclose(value, [1.0, 1.0])

    def test_intersection_direction_invariance(self):
        """Intersection is direction invariant

        This runs through a number of examples and checks that
        direction of lines don't matter.
        """

        line0 = [[0, 0], [100, 100]]

        common_end_point = [20, 150]

        for i in range(100):
            x = 20 + i * 1.0 / 100

            line1 = [[x, 0], common_end_point]
            p1 = intersection(line0, line1)

            # Swap direction of line1
            line1 = [common_end_point, [x, 0]]
            p2 = intersection(line0, line1)

            msg = ('Orientation of line should not matter.\n'
                   'However, segment [%f,%f], [%f, %f]' %
                   (x, 0, common_end_point[0], common_end_point[1]))
            msg += ' gave %s, \nbut when reversed we got %s' % (p1, p2)
            assert numpy.allclose(p1, p2), msg

            # Swap order of lines
            p3 = intersection(line1, line0)
            msg = 'Order of lines gave different results'
            assert numpy.allclose(p1, p3), msg

    test_intersection_direction_invariance.slow = True

    def test_no_intersection(self):
        """Lines that don't intersect return None as expected
        """

        line0 = [[-1, 1], [1, 1]]
        line1 = [[0, -1], [0, 0]]

        value = intersection(line0, line1)
        assert value is None

    def test_vectorised_intersection1(self):
        """Vectorised intersection of multiple lines works 1
        """

        line0 = [[0, 0], [24, 12]]

        # One way of building the array
        line1 = numpy.zeros(16).reshape(2, 2, 4)  # Three lines
        line1[0, 0, :] = [0, 24, 0, 15]   # x0
        line1[0, 1, :] = [12, 0, 24, 0]   # y0
        line1[1, 0, :] = [24, 0, 0, 5]    # x1
        line1[1, 1, :] = [0, 12, 12, 15]  # y1

        value = intersection(line0, line1)
        mask = - numpy.isnan(value[:, 0])
        v = value[mask]
        assert numpy.allclose(v,
                              [[12.0, 6.0],
                               [12.0, 6.0],
                               [11.25, 5.625]])

        # A more direct way of building the array
        line1 = [[[0, 24, 0, 15],    # x0
                  [12, 0, 24, 0]],   # y0
                 [[24, 0, 0, 5],     # x1
                  [0, 12, 12, 15]]]  # y1

        value = intersection(line0, line1)
        mask = - numpy.isnan(value[:, 0])
        v = value[mask]
        assert numpy.allclose(v,
                              [[12.0, 6.0],
                               [12.0, 6.0],
                               [11.25, 5.625]])

    def test_vectorised_intersection2(self):
        """Vectorised intersection of multiple lines works 2
        """

        # Common line segment to intersect with
        line0 = [[0, 0], [100, 100]]

        # Vectorised collection of line arguments
        N = 15  # Line 0 to 10 will intersect, 11 - 14 won't
        line1 = numpy.zeros(4 * N, numpy.float).reshape(2, 2, N)
        x0 = numpy.arange(N) * 10
        y0 = numpy.zeros(N)
        x1 = numpy.arange(N) * 10
        y1 = numpy.ones(N) * 100
        line1[0, 0, :] = x0
        line1[0, 1, :] = y0
        line1[1, 0, :] = x1
        line1[1, 1, :] = y1

        value = intersection(line0, line1)
        assert len(value.shape) == 2
        assert value.shape[0] == N
        assert value.shape[1] == 2

        for i in range(0, 11):
            assert value[i, 0] == i * 10

        for i in range(11, 15):
            assert numpy.all(numpy.isnan(value[i]))

    def test_intersection_parallel(self):
        """Parallel lines are correctly detected in intersection code
        """

        line0 = [[-1, 1], [1, 1]]
        line1 = [[-1, 0], [5, 0]]

        value = intersection(line0, line1)
        assert value is None

        line0 = [[0, 0], [10, 100]]
        line1 = [[-10, 5], [0, 105]]

        value = intersection(line0, line1)
        assert value is None

    def test_clip_line_by_polygon_simple(self):
        """Simple lines are clipped and classified by polygon
        """

        # Simplest case: Polygon is the unit square
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # Simple horizontal fully intersecting line
        line = [[-1, 0.5], [2, 0.5]]

        inside_line_segments, outside_line_segments = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_line_segments,
                              [[[0, 0.5], [1, 0.5]]])

        assert numpy.allclose(outside_line_segments,
                              [[[-1, 0.5], [0, 0.5]],
                               [[1, 0.5], [2, 0.5]]])

        # Simple horizontal line coinciding with polygon edge
        line = [[-1, 1], [2, 1]]

        inside_line_segments, outside_line_segments = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_line_segments,
                              [[[0, 1], [1, 1]]])

        assert numpy.allclose(outside_line_segments,
                              [[[-1, 1], [0, 1]],
                               [[1, 1], [2, 1]]])

        # Simple vertical fully intersecting line
        line = [[0.5, -1], [0.5, 2]]

        inside_line_segments, outside_line_segments = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_line_segments,
                              [[[0.5, 0], [0.5, 1]]])

        assert numpy.allclose(outside_line_segments,
                              [[[0.5, -1], [0.5, 0]],
                               [[0.5, 1], [0.5, 2]]])

        # Simple vertical line coinciding with polygon edge
        line = [[1, -1], [1, 2]]

        inside_line_segments, outside_line_segments = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_line_segments,
                              [[[1, 0], [1, 1]]])

        assert numpy.allclose(outside_line_segments,
                              [[[1, -1], [1, 0]],
                               [[1, 1], [1, 2]]])

        # Simple sloping fully intersecting line
        line = [[-1, 0.0], [2, 1.0]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines,
                              [[[0, 1.0 / 3], [1, 2.0 / 3]]])

        assert numpy.allclose(outside_lines,
                              [[[-1, 0], [0, 1.0 / 3]],
                               [[1, 2.0 / 3], [2, 1]]])

        # Simple sloping line coinciding with one edge, intersecting another
        line = [[-1, 0.0], [1, 2.0 / 3]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines,
                              [[[0, 1.0 / 3], [1, 2.0 / 3]]])

        assert numpy.allclose(outside_lines,
                              [[[-1, 0], [0, 1.0 / 3]]])

        # Diagonal line intersecting corners
        line = [[-1, -1], [2, 2]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines,
                              [[[0, 0], [1, 1]]])

        assert numpy.allclose(outside_lines,
                              [[[-1, -1], [0, 0]],
                               [[1, 1], [2, 2]]])

        # Diagonal line intersecting corners - other way
        line = [[-1, 2], [2, -1]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines,
                              [[[0, 1], [1, 0]]])

        assert numpy.allclose(outside_lines,
                              [[[-1, 2], [0, 1]],
                               [[1, 0], [2, -1]]])

        # Diagonal line coinciding with one corner
        line = [[-1, -1], [1, 1]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines,
                              [[[0, 0], [1, 1]]])

        assert numpy.allclose(outside_lines,
                              [[[-1, -1], [0, 0]]])

        # Very convoluted polygon
        polygon = [[0, 0], [10, 10], [15, 5], [20, 10], [25, 0],
                   [30, 10], [40, -10]]

        line = [[-10, 6], [60, 6]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines, [
            [[6, 6], [14, 6]], [[16, 6.], [22, 6]], [[28, 6], [32, 6]]])
        assert numpy.allclose(outside_lines, [
            [[-10, 6], [6, 6]], [[14, 6], [16, 6]], [[22, 6], [28, 6]],
            [[32, 6], [60, 6]]])

    def test_clip_line_by_polygon_already_inside(self):
        """Polygon line clipping works for special cases
        """

        line = [[1.5, 0.5], [2.5, 0.5]]
        polygon = [[1, 0], [3, 0], [2, 1]]

        # Assert that this line is fully inside polygon
        inside, outside = clip_line_by_polygon(line, polygon)
        assert len(outside) == 0
        assert len(inside) > 0

    def test_clip_composite_lines_by_polygon(self):
        """Composite lines are clipped and classified by polygon
        """

        # Simplest case: Polygon is the unit square
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # One line with same two segments changing direction inside polygon
        line = [[-1, 0.5], [0.5, 0.5], [0.5, 2]]

        inside_lines, outside_lines = \
            clip_line_by_polygon(line, polygon)

        assert numpy.allclose(inside_lines,
                              [[[0, 0.5], [0.5, 0.5], [0.5, 1]]])

        assert numpy.allclose(outside_lines, [
            [[-1, 0.5], [0, 0.5]], [[0.5, 1], [0.5, 2]]])

        # One line with multiple segments both inside and outside polygon
        line = [
            [-1, 0.5], [-0.5, 0.5], [0.5, 0.5], [1.0, 0.5], [1.5, 0.5],
            [2.0, 0.5]]

        inside_lines, outside_lines = clip_line_by_polygon(line, polygon)
        assert len(inside_lines) == 1
        assert numpy.allclose(inside_lines,
                              [[0, 0.5], [0.5, 0.5], [1.0, 0.5]])

    def test_clip_lines_by_polygon_multi(self):
        """Multiple composite lines are clipped and classified by polygon
        """

        # Simplest case: Polygon is the unit square
        polygon = [[0, 0], [1, 0], [1, 1], [0, 1]]

        # Two lines changing direction inside polygon
        lines = [[[-1, 0.5], [0.5, 0.5]],
                 [[0.5, 0.5], [0.5, 2]]]

        inside_lines, outside_lines = clip_lines_by_polygon(lines, polygon)

        assert numpy.allclose(inside_lines[0], [[0, 0.5], [0.5, 0.5]])
        assert numpy.allclose(inside_lines[1], [[0.5, 0.5], [0.5, 1]])

        assert numpy.allclose(outside_lines[0], [[-1, 0.5], [0, 0.5]])
        assert numpy.allclose(outside_lines[1], [[0.5, 1], [0.5, 2]])

        # Multiple lines with different number of segments
        lines = [[[-1, 0.5], [0.5, 0.5], [0.5, 2]],
                 [[-1, 0.0], [1, 2.0 / 3]]]

        inside_lines, outside_lines = \
            clip_lines_by_polygon(lines, polygon, check_input=True)

        assert len(inside_lines) == 2
        assert len(outside_lines) == 2

        for _, values in inside_lines.items():
            for line in values:
                assert isinstance(line, numpy.ndarray)
                assert len(line.shape) == 2
                assert line.shape[1] == 2

        for _, values in outside_lines.items():
            for line in values:
                assert isinstance(line, numpy.ndarray)
                assert len(line.shape) == 2
                assert line.shape[1] == 2

        assert numpy.allclose(inside_lines[0],
                              [[[0, 0.5], [0.5, 0.5], [0.5, 1]]])
        assert numpy.allclose(inside_lines[1],
                              [[[0, 1.0 / 3], [1, 2.0 / 3]]])

        assert numpy.allclose(outside_lines[0],
                              [[[-1, 0.5], [0, 0.5]],
                               [[0.5, 1], [0.5, 2]]])  # Two lines
        assert numpy.allclose(outside_lines[1], [[-1, 0], [0, 1.0 / 3]])

        # Test that lines dictionaries convert to geometries
        # (lists of Nx2 arrays)
        inside_geo = line_dictionary_to_geometry(inside_lines)
        outside_geo = line_dictionary_to_geometry(outside_lines)

        for line in inside_geo + outside_geo:
            assert isinstance(line, numpy.ndarray)
            assert len(line.shape) == 2
            assert line.shape[1] == 2

        assert numpy.allclose(inside_geo[0], [[0, 0.5], [0.5, 0.5], [0.5, 1]])

        assert numpy.allclose(inside_geo[1], [[0, 1.0 / 3], [1, 2.0 / 3]])

        assert numpy.allclose(outside_geo, [
            [[-1, 0.5], [0, 0.5]], [[0.5, 1], [0.5, 2]],
            [[-1, 0], [0, 1.0 / 3]]])

    def test_clip_lines_by_multiple_polygons(self):
        """Multiple composite lines are clipped by multiple polygons
        """

        # Test polys
        polygons = [[[0, 0], [1, 0], [1, 1], [0, 1]],  # Unit square
                    [[1, 0], [3, 0], [2, 1]],  # Adjacent triangle
                    [[0, 3], [1, 3], [0.5, 2],
                     [2, 2], [2, 4], [0, 4]],  # Convoluted
                    [[-1, -1], [5, -1], [5, 3], [5, 3]],  # Overlapping
                    [[-1, -1], [6, -1], [6, 6], [6, 6]]]  # Cover the others

        # Test lines
        input_lines = [[[0, 0.5], [4, 0.5]],
                       [[2, 0], [2, 5]],
                       [[0, 0], [5, 5]],
                       [[10, 10], [30, 10]],
                       [[-1, 0.5], [0.5, 0.5], [2.5, 3]],
                       [[0.5, 0.5], [0.5, 2]],
                       [[100, 100], [300, 100]],
                       [[0.3, 0.2], [0.7, 3], [1.0, 1.9]],
                       [[30, 10], [30, 20]]]

        # Vector(geometry=polygons).write_to_file('multiple_polygons.shp')
        # Vector(geometry=input_lines,
        #       geometry_type='line').write_to_file('input_lines.shp')
        lines_covered = clip_lines_by_polygons(input_lines, polygons)

        # Sanity checks
        assert len(lines_covered) == len(polygons)

        i = 0
        for lines in lines_covered:
            # filename = 'clipped_lines_%i.shp' % i
            # Vector(geometry=line_dictionary_to_geometry(lines),
            #       geometry_type='line').write_to_file(filename)
            i += 1

            assert len(lines) == len(input_lines)

        # Thorough check of all lines
        for i, polygon in enumerate(polygons):

            lines_in_polygon = lines_covered[i]
            for key in lines_in_polygon:
                for line in lines_covered[i][key]:

                    # Assert that this line is fully inside polygon
                    inside, outside = clip_line_by_polygon(line, polygon)
                    assert len(outside) == 0

                    # Line can be joined from separate segments but
                    # endpoints must match
                    for x in inside:
                        assert numpy.allclose(x[0], line[0])
                        assert numpy.allclose(x[-1], line[-1])

        # Spot checks

        # Polygon 2, line 1
        assert numpy.allclose(lines_covered[2][1][0],
                              [[2., 2.],
                               [2., 4.]])

        # Polygon 4, line 2
        # This one will fail if we ignore points_on_line check
        assert numpy.allclose(lines_covered[4][2][0],
                              [[0., 0.],
                               [5., 5.]])

        # Polygon 4, line 7
        assert numpy.allclose(lines_covered[4][7][0],
                              [[0.3, 0.2],
                               [0.31666667, 0.31666667]])

    def test_clip_lines_by_polygon_real_data(self):
        """Real roads are clipped by complex polygon
        """

        inside_lines, outside_lines = \
            clip_lines_by_polygon(TEST_LINES, TEST_POLYGON,
                                  check_input=True)

        # Convert dictionaries to lists of lines
        inside_line_segments = line_dictionary_to_geometry(inside_lines)
        outside_line_segments = line_dictionary_to_geometry(outside_lines)

        # These lines have compontes both inside and outside
        assert len(inside_line_segments) == 13
        assert len(outside_line_segments) == 17

        # Store for visual inspection by e.g. QGis
        # Set to True, run test and do
        # qgis test_polygon.shp test_lines.shp in_segments.shp out_segments.shp
        if False:
            Vector(geometry=[TEST_POLYGON],
                   geometry_type='polygon').write_to_file('test_polygon.shp')
            Vector(geometry=TEST_LINES,
                   geometry_type='line').write_to_file('test_lines.shp')
            Vector(geometry=inside_line_segments,
                   geometry_type='line').write_to_file('in_segments.shp')
            Vector(geometry=outside_line_segments,
                   geometry_type='line').write_to_file('out_segments.shp')

        # Check that midpoints of each segment are correctly placed
        for seg in inside_line_segments:
            N = len(seg)
            for i in range(N - 1):
                midpoint = (seg[i] + seg[i + 1]) / 2
                assert is_inside_polygon(midpoint, TEST_POLYGON)

        for seg in outside_line_segments:
            N = len(seg)
            for i in range(N - 1):
                midpoint = (seg[i] + seg[i + 1]) / 2
                assert not is_inside_polygon(midpoint, TEST_POLYGON)

        # Characterisation test based on visually verified result
        assert len(inside_line_segments) == 13
        assert len(outside_line_segments) == 17
        assert numpy.allclose(inside_line_segments[0],
                              [[122.23028405, -8.62598333],
                               [122.22879, -8.624855],
                               [122.22776827, -8.62420644]])

        assert numpy.allclose(inside_line_segments[-1],
                              [[122.247938, -8.632926],
                               [122.24793987, -8.63351817]])

        assert numpy.allclose(outside_line_segments[0],
                              [[122.231021, -8.626557],
                               [122.230563, -8.626194],
                               [122.23028405, -8.62598333]])

        assert numpy.allclose(outside_line_segments[-1],
                              [[122.24793987, -8.63351817],
                               [122.24794, -8.63356],
                               [122.24739, -8.63622]])

        # Not joined are (but that's OK)
        # [[122.231108, -8.626598], [122.231021, -8.626557]]
        # [[122.231021, -8.626557], [122.230284, -8.625983]]

        # Check dictionaries directly (same data):
        assert len(inside_lines) == 6
        assert len(outside_lines) == 6
        assert numpy.allclose(inside_lines[0][0],
                              [[122.23028405, -8.62598333],
                               [122.22879, -8.624855],
                               [122.22776827, -8.62420644]])

        assert numpy.allclose(inside_lines[5][0],
                              [[122.247938, -8.632926],
                               [122.24793987, -8.63351817]])

        assert numpy.allclose(outside_lines[0][0],
                              [[122.231021, -8.626557],
                               [122.230563, -8.626194],
                               [122.23028405, -8.62598333]])

        assert numpy.allclose(outside_lines[0][1],
                              [[122.22776827, -8.62420644],
                               [122.227536, -8.624059],
                               [122.226648, -8.623494],
                               [122.225775, -8.623022],
                               [122.224872, -8.622444],
                               [122.22423, -8.6221],
                               [122.221931, -8.621082],
                               [122.2217, -8.62098],
                               [122.220577, -8.620555],
                               [122.21958, -8.62103]])

        assert numpy.allclose(outside_lines[5][0],
                              [[122.24793987, -8.63351817],
                               [122.24794, -8.63356],
                               [122.24739, -8.63622]])

    test_clip_lines_by_polygon_real_data.slow = True

    def test_join_segments(self):
        """Consecutive line segments can be joined into continuous line
        """

        # Two segments forming one line
        segments = [[[-1, 0.5], [0.5, 0.5]],
                    [[0.5, 0.5], [0.5, 2]]]
        lines = join_line_segments(segments)

        assert len(lines) == 1
        assert numpy.allclose(lines[0], [[-1, 0.5], [0.5, 0.5], [0.5, 2]])

        # Longer line
        segments = [[[0.0, 0.5], [0.5, 0.5]],
                    [[0.5, 0.5], [0.5, 2.0]],
                    [[0.5, 2.0], [1.0, 2.0]],
                    [[1.0, 2.0], [2.0, 2.0]]]
        lines = join_line_segments(segments)
        assert len(lines) == 1
        assert numpy.allclose(lines[0], [[0.0, 0.5], [0.5, 0.5],
                                         [0.5, 2.0], [1.0, 2.0],
                                         [2.0, 2.0]])

        # Disjoint segment forming two multilines
        segments = [[[0.0, 0.5], [0.5, 0.5]],
                    [[0.5, 0.5], [0.5, 2.0]],
                    [[0.7, 2.0], [1.0, 2.0]],
                    [[1.0, 2.0], [2.0, 2.0]]]
        lines = join_line_segments(segments)
        assert len(lines) == 2
        assert numpy.allclose(lines[0], [[0.0, 0.5], [0.5, 0.5],
                                         [0.5, 2.0]])
        assert numpy.allclose(lines[1], [[0.7, 2.0], [1.0, 2.0],
                                         [2.0, 2.0]])

        # Another example
        segments = [[[0, 0.5], [0.5, 0.5]],
                    [[0.5, 0.5], [0.5, 1]],
                    [[0, 1.0 / 3], [1, 2.0 / 3]]]
        lines = join_line_segments(segments)
        assert len(lines) == 2
        assert numpy.allclose(lines[0], [[0.0, 0.5], [0.5, 0.5],
                                         [0.5, 1.0]])
        assert numpy.allclose(lines[1], [[0, 1.0 / 3], [1, 2.0 / 3]])

        # One with all segments separate
        segments = [[[-1, 0.5], [0, 0.5]],
                    [[0.5, 1], [0.5, 2]],
                    [[-1, 0], [0, 1.0 / 3]]]
        lines = join_line_segments(segments)
        assert len(lines) == 3
        for i in range(len(lines)):
            assert numpy.allclose(lines[i], segments[i])

if __name__ == '__main__':
    suite = unittest.makeSuite(TestPolygon, 'test')
    runner = unittest.TextTestRunner(verbosity=2)
    runner.run(suite)
