//
// Copyright (C) 2006 Andras Babos and Andras Varga
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
//

#include "OSPFNeighbor.h"
#include "OSPFNeighborState.h"
#include "OSPFNeighborStateDown.h"
#include "MessageHandler.h"
#include "OSPFArea.h"
#include "OSPFRouter.h"
#include <memory.h>

// FIXME!!! Should come from a global unique number generator module.
unsigned long OSPF::Neighbor::ddSequenceNumberInitSeed = 0;

OSPF::Neighbor::Neighbor(RouterID neighbor) :
    updateRetransmissionTimerActive(false),
    requestRetransmissionTimerActive(false),
    firstAdjacencyInited(false),
    ddSequenceNumber(0),
    neighborID(neighbor),
    neighborPriority(0),
    neighborIPAddress(OSPF::NULL_IPV4ADDRESS),
    neighborsDesignatedRouter(OSPF::NULL_DESIGNATEDROUTERID),
    neighborsBackupDesignatedRouter(OSPF::NULL_DESIGNATEDROUTERID),
    designatedRoutersSetUp(false),
    neighborsRouterDeadInterval(40),
    lastTransmittedDDPacket(NULL)
{
    memset(&lastReceivedDDPacket, 0, sizeof(OSPF::Neighbor::DDPacketID));
    // setting only I and M bits is invalid -> good initializer
    lastReceivedDDPacket.ddOptions.I_Init = true;
    lastReceivedDDPacket.ddOptions.M_More = true;
    inactivityTimer = new OSPFTimer();
    inactivityTimer->setTimerKind(NEIGHBOR_INACTIVITY_TIMER);
    inactivityTimer->setContextPointer(this);
    inactivityTimer->setName("OSPF::Neighbor::NeighborInactivityTimer");
    pollTimer = new OSPFTimer();
    pollTimer->setTimerKind(NEIGHBOR_POLL_TIMER);
    pollTimer->setContextPointer(this);
    pollTimer->setName("OSPF::Neighbor::NeighborPollTimer");
    ddRetransmissionTimer = new OSPFTimer();
    ddRetransmissionTimer->setTimerKind(NEIGHBOR_DD_RETRANSMISSION_TIMER);
    ddRetransmissionTimer->setContextPointer(this);
    ddRetransmissionTimer->setName("OSPF::Neighbor::NeighborDDRetransmissionTimer");
    updateRetransmissionTimer = new OSPFTimer();
    updateRetransmissionTimer->setTimerKind(NEIGHBOR_UPDATE_RETRANSMISSION_TIMER);
    updateRetransmissionTimer->setContextPointer(this);
    updateRetransmissionTimer->setName("OSPF::Neighbor::Neighbor::NeighborUpdateRetransmissionTimer");
    requestRetransmissionTimer = new OSPFTimer();
    requestRetransmissionTimer->setTimerKind(NEIGHBOR_REQUEST_RETRANSMISSION_TIMER);
    requestRetransmissionTimer->setContextPointer(this);
    requestRetransmissionTimer->setName("OSPF::Neighbor::NeighborRequestRetransmissionTimer");
    state = new OSPF::NeighborStateDown;
    previousState = NULL;
}

OSPF::Neighbor::~Neighbor()
{
    reset();
    MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    messageHandler->clearTimer(inactivityTimer);
    messageHandler->clearTimer(pollTimer);
    delete inactivityTimer;
    delete pollTimer;
    delete ddRetransmissionTimer;
    delete updateRetransmissionTimer;
    delete requestRetransmissionTimer;
    if (previousState != NULL) {
        delete previousState;
    }
    delete state;
}

void OSPF::Neighbor::changeState(NeighborState* newState, NeighborState* currentState)
{
    if (previousState != NULL) {
        delete previousState;
    }
    state = newState;
    previousState = currentState;
}

void OSPF::Neighbor::processEvent(OSPF::Neighbor::NeighborEventType event)
{
    state->processEvent(this, event);
}

void OSPF::Neighbor::reset()
{
    for (std::list<OSPFLSA*>::iterator retIt = linkStateRetransmissionList.begin();
         retIt != linkStateRetransmissionList.end();
         retIt++)
    {
        delete (*retIt);
    }
    linkStateRetransmissionList.clear();

    std::list<OSPFLSAHeader*>::iterator it;
    for (it = databaseSummaryList.begin(); it != databaseSummaryList.end(); it++) {
        delete (*it);
    }
    databaseSummaryList.clear();
    for (it = linkStateRequestList.begin(); it != linkStateRequestList.end(); it++)
    {
        delete (*it);
    }
    linkStateRequestList.clear();

    parentInterface->getArea()->getRouter()->getMessageHandler()->clearTimer(ddRetransmissionTimer);
    clearUpdateRetransmissionTimer();
    clearRequestRetransmissionTimer();

    if (lastTransmittedDDPacket != NULL) {
        delete lastTransmittedDDPacket;
        lastTransmittedDDPacket = NULL;
    }
}

void OSPF::Neighbor::initFirstAdjacency()
{
    ddSequenceNumber = getUniqueULong();
    firstAdjacencyInited = true;
}

unsigned long OSPF::Neighbor::getUniqueULong()
{
    // FIXME!!! Should come from a global unique number generator module.
    return (ddSequenceNumberInitSeed++);
}

OSPF::Neighbor::NeighborStateType OSPF::Neighbor::getState() const
{
    return state->getState();
}

const char* OSPF::Neighbor::getStateString(OSPF::Neighbor::NeighborStateType stateType)
{
    switch (stateType) {
        case DOWN_STATE:           return "Down";
        case ATTEMPT_STATE:        return "Attempt";
        case INIT_STATE:           return "Init";
        case TWOWAY_STATE:         return "TwoWay";
        case EXCHANGE_START_STATE: return "ExchangeStart";
        case EXCHANGE_STATE:       return "Exchange";
        case LOADING_STATE:        return "Loading";
        case FULL_STATE:           return "Full";
        default:                   ASSERT(false);
    }
    return "";
}

void OSPF::Neighbor::sendDatabaseDescriptionPacket(bool init)
{
    OSPFDatabaseDescriptionPacket* ddPacket = new OSPFDatabaseDescriptionPacket();

    ddPacket->setType(DATABASE_DESCRIPTION_PACKET);
    ddPacket->setRouterID(parentInterface->getArea()->getRouter()->getRouterID());
    ddPacket->setAreaID(parentInterface->getArea()->getAreaID());
    ddPacket->setAuthenticationType(parentInterface->getAuthenticationType());
    OSPF::AuthenticationKeyType authKey = parentInterface->getAuthenticationKey();
    for (int i = 0; i < 8; i++) {
        ddPacket->setAuthentication(i, authKey.bytes[i]);
    }

    if (parentInterface->getType() != OSPF::Interface::VIRTUAL) {
        ddPacket->setInterfaceMTU(parentInterface->getMTU());
    } else {
        ddPacket->setInterfaceMTU(0);
    }

    OSPFOptions options;
    memset(&options, 0, sizeof(OSPFOptions));
    options.E_ExternalRoutingCapability = parentInterface->getArea()->getExternalRoutingCapability();
    ddPacket->setOptions(options);

    ddPacket->setDdSequenceNumber(ddSequenceNumber);

    long maxPacketSize = ((IPV4_HEADER_LENGTH + OSPF_HEADER_LENGTH + OSPF_DD_HEADER_LENGTH + OSPF_LSA_HEADER_LENGTH) > parentInterface->getMTU()) ?
                          IPV4_DATAGRAM_LENGTH :
                          parentInterface->getMTU();

    if (init || databaseSummaryList.empty()) {
        ddPacket->setLsaHeadersArraySize(0);
    } else {
        // delete included LSAs from summary list
        // (they are still in lastTransmittedDDPacket)
        long packetSize = IPV4_HEADER_LENGTH + OSPF_HEADER_LENGTH + OSPF_DD_HEADER_LENGTH;
        while ((!databaseSummaryList.empty()) && (packetSize <= (maxPacketSize - OSPF_LSA_HEADER_LENGTH))) {
            unsigned long headerCount = ddPacket->getLsaHeadersArraySize();
            OSPFLSAHeader* lsaHeader = *(databaseSummaryList.begin());
            ddPacket->setLsaHeadersArraySize(headerCount + 1);
            ddPacket->setLsaHeaders(headerCount, *lsaHeader);
            delete lsaHeader;
            databaseSummaryList.pop_front();
            packetSize += OSPF_LSA_HEADER_LENGTH;
        }
    }

    OSPFDDOptions ddOptions;
    memset(&ddOptions, 0, sizeof(OSPFDDOptions));
    if (init) {
        ddOptions.I_Init = true;
        ddOptions.M_More = true;
        ddOptions.MS_MasterSlave = true;
    } else {
        ddOptions.I_Init = false;
        ddOptions.M_More = (databaseSummaryList.empty()) ? false : true;
        ddOptions.MS_MasterSlave = (databaseExchangeRelationship == OSPF::Neighbor::MASTER) ? true : false;
    }
    ddPacket->setDdOptions(ddOptions);

    ddPacket->setPacketLength(0); // TODO: Calculate correct length
    ddPacket->setChecksum(0); // TODO: Calculate correct cheksum(16-bit one's complement of the entire packet)

    OSPF::MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    int ttl = (parentInterface->getType() == OSPF::Interface::VIRTUAL) ? VIRTUAL_LINK_TTL : 1;
    if (parentInterface->getType() == OSPF::Interface::POINTTOPOINT) {
        messageHandler->sendPacket(ddPacket, OSPF::ALL_SPF_ROUTERS, parentInterface->getIfIndex(), ttl);
    } else {
        messageHandler->sendPacket(ddPacket, neighborIPAddress, parentInterface->getIfIndex(), ttl);
    }

    if (lastTransmittedDDPacket != NULL) {
        delete lastTransmittedDDPacket;
    }
    lastTransmittedDDPacket = new OSPFDatabaseDescriptionPacket(*ddPacket);
}

bool OSPF::Neighbor::retransmitDatabaseDescriptionPacket()
{
    if (lastTransmittedDDPacket != NULL) {
        OSPFDatabaseDescriptionPacket* ddPacket = new OSPFDatabaseDescriptionPacket(*lastTransmittedDDPacket);
        OSPF::MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
        int ttl = (parentInterface->getType() == OSPF::Interface::VIRTUAL) ? VIRTUAL_LINK_TTL : 1;

        if (parentInterface->getType() == OSPF::Interface::POINTTOPOINT) {
            messageHandler->sendPacket(ddPacket, OSPF::ALL_SPF_ROUTERS, parentInterface->getIfIndex(), ttl);
        } else {
            messageHandler->sendPacket(ddPacket, neighborIPAddress, parentInterface->getIfIndex(), ttl);
        }

        return true;
    } else {
        return false;
    }
}

void OSPF::Neighbor::createDatabaseSummary()
{
    OSPF::Area* area = parentInterface->getArea();
    unsigned long routerLSACount = area->getRouterLSACount();

    /* Note: OSPF specification says:
     * "LSAs whose age is equal to MaxAge are instead added to the neighbor's
     *  Link state retransmission list."
     * But this task has been already done during the aging of the database. (???)
     * So we'll skip this.
     */
    for (unsigned long i = 0; i < routerLSACount; i++) {
        if (area->getRouterLSA(i)->getHeader().getLsAge() < MAX_AGE) {
            OSPFLSAHeader* routerLSA = new OSPFLSAHeader(area->getRouterLSA(i)->getHeader());
            databaseSummaryList.push_back(routerLSA);
        }
    }

    unsigned long networkLSACount = area->getNetworkLSACount();
    for (unsigned long j = 0; j < networkLSACount; j++) {
        if (area->getNetworkLSA(j)->getHeader().getLsAge() < MAX_AGE) {
            OSPFLSAHeader* networkLSA = new OSPFLSAHeader(area->getNetworkLSA(j)->getHeader());
            databaseSummaryList.push_back(networkLSA);
        }
    }

    unsigned long summaryLSACount = area->getSummaryLSACount();
    for (unsigned long k = 0; k < summaryLSACount; k++) {
        if (area->getSummaryLSA(k)->getHeader().getLsAge() < MAX_AGE) {
            OSPFLSAHeader* summaryLSA = new OSPFLSAHeader(area->getSummaryLSA(k)->getHeader());
            databaseSummaryList.push_back(summaryLSA);
        }
    }

    if ((parentInterface->getType() != OSPF::Interface::VIRTUAL) &&
        (area->getExternalRoutingCapability()))
    {
        OSPF::Router* router = area->getRouter();
        unsigned long asExternalLSACount = router->getASExternalLSACount();

        for (unsigned long m = 0; m < asExternalLSACount; m++) {
            if (router->getASExternalLSA(m)->getHeader().getLsAge() < MAX_AGE) {
                OSPFLSAHeader* asExternalLSA = new OSPFLSAHeader(router->getASExternalLSA(m)->getHeader());
                databaseSummaryList.push_back(asExternalLSA);
            }
        }
    }
}

void OSPF::Neighbor::sendLinkStateRequestPacket()
{
    OSPFLinkStateRequestPacket* requestPacket = new OSPFLinkStateRequestPacket();

    requestPacket->setType(LINKSTATE_REQUEST_PACKET);
    requestPacket->setRouterID(parentInterface->getArea()->getRouter()->getRouterID());
    requestPacket->setAreaID(parentInterface->getArea()->getAreaID());
    requestPacket->setAuthenticationType(parentInterface->getAuthenticationType());
    OSPF::AuthenticationKeyType authKey = parentInterface->getAuthenticationKey();
    for (int i = 0; i < 8; i++) {
        requestPacket->setAuthentication(i, authKey.bytes[i]);
    }

    long maxPacketSize = ((IPV4_HEADER_LENGTH + OSPF_HEADER_LENGTH + OSPF_REQUEST_LENGTH) > parentInterface->getMTU()) ?
                          IPV4_DATAGRAM_LENGTH :
                          parentInterface->getMTU();

    if (linkStateRequestList.empty()) {
        requestPacket->setRequestsArraySize(0);
    } else {
        long packetSize = IPV4_HEADER_LENGTH + OSPF_HEADER_LENGTH;
        std::list<OSPFLSAHeader*>::iterator it = linkStateRequestList.begin();

        while ((it != linkStateRequestList.end()) && (packetSize <= (maxPacketSize - OSPF_REQUEST_LENGTH))) {
            unsigned long requestCount = requestPacket->getRequestsArraySize();
            OSPFLSAHeader* requestHeader = (*it);
            LSARequest request;

            request.lsType = requestHeader->getLsType();
            request.linkStateID = requestHeader->getLinkStateID();
            request.advertisingRouter = requestHeader->getAdvertisingRouter();

            requestPacket->setRequestsArraySize(requestCount + 1);
            requestPacket->setRequests(requestCount, request);

            packetSize += OSPF_REQUEST_LENGTH;
            it++;
        }
    }

    requestPacket->setPacketLength(0); // TODO: Calculate correct length
    requestPacket->setChecksum(0); // TODO: Calculate correct cheksum(16-bit one's complement of the entire packet)

    OSPF::MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    int ttl = (parentInterface->getType() == OSPF::Interface::VIRTUAL) ? VIRTUAL_LINK_TTL : 1;
    if (parentInterface->getType() == OSPF::Interface::POINTTOPOINT) {
        messageHandler->sendPacket(requestPacket, OSPF::ALL_SPF_ROUTERS, parentInterface->getIfIndex(), ttl);
    } else {
        messageHandler->sendPacket(requestPacket, neighborIPAddress, parentInterface->getIfIndex(), ttl);
    }
}

bool OSPF::Neighbor::needAdjacency()
{
    OSPF::Interface::OSPFInterfaceType interfaceType = parentInterface->getType();
    OSPF::RouterID routerID = parentInterface->getArea()->getRouter()->getRouterID();
    OSPF::DesignatedRouterID dRouter = parentInterface->getDesignatedRouter();
    OSPF::DesignatedRouterID backupDRouter = parentInterface->getBackupDesignatedRouter();

    if ((interfaceType == OSPF::Interface::POINTTOPOINT) ||
        (interfaceType == OSPF::Interface::POINTTOMULTIPOINT) ||
        (interfaceType == OSPF::Interface::VIRTUAL) ||
        (dRouter.routerID == routerID) ||
        (backupDRouter.routerID == routerID) ||
        ((neighborsDesignatedRouter.routerID == dRouter.routerID) ||
         (!designatedRoutersSetUp &&
          (neighborsDesignatedRouter.ipInterfaceAddress == dRouter.ipInterfaceAddress))) ||
        ((neighborsBackupDesignatedRouter.routerID == backupDRouter.routerID) ||
         (!designatedRoutersSetUp &&
          (neighborsBackupDesignatedRouter.ipInterfaceAddress == backupDRouter.ipInterfaceAddress))))
    {
        return true;
    } else {
        return false;
    }
}

/**
 * If the LSA is already on the retransmission list then it is replaced, else
 * a copy of the LSA is added to the end of the retransmission list.
 * @param lsa [in] The LSA to be added.
 */
void OSPF::Neighbor::addToRetransmissionList(OSPFLSA* lsa)
{
    std::list<OSPFLSA*>::iterator it;
    for (it = linkStateRetransmissionList.begin(); it != linkStateRetransmissionList.end(); it++) {
        if (((*it)->getHeader().getLinkStateID() == lsa->getHeader().getLinkStateID()) &&
            ((*it)->getHeader().getAdvertisingRouter().getInt() == lsa->getHeader().getAdvertisingRouter().getInt()))
        {
            break;
        }
    }

    OSPFLSA* lsaCopy = NULL;
    switch (lsa->getHeader().getLsType()) {
        case ROUTERLSA_TYPE:
            lsaCopy = new OSPFRouterLSA(*(check_and_cast<OSPFRouterLSA*> (lsa)));
            break;
        case NETWORKLSA_TYPE:
            lsaCopy = new OSPFNetworkLSA(*(check_and_cast<OSPFNetworkLSA*> (lsa)));
            break;
        case SUMMARYLSA_NETWORKS_TYPE:
        case SUMMARYLSA_ASBOUNDARYROUTERS_TYPE:
            lsaCopy = new OSPFSummaryLSA(*(check_and_cast<OSPFSummaryLSA*> (lsa)));
            break;
        case AS_EXTERNAL_LSA_TYPE:
            lsaCopy = new OSPFASExternalLSA(*(check_and_cast<OSPFASExternalLSA*> (lsa)));
            break;
        default:
            ASSERT(false); // error
            break;
    }

    if (it != linkStateRetransmissionList.end()) {
        delete (*it);
        *it = static_cast<OSPFLSA*> (lsaCopy);
    } else {
        linkStateRetransmissionList.push_back(static_cast<OSPFLSA*> (lsaCopy));
    }
}

void OSPF::Neighbor::removeFromRetransmissionList(OSPF::LSAKeyType lsaKey)
{
    std::list<OSPFLSA*>::iterator it = linkStateRetransmissionList.begin();
    while (it != linkStateRetransmissionList.end()) {
        if (((*it)->getHeader().getLinkStateID() == lsaKey.linkStateID) &&
            ((*it)->getHeader().getAdvertisingRouter().getInt() == lsaKey.advertisingRouter))
        {
            delete (*it);
            it = linkStateRetransmissionList.erase(it);
        } else {
            it++;
        }
    }
}

bool OSPF::Neighbor::isLinkStateRequestListEmpty(OSPF::LSAKeyType lsaKey) const
{
    for (std::list<OSPFLSA*>::const_iterator it = linkStateRetransmissionList.begin(); it != linkStateRetransmissionList.end(); it++) {
        const OSPFLSA* lsa = *it;
        if ((lsa->getHeader().getLinkStateID() == lsaKey.linkStateID) &&
            (lsa->getHeader().getAdvertisingRouter().getInt() == lsaKey.advertisingRouter))
        {
            return true;
        }
    }
    return false;
}

OSPFLSA* OSPF::Neighbor::findOnRetransmissionList(OSPF::LSAKeyType lsaKey)
{
    for (std::list<OSPFLSA*>::iterator it = linkStateRetransmissionList.begin(); it != linkStateRetransmissionList.end(); it++) {
        if (((*it)->getHeader().getLinkStateID() == lsaKey.linkStateID) &&
            ((*it)->getHeader().getAdvertisingRouter().getInt() == lsaKey.advertisingRouter))
        {
            return (*it);
        }
    }
    return NULL;
}

void OSPF::Neighbor::startUpdateRetransmissionTimer()
{
    MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    messageHandler->startTimer(updateRetransmissionTimer, parentInterface->getRetransmissionInterval());
    updateRetransmissionTimerActive = true;
}

void OSPF::Neighbor::clearUpdateRetransmissionTimer()
{
    MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    messageHandler->clearTimer(updateRetransmissionTimer);
    updateRetransmissionTimerActive = false;
}

void OSPF::Neighbor::addToRequestList(OSPFLSAHeader* lsaHeader)
{
    linkStateRequestList.push_back(new OSPFLSAHeader(*lsaHeader));
}

void OSPF::Neighbor::removeFromRequestList(OSPF::LSAKeyType lsaKey)
{
    std::list<OSPFLSAHeader*>::iterator it = linkStateRequestList.begin();
    while (it != linkStateRequestList.end()) {
        if (((*it)->getLinkStateID() == lsaKey.linkStateID) &&
            ((*it)->getAdvertisingRouter().getInt() == lsaKey.advertisingRouter))
        {
            delete (*it);
            it = linkStateRequestList.erase(it);
        } else {
            it++;
        }
    }

    if ((getState() == OSPF::Neighbor::LOADING_STATE) && (linkStateRequestList.empty())) {
        clearRequestRetransmissionTimer();
        processEvent(OSPF::Neighbor::LOADING_DONE);
    }
}

bool OSPF::Neighbor::isLSAOnRequestList(OSPF::LSAKeyType lsaKey) const
{
    for (std::list<OSPFLSAHeader*>::const_iterator it = linkStateRequestList.begin(); it != linkStateRequestList.end(); it++) {
        const OSPFLSAHeader* lsaHeader = *it;
        if ((lsaHeader->getLinkStateID() == lsaKey.linkStateID) &&
            (lsaHeader->getAdvertisingRouter().getInt() == lsaKey.advertisingRouter))
        {
            return true;
        }
    }
    return false;
}

OSPFLSAHeader* OSPF::Neighbor::findOnRequestList(OSPF::LSAKeyType lsaKey)
{
    for (std::list<OSPFLSAHeader*>::iterator it = linkStateRequestList.begin(); it != linkStateRequestList.end(); it++) {
        if (((*it)->getLinkStateID() == lsaKey.linkStateID) &&
            ((*it)->getAdvertisingRouter().getInt() == lsaKey.advertisingRouter))
        {
            return (*it);
        }
    }
    return NULL;
}

void OSPF::Neighbor::startRequestRetransmissionTimer()
{
    MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    messageHandler->startTimer(requestRetransmissionTimer, parentInterface->getRetransmissionInterval());
    requestRetransmissionTimerActive = true;
}

void OSPF::Neighbor::clearRequestRetransmissionTimer()
{
    MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    messageHandler->clearTimer(requestRetransmissionTimer);
    requestRetransmissionTimerActive = false;
}

void OSPF::Neighbor::addToTransmittedLSAList(OSPF::LSAKeyType lsaKey)
{
    TransmittedLSA transmit;

    transmit.lsaKey = lsaKey;
    transmit.age = 0;

    transmittedLSAs.push_back(transmit);
}

bool OSPF::Neighbor::isOnTransmittedLSAList(OSPF::LSAKeyType lsaKey) const
{
    for (std::list<TransmittedLSA>::const_iterator it = transmittedLSAs.begin(); it != transmittedLSAs.end(); it++) {
        if ((it->lsaKey.linkStateID == lsaKey.linkStateID) &&
            (it->lsaKey.advertisingRouter == lsaKey.advertisingRouter))
        {
            return true;
        }
    }
    return false;
}

void OSPF::Neighbor::ageTransmittedLSAList()
{
    std::list<TransmittedLSA>::iterator it = transmittedLSAs.begin();
    while ((it != transmittedLSAs.end()) && (it->age == MIN_LS_ARRIVAL)) {
        transmittedLSAs.pop_front();
        it = transmittedLSAs.begin();
    }
    for (it = transmittedLSAs.begin(); it != transmittedLSAs.end(); it++) {
        it->age++;
    }
}

void OSPF::Neighbor::retransmitUpdatePacket()
{
    OSPFLinkStateUpdatePacket* updatePacket = new OSPFLinkStateUpdatePacket();

    updatePacket->setType(LINKSTATE_UPDATE_PACKET);
    updatePacket->setRouterID(parentInterface->getArea()->getRouter()->getRouterID());
    updatePacket->setAreaID(parentInterface->getArea()->getAreaID());
    updatePacket->setAuthenticationType(parentInterface->getAuthenticationType());
    OSPF::AuthenticationKeyType authKey = parentInterface->getAuthenticationKey();
    for (int i = 0; i < 8; i++) {
        updatePacket->setAuthentication(i, authKey.bytes[i]);
    }

    bool packetFull = false;
    unsigned short lsaCount = 0;
    unsigned long packetLength = IPV4_HEADER_LENGTH + OSPF_LSA_HEADER_LENGTH;
    std::list<OSPFLSA*>::iterator it = linkStateRetransmissionList.begin();

    while (!packetFull && (it != linkStateRetransmissionList.end())) {
        LSAType lsaType = static_cast<LSAType> ((*it)->getHeader().getLsType());
        OSPFRouterLSA* routerLSA = (lsaType == ROUTERLSA_TYPE) ? dynamic_cast<OSPFRouterLSA*> (*it) : NULL;
        OSPFNetworkLSA* networkLSA = (lsaType == NETWORKLSA_TYPE) ? dynamic_cast<OSPFNetworkLSA*> (*it) : NULL;
        OSPFSummaryLSA* summaryLSA = ((lsaType == SUMMARYLSA_NETWORKS_TYPE) ||
                                            (lsaType == SUMMARYLSA_ASBOUNDARYROUTERS_TYPE)) ? dynamic_cast<OSPFSummaryLSA*> (*it) : NULL;
        OSPFASExternalLSA* asExternalLSA = (lsaType == AS_EXTERNAL_LSA_TYPE) ? dynamic_cast<OSPFASExternalLSA*> (*it) : NULL;
        long lsaSize = 0;
        bool includeLSA = false;

        switch (lsaType) {
            case ROUTERLSA_TYPE:
                if (routerLSA != NULL) {
                    lsaSize = calculateLSASize(routerLSA);
                }
                break;
            case NETWORKLSA_TYPE:
                if (networkLSA != NULL) {
                    lsaSize = calculateLSASize(networkLSA);
                }
                break;
            case SUMMARYLSA_NETWORKS_TYPE:
            case SUMMARYLSA_ASBOUNDARYROUTERS_TYPE:
                if (summaryLSA != NULL) {
                    lsaSize = calculateLSASize(summaryLSA);
                }
                break;
            case AS_EXTERNAL_LSA_TYPE:
                if (asExternalLSA != NULL) {
                    lsaSize = calculateLSASize(asExternalLSA);
                }
                break;
            default: break;
        }

        if (packetLength + lsaSize < parentInterface->getMTU()) {
            includeLSA = true;
            lsaCount++;
        } else {
            if ((lsaCount == 0) && (packetLength + lsaSize < IPV4_DATAGRAM_LENGTH)) {
                includeLSA = true;
                lsaCount++;
                packetFull = true;
            }
        }

        if (includeLSA) {
            switch (lsaType) {
                case ROUTERLSA_TYPE:
                    if (routerLSA != NULL) {
                        unsigned int routerLSACount = updatePacket->getRouterLSAsArraySize();

                        updatePacket->setRouterLSAsArraySize(routerLSACount + 1);
                        updatePacket->setRouterLSAs(routerLSACount, *routerLSA);

                        unsigned short lsAge = updatePacket->getRouterLSAs(routerLSACount).getHeader().getLsAge();
                        if (lsAge < MAX_AGE - parentInterface->getTransmissionDelay()) {
                            updatePacket->getRouterLSAs(routerLSACount).getHeader().setLsAge(lsAge + parentInterface->getTransmissionDelay());
                        } else {
                            updatePacket->getRouterLSAs(routerLSACount).getHeader().setLsAge(MAX_AGE);
                        }
                    }
                    break;
                case NETWORKLSA_TYPE:
                    if (networkLSA != NULL) {
                        unsigned int networkLSACount = updatePacket->getNetworkLSAsArraySize();

                        updatePacket->setNetworkLSAsArraySize(networkLSACount + 1);
                        updatePacket->setNetworkLSAs(networkLSACount, *networkLSA);

                        unsigned short lsAge = updatePacket->getNetworkLSAs(networkLSACount).getHeader().getLsAge();
                        if (lsAge < MAX_AGE - parentInterface->getTransmissionDelay()) {
                            updatePacket->getNetworkLSAs(networkLSACount).getHeader().setLsAge(lsAge + parentInterface->getTransmissionDelay());
                        } else {
                            updatePacket->getNetworkLSAs(networkLSACount).getHeader().setLsAge(MAX_AGE);
                        }
                    }
                    break;
                case SUMMARYLSA_NETWORKS_TYPE:
                case SUMMARYLSA_ASBOUNDARYROUTERS_TYPE:
                    if (summaryLSA != NULL) {
                        unsigned int summaryLSACount = updatePacket->getSummaryLSAsArraySize();

                        updatePacket->setSummaryLSAsArraySize(summaryLSACount + 1);
                        updatePacket->setSummaryLSAs(summaryLSACount, *summaryLSA);

                        unsigned short lsAge = updatePacket->getSummaryLSAs(summaryLSACount).getHeader().getLsAge();
                        if (lsAge < MAX_AGE - parentInterface->getTransmissionDelay()) {
                            updatePacket->getSummaryLSAs(summaryLSACount).getHeader().setLsAge(lsAge + parentInterface->getTransmissionDelay());
                        } else {
                            updatePacket->getSummaryLSAs(summaryLSACount).getHeader().setLsAge(MAX_AGE);
                        }
                    }
                    break;
                case AS_EXTERNAL_LSA_TYPE:
                    if (asExternalLSA != NULL) {
                        unsigned int asExternalLSACount = updatePacket->getAsExternalLSAsArraySize();

                        updatePacket->setAsExternalLSAsArraySize(asExternalLSACount + 1);
                        updatePacket->setAsExternalLSAs(asExternalLSACount, *asExternalLSA);

                        unsigned short lsAge = updatePacket->getAsExternalLSAs(asExternalLSACount).getHeader().getLsAge();
                        if (lsAge < MAX_AGE - parentInterface->getTransmissionDelay()) {
                            updatePacket->getAsExternalLSAs(asExternalLSACount).getHeader().setLsAge(lsAge + parentInterface->getTransmissionDelay());
                        } else {
                            updatePacket->getAsExternalLSAs(asExternalLSACount).getHeader().setLsAge(MAX_AGE);
                        }
                    }
                    break;
                default: break;
            }
        }

        it++;
    }

    updatePacket->setPacketLength(0); // TODO: Calculate correct length
    updatePacket->setChecksum(0); // TODO: Calculate correct cheksum(16-bit one's complement of the entire packet)

    OSPF::MessageHandler* messageHandler = parentInterface->getArea()->getRouter()->getMessageHandler();
    int ttl = (parentInterface->getType() == OSPF::Interface::VIRTUAL) ? VIRTUAL_LINK_TTL : 1;
    messageHandler->sendPacket(updatePacket, neighborIPAddress, parentInterface->getIfIndex(), ttl);
}

void OSPF::Neighbor::deleteLastSentDDPacket()
{
    if (lastTransmittedDDPacket != NULL) {
        delete lastTransmittedDDPacket;
        lastTransmittedDDPacket = NULL;
    }
}
