package msig

import (
	"fmt"
	"math/big"
	"strings"

	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient/gethclient"
	. "github.com/tranvictor/jarvis/networks"
	"github.com/tranvictor/jarvis/util"
	"github.com/tranvictor/jarvis/util/reader"
)

const GNOSIS_MULTISIG_ABI string = `[{"constant":true,"inputs":[{"name":"","type":"uint256"}],"name":"owners","outputs":[{"name":"","type":"address"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"owner","type":"address"}],"name":"removeOwner","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"transactionId","type":"uint256"}],"name":"revokeConfirmation","outputs":[],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"","type":"address"}],"name":"isOwner","outputs":[{"name":"","type":"bool"}],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"","type":"uint256"},{"name":"","type":"address"}],"name":"confirmations","outputs":[{"name":"","type":"bool"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"calcMaxWithdraw","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"pending","type":"bool"},{"name":"executed","type":"bool"}],"name":"getTransactionCount","outputs":[{"name":"count","type":"uint256"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"dailyLimit","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"lastDay","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"owner","type":"address"}],"name":"addOwner","outputs":[],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"transactionId","type":"uint256"}],"name":"isConfirmed","outputs":[{"name":"","type":"bool"}],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"transactionId","type":"uint256"}],"name":"getConfirmationCount","outputs":[{"name":"count","type":"uint256"}],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"","type":"uint256"}],"name":"transactions","outputs":[{"name":"destination","type":"address"},{"name":"value","type":"uint256"},{"name":"data","type":"bytes"},{"name":"executed","type":"bool"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"getOwners","outputs":[{"name":"","type":"address[]"}],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"from","type":"uint256"},{"name":"to","type":"uint256"},{"name":"pending","type":"bool"},{"name":"executed","type":"bool"}],"name":"getTransactionIds","outputs":[{"name":"_transactionIds","type":"uint256[]"}],"payable":false,"type":"function"},{"constant":true,"inputs":[{"name":"transactionId","type":"uint256"}],"name":"getConfirmations","outputs":[{"name":"_confirmations","type":"address[]"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"transactionCount","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"_required","type":"uint256"}],"name":"changeRequirement","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"transactionId","type":"uint256"}],"name":"confirmTransaction","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"destination","type":"address"},{"name":"value","type":"uint256"},{"name":"data","type":"bytes"}],"name":"submitTransaction","outputs":[{"name":"transactionId","type":"uint256"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"_dailyLimit","type":"uint256"}],"name":"changeDailyLimit","outputs":[],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"MAX_OWNER_COUNT","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"required","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"owner","type":"address"},{"name":"newOwner","type":"address"}],"name":"replaceOwner","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"transactionId","type":"uint256"}],"name":"executeTransaction","outputs":[],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"spentToday","outputs":[{"name":"","type":"uint256"}],"payable":false,"type":"function"},{"inputs":[{"name":"_owners","type":"address[]"},{"name":"_required","type":"uint256"},{"name":"_dailyLimit","type":"uint256"}],"type":"constructor"},{"payable":true,"type":"fallback"},{"anonymous":false,"inputs":[{"indexed":false,"name":"dailyLimit","type":"uint256"}],"name":"DailyLimitChange","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"sender","type":"address"},{"indexed":true,"name":"transactionId","type":"uint256"}],"name":"Confirmation","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"sender","type":"address"},{"indexed":true,"name":"transactionId","type":"uint256"}],"name":"Revocation","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"transactionId","type":"uint256"}],"name":"Submission","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"transactionId","type":"uint256"}],"name":"Execution","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"transactionId","type":"uint256"}],"name":"ExecutionFailure","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"sender","type":"address"},{"indexed":false,"name":"value","type":"uint256"}],"name":"Deposit","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"owner","type":"address"}],"name":"OwnerAddition","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"owner","type":"address"}],"name":"OwnerRemoval","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"required","type":"uint256"}],"name":"RequirementChange","type":"event"}]`

var OVERRIDECODE []byte = common.Hex2Bytes(`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`)

type MultisigContract struct {
	Address string
	Network Network
	reader  *reader.EthReader
	Abi     *abi.ABI
}

func (self *MultisigContract) Owners() ([]string, error) {
	owners := new([]common.Address)
	err := self.reader.ReadContractWithABI(
		owners,
		self.Address,
		self.Abi,
		"getOwners",
	)
	result := []string{}
	if err != nil {
		return result, err
	}
	for _, owner := range *owners {
		result = append(result, owner.Hex())
	}
	return result, nil
}

func (self *MultisigContract) IsExecuted(txid *big.Int) (bool, error) {
	_, _, _, executed, _, err := self.TransactionInfo(txid)
	return executed, err
}

func (self *MultisigContract) IsConfirmed(txid *big.Int) (bool, error) {
	r := new(bool)
	err := self.reader.ReadContractWithABI(
		r,
		self.Address,
		self.Abi,
		"isConfirmed",
		txid,
	)
	if err != nil {
		return false, err
	}
	return *r, err
}

func (self *MultisigContract) NOTransactions() (int64, error) {
	r := big.NewInt(0)
	err := self.reader.ReadContractWithABI(
		&r,
		self.Address,
		self.Abi,
		"transactionCount",
	)
	if err != nil {
		return 0, err
	}
	return r.Int64(), nil
}

func (self *MultisigContract) VoteRequirement() (int64, error) {
	r := big.NewInt(0)
	err := self.reader.ReadContractWithABI(
		&r,
		self.Address,
		self.Abi,
		"required",
	)
	if err != nil {
		return 0, err
	}
	return r.Int64(), nil
}

func (self *MultisigContract) TransactionInfo(txid *big.Int) (address string, value *big.Int, data []byte, executed bool, confirmations []string, err error) {
	type response struct {
		Destination *common.Address
		Value       *big.Int
		Data        *[]byte
		Executed    *bool
	}
	ret := response{
		&common.Address{},
		big.NewInt(0),
		new([]byte),
		new(bool),
	}
	err = self.reader.ReadContractWithABI(
		&ret,
		self.Address,
		self.Abi,
		"transactions",
		txid,
	)
	if err != nil {
		return "", big.NewInt(0), []byte{}, false, []string{}, err
	}
	signers := []common.Address{}
	err = self.reader.ReadContractWithABI(
		&signers,
		self.Address,
		self.Abi,
		"getConfirmations",
		txid,
	)
	if err != nil {
		return "", big.NewInt(0), []byte{}, false, []string{}, err
	}
	confirmations = []string{}
	for _, s := range signers {
		confirmations = append(confirmations, s.Hex())
	}
	return ret.Destination.Hex(), ret.Value, *ret.Data, *ret.Executed, confirmations, nil
}

func NewMultisigContract(address string, network Network) (*MultisigContract, error) {
	reader, err := util.EthReader(network)
	if err != nil {
		return nil, err
	}

	a, err := abi.JSON(strings.NewReader(GNOSIS_MULTISIG_ABI))
	if err != nil {
		return nil, err
	}
	return &MultisigContract{
		address,
		network,
		reader,
		&a,
	}, nil
}

func (self *MultisigContract) SimulateSubmit(from string, to string, value *big.Int, data []byte) error {
	owners, err := self.Owners()
	if err != nil {
		fmt.Printf("Couldn't get owners of the multisig: %s\n", err)
		return err
	}
	transactionID, err := self.NOTransactions()
	if err != nil {
		fmt.Printf("Couldn't get number of transactions of the multisig: %s\n", err)
		return err
	}

	stateDiff := make(map[common.Hash]common.Hash)
	isOnwner := false
	for _, owner := range owners {
		if strings.EqualFold(owner, from) {
			isOnwner = true
		} else {
			overrideConfirmation(stateDiff, transactionID, owner)
		}
	}
	if !isOnwner {
		return fmt.Errorf("Sender is not owner of the msig")
	}
	overrides := &map[common.Address]gethclient.OverrideAccount{
		common.HexToAddress(self.Address): {
			StateDiff: stateDiff,    // override that all owner approve to submit tradeID
			Code:      OVERRIDECODE, // override contract code to get revert msg if any
		},
	}
	txdata, err := self.Abi.Pack(
		"submitTransaction",
		common.HexToAddress(to),
		value,
		data,
	)
	if err != nil {
		return err
	}
	_, err = self.reader.EthCall(from, self.Address, txdata, overrides)
	return err
}

func overrideConfirmation(stateDiff map[common.Hash]common.Hash, transactionID int64, owner string) {
	// mapping (uint => mapping (address => bool)) public confirmations

	storageSlot := common.BigToHash(big.NewInt(1))
	transactionHash := common.BigToHash(big.NewInt(int64(transactionID)))
	ownerAddress := common.HexToAddress(owner)
	addressHash := common.BytesToHash(ownerAddress[:])
	fmt.Println(addressHash)

	mapSlot := crypto.Keccak256Hash(
		transactionHash[:],
		storageSlot[:],
	)
	confirmationSlot := crypto.Keccak256Hash(
		addressHash[:],
		mapSlot[:],
	)
	stateDiff[confirmationSlot] = common.BigToHash(big.NewInt(1))
}
