import { Injectable, Logger } from "@nestjs/common";
import { INTERNAL_SERVER_ERROR } from "../../common/helpers/constants";
import { IncidentUpdateDto, IncidentBatchUpdateDto } from "../../common/types/dto/incident.dto";
import { ApiResponse } from "../../common/types/dto/response.dto";
import { Incident } from "../../db/entities/incident.entity";
import { EntityManager } from "typeorm";
import { UserQueryService } from "../user/user-query/user-query.service";

@Injectable()
export class IncidentsService {
  private logger: Logger;
  constructor(private entityManger: EntityManager, private userQueryService: UserQueryService) {
    this.logger = new Logger(IncidentsService.name);
  }

  async updateIncident(
    incidentId: string,
    update: IncidentUpdateDto
  ): Promise<ApiResponse<unknown>> {
    if (!update) {
      return;
    }

    return await this.entityManger
      .transaction(async (manager) => {
        if (update.assignedId) {
          const user = await this.userQueryService.getDto(update.assignedId);
          await manager.getRepository(Incident).update({ id: incidentId }, { assigned: user });

          return new ApiResponse("success", "Incident assigned");
        }

        await manager.getRepository(Incident).update({ id: incidentId }, update);

        return new ApiResponse("success", "Incident updated");
      })
      .catch((err: Error) => {
        this.logger.error(`[${this.updateIncident.name}] Caused by: ${err}`);
        return new ApiResponse("error", INTERNAL_SERVER_ERROR, err);
      });
  }

  async updateBatchIncidents(update: IncidentBatchUpdateDto): Promise<ApiResponse<unknown>> {
    const { incidentsIds, ...rest } = update;

    if (!update) {
      return;
    }

    try {
      await this.entityManger
        .getRepository(Incident)
        .createQueryBuilder()
        .whereInIds(incidentsIds)
        .update(rest)
        .execute();

      return new ApiResponse("success", "Incidents updated");
    } catch (err) {
      this.logger.error(`[${this.updateBatchIncidents.name}] Caused by: ${err}`);
      return new ApiResponse("error", INTERNAL_SERVER_ERROR, err);
    }
  }

  async removeIncident(id: string): Promise<ApiResponse<unknown>> {
    try {
      await this.entityManger
        .getRepository(Incident)
        .createQueryBuilder("incident")
        .where("incident.id = :id", { id })
        .delete()
        .execute();

      return new ApiResponse("success", "Incident removed");
    } catch (err) {
      this.logger.error(`[${this.removeIncident.name}] Caused by: ${err}`);
      return new ApiResponse("error", INTERNAL_SERVER_ERROR, err);
    }
  }
}
