// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using System.Globalization;

namespace System.ComponentModel
{
    /// <devdoc>
    ///    <para>Provides a type
    ///       converter that can be used to populate a list box with available types.</para>
    /// </devdoc>
    public abstract class TypeListConverter : TypeConverter
    {
        private Type[] _types;

        /// <devdoc>
        /// <para>Initializes a new instance of the <see cref='System.ComponentModel.TypeListConverter'/> class using
        ///    the type array as the available types.</para>
        /// </devdoc>
        protected TypeListConverter(Type[] types)
        {
            _types = types;
        }

        /// <internalonly/>
        /// <devdoc>
        ///    <para>Gets a value indicating whether this converter
        ///       can convert an object in the given source type to an enumeration object using
        ///       the specified context.</para>
        /// </devdoc>
        public override bool CanConvertFrom(ITypeDescriptorContext context, Type sourceType)
        {
            if (sourceType == typeof(string))
            {
                return true;
            }
            return base.CanConvertFrom(context, sourceType);
        }

        /// <internalonly/>
        /// <devdoc>
        ///    <para>Converts the specified value object to an enumeration object.</para>
        /// </devdoc>
        public override object ConvertFrom(ITypeDescriptorContext context, CultureInfo culture, object value)
        {
            if (value is string)
            {
                foreach (Type t in _types)
                {
                    if (value.Equals(t.FullName))
                    {
                        return t;
                    }
                }
            }

            return base.ConvertFrom(context, culture, value);
        }

        /// <internalonly/>
        /// <devdoc>
        ///    <para>Converts the given value object to the specified destination type.</para>
        /// </devdoc>
        public override object ConvertTo(ITypeDescriptorContext context, CultureInfo culture, object value, Type destinationType)
        {
            if (destinationType == null)
            {
                throw new ArgumentNullException("destinationType");
            }

            if (destinationType == typeof(string))
            {
                if (value == null)
                {
                    return SR.none;
                }
                else
                {
                    return ((Type)value).FullName;
                }
            }

            return base.ConvertTo(context, culture, value, destinationType);
        }
    }
}

