﻿// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using System.Runtime.InteropServices;
using Test.Cryptography;
using Xunit;

namespace System.Security.Cryptography.X509Certificates.Tests
{
    internal static class TestData
    {
        public static byte[] MsCertificate = (
            "308204ec308203d4a003020102021333000000b011af0a8bd03b9fdd00010000" +
            "00b0300d06092a864886f70d01010505003079310b3009060355040613025553" +
            "311330110603550408130a57617368696e67746f6e3110300e06035504071307" +
            "5265646d6f6e64311e301c060355040a13154d6963726f736f667420436f7270" +
            "6f726174696f6e312330210603550403131a4d6963726f736f667420436f6465" +
            "205369676e696e6720504341301e170d3133303132343232333333395a170d31" +
            "34303432343232333333395a308183310b300906035504061302555331133011" +
            "0603550408130a57617368696e67746f6e3110300e060355040713075265646d" +
            "6f6e64311e301c060355040a13154d6963726f736f667420436f72706f726174" +
            "696f6e310d300b060355040b13044d4f5052311e301c060355040313154d6963" +
            "726f736f667420436f72706f726174696f6e30820122300d06092a864886f70d" +
            "01010105000382010f003082010a0282010100e8af5ca2200df8287cbc057b7f" +
            "adeeeb76ac28533f3adb407db38e33e6573fa551153454a5cfb48ba93fa837e1" +
            "2d50ed35164eef4d7adb137688b02cf0595ca9ebe1d72975e41b85279bf3f82d" +
            "9e41362b0b40fbbe3bbab95c759316524bca33c537b0f3eb7ea8f541155c0865" +
            "1d2137f02cba220b10b1109d772285847c4fb91b90b0f5a3fe8bf40c9a4ea0f5" +
            "c90a21e2aae3013647fd2f826a8103f5a935dc94579dfb4bd40e82db388f12fe" +
            "e3d67a748864e162c4252e2aae9d181f0e1eb6c2af24b40e50bcde1c935c49a6" +
            "79b5b6dbcef9707b280184b82a29cfbfa90505e1e00f714dfdad5c238329ebc7" +
            "c54ac8e82784d37ec6430b950005b14f6571c50203010001a38201603082015c" +
            "30130603551d25040c300a06082b06010505070303301d0603551d0e04160414" +
            "5971a65a334dda980780ff841ebe87f9723241f230510603551d11044a3048a4" +
            "463044310d300b060355040b13044d4f5052313330310603550405132a333135" +
            "39352b34666166306237312d616433372d346161332d613637312d3736626330" +
            "35323334346164301f0603551d23041830168014cb11e8cad2b4165801c9372e" +
            "331616b94c9a0a1f30560603551d1f044f304d304ba049a0478645687474703a" +
            "2f2f63726c2e6d6963726f736f66742e636f6d2f706b692f63726c2f70726f64" +
            "756374732f4d6963436f645369675043415f30382d33312d323031302e63726c" +
            "305a06082b06010505070101044e304c304a06082b06010505073002863e6874" +
            "74703a2f2f7777772e6d6963726f736f66742e636f6d2f706b692f6365727473" +
            "2f4d6963436f645369675043415f30382d33312d323031302e637274300d0609" +
            "2a864886f70d0101050500038201010031d76e2a12573381d59dc6ebf93ad444" +
            "4d089eee5edf6a5bb779cf029cbc76689e90a19c0bc37fa28cf14dba9539fb0d" +
            "e0e19bf45d240f1b8d88153a7cdbadceb3c96cba392c457d24115426300d0dff" +
            "47ea0307e5e4665d2c7b9d1da910fa1cb074f24f696b9ea92484daed96a0df73" +
            "a4ef6a1aac4b629ef17cc0147f48cd4db244f9f03c936d42d8e87ce617a09b68" +
            "680928f90297ef1103ba6752adc1e9b373a6d263cd4ae23ee4f34efdffa1e0bb" +
            "02133b5d20de553fa3ae9040313875285e04a9466de6f57a7940bd1fcde845d5" +
            "aee25d3ef575c7e6666360ccd59a84878d2430f7ef34d0631db142674a0e4bbf" +
            "3a0eefb6953aa738e4259208a6886682").HexToByteArray();

        public static readonly byte[] MsCertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIE7DCCA9SgAwIBAgITMwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQD
ExpNaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzlaFw0x
NDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNyb3NvZnQgQ29ycG9yYXRp
b24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/
re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01Fk7vTXrbE3aIsCzw
WVyp6+HXKXXkG4Unm/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhl
HSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMBNkf9L4JqgQP1
qTXclFed+0vUDoLbOI8S/uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZDC5UABbFP
ZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQU
WXGmWjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsTBE1P
UFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMw
NTIzNDRhZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9k
dWN0cy9NaWNDb2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBM
MEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRz
L01pY0NvZFNpZ1BDQV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEA
MdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/oozxTbqVOfsN
4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjofOYXoJto
aAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUo
XgSpRm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdKDku/
Og7vtpU6pzjkJZIIpohmgg==
-----END CERTIFICATE-----
");

        public const string PfxDataPassword = "12345";

        public static byte[] PfxData = (
            "308206a20201033082065e06092a864886f70d010701a082064f0482064b3082" +
            "0647308203c006092a864886f70d010701a08203b1048203ad308203a9308203" +
            "a5060b2a864886f70d010c0a0102a08202b6308202b2301c060a2a864886f70d" +
            "010c0103300e04085052002c7da2c2a6020207d0048202907d485e3bfc6e6457" +
            "c811394c145d0e8a18325646854e4ff0097bc5a98547f5ad616c8efda8505aa8" +
            "7564ed4800a3139759497c60c6688b51f376acae906429c8771cb1428226b68a" +
            "6297207bcc9dd7f9563478dd83880aab2304b545759b2275305df4eff9fac24a" +
            "3cc9d3b2d672efe45d8f48e24a16506c1d7566fc6d1b269fbf201b3ac3309d3e" +
            "bc6fd606257a7a707aa2f790ea3fe7a94a51138540c5319010cba6de9fb9d85f" +
            "cdc78da60e33df2f21c46fb9a8554b4f82e0a6edba4db5585d77d331d35daaed" +
            "51b6a5a3e000a299880fb799182c8ca3004b7837a9feb8bfc76778089993f3d1" +
            "1d70233608af7c50722d680623d2bf54bd4b1e7a604184d9f44e0af8099ffa47" +
            "1e5536e7902793829db9902ddb61264a62962950ad274ea516b2d44be9036530" +
            "016e607b73f341aeefed2211f6330364738b435b0d2ed6c57747f6c8230a053f" +
            "78c4dd65db83b26c6a47836a6cbbab92cbb262c6fb6d08632b4457f5fa8eabfa" +
            "65db34157e1d301e9085cc443582cdd15404314872748545eb3fc3c574882655" +
            "8c9a85f966e315775bbe9da34d1e8b6dadc3c9e120c6d6a2e1cffe4eb014c3ce" +
            "fbc19356ce33dac60f93d67a4de247b0dae13cd8b8c9f15604cc0ec9968e3ad7" +
            "f57c9f53c45e2ecb0a0945ec0ba04baa15b48d8596edc9f5fe9165a5d21949fb" +
            "5fe30a920ad2c0f78799f6443c300629b8ca4dca19b9dbf1e27aab7b12271228" +
            "119a95c9822be6439414beeae24002b46eb97e030e18bd810ade0bcf4213a355" +
            "038b56584b2fbcc3f5ea215d0cf667ffd823ea03ab62c3b193dfb4450aabb50b" +
            "af306e8088ee7384fa2fdff03e0dd7acd61832223e806a94d46e196462522808" +
            "3163f1caf333fdbbe2d54ca86968867ce0b6dd5e5b7f0633c6fab4a19cc14f64" +
            "5ec14d0b1436f7623181db301306092a864886f70d0109153106040401000000" +
            "305d06092b060104018237110131501e4e004d006900630072006f0073006f00" +
            "6600740020005300740072006f006e0067002000430072007900700074006f00" +
            "67007200610070006800690063002000500072006f0076006900640065007230" +
            "6506092a864886f70d01091431581e5600500076006b0054006d0070003a0034" +
            "0064006100340061003100650036002d0066003700380062002d003400360061" +
            "0035002d0039003800380033002d003500320063003800330032006100340063" +
            "0030006100623082027f06092a864886f70d010706a08202703082026c020100" +
            "3082026506092a864886f70d010701301c060a2a864886f70d010c0106300e04" +
            "08e0c117e67a75d8eb020207d080820238292882408b31826f0dc635f9bbe7c1" +
            "99a48a3b4fefc729dbf95508d6a7d04805a8dd612427f93124f522ac7d3c6f4d" +
            "db74d937f57823b5b1e8cfae4ece4a1fffd801558d77ba31985aa7f747d834cb" +
            "e84464ef777718c9865c819d6c9daa0fa25e2a2a80b3f2aaa67d40e382eb084c" +
            "ca85e314ea40c3ef3ed1593904d7a16f37807c99af06c917093f6c5aaebb12a6" +
            "c58c9956d4fbbdde1f1e389989c36e19dd38d4b978d6f47131e458ab68e237e4" +
            "0cb6a87f21c8773de845780b50995a51f041106f47c740b3bd946038984f1ac9" +
            "e91230616480962f11b0683f8802173c596c4bd554642f51a76f9dfff9053def" +
            "7b3c3f759fc7eeac3f2386106c4b8cb669589e004fb235f0357ea5cf0b5a6fc7" +
            "8a6d941a3ae44af7b601b59d15cd1ec61bccc481fbb83eae2f83153b41e71ef7" +
            "6a2814ab59347f116ab3e9c1621668a573013d34d13d3854e604286733c6bad0" +
            "f511d7f8fd6356f7c3198d0cb771af27f4b5a3c3b571fdd083fd68a9a1eea783" +
            "152c436f7513613a7e399a1da48d7e55db7504dc47d1145df8d7b6d32eaa4cce" +
            "e06f98bb3dda2cc0d0564a962f86dfb122e4f7e2ed6f1b509c58d4a3b2d0a687" +
            "88f7e313aecfbdef456c31b96fc13586e02aeb65807ed83bb0cb7c28f157bc95" +
            "c9c593c9194691539ae3c620ed1d4d4af0177f6b9483a5341d7b084bc5b425af" +
            "b658168ee2d8fb2bfab07a3ba061687a5ecd1f8da9001dd3e7be793923094abb" +
            "0f2cf4d24cb071b9e568b18336bb4dc541352c9785c48d0f0e53066eb2009efc" +
            "b3e5644ed12252c1bc303b301f300706052b0e03021a0414f90827ae93fd3a91" +
            "54c3c0840d7950b0e30ffbaf0414e147930b932899741c92d765226893877025" +
            "4a2b020207d0").HexToByteArray();

        public static byte[] StoreSavedAsPfxData = (
            "3082070406092a864886f70d010702a08206f5308206f10201013100300b0609" +
            "2a864886f70d010701a08206d9308201e530820152a0030201020210d5b5bc1c" +
            "458a558845bff51cb4dff31c300906052b0e03021d05003011310f300d060355" +
            "040313064d794e616d65301e170d3130303430313038303030305a170d313130" +
            "3430313038303030305a3011310f300d060355040313064d794e616d6530819f" +
            "300d06092a864886f70d010101050003818d0030818902818100b11e30ea8742" +
            "4a371e30227e933ce6be0e65ff1c189d0d888ec8ff13aa7b42b68056128322b2" +
            "1f2b6976609b62b6bc4cf2e55ff5ae64e9b68c78a3c2dacc916a1bc7322dd353" +
            "b32898675cfb5b298b176d978b1f12313e3d865bc53465a11cca106870a4b5d5" +
            "0a2c410938240e92b64902baea23eb093d9599e9e372e48336730203010001a3" +
            "46304430420603551d01043b3039801024859ebf125e76af3f0d7979b4ac7a96" +
            "a1133011310f300d060355040313064d794e616d658210d5b5bc1c458a558845" +
            "bff51cb4dff31c300906052b0e03021d0500038181009bf6e2cf830ed485b86d" +
            "6b9e8dffdcd65efc7ec145cb9348923710666791fcfa3ab59d689ffd7234b787" +
            "2611c5c23e5e0714531abadb5de492d2c736e1c929e648a65cc9eb63cd84e57b" +
            "5909dd5ddf5dbbba4a6498b9ca225b6e368b94913bfc24de6b2bd9a26b192b95" +
            "7304b89531e902ffc91b54b237bb228be8afcda26476308204ec308203d4a003" +
            "020102021333000000b011af0a8bd03b9fdd0001000000b0300d06092a864886" +
            "f70d01010505003079310b300906035504061302555331133011060355040813" +
            "0a57617368696e67746f6e3110300e060355040713075265646d6f6e64311e30" +
            "1c060355040a13154d6963726f736f667420436f72706f726174696f6e312330" +
            "210603550403131a4d6963726f736f667420436f6465205369676e696e672050" +
            "4341301e170d3133303132343232333333395a170d3134303432343232333333" +
            "395a308183310b3009060355040613025553311330110603550408130a576173" +
            "68696e67746f6e3110300e060355040713075265646d6f6e64311e301c060355" +
            "040a13154d6963726f736f667420436f72706f726174696f6e310d300b060355" +
            "040b13044d4f5052311e301c060355040313154d6963726f736f667420436f72" +
            "706f726174696f6e30820122300d06092a864886f70d01010105000382010f00" +
            "3082010a0282010100e8af5ca2200df8287cbc057b7fadeeeb76ac28533f3adb" +
            "407db38e33e6573fa551153454a5cfb48ba93fa837e12d50ed35164eef4d7adb" +
            "137688b02cf0595ca9ebe1d72975e41b85279bf3f82d9e41362b0b40fbbe3bba" +
            "b95c759316524bca33c537b0f3eb7ea8f541155c08651d2137f02cba220b10b1" +
            "109d772285847c4fb91b90b0f5a3fe8bf40c9a4ea0f5c90a21e2aae3013647fd" +
            "2f826a8103f5a935dc94579dfb4bd40e82db388f12fee3d67a748864e162c425" +
            "2e2aae9d181f0e1eb6c2af24b40e50bcde1c935c49a679b5b6dbcef9707b2801" +
            "84b82a29cfbfa90505e1e00f714dfdad5c238329ebc7c54ac8e82784d37ec643" +
            "0b950005b14f6571c50203010001a38201603082015c30130603551d25040c30" +
            "0a06082b06010505070303301d0603551d0e041604145971a65a334dda980780" +
            "ff841ebe87f9723241f230510603551d11044a3048a4463044310d300b060355" +
            "040b13044d4f5052313330310603550405132a33313539352b34666166306237" +
            "312d616433372d346161332d613637312d373662633035323334346164301f06" +
            "03551d23041830168014cb11e8cad2b4165801c9372e331616b94c9a0a1f3056" +
            "0603551d1f044f304d304ba049a0478645687474703a2f2f63726c2e6d696372" +
            "6f736f66742e636f6d2f706b692f63726c2f70726f64756374732f4d6963436f" +
            "645369675043415f30382d33312d323031302e63726c305a06082b0601050507" +
            "0101044e304c304a06082b06010505073002863e687474703a2f2f7777772e6d" +
            "6963726f736f66742e636f6d2f706b692f63657274732f4d6963436f64536967" +
            "5043415f30382d33312d323031302e637274300d06092a864886f70d01010505" +
            "00038201010031d76e2a12573381d59dc6ebf93ad4444d089eee5edf6a5bb779" +
            "cf029cbc76689e90a19c0bc37fa28cf14dba9539fb0de0e19bf45d240f1b8d88" +
            "153a7cdbadceb3c96cba392c457d24115426300d0dff47ea0307e5e4665d2c7b" +
            "9d1da910fa1cb074f24f696b9ea92484daed96a0df73a4ef6a1aac4b629ef17c" +
            "c0147f48cd4db244f9f03c936d42d8e87ce617a09b68680928f90297ef1103ba" +
            "6752adc1e9b373a6d263cd4ae23ee4f34efdffa1e0bb02133b5d20de553fa3ae" +
            "9040313875285e04a9466de6f57a7940bd1fcde845d5aee25d3ef575c7e66663" +
            "60ccd59a84878d2430f7ef34d0631db142674a0e4bbf3a0eefb6953aa738e425" +
            "9208a68866823100").HexToByteArray();


        public static byte[] StoreSavedAsCerData = (
            "308201e530820152a0030201020210d5b5bc1c458a558845bff51cb4dff31c30" +
            "0906052b0e03021d05003011310f300d060355040313064d794e616d65301e17" +
            "0d3130303430313038303030305a170d3131303430313038303030305a301131" +
            "0f300d060355040313064d794e616d6530819f300d06092a864886f70d010101" +
            "050003818d0030818902818100b11e30ea87424a371e30227e933ce6be0e65ff" +
            "1c189d0d888ec8ff13aa7b42b68056128322b21f2b6976609b62b6bc4cf2e55f" +
            "f5ae64e9b68c78a3c2dacc916a1bc7322dd353b32898675cfb5b298b176d978b" +
            "1f12313e3d865bc53465a11cca106870a4b5d50a2c410938240e92b64902baea" +
            "23eb093d9599e9e372e48336730203010001a346304430420603551d01043b30" +
            "39801024859ebf125e76af3f0d7979b4ac7a96a1133011310f300d0603550403" +
            "13064d794e616d658210d5b5bc1c458a558845bff51cb4dff31c300906052b0e" +
            "03021d0500038181009bf6e2cf830ed485b86d6b9e8dffdcd65efc7ec145cb93" +
            "48923710666791fcfa3ab59d689ffd7234b7872611c5c23e5e0714531abadb5d" +
            "e492d2c736e1c929e648a65cc9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca" +
            "225b6e368b94913bfc24de6b2bd9a26b192b957304b89531e902ffc91b54b237" +
            "bb228be8afcda26476").HexToByteArray();

        public static byte[] StoreSavedAsSerializedCerData = (
            "0200000001000000bc0000001c0000006c000000010000000000000000000000" +
            "00000000020000007b00370037004500420044003000320044002d0044003800" +
            "440045002d0034003700350041002d0038003800360037002d00440032003000" +
            "4200300030003600340045003400390046007d00000000004d00690063007200" +
            "6f0073006f006600740020005300740072006f006e0067002000430072007900" +
            "700074006f0067007200610070006800690063002000500072006f0076006900" +
            "64006500720000002000000001000000e9010000308201e530820152a0030201" +
            "020210d5b5bc1c458a558845bff51cb4dff31c300906052b0e03021d05003011" +
            "310f300d060355040313064d794e616d65301e170d3130303430313038303030" +
            "305a170d3131303430313038303030305a3011310f300d060355040313064d79" +
            "4e616d6530819f300d06092a864886f70d010101050003818d00308189028181" +
            "00b11e30ea87424a371e30227e933ce6be0e65ff1c189d0d888ec8ff13aa7b42" +
            "b68056128322b21f2b6976609b62b6bc4cf2e55ff5ae64e9b68c78a3c2dacc91" +
            "6a1bc7322dd353b32898675cfb5b298b176d978b1f12313e3d865bc53465a11c" +
            "ca106870a4b5d50a2c410938240e92b64902baea23eb093d9599e9e372e48336" +
            "730203010001a346304430420603551d01043b3039801024859ebf125e76af3f" +
            "0d7979b4ac7a96a1133011310f300d060355040313064d794e616d658210d5b5" +
            "bc1c458a558845bff51cb4dff31c300906052b0e03021d0500038181009bf6e2" +
            "cf830ed485b86d6b9e8dffdcd65efc7ec145cb9348923710666791fcfa3ab59d" +
            "689ffd7234b7872611c5c23e5e0714531abadb5de492d2c736e1c929e648a65c" +
            "c9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca225b6e368b94913bfc24de6b" +
            "2bd9a26b192b957304b89531e902ffc91b54b237bb228be8afcda26476").HexToByteArray();

        public static byte[] StoreSavedAsSerializedStoreData = (
            "00000000434552540200000001000000bc0000001c0000006c00000001000000" +
            "000000000000000000000000020000007b003700370045004200440030003200" +
            "44002d0044003800440045002d0034003700350041002d003800380036003700" +
            "2d004400320030004200300030003600340045003400390046007d0000000000" +
            "4d006900630072006f0073006f006600740020005300740072006f006e006700" +
            "2000430072007900700074006f00670072006100700068006900630020005000" +
            "72006f007600690064006500720000002000000001000000e9010000308201e5" +
            "30820152a0030201020210d5b5bc1c458a558845bff51cb4dff31c300906052b" +
            "0e03021d05003011310f300d060355040313064d794e616d65301e170d313030" +
            "3430313038303030305a170d3131303430313038303030305a3011310f300d06" +
            "0355040313064d794e616d6530819f300d06092a864886f70d01010105000381" +
            "8d0030818902818100b11e30ea87424a371e30227e933ce6be0e65ff1c189d0d" +
            "888ec8ff13aa7b42b68056128322b21f2b6976609b62b6bc4cf2e55ff5ae64e9" +
            "b68c78a3c2dacc916a1bc7322dd353b32898675cfb5b298b176d978b1f12313e" +
            "3d865bc53465a11cca106870a4b5d50a2c410938240e92b64902baea23eb093d" +
            "9599e9e372e48336730203010001a346304430420603551d01043b3039801024" +
            "859ebf125e76af3f0d7979b4ac7a96a1133011310f300d060355040313064d79" +
            "4e616d658210d5b5bc1c458a558845bff51cb4dff31c300906052b0e03021d05" +
            "00038181009bf6e2cf830ed485b86d6b9e8dffdcd65efc7ec145cb9348923710" +
            "666791fcfa3ab59d689ffd7234b7872611c5c23e5e0714531abadb5de492d2c7" +
            "36e1c929e648a65cc9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca225b6e36" +
            "8b94913bfc24de6b2bd9a26b192b957304b89531e902ffc91b54b237bb228be8" +
            "afcda264762000000001000000f0040000308204ec308203d4a0030201020213" +
            "33000000b011af0a8bd03b9fdd0001000000b0300d06092a864886f70d010105" +
            "05003079310b3009060355040613025553311330110603550408130a57617368" +
            "696e67746f6e3110300e060355040713075265646d6f6e64311e301c06035504" +
            "0a13154d6963726f736f667420436f72706f726174696f6e3123302106035504" +
            "03131a4d6963726f736f667420436f6465205369676e696e6720504341301e17" +
            "0d3133303132343232333333395a170d3134303432343232333333395a308183" +
            "310b3009060355040613025553311330110603550408130a57617368696e6774" +
            "6f6e3110300e060355040713075265646d6f6e64311e301c060355040a13154d" +
            "6963726f736f667420436f72706f726174696f6e310d300b060355040b13044d" +
            "4f5052311e301c060355040313154d6963726f736f667420436f72706f726174" +
            "696f6e30820122300d06092a864886f70d01010105000382010f003082010a02" +
            "82010100e8af5ca2200df8287cbc057b7fadeeeb76ac28533f3adb407db38e33" +
            "e6573fa551153454a5cfb48ba93fa837e12d50ed35164eef4d7adb137688b02c" +
            "f0595ca9ebe1d72975e41b85279bf3f82d9e41362b0b40fbbe3bbab95c759316" +
            "524bca33c537b0f3eb7ea8f541155c08651d2137f02cba220b10b1109d772285" +
            "847c4fb91b90b0f5a3fe8bf40c9a4ea0f5c90a21e2aae3013647fd2f826a8103" +
            "f5a935dc94579dfb4bd40e82db388f12fee3d67a748864e162c4252e2aae9d18" +
            "1f0e1eb6c2af24b40e50bcde1c935c49a679b5b6dbcef9707b280184b82a29cf" +
            "bfa90505e1e00f714dfdad5c238329ebc7c54ac8e82784d37ec6430b950005b1" +
            "4f6571c50203010001a38201603082015c30130603551d25040c300a06082b06" +
            "010505070303301d0603551d0e041604145971a65a334dda980780ff841ebe87" +
            "f9723241f230510603551d11044a3048a4463044310d300b060355040b13044d" +
            "4f5052313330310603550405132a33313539352b34666166306237312d616433" +
            "372d346161332d613637312d373662633035323334346164301f0603551d2304" +
            "1830168014cb11e8cad2b4165801c9372e331616b94c9a0a1f30560603551d1f" +
            "044f304d304ba049a0478645687474703a2f2f63726c2e6d6963726f736f6674" +
            "2e636f6d2f706b692f63726c2f70726f64756374732f4d6963436f6453696750" +
            "43415f30382d33312d323031302e63726c305a06082b06010505070101044e30" +
            "4c304a06082b06010505073002863e687474703a2f2f7777772e6d6963726f73" +
            "6f66742e636f6d2f706b692f63657274732f4d6963436f645369675043415f30" +
            "382d33312d323031302e637274300d06092a864886f70d010105050003820101" +
            "0031d76e2a12573381d59dc6ebf93ad4444d089eee5edf6a5bb779cf029cbc76" +
            "689e90a19c0bc37fa28cf14dba9539fb0de0e19bf45d240f1b8d88153a7cdbad" +
            "ceb3c96cba392c457d24115426300d0dff47ea0307e5e4665d2c7b9d1da910fa" +
            "1cb074f24f696b9ea92484daed96a0df73a4ef6a1aac4b629ef17cc0147f48cd" +
            "4db244f9f03c936d42d8e87ce617a09b68680928f90297ef1103ba6752adc1e9" +
            "b373a6d263cd4ae23ee4f34efdffa1e0bb02133b5d20de553fa3ae9040313875" +
            "285e04a9466de6f57a7940bd1fcde845d5aee25d3ef575c7e6666360ccd59a84" +
            "878d2430f7ef34d0631db142674a0e4bbf3a0eefb6953aa738e4259208a68866" +
            "82000000000000000000000000").HexToByteArray();

        public static byte[] DssCer = (
            "3082025d3082021da00302010202101e9ae1e91e07de8640ac7af21ac22e8030" +
            "0906072a8648ce380403300e310c300a06035504031303466f6f301e170d3135" +
            "303232343232313734375a170d3136303232343232313734375a300e310c300a" +
            "06035504031303466f6f308201b73082012c06072a8648ce3804013082011f02" +
            "818100871018cc42552d14a5a9286af283f3cfba959b8835ec2180511d0dceb8" +
            "b979285708c800fc10cb15337a4ac1a48ed31394072015a7a6b525986b49e5e1" +
            "139737a794833c1aa1e0eaaa7e9d4efeb1e37a65dbc79f51269ba41e8f0763aa" +
            "613e29c81c3b977aeeb3d3c3f6feb25c270cdcb6aee8cd205928dfb33c44d2f2" +
            "dbe819021500e241edcf37c1c0e20aadb7b4e8ff7aa8fde4e75d02818100859b" +
            "5aeb351cf8ad3fabac22ae0350148fd1d55128472691709ec08481584413e9e5" +
            "e2f61345043b05d3519d88c021582ccef808af8f4b15bd901a310fefd518af90" +
            "aba6f85f6563db47ae214a84d0b7740c9394aa8e3c7bfef1beedd0dafda079bf" +
            "75b2ae4edb7480c18b9cdfa22e68a06c0685785f5cfb09c2b80b1d05431d0381" +
            "8400028180089a43f439b924bef3529d8d6206d1fca56a55caf52b41d6ce371e" +
            "bf07bda132c8eadc040007fcf4da06c1f30504ebd8a77d301f5a4702f01f0d2a" +
            "0707ac1da38dd3251883286e12456234da62eda0df5fe2fa07cd5b16f3638bec" +
            "ca7786312da7d3594a4bb14e353884da0e9aecb86e3c9bdb66fca78ea85e1cc3" +
            "f2f8bf0963300906072a8648ce380403032f00302c021461f6d143a47a4f7e0e" +
            "0ef9848b7f83eacbf83ffd021420e2ac47e656874633e01b0d207a99280c1127" +
            "01").HexToByteArray();

        public static byte[] CertWithPolicies = (
            "308201f33082015ca0030201020210134fb7082cf69bbb4930bfc8e1ca446130" +
            "0d06092a864886f70d0101050500300e310c300a06035504031303466f6f301e" +
            "170d3135303330313232343735385a170d3136303330313034343735385a300e" +
            "310c300a06035504031303466f6f30819f300d06092a864886f70d0101010500" +
            "03818d0030818902818100c252d52fb96658ddbb7d19dd9caaf203ec0376f77c" +
            "3012bd93e14bb22a6ff2b5ce8060a197e3fd8289fbff826746baae0db8d68b47" +
            "a1cf13678717d7db9a16dab028927173a3e843b3a7df8c5a4ff675957ea20703" +
            "6389a60a83d643108bd1293e2135a672a1cff10b7d5b3c78ab44d35e20ca6a5c" +
            "5b6f714c5bfd66ed4307070203010001a3523050301b06092b06010401823714" +
            "02040e1e0c00480065006c006c006f0000301a06092b0601040182371507040d" +
            "300b060357080902010302010230150603551d20040e300c3004060262133004" +
            "06027021300d06092a864886f70d0101050500038181001be04e59fbea63acfb" +
            "c8b6fd3d02dd7442532344cfbc124e924c0bacf23865e4ce2f442ad60ae457d8" +
            "4f7a1f05d50fb867c20e778e412a25237054555669ced01c1ce1ba8e8e57510f" +
            "73e1167c920f78aa5415dc5281f0c761fb25bb1ebc707bc003dd90911e649915" +
            "918cfe4f3176972f8afdc1cccd9705e7fb307a0c17d273").HexToByteArray();

        public static byte[] CertWithTemplateData = (
            "308201dc30820145a00302010202105101b8242daf6cae4c53bac68a948b0130" +
            "0d06092a864886f70d0101050500300e310c300a06035504031303466f6f301e" +
            "170d3135303330313232333133395a170d3136303330313034333133395a300e" +
            "310c300a06035504031303466f6f30819f300d06092a864886f70d0101010500" +
            "03818d0030818902818100a6dcff50bd1fe420301fea5fa56be93a7a53f2599c" +
            "e453cf3422bec797bac0ed78a03090a3754569e6494bcd585ac16a5ea5086344" +
            "3f25521085ca09580579cf0b46bd6e50015319fba5d2bd3724c53b20cdddf604" +
            "74bd7ef426aead9ca5ffea275a4b2b1b6f87c203ab8783559b75e319722886fb" +
            "eb784f5f06823906b2a9950203010001a33b3039301b06092b06010401823714" +
            "02040e1e0c00480065006c006c006f0000301a06092b0601040182371507040d" +
            "300b0603570809020103020102300d06092a864886f70d010105050003818100" +
            "962594da079523c26e2d3fc573fd17189ca33bedbeb2c38c92508fc2a865973b" +
            "e85ba686f765101aea0a0391b22fcfa6c0760eece91a0eb75501bf6871553f8d" +
            "6b089cf2ea63c872e0b4a178795b71826c4569857b45994977895e506dfb8075" +
            "ed1b1096987f2c8f65f2d6bbc788b1847b6ba13bee17ef6cb9c6a3392e13003f").HexToByteArray();

        public static byte[] ComplexNameInfoCert = (
            "308204BE30820427A00302010202080123456789ABCDEF300D06092A864886F70" +
            "D01010505003081A43110300E06035504061307436F756E747279310E300C0603" +
            "550408130553746174653111300F060355040713084C6F63616C6974793111300" +
            "F060355040A13084578616D706C654F31123010060355040B13094578616D706C" +
            "654F55311E301C06035504031315636E2E6973737565722E6578616D706C652E6" +
            "F72673126302406092A864886F70D0109011617697373756572656D61696C4065" +
            "78616D706C652E6F7267301E170D3133313131323134313531365A170D3134313" +
            "231333135313631375A3081A63110300E06035504061307436F756E747279310E" +
            "300C0603550408130553746174653111300F060355040713084C6F63616C69747" +
            "93111300F060355040A13084578616D706C654F31123010060355040B13094578" +
            "616D706C654F55311F301D06035504031316636E2E7375626A6563742E6578616" +
            "D706C652E6F72673127302506092A864886F70D01090116187375626A65637465" +
            "6D61696C406578616D706C652E6F7267305C300D06092A864886F70D010101050" +
            "0034B003048024100DC6FBBDA0300520DFBC9F046CC865D8876AEAC353807EA84" +
            "F58F92FE45EE03C22E970CAF41031D47F97C8A5117C62718482911A8A31B58D92" +
            "328BA3CF9E605230203010001A382023730820233300B0603551D0F0404030200" +
            "B0301D0603551D250416301406082B0601050507030106082B060105050703023" +
            "081FD0603551D120481F53081F28217646E73312E6973737565722E6578616D70" +
            "6C652E6F72678217646E73322E6973737565722E6578616D706C652E6F7267811" +
            "569616E656D61696C31406578616D706C652E6F7267811569616E656D61696C32" +
            "406578616D706C652E6F7267A026060A2B060104018237140203A0180C1669737" +
            "375657275706E31406578616D706C652E6F7267A026060A2B0601040182371402" +
            "03A0180C1669737375657275706E32406578616D706C652E6F7267861F6874747" +
            "03A2F2F757269312E6973737565722E6578616D706C652E6F72672F861F687474" +
            "703A2F2F757269322E6973737565722E6578616D706C652E6F72672F308201030" +
            "603551D110481FB3081F88218646E73312E7375626A6563742E6578616D706C65" +
            "2E6F72678218646E73322E7375626A6563742E6578616D706C652E6F726781157" +
            "3616E656D61696C31406578616D706C652E6F7267811573616E656D61696C3240" +
            "6578616D706C652E6F7267A027060A2B060104018237140203A0190C177375626" +
            "A65637475706E31406578616D706C652E6F7267A027060A2B0601040182371402" +
            "03A0190C177375626A65637475706E32406578616D706C652E6F7267862068747" +
            "4703A2F2F757269312E7375626A6563742E6578616D706C652E6F72672F862068" +
            "7474703A2F2F757269322E7375626A6563742E6578616D706C652E6F72672F300" +
            "D06092A864886F70D0101050500038181005CD44A247FF4DFBF2246CC04D7D57C" +
            "EF2B6D3A4BC83FF685F6B5196B65AFC8F992BE19B688E53E353EEA8B63951EC40" +
            "29008DE8B851E2C30B6BF73F219BCE651E5972E62D651BA171D1DA9831A449D99" +
            "AF4E2F4B9EE3FD0991EF305ADDA633C44EB5E4979751280B3F54F9CCD561AC27D" +
            "3426BC6FF32E8E1AAF9F7C0150A726B").HexToByteArray();

        internal static readonly byte[] MultiPrivateKeyPfx = (
            "30820FD102010330820F9106092A864886F70D010701A0820F8204820F7E3082" +
            "0F7A3082076306092A864886F70D010701A0820754048207503082074C308203" +
            "A9060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E0408ED42EEFCD77BB2EB020207D00482029048F341D409492D23" +
            "D89C0C01DEE7EFFB6715B15D2BB558E9045D635CADFFFEC85C10A4849AB0657D" +
            "A17FE7EC578F779BA2DC129FA959664DC7E85DFD13CAC673E487208FE457223A" +
            "75732915FFCF3FF70F557B0846D62AD507300EA1770EDED82F7D8E6E75075728" +
            "A29D3BF829E75F09EF283A9DDEDDFBABC2E25698DA8C24E4FE34CD43C87554BF" +
            "55B1D4B2B0979F399AEC95B781C62CBE9E412329F9A9BCABF20F716A95F1D795" +
            "7C379A27587F6BBFA44A0B75FAAC15CA3730629C55E87990EE521BC4657EE2A4" +
            "41AF099A226D31707685A89A28EB27CA65512B70DEC09231369AA1A265D4F5C3" +
            "C5D17CB11DB54C70AB83EA28F4740D1F79D490F46F926FB267D5F0E4B2FE096D" +
            "F161A4FF9E9AC068EFCA999B3ED0A3BD05D8D1E3B67CF51E6A478154B427D87D" +
            "C861D0FE2A7A42600483D7B979DC71E8A00D0E805E3BB86E8673234DC1D14987" +
            "99272754A5FD5FEC118CF1E2B2A539B604FED5486A4E4D73FAAFF69023263B84" +
            "6870D6B8DB01E31CB3A1E4BA3588C1FA81C786745A33B95573D5381AB307827A" +
            "549A36AF535FD05E1247BB92C6C6FCB0E76E87F2E4C8136F37C9C19BE3001F59" +
            "FC5CB459C620B8E73711BF102D78F665F40E4D1A341370BC1FB7A5567C29359C" +
            "FFB938237002904BE59F5605AF96E8A670E2248AB71D27FE63E327077144F095" +
            "4CA815E0284E2FF5E1A11B2946276A99B91BF138A79B057436798AF72FD86842" +
            "881C5A5ECDA8A961A21553CC930703047F1F45699CEFEF26AAB6B7DBC65C8C62" +
            "4CA3286094596F2AA48268B9F5411058613185507332833AFB312D5780CEFF96" +
            "6DD05A2CB6E1B252D9656D8E92E63E6C0360F119232E954E11DE777D2DE1C208" +
            "F704DDB16E1351F49B42A859E3B6B2D94E1E2B3CD97F06B1123E9CCA049201E6" +
            "DB7273C0BDE63CC93181DF301306092A864886F70D0109153106040401000000" +
            "305B06092A864886F70D010914314E1E4C007B00310036004200340039004300" +
            "320045002D0036004400390043002D0034003200440042002D00410034004500" +
            "39002D003000320036003000430030004100450032003600300034007D306B06" +
            "092B0601040182371101315E1E5C004D006900630072006F0073006F00660074" +
            "00200045006E00680061006E006300650064002000430072007900700074006F" +
            "0067007200610070006800690063002000500072006F00760069006400650072" +
            "002000760031002E00303082039B060B2A864886F70D010C0A0102A08202B630" +
            "8202B2301C060A2A864886F70D010C0103300E04081F85B7ED57F6F934020207" +
            "D00482029051A5ADA683AAE06A699761CCF05CB081A4398A7B1256A25084DBE1" +
            "115BFAB07A5A9146BC22F2E4223FF25BCA1836AE218691815F20A27A1B98D1FC" +
            "78F84AFA7E90A55954EE5BEA47FFA35928A990CB47346767F6F4212DBCD03FFF" +
            "1E4D137979006B46B19A9FC3BC9B5036ED6F8582E2007D08DB94B2B576E15471" +
            "9CAC90DFB6F238CA875FCBEBCF9E9F933E4451E6A2B60C2A0A8A35B5FD20E5DD" +
            "A000008DCCE95BBDF604A8F93001F594E402FF8649A6582DE5901EDF9DED7D6F" +
            "9657C5A184D82690EFCFB2F25BFCE02BC56F0FF00595996EBF1BA25475AB6134" +
            "61280DD641186237D8A3AB257BD6FB1BDC3768B00719D233E0D5FD26D08BA6EA" +
            "B29D732B990FB9423E643E4663ABBA0D8885DD2A276EE02C92778261C7853F70" +
            "8E2B9AF8D2E96416F676D0191BD24D0C8430BD419049F43C8E2A0C32F862207B" +
            "3DA661577CE5933460D0EF69FAD7323098B55FEF3A9955FE632FBCE8452BB5F3" +
            "430AE2A9021EBF756CC7FDFC3E63581C8B0D7AB77760F447F868B5923614DAA9" +
            "C36AEBC67DC854B93C38E8A6D3AC11B1EE1D02855CE96ADEB840B626BFC4B3BF" +
            "D6487C9073F8A15F55BA945D58AD1636A7AED476EBDB5227A71144BF8745192E" +
            "F5CD177818F61836717ED9EB0A83BEEE582ADEDD407035E453083B17E7C23700" +
            "9D9F04F355CEAB0C0E9AD6F13A3B54459FA05B19E02275FE2588258B63A125F5" +
            "49D1B44C827CDC94260A02F4A1B42A30E675B9760D876685D6CA05C25803BDE1" +
            "F33D325CF6020A662B0F5DCCC8D77B941B273AC462F0D3E050CEB5AEF7107C45" +
            "372F7063EF1AB420CA555A6C9BE6E1067966755584346CDDE7C05B6132E553B1" +
            "1C374DB90B54E5C096062349A1F6CB78A1A2D995C483541750CFA956DEA0EB36" +
            "67DE7AD78931C65B6E039B5DE461810B68C344D2723181D1301306092A864886" +
            "F70D0109153106040402000000305B06092A864886F70D010914314E1E4C007B" +
            "00390044004500340033003500380036002D0039003100320043002D00340036" +
            "00370036002D0042003500410041002D00420046004200360030003900370030" +
            "0035003800350041007D305D06092B060104018237110131501E4E004D006900" +
            "630072006F0073006F006600740020005300740072006F006E00670020004300" +
            "72007900700074006F0067007200610070006800690063002000500072006F00" +
            "7600690064006500723082080F06092A864886F70D010706A0820800308207FC" +
            "020100308207F506092A864886F70D010701301C060A2A864886F70D010C0106" +
            "300E04089ADEE71816BCD023020207D0808207C851AA1EA533FECABB26D3846F" +
            "AEE8DEDB919C29F8B98BBBF785BC306C12A8ACB1437786C4689161683718BB7E" +
            "40EB60D9BE0C87056B5ECF20ACCB8BF7F36033B8FCB84ED1474E97DE0A8709B5" +
            "63B6CF8E69DF4B3F970C92324946723C32D08B7C3A76C871C6B6C8C56F2D3C4C" +
            "00B8A809E65A4EB5EFECC011E2B10F0E44ECDA07B325417B24924080844F6D7F" +
            "1F6E420346EA85825EB830C7E05A5383412A9502A51F1AC07F315ADE357F1F9F" +
            "B2E6427976E78B8FF9CD6C2F9841F2D84658AC8747694EFD0C451B7AC5B83D5F" +
            "0780808417501666BB452B53CEB0698162D94541DE181A7968DB139F17A1076E" +
            "DEB70B38B8881DBC6DE2B694070A5A1AA71E4CDFBF7F4D5DBCF16646768364D3" +
            "C74FA212E40CBE3BE7C51A74D271164D00E89F997FD418C51A7C2D73130D7C6F" +
            "CAA2CA65082CE38BFB753BB30CC71656529E8DBA4C4D0B7E1A79CF2A052FFEFA" +
            "2DEE3373115472AFD1F40A80B23AA6141D5CDE0A378FE6210D4EE69B8771D3E1" +
            "92FD989AEC14C26EA4845D261B8A45ABC1C8FA305449DCDEDA9882DD4DDC69B2" +
            "DE315645FBC3EE52090907E7687A22A63F538E030AB5A5413CA415F1D70E70CB" +
            "567261FB892A8B3BAFC72D632CD2FDCC0559E01D5C246CC27C934863CCFA5249" +
            "0E1F01D8D2D0AF2587E4D04011140A494FFA3CA42C5F645B94EE30100DE019B2" +
            "7F66FFC035E49A65B2A3F6CB14EB1E2FFF1F25B5C87481BD8506F307E0B042A2" +
            "C85B99ECA520B4AAC7DFF2B11C1213E4128A01765DDB27B867336B8CCF148CE7" +
            "38465D46E7A0BEA466CD8BBCCE2E11B16E0F9D24FF2F2D7C9F852779ADBB818F" +
            "87E4AFF7C21A9C2BC20D38209322A34B0B393B187C96583D3D73D9440F994B2F" +
            "320D3274848AB7167942179CFF725C2C7556CCC289A5E788C5B863E6FCDD5E4B" +
            "87E41458BEB3F43D14C7E5196C38CA36322F8B83064862178D58925AEF34F444" +
            "A31A4FB18431D7D37C65ED519643BC7BD025F801390430022253AAFCEA670726" +
            "512C3532EA9F410DB8AA6628CC455E4AB3F478A6981DB9180B7A2A24B365F375" +
            "54CE04B08F22B3539D98BF9A1AC623BBF9A08DBEC951E9730C131802B2C40750" +
            "AAE6A791B3219A96A5BAC7AE17A2F7EA02FF66D6FB36C2E6B6AB90D821A6322B" +
            "F3E8D82969756A474551DB9EAA8C587FC878F996F5FA1E1C39E983F164B0A678" +
            "97EB3755C378807FFDFE964C5C0F290784A08E8C925E85775A9B892E278F68C3" +
            "C1DE72622AC10EA56D88C909EF4AC9F47ED61376737C1E43DBF0F89337F0684F" +
            "A0B96E7A993EC328A6A5FBCDCB809ACBFDAE4ECE192A45480104ED12820238AB" +
            "6AC9C88CC9A82585FD29A81A7BC5BC591738A4D49A86D06B4E18BDC83DFFAA60" +
            "D8A0D4F70CC63D4E83812CB6753F3744545592D04223793E5B305125AAD8807A" +
            "753D235769BD0280E2DE808B0CEE2B98B0F5562FF9EF68161A6B7E08C8B10576" +
            "6EBCFC44AC858B1A89E34C099B194A8B24D1DBABC13909EFAF5B9A9E77AEAF7D" +
            "D9BE772FA01AB9518EB8864AE6D07D7DD7451797541D2F723BC71A9C14ED1D81" +
            "1594E2C4A57017D4CB90FD82C195FA9B823DF1E2FFD965E3139F9A6E8AAC36FA" +
            "39CFA4C52E85D2A661F9D0D466720C5AB7ECDE968FF51B535B019A3E9C76058E" +
            "6F673A49CDD89EA7EC998BDADE71186EA084020A897A328753B72E213A9D8244" +
            "3F7E34D94508199A2A63E71A12BD441C132201E9A3829B2727F23E65C519F4DA" +
            "2C40162A3A501B1BD57568ED75447FEAF8B42988CE25407644BFA0B76059D275" +
            "EC994BB336055E271751B32233D79A6E5E3AA700F3803CCA50586D28934E3D41" +
            "35FA043AF7DFAB977477283602B1739C4AF40E3856E75C34EB98C69A928ADE05" +
            "B67A679630EFA14E64B2957EDD1AB4EC0B0E7BC38D4851EBF6792833EACB62FB" +
            "6C862B089E3066AE5EAAFD2A8B7FC712DE9BD2F488222EEB1FB91B4E57C2D240" +
            "92818965621C123280453EDCFA2EC9D9B50AFA437D1ED09EC36FD232B169ED30" +
            "1E0DB0BABE562B67130F90EBC85D325A90931A5B5A94736A4B3AADB8CA295F59" +
            "AF7FF08CCFADE5AFBBC2346BC6D78D9E5F470E9BDFF547F2574B10A48DD9D56B" +
            "5B03E9E24D65C367B6E342A26A344111A66B1908EDAECD0834930DA74E1CFE2E" +
            "4B0636A7C18E51A27AD21992A2DCF466BAACAC227B90B5E61BED799C97DEE7ED" +
            "B33CCAF5DAD7AAD3CACCDE59478CF69AE64B9065FCB436E1993514C42872DD48" +
            "6ABB75A07A4ED46CDF0E12C0D73FAB83564CF1A814791971EC9C7C6A08A13CE0" +
            "453C2C3236C8B2E146D242E3D37A3ECF6C350D0B2AB956CB21057FDC630750A7" +
            "1C61C66DE3D4A6DB187BEE2F86DEB93E723C5943EA17E699E93555756920416B" +
            "D6B267A4CFAC4EE90E96A6419302B4C0A3B9705509CA09EE92F184FD2817BA09" +
            "BE29E465909DB6C93E3C1CAF6DC29E1A5838F3C32CCB220235EF829CD21D1B3E" +
            "960518A80D08AE7FF08D3AFB7451C823E9B8D49DAF66F503E4AE5399FECFC958" +
            "429D758C06EFF8338BC02457F6FE5053AA3C2F27D360058FD935663B55F026B5" +
            "04E39D86E7CE15F04B1C62BBFA0B1CA5E64FF0BD088D94FB1518E05B2F40BF9D" +
            "71C61FC43E3AF8440570C44030F59D14B8858B7B8506B136E7E39BB04F9AFEAF" +
            "2FA292D28A8822046CEFDE381F2399370BDE9B97BC700418585C31E9C353635A" +
            "DAA6A00A833899D0EDA8F5FFC558D822AEB99C7E35526F5297F333F9E758D4CD" +
            "53277316608B1F7DB6AC71309A8542A356D407531BA1D3071BA9DC02AE91C7DF" +
            "2561AEBC3845A118B00D21913B4A401DDDC40CE983178EF26C4A41343037301F" +
            "300706052B0E03021A041438351C5D7948F9BEA3ACECC0F54AF460EC01093B04" +
            "14B610EC75D16EA23BF253AAD061FAC376E1EAF684").HexToByteArray();

        internal static readonly byte[] EmptyPfx = (
            "304F020103301106092A864886F70D010701A004040230003037301F30070605" +
            "2B0E03021A0414822078BC83E955E314BDA908D76D4C5177CC94EB0414711018" +
            "F2897A44A90E92779CB655EA11814EC598").HexToByteArray();

        internal const string ChainPfxPassword = "test";

        internal static readonly byte[] ChainPfxBytes = (
            "308213D80201033082139406092A864886F70D010701A0821385048213813082" +
            "137D308203C606092A864886F70D010701A08203B7048203B3308203AF308203" +
            "AB060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E040811E8B9808BA6E96C020207D004820290D11DA8713602105C" +
            "95792D65BCDFC1B7E3708483BF6CD83008082F89DAE4D003F86081B153BD4D4A" +
            "C122E802752DEA29F07D0B7E8F0FB8A762B4CAA63360F9F72CA5846771980A6F" +
            "AE2643CD412E6E4A101625371BBD48CC6E2D25191D256B531B06DB7CDAC04DF3" +
            "E10C6DC556D5FE907ABF32F2966A561C988A544C19B46DF1BE531906F2CC2263" +
            "A301302A857075C7A9C48A395241925C6A369B60D176419D75E320008D5EFD91" +
            "5257B160F6CD643953E85F19EBE4E4F72B9B787CF93E95F819D1E43EF01CCFA7" +
            "48F0E7260734EA9BC6039BA7557BE6328C0149718A1D9ECF3355082DE697B6CD" +
            "630A9C224D831B7786C7E904F1EF2D9D004E0E825DD74AC4A576CDFCA7CECD14" +
            "D8E2E6CCAA3A302871AE0BA979BB25559215D771FAE647905878E797BBA9FC62" +
            "50F30F518A8008F5A12B35CE526E31032B56EFE5A4121E1E39DC7339A0CE8023" +
            "24CDDB7E9497BA37D8B9F8D826F901C52708935B4CA5B0D4D760A9FB33B0442D" +
            "008444D5AEB16E5C32187C7038F29160DD1A2D4DB1F9E9A6C035CF5BCED45287" +
            "C5DEBAB18743AAF90E77201FEA67485BA3BBCE90CEA4180C447EE588AC19C855" +
            "638B9552D47933D2760351174D9C3493DCCE9708B3EFE4BE398BA64051BF52B7" +
            "C1DCA44D2D0ED5A6CFB116DDA41995FA99373C254F3F3EBF0F0049F1159A8A76" +
            "4CFE9F9CC56C5489DD0F4E924158C9B1B626030CB492489F6AD0A9DCAF3E141D" +
            "B4D4821B2D8A384110B6B0B522F62A9DC0C1315A2A73A7F25F96C530E2F700F9" +
            "86829A839B944AE6758B8DD1A1E9257F91C160878A255E299C18424EB9983EDE" +
            "6DD1C5F4D5453DD5A56AC87DB1EFA0806E3DBFF10A9623FBAA0BAF352F50AB5D" +
            "B16AB1171145860D21E2AB20B45C8865B48390A66057DE3A1ABE45EA65376EF6" +
            "A96FE36285C2328C3181E1301306092A864886F70D0109153106040401000000" +
            "305D06092A864886F70D01091431501E4E006C0065002D006100340034003100" +
            "30003300610064002D0033003500620032002D0034003800340061002D003900" +
            "3600610036002D00650030006600610036006600330035006500650065003230" +
            "6B06092B0601040182371101315E1E5C004D006900630072006F0073006F0066" +
            "007400200045006E00680061006E006300650064002000430072007900700074" +
            "006F0067007200610070006800690063002000500072006F0076006900640065" +
            "0072002000760031002E003030820FAF06092A864886F70D010706A0820FA030" +
            "820F9C02010030820F9506092A864886F70D010701301C060A2A864886F70D01" +
            "0C0106300E0408FFCC41FD8C8414F6020207D080820F68092C6010873CF9EC54" +
            "D4676BCFB5FA5F523D03C981CB4A3DC096074E7D04365DDD1E80BF366B8F9EC4" +
            "BC056E8CE0CAB516B9C28D17B55E1EB744C43829D0E06217852FA99CCF549617" +
            "6DEF9A48967C1EEB4A384DB7783E643E35B5B9A50533B76B8D53581F02086B78" +
            "2895097860D6CA512514E10D004165C85E561DF5F9AEFD2D89B64F178A7385C7" +
            "FA40ECCA899B4B09AE40EE60DAE65B31FF2D1EE204669EFF309A1C7C8D7B0751" +
            "AE57276D1D0FB3E8344A801AC5226EA4ED97FCD9399A4EB2E778918B81B17FE4" +
            "F65B502595195C79E6B0E37EB8BA36DB12435587E10037D31173285D45304F6B" +
            "0056512B3E147D7B5C397709A64E1D74F505D2BD72ED99055161BC57B6200F2F" +
            "48CF128229EFBEBFC2707678C0A8C51E3C373271CB4FD8EF34A1345696BF3950" +
            "E8CE9831F667D68184F67FE4D30332E24E5C429957694AF23620EA7742F08A38" +
            "C9A517A7491083A367B31C60748D697DFA29635548C605F898B64551A48311CB" +
            "2A05B1ACA8033128D48E4A5AA263D970FE59FBA49017F29049CF80FFDBD19295" +
            "B421FEFF6036B37D2F8DC8A6E36C4F5D707FB05274CC0D8D94AFCC8C6AF546A0" +
            "CF49FBD3A67FB6D20B9FE6FDA6321E8ABF5F7CC794CFCC46005DC57A7BAFA899" +
            "54E43230402C8100789F11277D9F05C78DF0509ECFBF3A85114FD35F4F17E798" +
            "D60C0008064E2557BA7BF0B6F8663A6C014E0220693AE29E2AB4BDE5418B6108" +
            "89EC02FF5480BD1B344C87D73E6E4DB98C73F881B22C7D298059FE9D7ADA2192" +
            "BB6C87F8D25F323A70D234E382F6C332FEF31BB11C37E41903B9A59ADEA5E0CB" +
            "AB06DFB835257ABC179A897DEAD9F19B7DF861BE94C655DC73F628E065F921E5" +
            "DE98FFCBDF2A54AC01E677E365DD8B932B5BDA761A0032CE2127AB2A2B9DCB63" +
            "F1EA8A51FC360AB5BC0AD435F21F9B6842980D795A6734FDB27A4FA8209F7362" +
            "DD632FC5FB1F6DE762473D6EA68BFC4BCF983865E66E6D93159EFACC40AB31AA" +
            "178806CF893A76CAAA3279C988824A33AF734FAF8E21020D988640FAB6DB10DF" +
            "21D93D01776EEA5DAECF695E0C690ED27AD386E6F2D9C9482EA38946008CCB8F" +
            "0BD08F9D5058CF8057CA3AD50BB537116A110F3B3ACD9360322DB4D242CC1A6E" +
            "15FA2A95192FC65886BE2672031D04A4FB0B1F43AE8476CF82638B61B416AA97" +
            "925A0110B736B4D83D7977456F35D947B3D6C9571D8E2DA0E9DEE1E665A84425" +
            "9C17E01E044FAB898AA170F99157F7B525D524B01BD0710D23A7689A6157038A" +
            "0697BD48FFE0253ABD6F862093574B2FC9BA38E1A6EC60AF187F10D79FF71F7C" +
            "50E87A07CC0A51099899F7336FE742ADEF25E720B8E0F8781EC7957D414CF5D4" +
            "4D6998E7E35D2433AFD86442CCA637A1513BE3020B5334614277B3101ED7AD22" +
            "AFE50DE99A2AD0E690596C93B881E2962D7E52EE0A770FAF6917106A8FF0298D" +
            "F38D6DE926C30834C5D96854FFD053BDB020F7827FB81AD04C8BC2C773B2A59F" +
            "DD6DDF7298A052B3486E03FECA5AA909479DDC7FED972192792888F49C40F391" +
            "0140C5BE264D3D07BEBF3275117AF51A80C9F66C7028A2C3155414CF93999726" +
            "8A1F0AA9059CC3AA7C8BBEF880187E3D1BA8978CBB046E43289A020CAE11B251" +
            "40E2247C15A32CF70C7AA186CBB68B258CF2397D2971F1632F6EBC4846444DE4" +
            "45673B942F1F110C7D586B6728ECA5B0A62D77696BF25E21ED9196226E5BDA5A" +
            "80ECCC785BEEDE917EBC6FFDC2F7124FE8F719B0A937E35E9A720BB9ED72D212" +
            "13E68F058D80E9F8D7162625B35CEC4863BD47BC2D8D80E9B9048811BDD8CBB7" +
            "0AB215962CD9C40D56AE50B7003630AE26341C6E243B3D12D5933F73F78F15B0" +
            "14C5B1C36B6C9F410A77CA997931C8BD5CCB94C332F6723D53A4CCC630BFC9DE" +
            "96EFA7FDB66FA519F967D6A2DB1B4898BB188DEB98A41FFA7907AE7601DDE230" +
            "E241779A0FDF551FB84D80AAEE3D979F0510CD026D4AE2ED2EFB7468418CCDB3" +
            "BD2A29CD7C7DC6419B4637412304D5DA2DC178C0B4669CA8330B9713A812E652" +
            "E812135D807E361167F2A6814CEF2A8A9591EFE2C18216A517473B9C3BF2B751" +
            "E47844893DA30F7DCD4222D1A55D570C1B6F6A99AD1F9213BA8F84C0B14A6DED" +
            "6A26EAFF8F89DF733EEB44117DF0FD357186BA4A15BD5C669F60D6D4C3402832" +
            "2D4DDF035302131AB6FD08683804CC90C1791182F1AE3281EE69DDBBCC12B81E" +
            "60942FD082286B16BE27DC11E3BB0F18C281E02F3BA66E48C5FD8E8EA3B731BD" +
            "B12A4A3F2D9E1F833DD204372003532E1BB11298BDF5092F2959FC439E6BD2DC" +
            "6C37E3E775DCBE821B9CBB02E95D84C15E736CEA2FDDAD63F5CD47115B4AD552" +
            "27C2A02886CD2700540EBFD5BF18DC5F94C5874972FD5424FE62B30500B1A875" +
            "21EA3798D11970220B2BE7EFC915FCB7A6B8962F09ABA005861E839813EDA3E5" +
            "9F70D1F9C277B73928DFFC84A1B7B0F78A8B001164EB0824F2510885CA269FDC" +
            "BB2C3AE91BDE91A8BBC648299A3EB626E6F4236CCE79E14C803498562BAD6028" +
            "F5B619125F80925A2D3B1A56790795D04F417003A8E9E53320B89D3A3109B19B" +
            "B17B34CC9700DA138FABB5997EC34D0A44A26553153DBCFF8F6A1B5432B15058" +
            "F7AD87C6B37537796C95369DAD53BE5543D86D940892F93983153B4031D4FAB2" +
            "5DAB02C1091ACC1DAE2118ABD26D19435CD4F1A02BDE1896236C174743BCA6A3" +
            "3FB5429E627EB3FD9F513E81F7BD205B81AAE627C69CF227B043722FA0514139" +
            "347D202C9B7B4E55612FC27164F3B5F287F29C443793E22F6ED6D2F353ED82A9" +
            "F33EDBA8F5F1B2958F1D6A3943A9614E7411FDBCA597965CD08A8042307081BA" +
            "C5A070B467E52D5B91CA58F986C5A33502236B5BAE6DB613B1A408D16B29D356" +
            "0F1E94AD840CFA93E83412937A115ABF68322538DA8082F0192D19EAAA41C929" +
            "9729D487A9404ECDB6396DDA1534841EAE1E7884FA43574E213AE656116D9EF7" +
            "591AA7BDE2B44733DFE27AA59949E5DC0EE00FDF42130A748DDD0FB0053C1A55" +
            "986983C8B9CEAC023CAD7EDFFA1C20D3C437C0EF0FC9868D845484D8BE6538EA" +
            "ADA6365D48BA776EE239ED045667B101E3798FE53E1D4B9A2ACBBE6AF1E5C88A" +
            "3FB03AD616404013E249EC34458F3A7C9363E7772151119FE058BD0939BAB764" +
            "A2E545B0B2FDAA650B7E849C8DD4033922B2CAE46D0461C04A2C87657CB4C0FF" +
            "BA23DED69D097109EC8BFDC25BB64417FEEB32842DE3EFEF2BF4A47F08B9FCD1" +
            "907BC899CA9DA604F5132FB420C8D142D132E7E7B5A4BD0EF4A56D9E9B0ACD88" +
            "F0E862D3F8F0440954879FFE3AA7AA90573C6BFDC6D6474C606ACA1CD94C1C34" +
            "04349DD83A639B786AFCDEA1779860C05400E0479708F4A9A0DD51429A3F35FB" +
            "D5FB9B68CECC1D585F3E35B7BBFC469F3EAEEB8020A6F0C8E4D1804A3EB32EB3" +
            "909E80B0A41571B23931E164E0E1D0D05379F9FD3BF51AF04D2BE78BDB84BD78" +
            "7D419E85626297CB35FCFB6ED64042EAD2EBC17BB65677A1A33A5C48ADD28023" +
            "7FB2451D0EFB3A3C32354222C7AB77A3C92F7A45B5FB10092698D88725864A36" +
            "85FBDD0DC741424FCCD8A00B928F3638150892CAAB535CC2813D13026615B999" +
            "77F7B8240E914ACA0FF2DCB1A9274BA1F55DF0D24CCD2BAB7741C9EA8B1ECDE9" +
            "7477C45F88F034FDF73023502944AEE1FF370260C576992826C4B2E5CE992484" +
            "E3B85170FCCAC3413DC0FF6F093593219E637F699A98BD29E8EE4550C128CA18" +
            "2680FDA3B10BC07625734EE8A8274B43B170FC3AEC9AA58CD92709D388E166AB" +
            "4ADFD5A4876DC47C17DE51FDD42A32AF672515B6A81E7ABECFE748912B321AFD" +
            "0CBF4880298DD79403900A4002B5B436230EB6E49192DF49FAE0F6B60EBA75A5" +
            "4592587C141AD3B319129006367E9532861C2893E7A2D0D2832DF4377C31845C" +
            "B02A1D020282C3D2B7F77221F71FEA7FF0A988FEF15C4B2F6637159EEC5752D8" +
            "A7F4AB971117666A977370E754A4EB0DC52D6E8901DC60FCD87B5B6EF9A91AF8" +
            "D9A4E11E2FFDAB55FC11AF6EEB5B36557FC8945A1E291B7FF8931BE4A57B8E68" +
            "F04B9D4A9A02FC61AE913F2E2DDBEE42C065F4D30F568834D5BB15FDAF691F19" +
            "7EF6C25AE87D8E968C6D15351093AAC4813A8E7B191F77E6B19146F839A43E2F" +
            "40DE8BE28EB22C0272545BADF3BD396D383B8DA8388147100B347999DDC4125A" +
            "B0AA1159BC6776BD2BF51534C1B40522D41466F414BDE333226973BAD1E6D576" +
            "39D30AD94BEA1F6A98C047F1CE1294F0067B771778D59E7C722C73C2FF100E13" +
            "603206A694BF0ED07303BE0655DC984CA29893FD0A088B122B67AABDC803E73E" +
            "5729E868B1CA26F5D05C818D9832C70F5992E7D15E14F9775C6AD24907CF2F21" +
            "1CF87167861F94DCF9E3D365CB600B336D93AD44B8B89CA24E59C1F7812C84DB" +
            "E3EE57A536ED0D4BF948F7662E5BCBBB388C72243CFCEB720852D5A4A52F018C" +
            "2C087E4DB43410FE9ABA3A8EF737B6E8FFDB1AB9832EBF606ED5E4BD62A86BBC" +
            "AE115C67682EDEA93E7845D0D6962C146B411F7784545851D2F327BEC7E4344D" +
            "68F137CDA217A3F0FF3B752A34C3B5339C79CB8E1AC690C038E85D6FC1337909" +
            "0198D3555394D7A2159A23BD5EEF06EB0BCC729BB29B5BE911D02DA78FDA56F0" +
            "35E508C722139AD6F25A6C84BED0E98893370164B033A2B52BC40D9BF5163AF9" +
            "650AB55EABB23370492A7D3A87E17C11B4D07A7296273F33069C835FD208BA8F" +
            "989A3CF8659054E2CCCFB0C983531DC6590F27C4A1D2C3A780FE945F7E52BB9F" +
            "FD2E324640E3E348541A620CD62605BBDB284AF97C621A00D5D1D2C31D6BD611" +
            "49137B8A0250BC426417A92445A52574E999FB9102C16671914A1542E92DDE54" +
            "1B2A0457112AF936DA84707CADFEA43BFEDAE5F58859908640420948086E57FF" +
            "D1B867C241D40197CB0D4AD58BB69B3724772E0079406A1272858AAA620668F6" +
            "96955102639F3E95CFFC637EAF8AB54F0B5B2131AB292438D06E15F3826352DE" +
            "DC653DA5A4AACE2BB97061A498F3B6789A2310471B32F91A6B7A9944DDBB7031" +
            "525B3AE387214DC85A1C7749E9168F41272680D0B3C331D61175F23B623EEC40" +
            "F984C35C831268036680DE0821E5DEE5BB250C6984775D49B7AF94057371DB72" +
            "F81D2B0295FC6A51BCD00A697649D4346FDD59AC0DFAF21BFCC942C23C6134FF" +
            "BA2ABABC141FF700B52C5B26496BF3F42665A5B71BAC7F0C19870BD987389023" +
            "9C578CDDD8E08A1B0A429312FB24F151A11E4D180359A7FA043E8155453F6726" +
            "5CB2812B1C98C144E7675CFC86413B40E35445AE7710227D13DC0B5550C87010" +
            "B363C492DA316FB40D3928570BF71BF47638F1401549369B1255DB080E5DFA18" +
            "EA666B9ECBE5C9768C06B3FF125D0E94B98BB24B4FD44E770B78D7B336E0214F" +
            "D72E77C1D0BE9F313EDCD147957E3463C62E753C10BB98584C85871AAEA9D1F3" +
            "97FE9F1A639ADE31D40EAB391B03B588B8B031BCAC6C837C61B06E4B74505247" +
            "4D33531086519C39EDD6310F3079EB5AC83289A6EDCBA3DC97E36E837134F730" +
            "3B301F300706052B0E03021A04143EE801FCFB9F6CD2B975E2B2BB37DA8E6F29" +
            "369B0414DF1D90CD18B3FBC72226B3C66EC2CB1AB351D4D2020207D0").HexToByteArray();

        internal static readonly byte[] Pkcs7ChainDerBytes = (
            "30820E1606092A864886F70D010702A0820E0730820E030201013100300B0609" +
            "2A864886F70D010701A0820DEB3082050B30820474A003020102020A15EAA83A" +
            "000100009291300D06092A864886F70D010105050030818131133011060A0992" +
            "268993F22C6401191603636F6D31193017060A0992268993F22C64011916096D" +
            "6963726F736F667431143012060A0992268993F22C6401191604636F72703117" +
            "3015060A0992268993F22C64011916077265646D6F6E643120301E0603550403" +
            "13174D532050617373706F7274205465737420537562204341301E170D313330" +
            "3131303231333931325A170D3331313231333232323630375A308185310B3009" +
            "060355040613025553310B30090603550408130257413110300E060355040713" +
            "075265646D6F6E64310D300B060355040A130454455354310D300B060355040B" +
            "130454455354311330110603550403130A746573742E6C6F63616C3124302206" +
            "092A864886F70D010901161563726973706F70406D6963726F736F66742E636F" +
            "6D30819F300D06092A864886F70D010101050003818D0030818902818100B406" +
            "851089E9CF7CDB438DD77BEBD819197BEEFF579C35EF9C4652DF9E6330AA7E2E" +
            "24B181C59DA4AF10E97220C1DF99F66CE6E97247E9126A016AC647BD2EFD136C" +
            "31470C7BE01A20E381243BEEC8530B7F6466C50A051DCE37274ED7FF2AFFF4E5" +
            "8AABA61D5A448F4A8A9B3765D1D769F627ED2F2DE9EE67B1A7ECA3D288C90203" +
            "010001A38202823082027E300E0603551D0F0101FF0404030204F0301D060355" +
            "1D250416301406082B0601050507030106082B06010505070302301D0603551D" +
            "0E04160414FB3485708CBF6188F720EF948489405C8D0413A7301F0603551D23" +
            "0418301680146A6678620A4FF49CA8B75FD566348F3371E42B133081D0060355" +
            "1D1F0481C83081C53081C2A081BFA081BC865F687474703A2F2F707074657374" +
            "73756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E636F6D" +
            "2F43657274456E726F6C6C2F4D5325323050617373706F727425323054657374" +
            "25323053756225323043412831292E63726C865966696C653A2F2F5C5C707074" +
            "65737473756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E" +
            "636F6D5C43657274456E726F6C6C5C4D532050617373706F7274205465737420" +
            "5375622043412831292E63726C3082013806082B060105050701010482012A30" +
            "82012630819306082B06010505073002868186687474703A2F2F707074657374" +
            "73756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E636F6D" +
            "2F43657274456E726F6C6C2F70707465737473756263612E7265646D6F6E642E" +
            "636F72702E6D6963726F736F66742E636F6D5F4D5325323050617373706F7274" +
            "2532305465737425323053756225323043412831292E63727430818D06082B06" +
            "01050507300286818066696C653A2F2F5C5C70707465737473756263612E7265" +
            "646D6F6E642E636F72702E6D6963726F736F66742E636F6D5C43657274456E72" +
            "6F6C6C5C70707465737473756263612E7265646D6F6E642E636F72702E6D6963" +
            "726F736F66742E636F6D5F4D532050617373706F727420546573742053756220" +
            "43412831292E637274300D06092A864886F70D0101050500038181009DEBB8B5" +
            "A41ED54859795F68EF767A98A61EF7B07AAC190FCC0275228E4CAD360C9BA98B" +
            "0AE153C75522EEF42D400E813B4E49E7ACEB963EEE7B61D3C8DA05C183471544" +
            "725B2EBD1889877F62134827FB5993B8FDF618BD421ABA18D70D1C5B41ECDD11" +
            "695A48CB42EB501F96DA905471830C612B609126559120F6E18EA44830820358" +
            "308202C1A00302010202101B9671A4BC128B8341B0E314EAD9A191300D06092A" +
            "864886F70D01010505003081A13124302206092A864886F70D01090116156173" +
            "6D656D6F6E406D6963726F736F66742E636F6D310B3009060355040613025553" +
            "310B30090603550408130257413110300E060355040713075265646D6F6E6431" +
            "123010060355040A13094D6963726F736F667431163014060355040B130D5061" +
            "7373706F727420546573743121301F060355040313184D532050617373706F72" +
            "74205465737420526F6F74204341301E170D3035303132363031333933325A17" +
            "0D3331313231333232323630375A3081A13124302206092A864886F70D010901" +
            "161561736D656D6F6E406D6963726F736F66742E636F6D310B30090603550406" +
            "13025553310B30090603550408130257413110300E060355040713075265646D" +
            "6F6E6431123010060355040A13094D6963726F736F667431163014060355040B" +
            "130D50617373706F727420546573743121301F060355040313184D5320506173" +
            "73706F7274205465737420526F6F7420434130819F300D06092A864886F70D01" +
            "0101050003818D0030818902818100C4673C1226254F6BBD01B01D21BB05264A" +
            "9AA5B77AC51748EAC52048706DA6B890DCE043C6426FC44E76D70F9FE3A4AC85" +
            "5F533E3D08E140853DB769EE24DBDB7269FABEC0FDFF6ADE0AA85F0085B78864" +
            "58E7585E433B0924E81600433CB1177CE6AD5F2477B2A0E2D1A34B41F6C6F5AD" +
            "E4A9DD7D565C65F02C2AAA01C8E0C10203010001A3818E30818B301306092B06" +
            "0104018237140204061E0400430041300B0603551D0F040403020186300F0603" +
            "551D130101FF040530030101FF301D0603551D0E04160414F509C1D6267FC39F" +
            "CA1DE648C969C74FB111FE10301206092B060104018237150104050203010002" +
            "302306092B0601040182371502041604147F7A5208411D4607C0057C98F0C473" +
            "07010CB3DE300D06092A864886F70D0101050500038181004A8EAC73D8EA6D7E" +
            "893D5880945E0E3ABFC79C40BFA60A680CF8A8BF63EDC3AD9C11C081F1F44408" +
            "9581F5C8DCB23C0AEFA27571D971DBEB2AA9A1B3F7B9B0877E9311D36098A65B" +
            "7D03FC69A835F6C3096DEE135A864065F9779C82DEB0C777B9C4DB49F0DD11A0" +
            "EAB287B6E352F7ECA467D0D3CA2A8081119388BAFCDD25573082057C308204E5" +
            "A003020102020A6187C7F200020000001B300D06092A864886F70D0101050500" +
            "3081A13124302206092A864886F70D010901161561736D656D6F6E406D696372" +
            "6F736F66742E636F6D310B3009060355040613025553310B3009060355040813" +
            "0257413110300E060355040713075265646D6F6E6431123010060355040A1309" +
            "4D6963726F736F667431163014060355040B130D50617373706F727420546573" +
            "743121301F060355040313184D532050617373706F7274205465737420526F6F" +
            "74204341301E170D3039313032373231333133395A170D333131323133323232" +
            "3630375A30818131133011060A0992268993F22C6401191603636F6D31193017" +
            "060A0992268993F22C64011916096D6963726F736F667431143012060A099226" +
            "8993F22C6401191604636F727031173015060A0992268993F22C640119160772" +
            "65646D6F6E643120301E060355040313174D532050617373706F727420546573" +
            "742053756220434130819F300D06092A864886F70D010101050003818D003081" +
            "8902818100A6A4918F93C5D23B3C3A325AD8EC77043D207A0DDC294AD3F5BDE0" +
            "4033FADD4097BB1DB042B1D3B2F26A42CC3CB88FA9357710147AB4E1020A0DFB" +
            "2597AB8031DB62ABDC48398067EB79E4E2BBE5762F6B4C5EA7629BAC23F70269" +
            "06D46EC106CC6FBB4D143F7D5ADADEDE19B021EEF4A6BCB9D01DAEBB9A947703" +
            "40B748A3490203010001A38202D7308202D3300F0603551D130101FF04053003" +
            "0101FF301D0603551D0E041604146A6678620A4FF49CA8B75FD566348F3371E4" +
            "2B13300B0603551D0F040403020186301206092B060104018237150104050203" +
            "010001302306092B060104018237150204160414A0A485AE8296EA4944C6F6F3" +
            "886A8603FD07472C301906092B0601040182371402040C1E0A00530075006200" +
            "430041301F0603551D23041830168014F509C1D6267FC39FCA1DE648C969C74F" +
            "B111FE103081D60603551D1F0481CE3081CB3081C8A081C5A081C28663687474" +
            "703A2F2F70617373706F72747465737463612E7265646D6F6E642E636F72702E" +
            "6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F4D532532305061" +
            "7373706F727425323054657374253230526F6F7425323043412831292E63726C" +
            "865B66696C653A2F2F50415353504F52545445535443412E7265646D6F6E642E" +
            "636F72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F4D53" +
            "2050617373706F7274205465737420526F6F742043412831292E63726C308201" +
            "4406082B06010505070101048201363082013230819A06082B06010505073002" +
            "86818D687474703A2F2F70617373706F72747465737463612E7265646D6F6E64" +
            "2E636F72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F50" +
            "415353504F52545445535443412E7265646D6F6E642E636F72702E6D6963726F" +
            "736F66742E636F6D5F4D5325323050617373706F727425323054657374253230" +
            "526F6F7425323043412832292E63727430819206082B06010505073002868185" +
            "66696C653A2F2F50415353504F52545445535443412E7265646D6F6E642E636F" +
            "72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F50415353" +
            "504F52545445535443412E7265646D6F6E642E636F72702E6D6963726F736F66" +
            "742E636F6D5F4D532050617373706F7274205465737420526F6F742043412832" +
            "292E637274300D06092A864886F70D010105050003818100C44788F8C4F5C2DC" +
            "84976F66417CBAE19FBFA82C257DA4C7FED6267BC711D113C78B1C097154A62A" +
            "B462ADC84A434AEBAE38DEB9605FAB534A3CAF7B72C199448E58640388911296" +
            "115ED6B3478D0E741D990F2D59D66F12E58669D8983489AB0406E37462164B56" +
            "6AA1D9B273C406FA694A2556D1D3ACE723382C19871B8C143100").HexToByteArray();

        internal static readonly byte[] Pkcs7ChainPemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----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-----END PKCS7-----");

        internal static readonly byte[] Pkcs7EmptyPemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----
MCcGCSqGSIb3DQEHAqAaMBgCAQExADALBgkqhkiG9w0BBwGgAKEAMQA=
-----END PKCS7-----");

        internal static readonly byte[] Pkcs7EmptyDerBytes = (
            "302706092A864886F70D010702A01A30180201013100300B06092A864886F70D" +
            "010701A000A1003100").HexToByteArray();

        internal static readonly byte[] Pkcs7SingleDerBytes = (
            "3082021406092A864886F70D010702A0820205308202010201013100300B0609" +
            "2A864886F70D010701A08201E9308201E530820152A0030201020210D5B5BC1C" +
            "458A558845BFF51CB4DFF31C300906052B0E03021D05003011310F300D060355" +
            "040313064D794E616D65301E170D3130303430313038303030305A170D313130" +
            "3430313038303030305A3011310F300D060355040313064D794E616D6530819F" +
            "300D06092A864886F70D010101050003818D0030818902818100B11E30EA8742" +
            "4A371E30227E933CE6BE0E65FF1C189D0D888EC8FF13AA7B42B68056128322B2" +
            "1F2B6976609B62B6BC4CF2E55FF5AE64E9B68C78A3C2DACC916A1BC7322DD353" +
            "B32898675CFB5B298B176D978B1F12313E3D865BC53465A11CCA106870A4B5D5" +
            "0A2C410938240E92B64902BAEA23EB093D9599E9E372E48336730203010001A3" +
            "46304430420603551D01043B3039801024859EBF125E76AF3F0D7979B4AC7A96" +
            "A1133011310F300D060355040313064D794E616D658210D5B5BC1C458A558845" +
            "BFF51CB4DFF31C300906052B0E03021D0500038181009BF6E2CF830ED485B86D" +
            "6B9E8DFFDCD65EFC7EC145CB9348923710666791FCFA3AB59D689FFD7234B787" +
            "2611C5C23E5E0714531ABADB5DE492D2C736E1C929E648A65CC9EB63CD84E57B" +
            "5909DD5DDF5DBBBA4A6498B9CA225B6E368B94913BFC24DE6B2BD9A26B192B95" +
            "7304B89531E902FFC91B54B237BB228BE8AFCDA264763100").HexToByteArray();

        internal static readonly byte[] Pkcs7SinglePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----
MIICFAYJKoZIhvcNAQcCoIICBTCCAgECAQExADALBgkqhkiG9w0BBwGgggHpMIIB
5TCCAVKgAwIBAgIQ1bW8HEWKVYhFv/UctN/zHDAJBgUrDgMCHQUAMBExDzANBgNV
BAMTBk15TmFtZTAeFw0xMDA0MDEwODAwMDBaFw0xMTA0MDEwODAwMDBaMBExDzAN
BgNVBAMTBk15TmFtZTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAsR4w6odC
SjceMCJ+kzzmvg5l/xwYnQ2Ijsj/E6p7QraAVhKDIrIfK2l2YJtitrxM8uVf9a5k
6baMeKPC2syRahvHMi3TU7MomGdc+1spixdtl4sfEjE+PYZbxTRloRzKEGhwpLXV
CixBCTgkDpK2SQK66iPrCT2VmenjcuSDNnMCAwEAAaNGMEQwQgYDVR0BBDswOYAQ
JIWevxJedq8/DXl5tKx6lqETMBExDzANBgNVBAMTBk15TmFtZYIQ1bW8HEWKVYhF
v/UctN/zHDAJBgUrDgMCHQUAA4GBAJv24s+DDtSFuG1rno3/3NZe/H7BRcuTSJI3
EGZnkfz6OrWdaJ/9cjS3hyYRxcI+XgcUUxq6213kktLHNuHJKeZIplzJ62PNhOV7
WQndXd9du7pKZJi5yiJbbjaLlJE7/CTeayvZomsZK5VzBLiVMekC/8kbVLI3uyKL
6K/NomR2MQA=
-----END PKCS7-----");

        internal static readonly byte[] MicrosoftDotComSslCertBytes = (
            "308205943082047CA00302010202103DF70C5D9903F8D8868B9B8CCF20DF6930" +
            "0D06092A864886F70D01010B05003077310B3009060355040613025553311D30" +
            "1B060355040A131453796D616E74656320436F72706F726174696F6E311F301D" +
            "060355040B131653796D616E746563205472757374204E6574776F726B312830" +
            "260603550403131F53796D616E74656320436C61737320332045562053534C20" +
            "4341202D204733301E170D3134313031353030303030305A170D313631303135" +
            "3233353935395A3082010F31133011060B2B0601040182373C02010313025553" +
            "311B3019060B2B0601040182373C0201020C0A57617368696E67746F6E311D30" +
            "1B060355040F131450726976617465204F7267616E697A6174696F6E31123010" +
            "06035504051309363030343133343835310B3009060355040613025553310E30" +
            "0C06035504110C0539383035323113301106035504080C0A57617368696E6774" +
            "6F6E3110300E06035504070C075265646D6F6E643118301606035504090C0F31" +
            "204D6963726F736F667420576179311E301C060355040A0C154D6963726F736F" +
            "667420436F72706F726174696F6E310E300C060355040B0C054D53434F4D311A" +
            "301806035504030C117777772E6D6963726F736F66742E636F6D30820122300D" +
            "06092A864886F70D01010105000382010F003082010A0282010100A46861FA9D" +
            "5DB763633BF5A64EF6E7C2C2367F48D2D46643A22DFCFCCB24E58A14D0F06BDC" +
            "956437F2A56BA4BEF70BA361BF12964A0D665AFD84B0F7494C8FA4ABC5FCA2E0" +
            "17C06178AEF2CDAD1B5F18E997A14B965C074E8F564970607276B00583932240" +
            "FE6E2DD013026F9AE13D7C91CC07C4E1E8E87737DC06EF2B575B89D62EFE4685" +
            "9F8255A123692A706C68122D4DAFE11CB205A7B3DE06E553F7B95F978EF8601A" +
            "8DF819BF32040BDF92A0DE0DF269B4514282E17AC69934E8440A48AB9D1F5DF8" +
            "9A502CEF6DFDBE790045BD45E0C94E5CA8ADD76A013E9C978440FC8A9E2A9A49" +
            "40B2460819C3E302AA9C9F355AD754C86D3ED77DDAA3DA13810B4D0203010001" +
            "A38201803082017C30310603551D11042A302882117777772E6D6963726F736F" +
            "66742E636F6D821377777771612E6D6963726F736F66742E636F6D3009060355" +
            "1D1304023000300E0603551D0F0101FF0404030205A0301D0603551D25041630" +
            "1406082B0601050507030106082B0601050507030230660603551D20045F305D" +
            "305B060B6086480186F84501071706304C302306082B06010505070201161768" +
            "747470733A2F2F642E73796D63622E636F6D2F637073302506082B0601050507" +
            "020230191A1768747470733A2F2F642E73796D63622E636F6D2F727061301F06" +
            "03551D230418301680140159ABE7DD3A0B59A66463D6CF200757D591E76A302B" +
            "0603551D1F042430223020A01EA01C861A687474703A2F2F73722E73796D6362" +
            "2E636F6D2F73722E63726C305706082B06010505070101044B3049301F06082B" +
            "060105050730018613687474703A2F2F73722E73796D63642E636F6D30260608" +
            "2B06010505073002861A687474703A2F2F73722E73796D63622E636F6D2F7372" +
            "2E637274300D06092A864886F70D01010B0500038201010015F8505B627ED7F9" +
            "F96707097E93A51E7A7E05A3D420A5C258EC7A1CFE1843EC20ACF728AAFA7A1A" +
            "1BC222A7CDBF4AF90AA26DEEB3909C0B3FB5C78070DAE3D645BFCF840A4A3FDD" +
            "988C7B3308BFE4EB3FD66C45641E96CA3352DBE2AEB4488A64A9C5FB96932BA7" +
            "0059CE92BD278B41299FD213471BD8165F924285AE3ECD666C703885DCA65D24" +
            "DA66D3AFAE39968521995A4C398C7DF38DFA82A20372F13D4A56ADB21B582254" +
            "9918015647B5F8AC131CC5EB24534D172BC60218A88B65BCF71C7F388CE3E0EF" +
            "697B4203720483BB5794455B597D80D48CD3A1D73CBBC609C058767D1FF060A6" +
            "09D7E3D4317079AF0CD0A8A49251AB129157F9894A036487").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComIssuerBytes = (
            "3082052B30820413A00302010202107EE14A6F6FEFF2D37F3FAD654D3ADAB430" +
            "0D06092A864886F70D01010B05003081CA310B30090603550406130255533117" +
            "3015060355040A130E566572695369676E2C20496E632E311F301D060355040B" +
            "1316566572695369676E205472757374204E6574776F726B313A303806035504" +
            "0B1331286329203230303620566572695369676E2C20496E632E202D20466F72" +
            "20617574686F72697A656420757365206F6E6C79314530430603550403133C56" +
            "6572695369676E20436C6173732033205075626C6963205072696D6172792043" +
            "657274696669636174696F6E20417574686F72697479202D204735301E170D31" +
            "33313033313030303030305A170D3233313033303233353935395A3077310B30" +
            "09060355040613025553311D301B060355040A131453796D616E74656320436F" +
            "72706F726174696F6E311F301D060355040B131653796D616E74656320547275" +
            "7374204E6574776F726B312830260603550403131F53796D616E74656320436C" +
            "61737320332045562053534C204341202D20473330820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100D8A1657423E82B64E232D7" +
            "33373D8EF5341648DD4F7F871CF84423138EFB11D8445A18718E601626929BFD" +
            "170BE1717042FEBFFA1CC0AAA3A7B571E8FF1883F6DF100A1362C83D9CA7DE2E" +
            "3F0CD91DE72EFB2ACEC89A7F87BFD84C041532C9D1CC9571A04E284F84D935FB" +
            "E3866F9453E6728A63672EBE69F6F76E8E9C6004EB29FAC44742D27898E3EC0B" +
            "A592DCB79ABD80642B387C38095B66F62D957A86B2342E859E900E5FB75DA451" +
            "72467013BF67F2B6A74D141E6CB953EE231A4E8D48554341B189756A4028C57D" +
            "DDD26ED202192F7B24944BEBF11AA99BE3239AEAFA33AB0A2CB7F46008DD9F1C" +
            "CDDD2D016680AFB32F291D23B88AE1A170070C340F0203010001A382015D3082" +
            "0159302F06082B0601050507010104233021301F06082B060105050730018613" +
            "687474703A2F2F73322E73796D63622E636F6D30120603551D130101FF040830" +
            "060101FF02010030650603551D20045E305C305A0604551D2000305230260608" +
            "2B06010505070201161A687474703A2F2F7777772E73796D617574682E636F6D" +
            "2F637073302806082B06010505070202301C1A1A687474703A2F2F7777772E73" +
            "796D617574682E636F6D2F72706130300603551D1F042930273025A023A02186" +
            "1F687474703A2F2F73312E73796D63622E636F6D2F706361332D67352E63726C" +
            "300E0603551D0F0101FF04040302010630290603551D1104223020A41E301C31" +
            "1A30180603550403131153796D616E746563504B492D312D353333301D060355" +
            "1D0E041604140159ABE7DD3A0B59A66463D6CF200757D591E76A301F0603551D" +
            "230418301680147FD365A7C2DDECBBF03009F34339FA02AF333133300D06092A" +
            "864886F70D01010B050003820101004201557BD0161A5D58E8BB9BA84DD7F3D7" +
            "EB139486D67F210B47BC579B925D4F059F38A4107CCF83BE0643468D08BC6AD7" +
            "10A6FAABAF2F61A863F265DF7F4C8812884FB369D9FF27C00A97918F56FB89C4" +
            "A8BB922D1B73B0C6AB36F4966C2008EF0A1E6624454F670040C8075474333BA6" +
            "ADBB239F66EDA2447034FB0EEA01FDCF7874DFA7AD55B75F4DF6D63FE086CE24" +
            "C742A9131444354BB6DFC960AC0C7FD993214BEE9CE4490298D3607B5CBCD530" +
            "2F07CE4442C40B99FEE69FFCB07886516DD12C9DC696FB8582BB042FF76280EF" +
            "62DA7FF60EAC90B856BD793FF2806EA3D9B90F5D3A071D9193864B294CE1DCB5" +
            "E1E0339DB3CB36914BFEA1B4EEF0F9").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComRootBytes = (
            "308204D3308203BBA003020102021018DAD19E267DE8BB4A2158CDCC6B3B4A30" +
            "0D06092A864886F70D01010505003081CA310B30090603550406130255533117" +
            "3015060355040A130E566572695369676E2C20496E632E311F301D060355040B" +
            "1316566572695369676E205472757374204E6574776F726B313A303806035504" +
            "0B1331286329203230303620566572695369676E2C20496E632E202D20466F72" +
            "20617574686F72697A656420757365206F6E6C79314530430603550403133C56" +
            "6572695369676E20436C6173732033205075626C6963205072696D6172792043" +
            "657274696669636174696F6E20417574686F72697479202D204735301E170D30" +
            "36313130383030303030305A170D3336303731363233353935395A3081CA310B" +
            "300906035504061302555331173015060355040A130E566572695369676E2C20" +
            "496E632E311F301D060355040B1316566572695369676E205472757374204E65" +
            "74776F726B313A3038060355040B133128632920323030362056657269536967" +
            "6E2C20496E632E202D20466F7220617574686F72697A656420757365206F6E6C" +
            "79314530430603550403133C566572695369676E20436C617373203320507562" +
            "6C6963205072696D6172792043657274696669636174696F6E20417574686F72" +
            "697479202D20473530820122300D06092A864886F70D01010105000382010F00" +
            "3082010A0282010100AF240808297A359E600CAAE74B3B4EDC7CBC3C451CBB2B" +
            "E0FE2902F95708A364851527F5F1ADC831895D22E82AAAA642B38FF8B955B7B1" +
            "B74BB3FE8F7E0757ECEF43DB66621561CF600DA4D8DEF8E0C362083D5413EB49" +
            "CA59548526E52B8F1B9FEBF5A191C23349D843636A524BD28FE870514DD18969" +
            "7BC770F6B3DC1274DB7B5D4B56D396BF1577A1B0F4A225F2AF1C926718E5F406" +
            "04EF90B9E400E4DD3AB519FF02BAF43CEEE08BEB378BECF4D7ACF2F6F03DAFDD" +
            "759133191D1C40CB7424192193D914FEAC2A52C78FD50449E48D6347883C6983" +
            "CBFE47BD2B7E4FC595AE0E9DD4D143C06773E314087EE53F9F73B8330ACF5D3F" +
            "3487968AEE53E825150203010001A381B23081AF300F0603551D130101FF0405" +
            "30030101FF300E0603551D0F0101FF040403020106306D06082B060105050701" +
            "0C0461305FA15DA05B3059305730551609696D6167652F6769663021301F3007" +
            "06052B0E03021A04148FE5D31A86AC8D8E6BC3CF806AD448182C7B192E302516" +
            "23687474703A2F2F6C6F676F2E766572697369676E2E636F6D2F76736C6F676F" +
            "2E676966301D0603551D0E041604147FD365A7C2DDECBBF03009F34339FA02AF" +
            "333133300D06092A864886F70D0101050500038201010093244A305F62CFD81A" +
            "982F3DEADC992DBD77F6A5792238ECC4A7A07812AD620E457064C5E797662D98" +
            "097E5FAFD6CC2865F201AA081A47DEF9F97C925A0869200DD93E6D6E3C0D6ED8" +
            "E606914018B9F8C1EDDFDB41AAE09620C9CD64153881C994EEA284290B136F8E" +
            "DB0CDD2502DBA48B1944D2417A05694A584F60CA7E826A0B02AA251739B5DB7F" +
            "E784652A958ABD86DE5E8116832D10CCDEFDA8822A6D281F0D0BC4E5E71A2619" +
            "E1F4116F10B595FCE7420532DBCE9D515E28B69E85D35BEFA57D4540728EB70E" +
            "6B0E06FB33354871B89D278BC4655F0D86769C447AF6955CF65D320833A454B6" +
            "183F685CF2424A853854835FD1E82CF2AC11D6A8ED636A").HexToByteArray();

        internal static readonly ECDsaCngKeyValues ECDsaCng256PublicKey = 
            new ECDsaCngKeyValues()
            {
                QX = "448d98ee08aeba0d8b40f3c6dbd500e8b69f07c70c661771655228ea5a178a91".HexToByteArray(),
                QY = "0ef5cb1759f6f2e062021d4f973f5bb62031be87ae915cff121586809e3219af".HexToByteArray(),
                D = "692837e9cf613c0e290462a6f08faadcc7002398f75598d5554698a0cb51cf47".HexToByteArray(),
            };
    
        internal static readonly byte[] ECDsa256Certificate =
             ("308201223081c9a00302010202106a3c9e85ba6af1ac4f08111d8bdda340300906072a8648ce3d0401301431123010060355"
            + "04031309456332353655736572301e170d3135303931303231333533305a170d3136303931303033333533305a3014311230"
            + "10060355040313094563323536557365723059301306072a8648ce3d020106082a8648ce3d03010703420004448d98ee08ae"
            + "ba0d8b40f3c6dbd500e8b69f07c70c661771655228ea5a178a910ef5cb1759f6f2e062021d4f973f5bb62031be87ae915cff"
            + "121586809e3219af300906072a8648ce3d04010349003046022100f221063dca71955d17c8f0e0f63a144c4065578fd9f68e"
            + "1ae6a7683e209ea742022100ed1db6a8be27cfb20ab43e0ca061622ceff26f7249a0f791e4d6be1a4e52adfa").HexToByteArray();
    
        internal static readonly ECDsaCngKeyValues ECDsaCng384PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "c59eca607aa5559e6b2f8ac2eeb12d9ab47f420feabeb444c3f71520d7f2280439979323ab5a67344811d296fef6d1bd".HexToByteArray(),
                QY = "d15f307cc6cc6c8baeeeb168bfb02c34d6eb0621efb3d06ad31c06b29eaf6ec2ec67bf288455e729d82e5a6439f70901".HexToByteArray(),
                D = "f55ba33e28cea32a014e2fe1213bb4d41cef361f1fee022116b15be50feb96bc946b10a46a9a7a94176787e0928a3e1d".HexToByteArray(),
            };
    
        internal static readonly byte[] ECDsa384Certificate =
             ("3082015f3081e6a00302010202101e78eb573e70a2a64744672296988ad7300906072a8648ce3d0401301431123010060355"
            + "04031309456333383455736572301e170d3135303931303231333634365a170d3136303931303033333634365a3014311230"
            + "10060355040313094563333834557365723076301006072a8648ce3d020106052b8104002203620004c59eca607aa5559e6b"
            + "2f8ac2eeb12d9ab47f420feabeb444c3f71520d7f2280439979323ab5a67344811d296fef6d1bdd15f307cc6cc6c8baeeeb1"
            + "68bfb02c34d6eb0621efb3d06ad31c06b29eaf6ec2ec67bf288455e729d82e5a6439f70901300906072a8648ce3d04010369"
            + "003066023100a8fbaeeae61953897eae5f0beeeffaca48e89bc0cb782145f39f4ba5b03390ce6a28e432e664adf5ebc6a802"
            + "040b238b023100dcc19109383b9482fdda68f40a63ee41797dbb8f25c0284155cc4238d682fbb3fb6e86ea0933297e850a26"
            + "16f6c39bbf").HexToByteArray();
    
        internal static readonly ECDsaCngKeyValues ECDsaCng521PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "0134af29d1fe5e581fd2ff6194263abcb6f8cb4d9c08bdb384ede9b8663ae2f4e1af6c85eacc69dc768fbfcd856630792e05484cefb1fefb693081dc6490dac579c0".HexToByteArray(),
                QY = "00bfe103f53cbcb039873b1a3e81a9da9abd71995e722318367281d30b35a338bf356662342b653eff38e85881863b7128ddbb856d8ae158365550bb6330b93d4ef0".HexToByteArray(),
                D = "0153603164bcef5c9f62388d06dcbf5681479be4397c07ff6f44bb848465e3397537d5f61abc7bc9266d4df6bae1df4847fcfd3dabdda37a2fe549b821ea858d088d".HexToByteArray(),
            };
    
        internal static readonly byte[] ECDsa521Certificate =
             ("308201a93082010ca00302010202102c3134fe79bb9daa48df6431f4c1e4f3300906072a8648ce3d04013014311230100603"
            + "5504031309456335323155736572301e170d3135303931303231333832305a170d3136303931303033333832305a30143112"
            + "30100603550403130945633532315573657230819b301006072a8648ce3d020106052b8104002303818600040134af29d1fe"
            + "5e581fd2ff6194263abcb6f8cb4d9c08bdb384ede9b8663ae2f4e1af6c85eacc69dc768fbfcd856630792e05484cefb1fefb"
            + "693081dc6490dac579c000bfe103f53cbcb039873b1a3e81a9da9abd71995e722318367281d30b35a338bf356662342b653e"
            + "ff38e85881863b7128ddbb856d8ae158365550bb6330b93d4ef0300906072a8648ce3d040103818b0030818702420090bdf5"
            + "dfb328501910da4b02ba3ccd41f2bb073608c55f0f2b2e1198496c59b44db9e516a6a63ba7841d22cf590e39d3f09636d0eb"
            + "cd59a92c105f499e1329615602414285111634719b9bbd10eb7d08655b2fa7d7eb5e225bfdafef15562ae2f9f0c6a943a7bd"
            + "f0e39223d807b5e2e617a8e424294d90869567326531bcad0f893a0f3a").HexToByteArray();
    
        internal struct ECDsaCngKeyValues
        {
            public byte[] QX;
            public byte[] QY;
            public byte[] D;
        }
    }
}
