// Copyright (c) 2018 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package storage

import (
	"fmt"
	"sync"
	"sync/atomic"
	"testing"
	"time"

	"github.com/m3db/m3/src/dbnode/namespace"
	"github.com/m3db/m3/src/dbnode/runtime"
	"github.com/m3db/m3/src/dbnode/storage/index"
	"github.com/m3db/m3/src/m3ninx/doc"
	xclock "github.com/m3db/m3/src/x/clock"
	"github.com/m3db/m3/src/x/context"
	"github.com/m3db/m3/src/x/ident"
	xtime "github.com/m3db/m3/src/x/time"

	"github.com/fortytw2/leaktest"
	"github.com/golang/mock/gomock"
	"github.com/m3db/m3/src/dbnode/storage/series"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestShardInsertNamespaceIndex(t *testing.T) {
	defer leaktest.CheckTimeout(t, 2*time.Second)()
	opts := DefaultTestOptions()

	lock := sync.Mutex{}
	indexWrites := []doc.Document{}

	now := time.Now()
	blockSize := namespace.NewIndexOptions().BlockSize()

	blockStart := xtime.ToUnixNano(now.Truncate(blockSize))

	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	idx := NewMocknamespaceIndex(ctrl)
	idx.EXPECT().BlockStartForWriteTime(gomock.Any()).Return(blockStart).AnyTimes()
	idx.EXPECT().WriteBatch(gomock.Any()).Do(
		func(batch *index.WriteBatch) {

			lock.Lock()
			indexWrites = append(indexWrites, batch.PendingDocs()...)
			lock.Unlock()
			for i, e := range batch.PendingEntries() {
				e.OnIndexSeries.OnIndexSuccess(blockStart)
				e.OnIndexSeries.OnIndexFinalize(blockStart)
				batch.PendingEntries()[i].OnIndexSeries = nil
			}
		}).Return(nil).AnyTimes()

	shard := testDatabaseShardWithIndexFn(t, opts, idx)
	shard.SetRuntimeOptions(runtime.NewOptions().SetWriteNewSeriesAsync(false))
	defer shard.Close()

	ctx := context.NewContext()
	defer ctx.Close()

	_, wasWritten, err := shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now, 1.0, xtime.Second, nil, series.WriteOptions{})
	require.NoError(t, err)
	require.True(t, wasWritten)

	_, wasWritten, err = shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now, 2.0, xtime.Second, nil, series.WriteOptions{})
	require.NoError(t, err)
	require.True(t, wasWritten)

	_, wasWritten, err = shard.Write(
		ctx, ident.StringID("baz"), now, 1.0, xtime.Second, nil, series.WriteOptions{})
	require.NoError(t, err)
	require.True(t, wasWritten)

	lock.Lock()
	defer lock.Unlock()

	require.Len(t, indexWrites, 1)
	require.Equal(t, []byte("foo"), indexWrites[0].ID)
	require.Equal(t, []byte("name"), indexWrites[0].Fields[0].Name)
	require.Equal(t, []byte("value"), indexWrites[0].Fields[0].Value)
}

func TestShardAsyncInsertNamespaceIndex(t *testing.T) {
	defer leaktest.CheckTimeout(t, 2*time.Second)()

	opts := DefaultTestOptions()
	lock := sync.RWMutex{}
	indexWrites := []doc.Document{}

	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	idx := NewMocknamespaceIndex(ctrl)
	idx.EXPECT().WriteBatch(gomock.Any()).Do(
		func(batch *index.WriteBatch) {
			lock.Lock()
			indexWrites = append(indexWrites, batch.PendingDocs()...)
			lock.Unlock()
		}).Return(nil).AnyTimes()

	shard := testDatabaseShardWithIndexFn(t, opts, idx)
	shard.SetRuntimeOptions(runtime.NewOptions().SetWriteNewSeriesAsync(true))
	defer shard.Close()

	ctx := context.NewContext()
	defer ctx.Close()
	now := time.Now()
	_, wasWritten, err := shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now, 1.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	_, wasWritten, err = shard.Write(ctx, ident.StringID("bar"), now,
		1.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	_, wasWritten, err = shard.WriteTagged(ctx, ident.StringID("baz"),
		ident.NewTagsIterator(ident.NewTags(
			ident.StringTag("all", "tags"),
			ident.StringTag("should", "be-present"),
		)),
		now, 1.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	for {
		lock.RLock()
		l := len(indexWrites)
		lock.RUnlock()
		if l == 2 {
			break
		}
		time.Sleep(10 * time.Millisecond)
	}
	lock.Lock()
	defer lock.Unlock()

	assert.Len(t, indexWrites, 2)
	for _, w := range indexWrites {
		if string(w.ID) == "foo" {
			assert.Equal(t, 1, len(w.Fields))
			assert.Equal(t, "name", string(w.Fields[0].Name))
			assert.Equal(t, "value", string(w.Fields[0].Value))
		} else if string(w.ID) == "baz" {
			assert.Equal(t, 2, len(w.Fields))
			assert.Equal(t, "all", string(w.Fields[0].Name))
			assert.Equal(t, "tags", string(w.Fields[0].Value))
			assert.Equal(t, "should", string(w.Fields[1].Name))
			assert.Equal(t, "be-present", string(w.Fields[1].Value))
		} else {
			assert.Fail(t, "unexpected write", w)
		}
	}
}

func TestShardAsyncIndexOnlyWhenNotIndexed(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	defer leaktest.CheckTimeout(t, 2*time.Second)()

	var numCalls int32
	opts := DefaultTestOptions()
	blockSize := time.Hour
	now := time.Now()
	nextWriteTime := now.Truncate(blockSize)
	idx := NewMocknamespaceIndex(ctrl)
	idx.EXPECT().BlockStartForWriteTime(gomock.Any()).
		DoAndReturn(func(t time.Time) xtime.UnixNano {
			return xtime.ToUnixNano(t.Truncate(blockSize))
		}).
		AnyTimes()
	idx.EXPECT().WriteBatch(gomock.Any()).Do(
		func(batch *index.WriteBatch) {
			if batch.Len() == 0 {
				panic(fmt.Errorf("expected batch of len 1")) // panic to avoid goroutine exit from require
			}
			onIdx := batch.PendingEntries()[0].OnIndexSeries
			onIdx.OnIndexSuccess(xtime.ToUnixNano(nextWriteTime)) // i.e. mark that the entry should not be indexed for an hour at least
			onIdx.OnIndexFinalize(xtime.ToUnixNano(nextWriteTime))
			current := atomic.AddInt32(&numCalls, 1)
			if current > 1 {
				panic("only need to index when not-indexed")
			}
		}).Return(nil)

	shard := testDatabaseShardWithIndexFn(t, opts, idx)
	shard.SetRuntimeOptions(runtime.NewOptions().SetWriteNewSeriesAsync(true))
	defer shard.Close()

	ctx := context.NewContext()
	defer ctx.Close()

	_, wasWritten, err := shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now, 1.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	for {
		if l := atomic.LoadInt32(&numCalls); l == 1 {
			break
		}
		time.Sleep(10 * time.Millisecond)
	}

	// ensure we don't index once we have already indexed
	_, wasWritten, err = shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now.Add(time.Second), 2.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	// ensure attempting to write same point yields false and does not write
	_, wasWritten, err = shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now.Add(time.Second), 2.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.False(t, wasWritten)

	l := atomic.LoadInt32(&numCalls)
	assert.Equal(t, int32(1), l)

	entry, _, err := shard.tryRetrieveWritableSeries(ident.StringID("foo"))
	assert.NoError(t, err)
	assert.True(t, entry.IndexedForBlockStart(xtime.ToUnixNano(nextWriteTime)))
}

func TestShardAsyncIndexIfExpired(t *testing.T) {
	defer leaktest.CheckTimeout(t, 2*time.Second)()

	var numCalls int32

	// Make now not rounded exactly to the block size
	blockSize := time.Minute
	now := time.Now().Truncate(blockSize).Add(time.Second)

	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	idx := NewMocknamespaceIndex(ctrl)
	idx.EXPECT().BlockStartForWriteTime(gomock.Any()).
		DoAndReturn(func(t time.Time) xtime.UnixNano {
			return xtime.ToUnixNano(t.Truncate(blockSize))
		}).
		AnyTimes()
	idx.EXPECT().WriteBatch(gomock.Any()).
		Return(nil).
		Do(func(batch *index.WriteBatch) {
			for _, b := range batch.PendingEntries() {
				blockStart := b.Timestamp.Truncate(blockSize)
				b.OnIndexSeries.OnIndexSuccess(xtime.ToUnixNano(blockStart))
				b.OnIndexSeries.OnIndexFinalize(xtime.ToUnixNano(blockStart))
				atomic.AddInt32(&numCalls, 1)
			}
		}).
		AnyTimes()

	opts := DefaultTestOptions()
	shard := testDatabaseShardWithIndexFn(t, opts, idx)
	shard.SetRuntimeOptions(runtime.NewOptions().SetWriteNewSeriesAsync(true))
	defer shard.Close()

	ctx := context.NewContext()
	defer ctx.Close()

	_, wasWritten, err := shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		now, 1.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	// wait till we're done indexing.
	indexed := xclock.WaitUntil(func() bool {
		return atomic.LoadInt32(&numCalls) == 1
	}, 2*time.Second)
	assert.True(t, indexed)

	// ensure we index because it's expired
	nextWriteTime := now.Add(blockSize)
	_, wasWritten, err = shard.WriteTagged(ctx, ident.StringID("foo"),
		ident.NewTagsIterator(ident.NewTags(ident.StringTag("name", "value"))),
		nextWriteTime, 2.0, xtime.Second, nil, series.WriteOptions{})
	assert.NoError(t, err)
	assert.True(t, wasWritten)

	// wait till we're done indexing.
	reIndexed := xclock.WaitUntil(func() bool {
		return atomic.LoadInt32(&numCalls) == 2
	}, 2*time.Second)
	assert.True(t, reIndexed)

	entry, _, err := shard.tryRetrieveWritableSeries(ident.StringID("foo"))
	assert.NoError(t, err)

	// make sure we indexed the second write
	assert.True(t, entry.IndexedForBlockStart(
		xtime.ToUnixNano(nextWriteTime.Truncate(blockSize))))
}

// TODO(prateek): wire tests above to use the field `ts`
// nolint
type testIndexWrite struct {
	id   ident.ID
	tags ident.Tags
	ts   time.Time
}
