// Copyright (c) 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package static

import (
	"testing"

	"github.com/stretchr/testify/require"
	yaml "gopkg.in/yaml.v2"
)

func TestValidationResultUnmarshalYAML(t *testing.T) {
	inputs := []string{
		"invalid",
		"valid",
	}

	for _, input := range inputs {
		var res ValidationResult
		require.NoError(t, yaml.Unmarshal([]byte(input), &res))
		require.Equal(t, string(res), input)
	}
}

func TestValidationResultUnmarshalYAMLError(t *testing.T) {
	inputs := []string{
		"foo",
		"blah",
	}
	for _, input := range inputs {
		var res ValidationResult
		require.Error(t, yaml.Unmarshal([]byte(input), &res))
	}
}

func TestValidatorAlwaysValid(t *testing.T) {
	cfgStr := `
validationResult: valid
`
	var cfg NamespaceValidatorConfiguration
	require.NoError(t, yaml.Unmarshal([]byte(cfgStr), &cfg))
	validator := cfg.NewNamespaceValidator()

	inputs := []string{
		"foo",
		"bar",
		"baz",
	}
	for _, input := range inputs {
		require.NoError(t, validator.Validate(input))
	}
}

func TestValidatorAlwaysInvalid(t *testing.T) {
	cfgStr := `
validationResult: invalid
`
	var cfg NamespaceValidatorConfiguration
	require.NoError(t, yaml.Unmarshal([]byte(cfgStr), &cfg))
	validator := cfg.NewNamespaceValidator()

	inputs := []string{
		"foo",
		"bar",
		"baz",
	}
	for _, input := range inputs {
		require.Error(t, validator.Validate(input))
	}
}
