// Copyright (c) 2018 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package placement

import (
	"encoding/json"
	"net/http"
	"path"
	"time"

	"github.com/m3db/m3/src/cluster/kv"
	"github.com/m3db/m3/src/query/api/v1/handler"
	"github.com/m3db/m3/src/query/util/logging"
	xhttp "github.com/m3db/m3/src/x/net/http"

	"go.uber.org/zap"
)

const (
	// DeleteAllHTTPMethod is the HTTP method used with this resource.
	DeleteAllHTTPMethod = http.MethodDelete
)

var (
	// DeprecatedM3DBDeleteAllURL is the old url for the handler to delete all placements, maintained
	// for backwards compatibility.
	DeprecatedM3DBDeleteAllURL = path.Join(handler.RoutePrefixV1, PlacementPathName)

	// M3DBDeleteAllURL is the url for the handler to delete all placements (with the DELETE method)
	// for the M3DB service.
	M3DBDeleteAllURL = path.Join(handler.RoutePrefixV1, M3DBServicePlacementPathName)

	// M3AggDeleteAllURL is the url for the handler to delete all placements (with the DELETE method)
	// for the M3Agg service.
	M3AggDeleteAllURL = path.Join(handler.RoutePrefixV1, M3AggServicePlacementPathName)

	// M3CoordinatorDeleteAllURL is the url for the handler to delete all placements (with the DELETE method)
	// for the M3Coordinator service.
	M3CoordinatorDeleteAllURL = path.Join(handler.RoutePrefixV1, M3CoordinatorServicePlacementPathName)
)

// DeleteAllHandler is the handler to delete all placements.
type DeleteAllHandler Handler

// NewDeleteAllHandler returns a new instance of DeleteAllHandler.
func NewDeleteAllHandler(opts HandlerOptions) *DeleteAllHandler {
	return &DeleteAllHandler{HandlerOptions: opts, nowFn: time.Now}
}

func (h *DeleteAllHandler) ServeHTTP(
	svc handler.ServiceNameAndDefaults,
	w http.ResponseWriter,
	r *http.Request,
) {
	var (
		ctx    = r.Context()
		logger = logging.WithContext(ctx, h.instrumentOptions)
		opts   = handler.NewServiceOptions(svc, r.Header, h.m3AggServiceOptions)
	)

	service, err := Service(h.clusterClient, opts, h.nowFn(), nil)
	if err != nil {
		xhttp.Error(w, err, http.StatusInternalServerError)
		return
	}

	if err := service.Delete(); err != nil {
		if err == kv.ErrNotFound {
			logger.Info("cannot delete placement",
				zap.String("service", svc.ServiceName),
				zap.Error(err))
			xhttp.Error(w, err, http.StatusNotFound)
			return
		}
		logger.Error("error deleting placement", zap.Error(err))
		xhttp.Error(w, err, http.StatusInternalServerError)
		return
	}

	json.NewEncoder(w).Encode(struct {
		Deleted bool `json:"deleted"`
	}{
		Deleted: true,
	})
}
