"""This is the only module outside the ``blocks`` package with
dependency on the Blocks framework. It contains the neural machine
translation predictor nmt. Code is partially taken from the neural
machine translation example in blocks.

https://github.com/mila-udem/blocks-examples/tree/master/machine_translation

Note that using this predictor slows down decoding compared to the
original NMT decoding because search cannot be parallelized. However,
it is much more flexible as it can be combined with other predictors.
"""

import copy
import logging

from blocks.search import BeamSearch

from cam.sgnmt import utils
from cam.sgnmt.blocks.model  import LoadNMTUtils, NMTModel
from cam.sgnmt.blocks.sparse_search import SparseBeamSearch
from cam.sgnmt.misc import sparse
from cam.sgnmt.misc.sparse import FlatSparseFeatMap
from cam.sgnmt.misc.trie import SimpleTrie
from cam.sgnmt.predictors.core import Predictor, UnboundedVocabularyPredictor
import numpy as np


NEG_INF = float("-inf")


class MyopticSearch(BeamSearch):
    """This class hacks into blocks beam search to leverage off the 
    initialization routines. Note that this has nothing to do with 
    SGNMTs high level decoding in ``cam.sgnmt.decoding``. We basically
    replace the ``search()`` with single_step_``decoding()`` which 
    generates the posteriors for the next word. Thus, it fits in the 
    predictor framework. We try to use ``BeamSearch`` functionality
    wherever possible.
    """
    def __init__(self, samples):
        """Calls the ``BeamSearch`` constructor """
        super(MyopticSearch, self).__init__(samples)


class MyopticSparseSearch(SparseBeamSearch):
    """Variant of ``MyopticSearch`` for target side sparse features.
    """
    def __init__(self, samples, trg_sparse_feat_map):
        """Calls the ``SparseBeamSearch`` constructor """
        super(MyopticSparseSearch, self).__init__(samples, trg_sparse_feat_map)


class BlocksNMTPredictor(Predictor):
    """This is the neural machine translation predictor. The predicted
    posteriors are equal to the distribution generated by the decoder
    network in NMT. This predictor heavily relies on the NMT example in
    blocks. Note that this predictor cannot be used in combination with
    a target side sparse feature map. See 
    ``BlocksUnboundedNMTPredictor`` for that case.
    """
    
    def __init__(self, nmt_model_path, gnmt_beta, enable_cache, config):
        """Creates a new NMT predictor.
        
        Args:
            nmt_model_path (string):  Path to the NMT model file (.npz)
            gnmt_beta (float): If greater than 0.0, add a Google NMT
                               style coverage penalization term (Wu et
                               al., 2016) to the predictive scores
            enable_cache (bool):  The NMT predictor usually has a very
                                  limited vocabulary size, and a large
                                  number of UNKs in hypotheses. This
                                  enables reusing already computed
                                  predictor states for hypotheses which
                                  differ only by NMT OOV words.
            config (dict): NMT configuration
        
        Raises:
            ValueError. If a target sparse feature map is defined
        """
        super(BlocksNMTPredictor, self).__init__()
        self.gnmt_beta = gnmt_beta
        self.add_gnmt_coverage_term = gnmt_beta > 0.0
        self.config = copy.deepcopy(config)
        self.enable_cache = enable_cache
        self.set_up_predictor(nmt_model_path)
        self.src_eos = self.src_sparse_feat_map.word2dense(utils.EOS_ID)
    
    def set_up_predictor(self, nmt_model_path):
        """Initializes the predictor with the given NMT model. Code 
        following ``blocks.machine_translation.main``. 
        """
        self.src_vocab_size = self.config['src_vocab_size']
        self.trgt_vocab_size = self.config['trg_vocab_size']
        self.nmt_model = NMTModel(self.config)
        self.nmt_model.set_up()
        loader = LoadNMTUtils(nmt_model_path,
                              self.config['saveto'],
                              self.nmt_model.search_model)
        loader.load_weights()
            
        self.best_models = []
        self.val_bleu_curve = []
        self.src_sparse_feat_map = self.config['src_sparse_feat_map'] \
                if self.config['src_sparse_feat_map'] else FlatSparseFeatMap()
        if self.config['trg_sparse_feat_map']:
            logging.fatal("Cannot use bounded vocabulary predictor with "
                          "a target sparse feature map. Ignoring...")
        self.search_algorithm = MyopticSearch(samples=self.nmt_model.samples)
        self.search_algorithm.compile()
        

    def initialize(self, src_sentence):
        """Runs the encoder network to create the source annotations
        for the source sentence. If the cache is enabled, empty the
        cache.
        
        Args:
            src_sentence (list): List of word ids without <S> and </S>
                                 which represent the source sentence.
        """
        self.contexts = None
        self.states = None 
        self.posterior_cache = SimpleTrie()
        self.states_cache = SimpleTrie()
        self.consumed = []
        seq = self.src_sparse_feat_map.words2dense(
                    utils.oov_to_unk(src_sentence,
                                     self.src_vocab_size)) + [self.src_eos]
        if self.src_sparse_feat_map.dim > 1: # sparse src feats
            input_ = np.transpose(np.tile(seq, (1, 1, 1)), (2,0,1))
        else: # word ids on the source side
            input_ = np.tile(seq, (1, 1))
        
        input_values={self.nmt_model.sampling_input: input_}
        self.contexts, self.states, _ = self.search_algorithm.compute_initial_states_and_contexts(
            input_values)
        self.attention_records = (1 + len(src_sentence)) * [0.0]
    
    def is_history_cachable(self):
        """Returns true if cache is enabled and history contains UNK """
        if not self.enable_cache:
            return False
        for w in self.consumed:
            if w == utils.UNK_ID:
                return True
        return False

    def predict_next(self):
        """Uses cache or runs the decoder network to get the 
        distribution over the next target words.
        
        Returns:
            np array. Full distribution over the entire NMT vocabulary
            for the next target token.
        """
        use_cache = self.is_history_cachable()
        if use_cache:
            posterior = self.posterior_cache.get(self.consumed)
            if not posterior is None:
                logging.debug("Loaded NMT posterior from cache for %s" % 
                                self.consumed)
                return self._add_gnmt_beta(posterior)
        # logprobs are negative log probs, i.e. greater than 0
        logprobs = self.search_algorithm.compute_logprobs(self.contexts,
                                                          self.states)
        posterior = np.multiply(logprobs[0], -1.0)
        if use_cache:
            self.posterior_cache.add(self.consumed, posterior)
        return self._add_gnmt_beta(posterior)
    
    def _add_gnmt_beta(self, posterior):
        """Adds the GNMT coverage penalization term to EOS in 
        ``posterior``
        """
        if self.add_gnmt_coverage_term:
            posterior[utils.EOS_ID] += self.gnmt_beta * sum([np.log(max(0.0001,
                                                                        p)) 
                                for p in self.attention_records if p < 1.0])
        return posterior
        
    def get_unk_probability(self, posterior):
        """Returns the UNK probability defined by NMT. """
        return posterior[utils.UNK_ID] if len(posterior) > utils.UNK_ID else NEG_INF
    
    def consume(self, word):
        """Feeds back ``word`` to the decoder network. This includes 
        embedding of ``word``, running the attention network and update
        the recurrent decoder layer.
        """
        if word >= self.trgt_vocab_size:
            word = utils.UNK_ID
        self.consumed.append(word)
        use_cache = self.is_history_cachable()
        if use_cache:
            s = self.states_cache.get(self.consumed)
            if not s is None:
                logging.debug("Loaded NMT decoder states from cache for %s" % 
                                    self.consumed)
                self.states = copy.deepcopy(s)
                return
        self.states.update(self.search_algorithm.compute_next_states(
                self.contexts, self.states, [word]))
        if use_cache:
            self.states_cache.add(self.consumed, copy.deepcopy(self.states))
        if self.add_gnmt_coverage_term: # Keep track of attentions
            for pos,att in enumerate(self.states['weights'][0]):
                self.attention_records[pos] += att
    
    def get_state(self):
        """The NMT predictor state consists of the decoder network 
        state, and (for caching) the current history of consumed words
        """
        return self.states,self.consumed,self.attention_records
    
    def set_state(self, state):
        """Set the NMT predictor state. """
        self.states,self.consumed,self.attention_records = state

    def is_equal(self, state1, state2):
        """Returns true if the history is the same """
        _,consumed1,_ = state1
        _,consumed2,_ = state2
        return consumed1 == consumed2
    

class BlocksUnboundedNMTPredictor(BlocksNMTPredictor,
                                  UnboundedVocabularyPredictor):
    """This is a version of the NMT predictor which assumes an 
    unbounded vocabulary. Therefore, this predictor can only be used 
    when other predictors (like fst) define the words to score. Using
    this predictor is mandatory when a target sparse feature map is
    provided.
    """
    
    def __init__(self, nmt_model_path, gnmt_beta, config):
        """Creates a new NMT predictor with unbounded vocabulary.
        
        Args:
            nmt_model_path (string):  Path to the NMT model file (.npz)
            config (dict): NMT configuration, 
        """
        super(BlocksUnboundedNMTPredictor, self).__init__(nmt_model_path, 
                                                          gnmt_beta,
                                                          False, 
                                                          config)
    
    def set_up_predictor(self, nmt_model_path):
        """Initializes the predictor with the given NMT model. Code 
        following ``blocks.machine_translation.main``. 
        """
        self.src_vocab_size = self.config['src_vocab_size']
        self.trgt_vocab_size = self.config['trg_vocab_size']
        self.nmt_model = NMTModel(self.config)
        self.nmt_model.set_up()
        loader = LoadNMTUtils(nmt_model_path,
                              self.config['saveto'],
                              self.nmt_model.search_model)
        loader.load_weights()
            
        self.best_models = []
        self.val_bleu_curve = []
        self.src_sparse_feat_map = self.config['src_sparse_feat_map'] \
                if self.config['src_sparse_feat_map'] else FlatSparseFeatMap()
        if self.config['trg_sparse_feat_map']:
            self.trg_sparse_feat_map = self.config['trg_sparse_feat_map']
            self.search_algorithm = MyopticSparseSearch(
                                samples=self.nmt_model.samples,
                                trg_sparse_feat_map=self.trg_sparse_feat_map)
        else:
            self.trg_sparse_feat_map = FlatSparseFeatMap()
            self.search_algorithm = MyopticSearch(samples=self.nmt_model.samples)
        self.search_algorithm.compile()

    def predict_next(self, words):
        """Uses cache or runs the decoder network to get the 
        distribution over the next target words.
        
        Returns:
            np array. Full distribution over the entire NMT vocabulary
            for the next target token.
        """
        logprobs = self.search_algorithm.compute_logprobs(self.contexts,
                                                          self.states)
        if self.trg_sparse_feat_map.dim > 1:
            return {w: -sparse.dense_euclidean2(
                                        logprobs[0],
                                        self.trg_sparse_feat_map.word2dense(w))
                        for w in words}
        else:
            # logprobs are negative log probs, i.e. greater than 0
            posterior = np.multiply(logprobs[0], -1.0)
            return {w: posterior[w] for w in words}

    def get_unk_probability(self, posterior):
        """Returns negative inf as this is a unbounded predictor. """
        return NEG_INF
    
    def consume(self, word):
        """Feeds back ``word`` to the decoder network. This includes 
        embedding of ``word``, running the attention network and update
        the recurrent decoder layer.
        """
        if word >= self.trgt_vocab_size:
            word = utils.UNK_ID
        self.consumed.append(word)
        self.states.update(self.search_algorithm.compute_next_states(
                                self.contexts,
                                self.states,
                                [self.trg_sparse_feat_map.word2dense(word)]))

    def is_equal(self, state1, state2):
        """Returns true if the history is the same """
        _,consumed1 = state1
        _,consumed2 = state2
        return consumed1 == consumed2
