<?php

declare(strict_types=1);

namespace Pgs\HashIdBundle\Tests\Attribute;

use ReflectionClass;
use Attribute;
use Pgs\HashIdBundle\Attribute\Hash;
use PHPUnit\Framework\TestCase;

class HashAttributeTest extends TestCase
{
    public function testAttributeWithSingleParameter(): void
    {
        $attribute = new Hash('id');

        self::assertInstanceOf(Hash::class, $attribute);
        self::assertSame(['id'], $attribute->getParameters());
    }

    public function testAttributeWithMultipleParameters(): void
    {
        $attribute = new Hash(['id', 'parentId', 'userId']);

        self::assertInstanceOf(Hash::class, $attribute);
        self::assertSame(['id', 'parentId', 'userId'], $attribute->getParameters());
    }

    public function testAttributeTargets(): void
    {
        $reflectionClass = new ReflectionClass(Hash::class);
        $attributes = $reflectionClass->getAttributes(Attribute::class);

        self::assertCount(1, $attributes);

        $attributeInstance = $attributes[0]->newInstance();
        // Hash attribute supports CLASS and METHOD targets and is repeatable
        self::assertSame(Attribute::TARGET_CLASS | Attribute::TARGET_METHOD | Attribute::IS_REPEATABLE, $attributeInstance->flags);
    }

    public function testAttributeOnMethod(): void
    {
        $controller = new class() {
            #[Hash('id')]
            public function show(int $id): void
            {
            }

            #[Hash(['id', 'categoryId'])]
            public function showInCategory(int $id, int $categoryId): void
            {
            }
        };

        $reflectionClass = new ReflectionClass($controller);

        // Test first method
        $showMethod = $reflectionClass->getMethod('show');
        $hashAttributes = $showMethod->getAttributes(Hash::class);
        self::assertCount(1, $hashAttributes);

        $hashAttribute = $hashAttributes[0]->newInstance();
        self::assertSame(['id'], $hashAttribute->getParameters());

        // Test second method
        $showInCategoryMethod = $reflectionClass->getMethod('showInCategory');
        $hashAttributes = $showInCategoryMethod->getAttributes(Hash::class);
        self::assertCount(1, $hashAttributes);

        $hashAttribute = $hashAttributes[0]->newInstance();
        self::assertSame(['id', 'categoryId'], $hashAttribute->getParameters());
    }

    public function testAttributeAbsenceOnMethod(): void
    {
        $controller = new class() {
            public function index(): void
            {
            }
        };

        $reflectionClass = new ReflectionClass($controller);
        $indexMethod = $reflectionClass->getMethod('index');
        $hashAttributes = $indexMethod->getAttributes(Hash::class);

        self::assertCount(0, $hashAttributes);
    }
}
