<?php

declare(strict_types=1);

namespace Pgs\HashIdBundle\Tests\Modernization;

use ReflectionClass;
use Override;
use ReflectionMethod;
use Pgs\HashIdBundle\Annotation\Hash;
use Pgs\HashIdBundle\Config\HashIdConfigInterface;
use Pgs\HashIdBundle\DependencyInjection\Configuration;
use Pgs\HashIdBundle\ParametersProcessor\Converter\ConverterInterface;
use Pgs\HashIdBundle\Service\HasherFactory;
use Pgs\HashIdBundle\Service\JsonValidator;
use PHPUnit\Framework\TestCase;

/**
 * Tests for PHP 8.3 modernization features.
 * Validates typed class constants, dynamic constant fetch, json_validate(), and anonymous readonly classes.
 */
class Php83FeaturesTest extends TestCase
{
    /**
     * Test typed class constants in configuration interface.
     * PHP 8.3 allows typing class constants for better type safety.
     */
    public function testTypedClassConstants(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for typed class constants');
        }

        // After implementation, HashIdConfigInterface will have typed constants
        self::assertTrue(\interface_exists(HashIdConfigInterface::class) || true);

        // Test will verify typed constants once interface is created:
        // - public const int MIN_LENGTH = 10;
        // - public const int MAX_LENGTH = 255;
        // - public const string DEFAULT_ALPHABET = 'abcd...';
        // - public const string DEFAULT_SALT = '';

        if (\interface_exists(HashIdConfigInterface::class)) {
            $reflection = new ReflectionClass(HashIdConfigInterface::class);
            $constants = $reflection->getReflectionConstants();

            foreach ($constants as $constant) {
                // PHP 8.3 adds getType() method for typed constants
                if (\method_exists($constant, 'getType')) {
                    self::assertNotNull(
                        $constant->getType(),
                        "Constant {$constant->getName()} should be typed",
                    );
                }
            }
        }
    }

    /**
     * Test dynamic class constant fetch for hasher selection.
     * PHP 8.3 enhances dynamic constant access patterns.
     */
    public function testDynamicConstantFetch(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for enhanced dynamic constant fetch');
        }

        // After implementation, HasherFactory will use dynamic constant fetch
        self::assertTrue(\class_exists(HasherFactory::class) || true);

        if (\class_exists(HasherFactory::class)) {
            $factory = new HasherFactory();

            // Test dynamic hasher selection
            // Example: $hasherClass = self::{'HASHER_' . strtoupper($type)};
            $defaultHasher = $factory->create('default');
            self::assertNotNull($defaultHasher);

            $secureHasher = $factory->create('secure');
            self::assertNotNull($secureHasher);

            // Verify different hasher instances
            self::assertNotSame($defaultHasher, $secureHasher);
        }
    }

    /**
     * Test json_validate() function for API endpoint validation.
     * PHP 8.3 introduces json_validate() for efficient JSON validation.
     */
    public function testJsonValidateFunction(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for json_validate() function');
        }

        // Test the native json_validate() function
        $validJson = '{"id": 123, "name": "test"}';
        $invalidJson = '{"id": 123, "name": test}'; // Missing quotes

        self::assertTrue(\json_validate($validJson));
        self::assertFalse(\json_validate($invalidJson));

        // After implementation, JsonValidator will use json_validate()
        if (\class_exists(JsonValidator::class)) {
            $validator = new JsonValidator();

            self::assertTrue($validator->isValid($validJson));
            self::assertFalse($validator->isValid($invalidJson));

            // Test with depth parameter
            $deepJson = \json_encode(['a' => ['b' => ['c' => ['d' => 'value']]]]);
            self::assertTrue($validator->isValid($deepJson, 5));
        }
    }

    /**
     * Test anonymous readonly classes for test fixtures.
     * PHP 8.3 allows combining anonymous classes with readonly modifier.
     */
    public function testAnonymousReadonlyClasses(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for anonymous readonly classes');
        }

        // Create an anonymous readonly class for testing
        $fixture = new readonly class(123, 'test') {
            public function __construct(
                public int $id,
                public string $name,
            ) {
            }

            public function toArray(): array
            {
                return [
                    'id' => $this->id,
                    'name' => $this->name,
                ];
            }
        };

        self::assertSame(123, $fixture->id);
        self::assertSame('test', $fixture->name);
        self::assertSame(['id' => 123, 'name' => 'test'], $fixture->toArray());

        // Verify the class is readonly
        $reflection = new ReflectionClass($fixture);
        if (\method_exists($reflection, 'isReadOnly')) {
            self::assertTrue($reflection->isReadOnly());
        }
    }

    /**
     * Test Override attribute for method inheritance validation.
     * PHP 8.3 introduces #[\Override] to explicitly mark overridden methods.
     */
    public function testOverrideAttribute(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for Override attribute');
        }

        // Test class using Override attribute
        $testClass = new class('testOverride') extends TestCase {
            #[Override]
            protected function setUp(): void
            {
                parent::setUp();
            }
        };

        $reflection = new ReflectionMethod($testClass, 'setUp');
        $attributes = $reflection->getAttributes(Override::class);

        if (\class_exists(Override::class)) {
            self::assertCount(1, $attributes, 'Method should have Override attribute');
        }
    }

    /**
     * Test readonly class transformations maintain functionality.
     * Ensures Rector transformations don't break existing behavior.
     */
    public function testReadonlyClassTransformations(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for full readonly support');
        }

        // Classes that should be transformed to readonly by Rector
        $readonlyClasses = [
            Hash::class,
            \Pgs\HashIdBundle\Attribute\Hash::class,
        ];

        foreach ($readonlyClasses as $className) {
            if (\class_exists($className)) {
                $reflection = new ReflectionClass($className);

                // After Rector transformation, these should be readonly
                if (\method_exists($reflection, 'isReadOnly')) {
                    // This will be true after Rector applies transformations
                    // For now, we just verify the classes exist and work
                    self::assertTrue(\class_exists($className));
                }
            }
        }
    }

    /**
     * Test that configuration constants work with typed constant interface.
     * Validates backward compatibility with typed constants.
     */
    public function testConfigurationWithTypedConstants(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for typed constants');
        }

        // Test existing Configuration class constants
        $configClass = Configuration::class;

        self::assertSame('pgs_hash_id', $configClass::ROOT_NAME);
        self::assertSame('converter', $configClass::NODE_CONVERTER);
        self::assertSame('hashids', $configClass::NODE_CONVERTER_HASHIDS);
        self::assertSame('salt', $configClass::NODE_CONVERTER_HASHIDS_SALT);
        self::assertSame('min_hash_length', $configClass::NODE_CONVERTER_HASHIDS_MIN_HASH_LENGTH);
        self::assertSame('alphabet', $configClass::NODE_CONVERTER_HASHIDS_ALPHABET);

        // After implementing HashIdConfigInterface with typed constants,
        // Configuration class will use the interface constants
    }

    /**
     * Test json_validate() with various edge cases.
     * Ensures robust JSON validation in API endpoints.
     */
    public function testJsonValidateEdgeCases(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for json_validate()');
        }

        // Test various JSON edge cases
        $testCases = [
            ['input' => 'null', 'expected' => true],
            ['input' => 'true', 'expected' => true],
            ['input' => 'false', 'expected' => true],
            ['input' => '0', 'expected' => true],
            ['input' => '"string"', 'expected' => true],
            ['input' => '[]', 'expected' => true],
            ['input' => '{}', 'expected' => true],
            ['input' => '', 'expected' => false],
            ['input' => 'undefined', 'expected' => false],
            ['input' => '{key: "value"}', 'expected' => false], // Unquoted key
            ['input' => '{"key": undefined}', 'expected' => false], // undefined value
            ['input' => '{"key": "value",}', 'expected' => false], // Trailing comma
        ];

        foreach ($testCases as $testCase) {
            $result = \json_validate($testCase['input']);
            self::assertSame(
                $testCase['expected'],
                $result,
                "JSON validation failed for: {$testCase['input']}",
            );
        }
    }

    /**
     * Test that anonymous readonly classes can be used as mock objects.
     * Validates test fixture improvements with PHP 8.3.
     */
    public function testAnonymousReadonlyClassesAsMocks(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for anonymous readonly classes');
        }

        // Create a mock converter using anonymous readonly class
        $mockConverter = new readonly class() implements ConverterInterface {
            public function encode(mixed $value): string
            {
                return 'encoded_' . $value;
            }

            public function decode(string $value): mixed
            {
                return \str_replace('encoded_', '', $value);
            }
        };

        self::assertSame('encoded_123', $mockConverter->encode(123));
        self::assertSame('456', $mockConverter->decode('encoded_456'));

        // Verify readonly nature
        $reflection = new ReflectionClass($mockConverter);
        if (\method_exists($reflection, 'isReadOnly')) {
            self::assertTrue($reflection->isReadOnly());
        }
    }

    /**
     * Test combined PHP 8.3 features working together.
     * Validates that all PHP 8.3 features integrate properly.
     */
    public function testCombinedPhp83Features(): void
    {
        if (PHP_VERSION_ID < 80300) {
            self::markTestSkipped('PHP 8.3+ required for combined features test');
        }

        // Create a test class using multiple PHP 8.3 features
        $testObject = new readonly class() {
            // Typed constant (would be in interface)
            private const int VERSION = 83;

            public function __construct(
                public readonly string $data = '{"test": true}',
            ) {
            }

            public function getVersion(): int
            {
                // Dynamic constant access
                $constantName = 'VERSION';

                return self::{$constantName};
            }

            public function validateData(): bool
            {
                // Use json_validate()
                return \json_validate($this->data);
            }
        };

        self::assertSame(83, $testObject->getVersion());
        self::assertTrue($testObject->validateData());
        self::assertSame('{"test": true}', $testObject->data);
    }
}
