﻿using System.Threading.Tasks;
using FluentAssertions;
using Microsoft.CodeAnalysis;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Uno.Extensions.Core.Tests.Utils;

namespace Uno.Extensions.Core.Tests.PropertySelector;

[TestClass]
public class Given_PS0006
{
	[TestMethod]
	public async Task When_Fail()
	{
		var compilation = GenerationTestHelper.CreateCompilationWithAnalyzers($@"
			using Uno.Extensions.Edition;
			using System.Runtime.CompilerServices;

			namespace TestNamespace;

			public record RootEntity(SubEntity? Child);
			public record SubEntity(string Value);

			public class SUTClass
			{{
				public void Test()
				{{
					SUTMethod(e => e.Child?.Value);
				}}

				public void SUTMethod(PropertySelector<RootEntity, string> selector, [CallerFilePath] string path = """", [CallerLineNumber] int line = -1)
				{{
				}}
			}}
			");

		var diagnostics = await compilation.GetAnalyzerDiagnosticsAsync();
		diagnostics.Length.Should().Be(1);

		var pathDiag = diagnostics[0];
		pathDiag.Id.Should().Be("PS0006");
		pathDiag.Location.GetLineSpan().StartLinePosition.Line.Should().Be(13);
		pathDiag.Location.GetLineSpan().StartLinePosition.Character.Should().Be(20);

		var expectedCode = @"//----------------------
// <auto-generated>
//	Generated by the PropertySelectorsGenerationTool v1. DO NOT EDIT!
//	Manual changes to this file will be overwritten if the code is regenerated.
// </auto-generated>
//----------------------
#pragma warning disable

namespace Tests.__PropertySelectors
{
	/// <summary>
	/// Auto registration class for PropertySelector used in <global namespace>.
	/// </summary>
	[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
	internal static class _14_5
	{
		/// <summary>
		/// Register the value providers for the PropertySelectors used to invoke 'SUTMethod'
		/// in  on line 14.
		/// </summary>
		/// <remarks>
		/// This method is flagged with the [ModuleInitializerAttribute] which means that it will be invoked by the runtime when the module is being loaded.
		/// You should not have to use it at any time.
		/// </remarks>
		[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
		[global::System.Runtime.CompilerServices.ModuleInitializerAttribute]
		internal static void Register()
		{
			global::Uno.Extensions.Edition.PropertySelectors.Register(
				@""selector14"",
				new global::Uno.Extensions.Edition.ValueAccessor<TestNamespace.RootEntity, string>(
			"".Child?.Value"",
			entity => entity.Child?.Value,
			(current_0, updated_2) =>
			{
				var current_1 = current_0.Child/* Rule PS006 failed. */;
				var updated_1 = current_1 with { Value = updated_2 };
				var updated_0 = current_0 with { Child = updated_1 };

				return updated_0;
			}));
		}
	}
}";
		GenerationTestHelper.RunGeneratorTwice(
			compilation.Compilation,
			run1 => GenerationTestHelper.AssertRunReason(run1, IncrementalStepRunReason.New),
			run2 => GenerationTestHelper.AssertRunReason(run2, IncrementalStepRunReason.Cached),
			expectedCode);
	}
}
