﻿using System.Threading.Tasks;
using FluentAssertions;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.Diagnostics;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Uno.Extensions.Core.Tests.Utils;

namespace Uno.Extensions.Core.Tests.PropertySelector;

[TestClass]
public class Given_PS0102
{
	[TestMethod]
	public async Task When_Path()
	{
		var compilation = TestInvoke(@"
			var path = ""42"";
			SUTMethod(e => e.Value, path);");

		var diagnostics = await compilation.GetAnalyzerDiagnosticsAsync();
		diagnostics.Length.Should().Be(1);
		var diag = diagnostics[0];

		diag.Id.Should().Be("PS0102");
		diag.Location.GetLineSpan().StartLinePosition.Line.Should().Be(14);
		diag.Location.GetLineSpan().StartLinePosition.Character.Should().Be(27);

		var expectedCode = @"//----------------------
// <auto-generated>
//	Generated by the PropertySelectorsGenerationTool v1. DO NOT EDIT!
//	Manual changes to this file will be overwritten if the code is regenerated.
// </auto-generated>
//----------------------
#pragma warning disable

namespace Tests.__PropertySelectors
{
	/// <summary>
	/// Auto registration class for PropertySelector used in <global namespace>.
	/// </summary>
	[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
	internal static class _15_3
	{
		/// <summary>
		/// Register the value providers for the PropertySelectors used to invoke 'SUTMethod'
		/// in  on line 15.
		/// </summary>
		/// <remarks>
		/// This method is flagged with the [ModuleInitializerAttribute] which means that it will be invoked by the runtime when the module is being loaded.
		/// You should not have to use it at any time.
		/// </remarks>
		[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
		[global::System.Runtime.CompilerServices.ModuleInitializerAttribute]
		internal static void Register()
		{
			global::Uno.Extensions.Edition.PropertySelectors.Register(
				@""selector15"",
				new global::Uno.Extensions.Edition.ValueAccessor<TestNamespace.Entity, string>(
				"".Value"",
				entity => entity.Value,
				(current, updated_Value) => (current  ?? new(default(string?))) with { Value = updated_Value }));
		}
	}
}";
		GenerationTestHelper.RunGeneratorTwice(
			compilation.Compilation,
			run1 => GenerationTestHelper.AssertRunReason(run1, IncrementalStepRunReason.New),
			run2 => GenerationTestHelper.AssertRunReason(run2, IncrementalStepRunReason.Cached),
			expectedCode);
	}

	[TestMethod]
	public async Task When_Line()
	{
		var compilation = TestInvoke(@"
			var line = 42;
			SUTMethod(e => e.Value, line: line);");

		var diagnostics = await compilation.GetAnalyzerDiagnosticsAsync();
		diagnostics.Length.Should().Be(1);
		var diag = diagnostics[0];

		diag.Id.Should().Be("PS0102");
		diag.Location.GetLineSpan().StartLinePosition.Line.Should().Be(14);
		diag.Location.GetLineSpan().StartLinePosition.Character.Should().Be(27);

		var expectedCode = @"//----------------------
// <auto-generated>
//	Generated by the PropertySelectorsGenerationTool v1. DO NOT EDIT!
//	Manual changes to this file will be overwritten if the code is regenerated.
// </auto-generated>
//----------------------
#pragma warning disable

namespace Tests.__PropertySelectors
{
	/// <summary>
	/// Auto registration class for PropertySelector used in <global namespace>.
	/// </summary>
	[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
	internal static class _15_3
	{
		/// <summary>
		/// Register the value providers for the PropertySelectors used to invoke 'SUTMethod'
		/// in  on line 15.
		/// </summary>
		/// <remarks>
		/// This method is flagged with the [ModuleInitializerAttribute] which means that it will be invoked by the runtime when the module is being loaded.
		/// You should not have to use it at any time.
		/// </remarks>
		[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
		[global::System.Runtime.CompilerServices.ModuleInitializerAttribute]
		internal static void Register()
		{
			global::Uno.Extensions.Edition.PropertySelectors.Register(
				@""selector15"",
				new global::Uno.Extensions.Edition.ValueAccessor<TestNamespace.Entity, string>(
				"".Value"",
				entity => entity.Value,
				(current, updated_Value) => (current  ?? new(default(string?))) with { Value = updated_Value }));
		}
	}
}";
		GenerationTestHelper.RunGeneratorTwice(
			compilation.Compilation,
			run1 => GenerationTestHelper.AssertRunReason(run1, IncrementalStepRunReason.New),
			run2 => GenerationTestHelper.AssertRunReason(run2, IncrementalStepRunReason.Cached),
			expectedCode);
	}

	[TestMethod]
	public async Task When_PathAndLine()
	{
		var compilation = TestInvoke(@"
			var path = ""42"";
			var line = 42;
			SUTMethod(e => e.Value, path, line);");

		var diagnostics = await compilation.GetAnalyzerDiagnosticsAsync();
		diagnostics.Length.Should().Be(2);

		var pathDiag = diagnostics[0];
		pathDiag.Id.Should().Be("PS0102");
		pathDiag.Location.GetLineSpan().StartLinePosition.Line.Should().Be(15);
		pathDiag.Location.GetLineSpan().StartLinePosition.Character.Should().Be(27);

		var lineDiag = diagnostics[1];
		lineDiag.Id.Should().Be("PS0102");
		lineDiag.Location.GetLineSpan().StartLinePosition.Line.Should().Be(15);
		lineDiag.Location.GetLineSpan().StartLinePosition.Character.Should().Be(33);

		var expectedCode = @"//----------------------
// <auto-generated>
//	Generated by the PropertySelectorsGenerationTool v1. DO NOT EDIT!
//	Manual changes to this file will be overwritten if the code is regenerated.
// </auto-generated>
//----------------------
#pragma warning disable

namespace Tests.__PropertySelectors
{
	/// <summary>
	/// Auto registration class for PropertySelector used in <global namespace>.
	/// </summary>
	[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
	internal static class _16_3
	{
		/// <summary>
		/// Register the value providers for the PropertySelectors used to invoke 'SUTMethod'
		/// in  on line 16.
		/// </summary>
		/// <remarks>
		/// This method is flagged with the [ModuleInitializerAttribute] which means that it will be invoked by the runtime when the module is being loaded.
		/// You should not have to use it at any time.
		/// </remarks>
		[global::System.ComponentModel.EditorBrowsableAttribute(global::System.ComponentModel.EditorBrowsableState.Never)]
		[global::System.Runtime.CompilerServices.ModuleInitializerAttribute]
		internal static void Register()
		{
			global::Uno.Extensions.Edition.PropertySelectors.Register(
				@""selector16"",
				new global::Uno.Extensions.Edition.ValueAccessor<TestNamespace.Entity, string>(
				"".Value"",
				entity => entity.Value,
				(current, updated_Value) => (current  ?? new(default(string?))) with { Value = updated_Value }));
		}
	}
}";
		GenerationTestHelper.RunGeneratorTwice(
			compilation.Compilation,
			run1 => GenerationTestHelper.AssertRunReason(run1, IncrementalStepRunReason.New),
			run2 => GenerationTestHelper.AssertRunReason(run2, IncrementalStepRunReason.Cached),
			expectedCode);
	}

	private CompilationWithAnalyzers TestInvoke(string invocation)
		=> GenerationTestHelper.CreateCompilationWithAnalyzers($@"
			using Uno.Extensions.Edition;
			using System.Runtime.CompilerServices;

			namespace TestNamespace;

			public record Entity(string? Value);

			public class SUTClass
			{{
				public void Test()
				{{
					{invocation}
				}}

				public void SUTMethod(PropertySelector<Entity, string> selector, [CallerFilePath] string path = """", [CallerLineNumber] int line = -1)
				{{
				}}
			}}
			");
}
