﻿using System;
using System.Collections.Immutable;
using System.Diagnostics.CodeAnalysis;
using System.Linq;

namespace Uno.Extensions.Equality;

/// <summary>
/// A registry of default <see cref="IEqualityComparer{T}"/> which will consider as equals to instance of T as soon has their key identifier are equals.
/// </summary>
/// <remarks>
/// This registry is expected to be automatically full-filled by the code generated by the KeyEqualityGenerationTool for all types that explicitly
/// implements <see cref="IKeyEquatable{T}"/> or types that are eligible for its implementation generation (cf. ImplicitKeysAttribute).
/// </remarks>
public static class KeyEqualityComparer
{
	private static ImmutableHashSet<IKeyEqualityProvider> _providers = ImmutableHashSet<IKeyEqualityProvider>.Empty;

	/// <summary>
	/// Registers an instance of <see cref="IEqualityComparer{T}"/> that should be used for default key comparison.
	/// </summary>
	/// <typeparam name="T">Type of the compared objects.</typeparam>
	/// <param name="instance">The <see cref="IEqualityComparer{T}"/> to compare key of 2 instances of <typeparamref name="T"/>.</param>
	[EditorBrowsable(EditorBrowsableState.Advanced)]
	public static void Register<T>(IEqualityComparer<T> instance)
		where T : IKeyEquatable<T>
		=> Handler<T>.KeyComparer = instance;

	/// <summary>
	/// Registers a type for default key comparison.
	/// </summary>
	/// <remarks>This should not be used since the discovery is expected to be generated at compile time by the KeyEqualityGenerationTool.</remarks>
	/// <typeparam name="T">Type of the compared objects.</typeparam>
	[EditorBrowsable(EditorBrowsableState.Never)]
	public static void Register<T>()
		where T : IKeyEquatable<T>
		=> Handler<T>.KeyComparer = new KeyEqualityComparer<T>();

	/// <summary>
	/// Registers a provider of default key comparison.
	/// </summary>
	/// <remarks>This should not be used since the discovery is expected to be generated at compile time by the KeyEqualityGenerationTool.</remarks>
	[EditorBrowsable(EditorBrowsableState.Never)]
	public static void Register(IKeyEqualityProvider provider)
		=> ImmutableInterlocked.Update(ref _providers, (providers, p) => providers.Add(p), provider);

	/// <summary>
	/// Gets an <see cref="IEqualityComparer{T}"/> which compares only the key,
	/// if <typeparamref name="T"/> implements <see cref="IKeyEquatable{T}"/>.
	/// </summary>
	/// <returns>
	/// An <see cref="IEqualityComparer{T}"/> which compares only the key,
	/// if <typeparamref name="T"/> implements <see cref="IKeyEquatable{T}"/>,
	/// `null` otherwise.
	/// </returns>
	public static IEqualityComparer<T>? Find<T>()
		=> Handler<T>.KeyComparer;

	private class Handler<T>
	{
		private static bool _isInit;
		private static IEqualityComparer<T>? _keyComparer;

		public static IEqualityComparer<T>? KeyComparer
		{
			get
			{
				if (!_isInit)
				{
					_isInit = true;
					var type = typeof(T);
					foreach (var provider in _providers)
					{
						try
						{
							if (provider.TryGet(type) is { } comparer)
							{
								Interlocked.CompareExchange(ref _keyComparer, comparer.ToEqualityComparer<T>(), null);
								break;
							}
						}
						catch (Exception) { }
					}
				}

				return _keyComparer;
			}
			set
			{
				_isInit = true;
				_keyComparer = value;
			}
		}
	}
}
