﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace Uno.Extensions.Reactive.Core;

internal static class FeedHelper
{
	/// <summary>
	/// Invokes an async method in reaction to a message from a parent feed
	/// </summary>
	/// <typeparam name="TParent">Value type of the parent feed</typeparam>
	/// <typeparam name="TResult">Value type of the resulting feed</typeparam>
	/// <param name="msgManager">The message manager used to produce new values messages.</param>
	/// <param name="parentMsg">Optionally provide the message from a parent feed which needs to be pushed to the <paramref name="msgManager"/> (cf. Remarks).</param>
	/// <param name="dataProvider">The async method to invoke.</param>
	/// <param name="extraConfig">Additional configure step for the messages produced while invoking the async method.</param>
	/// <param name="context">The context to use to invoke the async method.</param>
	/// <param name="ct">The enumerator cancellation.</param>
	/// <returns>
	/// An async enumerable that produces message that has to be forwarded (and stored as new current) by the resulting feed
	/// to track the progress the of the async method invocation.
	/// In best cases this will contains only one message (e.g. <paramref name="dataProvider"/> runs sync),
	/// and currently it will produces a maximum of 2 messages.
	/// </returns>
	/// <remarks>
	/// If this manager is used with a parent feed, instead of manually updating the parent of the
	/// <paramref name="msgManager"/> and then call this invoke helper,
	/// it's preferable to provide the <paramref name="parentMsg"/> to this helper,
	/// so the parent message will be updated in sync with other messages generated by this method,
	/// reducing the total number of messages generated.
	/// </remarks>
	public static async Task InvokeAsync<TParent, TResult>(
		MessageManager<TParent, TResult> msgManager,
		Message<TParent>? parentMsg,
		AsyncFunc<Option<TResult>> dataProvider,
		Action<MessageBuilder<TParent, TResult>>? extraConfig,
		SourceContext context,
		CancellationToken ct)
	{
		// Note: We DO NOT register the 'message' update transaction into ct.Register, so in case of a "Last wins" usage of this,
		//		 we allow the next updater to really preserver the pending progress axis.
		using var message = msgManager.BeginUpdate(preservePendingAxes: MessageAxis.Progress);
		using var _ = context.AsCurrent();

		ValueTask<Option<TResult>> dataTask = default;
		try
		{
			dataTask = dataProvider(ct);
		}
		catch (OperationCanceledException) when (ct.IsCancellationRequested)
		{
			return;
		}
		catch (Exception error)
		{
			message.Commit(
				static (m, @params) => m.With(@params.parentMsg).Apply(@params.extraConfig).Error(@params.error),
				(parentMsg, error, extraConfig));

			return;
		}

		// If we are not yet and the 'dataTask' is really async, we need to send a new message flagged as transient
		// Note: This check is not "atomic", but it's valid as it only enables a fast path.
		if (!message.Local.Current.IsTransient)
		{
			// As lot of async methods are actually not really async but only re-scheduled,
			// we try to avoid the transient state by delaying a bit the message.
			for (var i = 0; !dataTask.IsCompleted && !ct.IsCancellationRequested && i < 5; i++)
			{
				await Task.Yield();
			}

			if (ct.IsCancellationRequested)
			{
				return;
			}

			// The 'valueProvider' is not completed yet, so we need to flag the current value as transient.
			// Note: We also provide the parentMsg which will be applied
			if (!dataTask.IsCompleted)
			{
				message.Update(
					static (msg, @params) =>
					{
						var builder = msg.With(@params.parentMsg);
						@params.extraConfig?.Invoke(builder.Inner);
						builder.SetTransient(MessageAxis.Progress, MessageAxis.Progress.ToMessageValue(true));
						return builder;
					},
					(parentMsg, extraConfig),
					ct);
			}
		}

		try
		{
			var data = await dataTask.ConfigureAwait(false);

			// Clear the local error if any.
			// Note: Thanks to the MessageManager, this will NOT erase the parent's error!
			message.Commit(
				static (msg, @params) => msg.With(@params.parentMsg).Apply(@params.extraConfig).Data(@params.data).Error(null),
				(parentMsg, data, extraConfig));
		}
		catch (OperationCanceledException) when (ct.IsCancellationRequested)
		{
		}
		catch (Exception error)
		{
			message.Commit(
				static (msg, @params) => msg.With(@params.parentMsg).Apply(@params.extraConfig).Error(@params.error),
				(parentMsg, error, extraConfig));
		}
	}

	public static Exception? AggregateErrors(Exception? error1, Exception? error2)
	{
		if (error1 is null)
		{
			return error2;
		}

		if (error2 is null)
		{
			return error1;
		}

		List<Exception> errors = new();
		if (error1 is AggregateException aggregate1)
		{
			errors.AddRange(aggregate1.InnerExceptions);
		}
		else
		{
			errors.Add(error1);
		}
		if (error2 is AggregateException aggregate2)
		{
			errors.AddRange(aggregate2.InnerExceptions);
		}
		else
		{
			errors.Add(error1);
		}

		return new AggregateException(errors);
	}

	public static Exception AggregateErrors(IReadOnlyCollection<Exception> errors)
	{
		var flattened = errors
			.SelectMany(error => error switch
			{
				null => Enumerable.Empty<Exception>(),
				AggregateException aggregate => aggregate.InnerExceptions,
				_ => new[] { error }
			})
			.Distinct();

		return new AggregateException(flattened);
	}

	internal static string GetDebugIdentifier(object? feed)
		=> feed is null ? "--null--" : $"{feed.GetType().Name}_{feed.GetHashCode():X8}";
}
