
![logo](img/logo-small.png)

## Codon-Aware aligner (`cawlign`)

A standalone C++ port of `bealign`, a part of the `BioExt` (https://github.com/veg/BioExt) package. The purpose of this program is to align/map sequences in a `FASTA` file to a reference sequence and output the alignment as another `FASTA` file. The underlying code was originally designed to map HIV-1 reads generated by a 454 instrument to references while accounting for homopolymer errors (e.g. [PMID 21632754](https://pubmed.ncbi.nlm.nih.gov/21632754/)). It has since become a part of the [HIV-TRACE](https://pubmed.ncbi.nlm.nih.gov/29401317/)

#### Key features.

1. Lightweight and self-contained. 
2. Allows codon-aware alignment while handling out-of-frame events. 
3. Flexible specification of scoring matrices.
4. OpenMP support. 

#### Installation.

`conda install -c bioconda cawlign`


#### Building from source

Requires `CMake`.

```
$git clone https://github.com/veg/cawlign
$cd cawlign
$cmake ./
$make 
$make install
```

#### Usage

```
usage: cawlign [-h] [-v] [-o OUTPUT] [-r REFERENCE] [-s SCORE] [-t DATATYPE] [-l LOCAL_ALIGNMENT] [-f FORMAT] [-S SPACE] [-a] [-q] [-I] [FASTA]

perform a pairwise alignment between a reference sequence and a set of other sequences

optional arguments:
  -h, --help               show this help message and exit
  -v, --version            show "cawlign" version 
  -o OUTPUT                direct the output to a file named OUTPUT (default=stdout)
  -r REFERENCE             read the reference sequence from this file (default="HXB2_pol")
                           first checks to see if the filepath exists, if not looks inside the res/references directory
                           relative to the install path (/usr/local/share/cawlign by default)
  -s SCORE                 read the scoring matrices and options from this file (default="Nucleotide-BLAST")
                           first checks to see if the filepath exists, if not looks inside the res/scoring directory
                           relative to the install path (/usr/local/share/cawlign by default)
                           see INSERT URL later for how to read and create alignment scoring files
  -t DATATYPE              datatype (default=nucleotide)
                           nucleotide : align sequences in the nucleotide space;
                           protein    : align sequences in the protein space;
                           codon: align sequences in the codon space (reference must be in frame; stop codons are defined in the scoring file);
  -l LOCAL_ALIGNMENT       global/local alignment (default=trim)
                           global : full string alignment; all gaps in the alignments are scored the same
                           local  : partial string local (smith-waterman type) alignment which maximizes the alignment score
  -f FORMAT                controls the format of the output (default=refmap)
                           refmap   : aligns query sequences to the reference and does NOT retain instertions relative to the reference;
                           pairwise : aligns query sequences to the reference and DOES retain instertions relative to the reference;
                                      no MSA is generated, but rather pair-wise alignments are all reported (2x the number of sequences);
  -S SPACE                 which version of the algorithm to use (an integer >0, default=quadratic):
                           quadratic : build the entire dynamic programming matrix (NxM);
                           linear    : use the divide and conquer recursion to keep only 6 columns in memory (~ max (N,M));
                                       NOT IMPLEMENTED FOR CODON DATA
  -a                       do NOT use affine gap scoring (use by default)
  -I                       write out the reference sequence for refmap and refalign output options (default = no) 
  FASTA                    read sequences to compare from this file (default=stdin)
```


#### Example use

##### Align a specified sequence to the default reference (HIV-1 `HXB2 pol`), in nucleotide space, using the default nucleotide scoring matrix.

```
$cawlign test/cases/MZ766877.fa 

>MZ766877.1
CCTCAGATCACTCTTTGGCAGCGGCCCCTTGTCTCAATAAAAGTAGGGGGCCAAATAAGAGAGGCTCTTTTAGATACAGG
AGCAGATGATACAGTATTAGAAGACATGA-
TTTGCCAAGAAAATAGAAACCAAAAATGATAGAAGGAATTAGAAGTTTTATCAAAGTAAAACAGTATGAACAAATACCTA
TAGAAATTTGTA-
AGAAAAAGCTATAAGTACAGTATTAGTAAGACCTACACCTGTCAACATAATTAGAAAGAATATGTTGACTCAGCTCGGAT
GTACACTAAATTTTCCAATTAGTCCCATTGAAACTGTACCAGTAAAACTAAAACCAGGAATAGATAGTCCAAAAATTAAA
CAGTGGCCATTGACAGAAGAAAAAATAAAAGCATTAATAGCAATTTGTGAAGAAATGGAGAAAGAAAGAAAAATTACAAA
AGTAGGGCCTGAAAACCCATACAACACTCCAGTGTTTGCCATAAAAAAGAAGGACAGCACTAAGTGGAGAAAATTAGTAG
ATTTCAAAGAACTCAATAAAAGAACTCAAGACTTTTAAGAAGTTCAATTAAGAATACCACACCCAGCAAAGTTAAAGAAG
AGAAAATCAGTGACAGTGCTAGATGTAGGAGATGCATATTTTTCAGTCCCTTTAGATGAAAGCTTCAGGAAGTATACTGC
ATTCACCATACCTAGTATAAATAATGAAACACCAGGAATTAGATACCAATATAATGTGCTTCCACAAGGATAGAAAAGAT
CACCAGCAATATTCCAGAGTAGCATGACAAAAATCTTAAAGCCCTTTAGGACAAAAAATCCAGACATAGTTATCTATCAA
TATATAGATGACTTGTATGTAGGCTCTGACTTAGAAATAAGGCAACATAGGGCAAAGGTAGAGAAGCTAAGAGAACATTT
GTTGAGATGAGGACTTACCACACCAGACAAAAAGCATCAGAAAGAGCCCCCATTTCTTTAGATAAAGTATGAACTCCATC
CTGACAAATGGACAGTACAACCTATAACGCTGCCAGAAAAGAACAGCTGGACTGTCAATGATATACAGAAGTTAGTAAGA
AAATTAAACTAGGCCAGCCAGATCTATGCAAAGATTA--
GTAAAACAACTGTGCAAACTCCTTAGGGGGACTAAAGCACTAACAGACATAGTACCACTGACTAAAGAAGCAGAATTAGA
ATTAGCAGAGAACAGAGAAATTCTAAAAGAACCAGTACATAAGGTATATTATGACCCATCAAAAGACTTAATAGCTAAAA
TACAGAAACAGGGGCATGGCCAATGGACATACCAAATTTACCAAGAGCCGTTCAAAAATCTGAAAACAGGGAAGTATGCA
AAAATGAAGCATGCCCACACTAATGATGTGAAACAGTTAACAAAGGCAGTGCAAAAAATAACTCTAGAAAGCATAGTAAT
ATGAAGAAAAACTCCTAAATTTAGATTACCCATCCAAAAAGAAACGTAGGAGACATAGTGGACAGACTATTAGCAAGCCA
CCTAGATTCCTAAGTAGGAGTTTGTTAATACCCCTCCCCTAGTAAAATTATAGTATCAGCTAGAAAAAGAACCCATAGTA
AGAGCAGAAACTTTTTATGTAGATAGAGCAGCTAATAAAGAAACTAAAGCAGGAAAAGCAAAGTATGTTACTGACAGAGG
AAGGCAAAAAGTTGTTTCCCTAACTGAAACAACAAACCAGAAAACTGAATTACAAGCAATTCAGCTAGCTTTACAAGATT
CAGGATCAGAAGTAAACATAGTAACAGACTCACAGTATGCATTAAGAATCATTCAAGGACAACCAGATAAGAGTGAATCA
GAGATAGTCAACCAAATAATAGAACAATTAATAAACAAGGAGAGGGTATACCTGTCATAGGTACCAGCACATAAAGGAAT
TAGAGGAAATGAACAAGTAAATAAATTAGTAAGTAATAGAATCAGGAAAGTGCTATTTCTAGATAAAATAAATAAGGCTC
AAGAAGAGCATGAAAAATATCACAGCAATTGAAGAGCAATGGCTAGTAAGTTTAACCTGCCACCAGTTGTAGCAAAAGAA
ATAGTAGCTAGCTGTGATAAGTGTCAGCTAAAAAGGGAAGCCATACATAGACAAGTAGACTGTAGTCCAAAGATATGGCA
ATTAGATTGTACACATCTAGAAAGGAAAGTCATCCTAGTAGCAGTCCATGTAGCCAGTGGCTATATAGAAGCAGAAGTTA
TCCCAGCAGAAACAGGACAAGAAACAGCATACTATATACTAAAATTAGCAGGAAGATGGCCAGTCAAAACCATACATACA
GACAATGGCACTAATTTCACCAGTGCTGCAGTTAAAGCAGCCTGCTGGTGGGCAAGTATCCAACAAGAATTTAGAATTCC
CTACAATCCCCAAAGTCAAGAAGTAGTAGAATCCATGAATAAAGAATTAAAGAAAATTATAGGGCAAGTAAGAGAGCAAG
CTGAGCACCTTAAGACAGCAGTACAAATAGCAGTATTCATTCACAATTTTAAAAGAAAAGGGGGGATTAGGGGGTATAGT
GCAGGGGAAAGAATAATAGACATAATAGCAACAGACATACAGACTAAAGAATTACAAAAACAAATTACAAAAATCCAAAA
TTTTCGGGTTTATTACAGAGACAGCAGAGATCCTATTTGGAAAGGACCAGCCAAACTACTGTGGAAAGGTGAAGGGGCAG
TAGTAATACAAGATAATAGTGACATAAAAGTAGTACCAAGGAGGAAAGTAAAAATCATTAAGGACTATGGAAAACAGATG
GCAGGTGCTGATTGTGTGGCAGGTAGACAGGATGAGGAT

```

##### Align a specified sequence to the default reference (HIV-1 `HXB2 pol`), in nucleotide space, using the default nucleotide scoring matrix, and a linear space algorithm

```
cawlign -S linear test/cases/MZ766877.fa
>MZ766877.1
CCTCAGATCACTCTTTGGCAGCGGCCCCTTGTCTCAATAAAAGTAGGGGGCCAAATAAGAGAGGCTCTTTTAGATACAGG
AGCAGATGATACAGTATTAGAAGACAT-
AGTTTGCCAAGAAAATAGAAACCAAAAATGATAGAAGGAATTAGAAGTTTTATCAAAGTAAAACAGTATGAACAAATACC
TATAGAAATTTGT-
AAGAAAAAGCTATAAGTACAGTATTAGTAAGACCTACACCTGTCAACATAATTAGAAAGAATATGTTGACTCAGCTCGGA
TGTACACTAAATTTTCCAATTAGTCCCATTGAAACTGTACCAGTAAAACTAAAACCAGGAATAGATAGTCCAAAAATTAA
ACAGTGGCCATTGACAGAAGAAAAAATAAAAGCATTAATAGCAATTTGTGAAGAAATGGAGAAAGAAAGAAAAATTACAA
AAGTAGGGCCTGAAAACCCATACAACACTCCAGTGTTTGCCATAAAAAAGAAGGACAGCACTAAGTGGAGAAAATTAGTA
GATTTCAAAGAACTCAATAAAAGAACTCAAGACTTTTAAGAAGTTCAATTAAGAATACCACACCCAGCAAAGTTAAAGAA
GAGAAAATCAGTGACAGTGCTAGATGTAGGAGATGCATATTTTTCAGTCCCTTTAGATGAAAGCTTCAGGAAGTATACTG
CATTCACCATACCTAGTATAAATAATGAAACACCAGGAATTAGATACCAATATAATGTGCTTCCACAAGGATAGAAAAGA
TCACCAGCAATATTCCAGAGTAGCATGACAAAAATCTTAAAGCCCTTTAGGACAAAAAATCCAGACATAGTTATCTATCA
ATATATAGATGACTTGTATGTAGGCTCTGACTTAGAAATAAGGCAACATAGGGCAAAGGTAGAGAAGCTAAGAGAACATT
TGTTGAGATGAGGACTTACCACACCAGACAAAAAGCATCAGAAAGAGCCCCCATTTCTTTAGATAAAGTATGAACTCCAT
CCTGACAAATGGACAGTACAACCTATAACGCTGCCAGAAAAGAACAGCTGGACTGTCAATGATATACAGAAGTTAGTAAG
AAAATTAAACTAGGCCAGCCAGATCTATGCAAAGATT--
AGTAAAACAACTGTGCAAACTCCTTAGGGGGACTAAAGCACTAACAGACATAGTACCACTGACTAAAGAAGCAGAATTAG
AATTAGCAGAGAACAGAGAAATTCTAAAAGAACCAGTACATAAGGTATATTATGACCCATCAAAAGACTTAATAGCTAAA
ATACAGAAACAGGGGCATGGCCAATGGACATACCAAATTTACCAAGAGCCGTTCAAAAATCTGAAAACAGGGAAGTATGC
AAAAATGAAGCATGCCCACACTAATGATGTGAAACAGTTAACAAAGGCAGTGCAAAAAATAACTCTAGAAAGCATAGTAA
TATGAAGAAAAACTCCTAAATTTAGATTACCCATCCAAAAAGAAACGTAGGAGACATAGTGGACAGACTATTAGCAAGCC
ACCTAGATTCCTAAGTAGGAGTTTGTTAATACCCCTCCCCTAGTAAAATTATAGTATCAGCTAGAAAAAGAACCCATAGT
AAGAGCAGAAACTTTTTATGTAGATAGAGCAGCTAATAAAGAAACTAAAGCAGGAAAAGCAAAGTATGTTACTGACAGAG
GAAGGCAAAAAGTTGTTTCCCTAACTGAAACAACAAACCAGAAAACTGAATTACAAGCAATTCAGCTAGCTTTACAAGAT
TCAGGATCAGAAGTAAACATAGTAACAGACTCACAGTATGCATTAAGAATCATTCAAGGACAACCAGATAAGAGTGAATC
AGAGATAGTCAACCAAATAATAGAACAATTAATAAACAAGGAGAGGGTATACCTGTCATAGGTACCAGCACATAAAGGAA
TTAGAGGAAATGAACAAGTAAATAAATTAGTAAGTAATAGAATCAGGAAAGTGCTATTTCTAGATAAAATAAATAAGGCT
CAAGAAGAGCATGAAAAATATCACAGCAATTGAAGAGCAATGGCTAGTAAGTTTAACCTGCCACCAGTTGTAGCAAAAGA
AATAGTAGCTAGCTGTGATAAGTGTCAGCTAAAAAGGGAAGCCATACATAGACAAGTAGACTGTAGTCCAAAGATATGGC
AATTAGATTGTACACATCTAGAAAGGAAAGTCATCCTAGTAGCAGTCCATGTAGCCAGTGGCTATATAGAAGCAGAAGTT
ATCCCAGCAGAAACAGGACAAGAAACAGCATACTATATACTAAAATTAGCAGGAAGATGGCCAGTCAAAACCATACATAC
AGACAATGGCACTAATTTCACCAGTGCTGCAGTTAAAGCAGCCTGCTGGTGGGCAAGTATCCAACAAGAATTTAGAATTC
CCTACAATCCCCAAAGTCAAGAAGTAGTAGAATCCATGAATAAAGAATTAAAGAAAATTATAGGGCAAGTAAGAGAGCAA
GCTGAGCACCTTAAGACAGCAGTACAAATAGCAGTATTCATTCACAATTTTAAAAGAAAAGGGGGGATTAGGGGGTATAG
TGCAGGGGAAAGAATAATAGACATAATAGCAACAGACATACAGACTAAAGAATTACAAAAACAAATTACAAAAATCCAAA
ATTTTCGGGTTTATTACAGAGACAGCAGAGATCCTATTTGGAAAGGACCAGCCAAACTACTGTGGAAAGGTGAAGGGGCA
GTAGTAATACAAGATAATAGTGACATAAAAGTAGTACCAAGGAGGAAAGTAAAAATCATTAAGGACTATGGAAAACAGAT
GGCAGGTGCTGATTGTGTGGCAGGTAGACAGGATGAGGAT
```

##### Align a specified sequence to the default reference (HIV-1 `HXB2 pol`), in codon space, using the `BLOSUM62` scoring matrix

```
$cawlign -t codon -s BLOSUM62 test/cases/MZ766877.fa
>MZ766877.1
CCTCAGATCACTCTTTGGCAGCGGCCCCTTGTCTCAATAAAAGTAGGGGGCCAAATAAGAGAGGCTCTTTTAGATACAGG
AGCAGATGATACAGTATTAGAAGACATAGATTTGCCA---
AGAAAAAAACCAAAAATGATAGAAGGAATTAGAAGTTTTATCAAAGTAAAACAGTATGAACAAATACCTATAGAAATTTG
TAAGAAAAAGGCTATAAGTACAGTATTAGTAAGACCTACACCTGTCAACATAATTAGAAAGAATATGTTGACTCAGCTCG
GATGTACACTAAATTTTCCAATTAGTCCCATTGAAACTGTACCAGTAAAACTAAAACCAGGAATAGATAGTCCAAAAATT
AAACAGTGGCCATTGACAGAAGAAAAAATAAAAGCATTAATAGCAATTTGTGAAGAAATGGAGAAAGAAAGAAAAATTAC
AAAAGTAGGGCCTGAAAACCCATACAACACTCCAGTGTTTGCCATAAAAAAGAAGGACAGCACTAAGTGGAGAAAATTAG
TAGATTTCAAAGAACTCAATAAAAGAACTCAAGACTTT---
GAAGTTCAATTAAGAATACCACACCCAGCAAAGTTAAAGAAGAGAAAATCAGTGACAGTGCTAGATGTAGGAGATGCATA
TTTTTCAGTCCCTTTAGATGAAAGCTTCAGGAAGTATACTGCATTCACCATACCTAGTATAAATAATGAAACACCAGGAA
TTAGATACCAATATAATGTGCTTCCACAAGGA---
AAAAGATCACCAGCAATATTCCAGAGTAGCATGACAAAAATCTTAAAGCCCTTTAGGACAAAAAATCCAGACATAGTTAT
CTATCAATATATAGATGACTTGTATGTAGGCTCTGACTTAGAAATAAGGCAACATAGGGCAAAGGTAGAGAAGCTAAGAG
AACATTTGTTGAGA---GGACTTACCACACCAGACAAAAAGCATCAGAAAGAGCCCCCATTTCTT---
ATAAAGTATGAACTCCATCCTGACAAATGGACAGTACAACCTATAACGCTGCCAGAAAAGAACAGCTGGACTGTCAATGA
TATACAGAAGTTAGTAAGAAAATTAAAC---
GCCAGCCAGATCTATGCAAAGATTAGTACAAAACAACTGTGCAAACTCCTTAGGGGGACTAAAGCACTAACAGACATAGT
ACCACTGACTAAAGAAGCAGAATTAGAATTAGCAGAGAACAGAGAAATTCTAAAAGAACCAGTACATAAGGTATATTATG
ACCCATCAAAAGACTTAATAGCTAAAATACAGAAACAGGGGCATGGCCAATGGACATACCAAATTTACCAAGAGCCGTTC
AAAAATCTGAAAACAGGGAAGTATGCAAAAATGAAGCATGCCCACACTAATGATGTGAAACAGTTAACAAAGGCAGTGCA
AAAAATAACTCTAGAAAGCATAGTAATA------AAAACTCCTAAATTTAGATTACCCATCCAAAAAGAAACG-------
--------------------------------------------GAGTTTGTTAATACCCCTCCCCTAGTAAAATTA---
TATCAGCTAGAAAAAGAACCCATAGTAAGAGCAGAAACTTTTTATGTAGATAGAGCAGCTAATAAAGAAACTAAAGCAGG
AAAAGCAAAGTATGTTACTGACAGAGGAAGGCAAAAAGTTGTTTCCCTAACTGAAACAACAAACCAGAAAACTGAATTAC
AAGCAATTCAGCTAGCTTTACAAGATTCAGGATCAGAAGTAAACATAGTAACAGACTCACAGTATGCATTAAGAATCATT
CAAGGACAACCAGATAAGAGTGAATCAGAGATAGTCAACCAAATAATAGAACAATTAATAAACAAGGAGAGGGTATACCT
GTCA---
GTACCAGCACATAAAGGAATTAGAGGAAATGAACAAGTAAATAAATTAGTAAGTAATAGAATCAGGAAAGTGCTATTTCT
AGATAAAATAAATAAGGCTCAAGAAGAGCATGAAAAATATCACAGCAAT---
AGAGCAATGGCTAGTAAGTTTAACCTGCCACCAGTTGTAGCAAAAGAAATAGTAGCTAGCTGTGATAAGTGTCAGCTAAA
AAGGGAAGCCATACATAGACAAGTAGACTGTAGTCCAAAGATATGGCAATTAGATTGTACACATCTAGAAAGGAAAGTCA
TCCTAGTAGCAGTCCATGTAGCCAGTGGCTATATAGAAGCAGAAGTTATCCCAGCAGAAACAGGACAAGAAACAGCATAC
TATATACTAAAATTAGCAGGAAGATGGCCAGTCAAAACCATACATACAGACAATGGCACTAATTTCACCAGTGCTGCAGT
TAAAGCAGCCTGCTGGTGGGCAAGTATCCAACAAGAATTTAGAATTCCCTACAATCCCCAAAGTCAAGAAGTAGTAGAAT
CCATGAATAAAGAATTAAAGAAAATTATAGGGCAAGTAAGAGAGCAAGCTGAGCACCTTAAGACAGCAGTACAAATAGCA
GTATTCATTCACAATTTTAAAAGAAAAGGGGGGATTAGGGGGTATAGTGCAGGGGAAAGAATAATAGACATAATAGCAAC
AGACATACAGACTAAAGAATTACAAAAACAAATTACAAAAATCCAAAATTTTCGGGTTTATTACAGAGACAGCAGAGATC
CTATTTGGAAAGGACCAGCCAAACTACTGTGGAAAGGTGAAGGGGCAGTAGTAATACAAGATAATAGTGACATAAAAGTA
GTACCAAGGAGGAAAGTAAAAATCATTAAGGACTATGGAAAACAGATGGCAGGTGCTGATTGTGTGGCAGGTAGACAGGA
TGAGGAT
```

##### Align a specified sequence to a specific  reference (HIV-1 `HXB2 rt`), in codon space, using the `BLOSUM62` scoring matrix and include the reference sequence (NOT including insertions in the reference)

```
>HXB2_rt
CCCATTAGCCCTATTGAGACTGTACCAGTAAAATTAAAGCCAGGAATGGATGGCCCAAAAGTTAAACAATGGCCATTGACAGAAGAAAAAATAAAAGCATTAGTAGAAATTTGTACAGAGATGGAAAAGGAAGGGAAAATTTCAAAAATTGGGCCTGAAAATCCATACAATACTCCAGTATTTGCCATAAAGAAAAAAGACAGTACTAAATGGAGAAAATTAGTAGATTTCAGAGAACTTAATAAGAGAACTCAAGACTTCTGGGAAGTTCAATTAGGAATACCACATCCCGCAGGGTTAAAAAAGAAAAAATCAGTAACAGTACTGGATGTGGGTGATGCATATTTTTCAGTTCCCTTAGATGAAGACTTCAGGAAGTATACTGCATTTACCATACCTAGTATAAACAATGAGACACCAGGGATTAGATATCAGTACAATGTGCTTCCACAGGGATGGAAAGGATCACCAGCAATATTCCAAAGTAGCATGACAAAAATCTTAGAGCCTTTTAGAAAACAAAATCCAGACATAGTTATCTATCAATACATGGATGATTTGTATGTAGGATCTGACTTAGAAATAGGGCAGCATAGAACAAAAATAGAGGAGCTGAGACAACATCTGTTGAGGTGGGGACTTACCACACCAGACAAAAAACATCAGAAAGAACCTCCATTCCTTTGGATGGGTTATGAACTCCATCCTGATAAATGGACAGTACAGCCTATAGTGCTGCCAGAAAAAGACAGCTGGACTGTCAATGACATACAGAAGTTAGTGGGGAAATTGAATTGGGCAAGTCAGATTTACCCAGGGATTAAAGTAAGGCAATTATGTAAACTCCTTAGAGGAACCAAAGCACTAACAGAAGTAATACCACTAACAGAAGAAGCAGAGCTAGAACTGGCAGAAAACAGAGAGATTCTAAAAGAACCAGTACATGGAGTGTATTATGACCCATCAAAAGACTTAATAGCAGAAATACAGAAGCAGGGGCAAGGCCAATGGACATATCAAATTTATCAAGAGCCATTTAAAAATCTGAAAACAGGAAAATATGCAAGAATGAGGGGTGCCCACACTAATGATGTAAAACAATTAACAGAGGCAGTGCAAAAAATAACCACAGAAAGCATAGTAATATGGGGAAAGACTCCTAAATTTAAACTGCCCATACAAAAGGAAACATGGGAAACATGGTGGACAGAGTATTGGCAAGCCACCTGGATTCCTGAGTGGGAGTTTGTTAATACCCCTCCCTTAGTGAAATTATGGTACCAGTTAGAGAAAGAACCCATAGTAGGAGCAGAAACCTTC
>MZ766877.1
CCAATTAGTCCCATTGAAACTGTACCAGTAAAACTAAAACCAGGAATAGATAGTCCAAAAATTAAACAGTGGCCATTGACAGAAGAAAAAATAAAAGCATTAATAGCAATTTGTGAAGAAATGGAGAAAGAAAGAAAAATTACAAAAGTAGGGCCTGAAAACCCATACAACACTCCAGTGTTTGCCATAAAAAAGAAGGACAGCACTAAGTGGAGAAAATTAGTAGATTTCAAAGAACTCAATAAAAGAACTCAAGACTTT---GAAGTTCAATTAAGAATACCACACCCAGCAAAGTTAAAGAAGAGAAAATCAGTGACAGTGCTAGATGTAGGAGATGCATATTTTTCAGTCCCTTTAGATGAAAGCTTCAGGAAGTATACTGCATTCACCATACCTAGTATAAATAATGAAACACCAGGAATTAGATACCAATATAATGTGCTTCCACAAGGA---AAAAGATCACCAGCAATATTCCAGAGTAGCATGACAAAAATCTTAAAGCCCTTTAGGACAAAAAATCCAGACATAGTTATCTATCAATATATAGATGACTTGTATGTAGGCTCTGACTTAGAAATAAGGCAACATAGGGCAAAGGTAGAGAAGCTAAGAGAACATTTGTTGAGA---GGACTTACCACACCAGACAAAAAGCATCAGAAAGAGCCCCCATTTCTT---ATAAAGTATGAACTCCATCCTGACAAATGGACAGTACAACCTATAACGCTGCCAGAAAAGAACAGCTGGACTGTCAATGATATACAGAAGTTAGTAAGAAAATTAAAC---GCCAGCCAGATCTATGCAAAGATTAGTACAAAACAACTGTGCAAACTCCTTAGGGGGACTAAAGCACTAACAGACATAGTACCACTGACTAAAGAAGCAGAATTAGAATTAGCAGAGAACAGAGAAATTCTAAAAGAACCAGTACATAAGGTATATTATGACCCATCAAAAGACTTAATAGCTAAAATACAGAAACAGGGGCATGGCCAATGGACATACCAAATTTACCAAGAGCCGTTCAAAAATCTGAAAACAGGGAAGTATGCAAAAATGAAGCATGCCCACACTAATGATGTGAAACAGTTAACAAAGGCAGTGCAAAAAATAACTCTAGAAAGCATAGTAATA------AAAACTCCTAAATTTAGATTACCCATCCAAAAAGAAACG---------------------------------------------------GAGTTTGTTAATACCCCTCCCCTAGTAAAATTA---TATCAGCTAGAAAAAGAACCCATAGTAAGAGCAGAAACTTTT


```

##### Align a specified sequence to a specific  reference (HIV-1 `HXB2 rt`), in codon space, using the `BLOSUM62` scoring matrix report pairwise alignment (including insertions in the reference)

```
cawlign -t codon -r HXB2_rt -f pairwise -s BLOSUM62 test/cases/MZ766877.fa 
>HXB2_rt
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------CCCATTAGCCCTATTGAGACTGTACCAGTAAAATTAAAGCCAGGAATGGATGGCCCAAAAGTTAAACAATGGCCATTGACAGAAGAAAAAATAAAAGCATTAGTAGAAATTTGTACAGAGATGGAAAAGGAAGGGAAAATTTCAAAAATTGGGCCTGAAAATCCATACAATACTCCAGTATTTGCCATAAAGAAAAAAGACAGTACTAAATGGAGAAAATTAGTAGATTTCAGAGAACTTAATAAGAGAACTCAAGACTTC---TGGGAAGTTCAATTAGGAATACCACATCCCGCAGGGTTAAAAAAGAAAAAATCAGTAACAGTACTGGATGTGGGTGATGCATATTTTTCAGTTCCCTTAGATGAAGACTTCAGGAAGTATACTGCATTTACCATACCTAGTATAAACAATGAGACACCAGGGATTAGATATCAGTACAATGTGCTTCCACAGGGA---TGGAAAGGATCACCAGCAATATTCCAAAGTAGCATGACAAAAATCTTAGAGCCTTTTAGAAAACAAAATCCAGACATAGTTATCTATCAATACATGGATGATTTGTATGTAGGATCTGACTTAGAAATAGGGCAGCATAGAACAAAAATAGAGGAGCTGAGACAACATCTGTTGAGG---TGGGGACTTACCACACCAGACAAAAAACATCAGAAAGAACCTCCATTCCTT---TGGATGGGTTATGAACTCCATCCTGATAAATGGACAGTACAGCCTATAGTGCTGCCAGAAAAAGACAGCTGGACTGTCAATGACATACAGAAGTTAGTGGGGAAATTGAAT---TGGGCAAGTCAGATTTACCCAGGGATTAAAGTAAGGCAATTATGTAAACTCCTTAGAGGAACCAAAGCACTAACAGAAGTAATACCACTAACAGAAGAAGCAGAGCTAGAACTGGCAGAAAACAGAGAGATTCTAAAAGAACCAGTACATGGAGTGTATTATGACCCATCAAAAGACTTAATAGCAGAAATACAGAAGCAGGGGCAAGGCCAATGGACATATCAAATTTATCAAGAGCCATTTAAAAATCTGAAAACAGGAAAATATGCAAGAATGAGGGGTGCCCACACTAATGATGTAAAACAATTAACAGAGGCAGTGCAAAAAATAACCACAGAAAGCATAGTAATA------TGGGGAAAGACTCCTAAATTTAAACTGCCCATACAAAAGGAAACA---------------------------------------------------TGGGAAACATGGTGGACAGAGTATTGGCAAGCCACCTGGATTCCTGAGTGGGAGTTTGTTAATACCCCTCCCTTAGTGAAATTA---TGGTACCAGTTAGAGAAAGAACCCATAGTAGGAGCAGAAACCTTC--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>MZ766877.1
GCGCCCGAACAGGGACTTGAAAGCGAAAGTGAGACCAGAGAAGATCTCTCGACGCAGGACTCGGCTTGCTGAAGTGCACTCGGCAAGAGGCGAGAGCGGCGGCTGGTGAGTACGCCAAATTTTATTTGACTAGCGGAGGCTAGAAGGAGAGAGATAGGTGCGAGAGCGTCAATATTAAGAGGAGGAAAATTAGATTCATAGGAGAGAATTAAGTTAAGGCCAGGGGGAAAGAAATGTTATCGGCTAAAACACTTAGTATGGGCAAGCAGGGAGCTGGACAGATTTGCACTTAACCCTGGCCTTTTAGAAACGTCAGATGGCTGTAAACAAATACTAAGACAGCTTCAGCCAGCTCTTCAGACAGGAACAGAGGAAATTAAATCATTATTCAACACAGTAGCAACTCTTTATTGTGTACATCAAGGGATAAAGGTACAAGACACCAAAGAAGCCTTAGACAAGATAGAGGAAGAACAAAAACAAAGTCAGAAAAAGGCACAACAAGCAGAAGCGGCTGACAAAGGAAAGGTCAGTCAAAATTATCCTATAGTGCAGAATCTCCAAGGGCAAATGGTACACCAGGCCCTGTCACCTAGAACTTTGAATGCATAGGTAAAAGTAGTAGAGGAGAAAAGTTTTAACCCTGAGGTAATACCCATGTTTTCAGCATTATCAGAGGGAGCCACCCCATCAGATTTAAACACCATGTTAAATACAATAGGGGGACATCAAGCAGCCATGCAAATGTTAAAGGATACTATCAATGAGGAGGCTGCAGAATAGGACAGATTACATCCAGTACAGGCAGGGCCTGTTGCACCAGGCCAAATGAGGGACCCAAAGGGAAGTGACATAGCAAGAACTACTAGTACCCTTCAGGAACAAATAGGATGGATGACAAGCAACCCACCTATGCCAGTGGGAGACATCTATAAAAGATAGATAATTCTGAGGTTAAATAAAATAGTAAGAATGTATAGCCCTGTCAGCATTTTGGACATAAAACAAGGGCCAAAGGAACCCTTTAGAGACTATGTAGATCGGTTCTTTAAAACCTTAAGAGCTGAACAAGCTACACAAGATGTAAAAAATTAGATGACAGACACCTTGTTAGTCCAAAATGCGAACCCAGACTGTAAGACCATTTTAAGAGCATTAAGACCAGAGGCTACATTAGAAGAAATGATGACAGCATGTCAAGGAGTGGGAGGACCTGGCCACAAAGCAAAAGTCTTGGCTGAAGCAATGAGCCAGGTAGGCAATACAAACATAATGATGCAGAGAAGCAATTTTAGAAGCAATAAAAGAATTGTTAAGTGTTTCAACTGTGGCAAGGAGGGGCACATAGCCAAAAATTGCAGGGCCCCTAAGAAAAAAGGCTGTTAGAAATGTGGAAAAGAAGGACACCAAATGAAGGATTGCTCTGAAAGGCAGGCTAATTTTTTAAGGAAAATTTGGCCTTCCCACAAGGGGAGGCCAGGAAATTTCCTTCAGAGCAGACCAAGGCCAACAGCCCCACCAGTAAAGCCAACAGCTCCACCAGTAGAGAGCTTCAGGTTCGAGGAGACAACCCCCTCGATGAAACCAGAGTCGAAGGACGAGGACGCTTTAACTTCCCTCAGATCACTCTTTGGCAGCGGCCCCTTGTCTCAATAAAAGTAGGGGGCCAAATAAGAGAGGCTCTTTTAGATACAGGAGCAGATGATACAGTATTAGAAGACATAGATTTGCCAAGAAAATAGAAACCAAAAATGATAGAAGGAATTAGAAGTTTTATCAAAGTAAAACAGTATGAACAAATACCTATAGAAATTTGTAAGAAAAAGGCTATAAGTACAGTATTAGTAAGACCTACACCTGTCAACATAATTAGAAAGAATATGTTGACTCAGCTCGGATGTACACTAAATTTTCCAATTAGTCCCATTGAAACTGTACCAGTAAAACTAAAACCAGGAATAGATAGTCCAAAAATTAAACAGTGGCCATTGACAGAAGAAAAAATAAAAGCATTAATAGCAATTTGTGAAGAAATGGAGAAAGAAAGAAAAATTACAAAAGTAGGGCCTGAAAACCCATACAACACTCCAGTGTTTGCCATAAAAAAGAAGGACAGCACTAAGTGGAGAAAATTAGTAGATTTCAAAGAACTCAATAAAAGAACTCAAGACTTTTAA---GAAGTTCAATTAAGAATACCACACCCAGCAAAGTTAAAGAAGAGAAAATCAGTGACAGTGCTAGATGTAGGAGATGCATATTTTTCAGTCCCTTTAGATGAAAGCTTCAGGAAGTATACTGCATTCACCATACCTAGTATAAATAATGAAACACCAGGAATTAGATACCAATATAATGTGCTTCCACAAGGATAG---AAAAGATCACCAGCAATATTCCAGAGTAGCATGACAAAAATCTTAAAGCCCTTTAGGACAAAAAATCCAGACATAGTTATCTATCAATATATAGATGACTTGTATGTAGGCTCTGACTTAGAAATAAGGCAACATAGGGCAAAGGTAGAGAAGCTAAGAGAACATTTGTTGAGATGA---GGACTTACCACACCAGACAAAAAGCATCAGAAAGAGCCCCCATTTCTTTAG---ATAAAGTATGAACTCCATCCTGACAAATGGACAGTACAACCTATAACGCTGCCAGAAAAGAACAGCTGGACTGTCAATGATATACAGAAGTTAGTAAGAAAATTAAACTAG---GCCAGCCAGATCTATGCAAAGATTAGTACAAAACAACTGTGCAAACTCCTTAGGGGGACTAAAGCACTAACAGACATAGTACCACTGACTAAAGAAGCAGAATTAGAATTAGCAGAGAACAGAGAAATTCTAAAAGAACCAGTACATAAGGTATATTATGACCCATCAAAAGACTTAATAGCTAAAATACAGAAACAGGGGCATGGCCAATGGACATACCAAATTTACCAAGAGCCGTTCAAAAATCTGAAAACAGGGAAGTATGCAAAAATGAAGCATGCCCACACTAATGATGTGAAACAGTTAACAAAGGCAGTGCAAAAAATAACTCTAGAAAGCATAGTAATATGAAGA------AAAACTCCTAAATTTAGATTACCCATCCAAAAAGAAACGTAGGAGACATAGTGGACAGACTATTAGCAAGCCACCTAGATTCCTAAGTAG---------------------------------------------------GAGTTTGTTAATACCCCTCCCCTAGTAAAATTATAG---TATCAGCTAGAAAAAGAACCCATAGTAAGAGCAGAAACTTTTTATGTAGATAGAGCAGCTAATAAAGAAACTAAAGCAGGAAAAGCAAAGTATGTTACTGACAGAGGAAGGCAAAAAGTTGTTTCCCTAACTGAAACAACAAACCAGAAAACTGAATTACAAGCAATTCAGCTAGCTTTACAAGATTCAGGATCAGAAGTAAACATAGTAACAGACTCACAGTATGCATTAAGAATCATTCAAGGACAACCAGATAAGAGTGAATCAGAGATAGTCAACCAAATAATAGAACAATTAATAAACAAGGAGAGGGTATACCTGTCATAGGTACCAGCACATAAAGGAATTAGAGGAAATGAACAAGTAAATAAATTAGTAAGTAATAGAATCAGGAAAGTGCTATTTCTAGATAAAATAAATAAGGCTCAAGAAGAGCATGAAAAATATCACAGCAATTGAAGAGCAATGGCTAGTAAGTTTAACCTGCCACCAGTTGTAGCAAAAGAAATAGTAGCTAGCTGTGATAAGTGTCAGCTAAAAAGGGAAGCCATACATAGACAAGTAGACTGTAGTCCAAAGATATGGCAATTAGATTGTACACATCTAGAAAGGAAAGTCATCCTAGTAGCAGTCCATGTAGCCAGTGGCTATATAGAAGCAGAAGTTATCCCAGCAGAAACAGGACAAGAAACAGCATACTATATACTAAAATTAGCAGGAAGATGGCCAGTCAAAACCATACATACAGACAATGGCACTAATTTCACCAGTGCTGCAGTTAAAGCAGCCTGCTGGTGGGCAAGTATCCAACAAGAATTTAGAATTCCCTACAATCCCCAAAGTCAAGAAGTAGTAGAATCCATGAATAAAGAATTAAAGAAAATTATAGGGCAAGTAAGAGAGCAAGCTGAGCACCTTAAGACAGCAGTACAAATAGCAGTATTCATTCACAATTTTAAAAGAAAAGGGGGGATTAGGGGGTATAGTGCAGGGGAAAGAATAATAGACATAATAGCAACAGACATACAGACTAAAGAATTACAAAAACAAATTACAAAAATCCAAAATTTTCGGGTTTATTACAGAGACAGCAGAGATCCTATTTGGAAAGGACCAGCCAAACTACTGTGGAAAGGTGAAGGGGCAGTAGTAATACAAGATAATAGTGACATAAAAGTAGTACCAAGGAGGAAAGTAAAAATCATTAAGGACTATGGAAAACAGATGGCAGGTGCTGATTGTGTGGCAGGTAGACAGGATGAGGATTAGAACATGGAATAGTCTAGTGAAGCACCATATGTATATGTCAAAGAGAGCCAGTGGCTGGTTTTACAGACACCATTATGAAAGTAGGCATCCAAAAGTAAGTTCAGAAACACACATCCCATTAGGGGAGGCTAAATTAGTCATAACAACATATTAGGGTTTGCAAACAGGAGAAAGAGAATGGCATTTAGGTCATGGAGTCTCCATAGAATGGAGATTGAGAAGATATAGCACACAAGTAGACCCTGGCCTGGCAGACCAGCTAATCCATATGCATTATTTTGATTGTTTTGCAGACTCTGCCATAAGACAAGCCATATTAGGACACATAGTTATTCCTAGGTGTGATTATCAAGCAGGACATAATAAGGTAGGATCTCTACAATACTTGGCACTGACAGCATTAGTAAAACCAAAAAAGAGAAAGCCACCCCTGCCTAGTGTTAGGAAGTTAGTAGAAGATAGATAGAACAAGCCCCAGAGAACCAAGGGCCGCAGAAGGAACCATACAATGAATGGGCACTAGAGCTTTTAGAGGAACTCAAGCAGGAAGCTGTCAGACACTTTCCTAGAGTATGGCTCCATAGCTTAGGACAGTATATCTATGAAACATATGGAGACACTTAGACAGAAGTTGAAGCTTTGATAAGAATACTGCAACAACTATTGTTTATTCATTTCAGAATTAGGTGCCAGCATAGCAGAATAGGCATTATACCACATAGAAGAGCAAGAAATGGAGCTAATAGATCCTAACCTAGAGCCCTGGAAGCATCCAGGAAGTCAGCCTACAACCCCTTGTACTCCATGCTATTGTAAAAGATGCAGCTATCATTGTTTAGTTTGCTTTCAGAAAAAAGGCTTAGGCATTTACTATGGCAGAAAGAAGCGGAGACAGCGACGAAGCACTCCTCCAAGCAATAAGGATCATCAAAATCCTATACCAGAGCAGTAAGTACCATATAGTAGATGTAATGTTAGATCTAGATTATAGAATAGGAGTAGCAGCACTTGTAGTAGCACTAATCATAGCAATAATTGTGTGGATAATAGTATATCTAGAATATAGAAAACTAGTAAAACAAAAGAGAATAGATTAGTTAATTAAGAGAATTAAAGAAAAAGAAGAAGACAGTGGCAATGAGAGTGAGGGAGATACTGAGGAATTGGCAACAATAGTAGATATAGGGCATCTTAAGCTTTTGGCTGCTAATGAGTTGTAATGTGAGAGGAAATTTGTAGGTCACAGTCTATTATGGAGTACCTGTGTGGAAAGAAGCAAAAACTACTCTATTCTGTGCATCAGATGCTAAAGCATATGAGACAGAAGCGCATAATGTCTAGGCTACACATGCCTGTGTACCCACAGACCCTAACCCACAGGAAATAGTTTTAGAAAATGTAACAGAAAATTTTAATATGTAAGAAAATAGTATGGTAGATCAGATGCATAAAGATGTAATCAGTTTATAAGATCAAAGCCTAAAGCCATGTGTAAAGATGACCCCACTCTGTGTCACTCTACATTGTACAAATGTAACAGGTAATGATAGCCGTACTGTTGACAATGACACCATAAGAGAAATAAAAAATTGCTCTTTCAATGCAACTACAGAAATAAAAGATAAGATAAAGAAGGAGTATGCACTTTTCTATAGACTTGATGTAGTACCACTTAAGGACACCAACTCTAGTGAATATATATTAATAAATTGTAATTCTTCAACCATATCACAGGCCTGTCCAAAGATCTCTTTTGACCCAATTCCTATACATTATTGTGCTCCGGCTAGTTATGCGATTCTAAAATGTAATAATAAAACATTTAATAGGTCAGGACCATGTCAGAATGTCAGTACAGTACAATGTACACATAGAATTAAGCCAGTGGTATCAACTCAATTACTGTTAAATAGTAGCCTAGCAGAAGAAGAGATAATAATCAGATCTAAGAATCTGAGCGAGATTAGACACACAATAATAGTACATCTTAATAAATCTGTAAAGATTATATGTACAAGACCCAACAATAATACAAGAAAAAGTGTAAAGATAGGACCAAGACAGACATTCTATGCAACAGAAGATATAATAAGAGACATAAGACAAGCACACTGTAACATTAATAAAAAGAACTAGACTGAAACCTTAGAAAGGGTAGGGGAAAAATTAAAGGAACACTTCCCTAATAAAACAATAACATTTAAACCATCCGCAGGAAAGGACCCAGAAGTCACAACACATATGTTTAATTGTAGAGGAGAATTTTTCTATTGCAACACATCAGGCCTGTTTAATAGTACATTTAATGGTACACACCTGAATAATACATTCAGGCCTAATAGTACAGACATCATCACACTCCAATGCAGAATAAAACAAATTGTAAACATGTGGCAGGAAGTAAGACGAGCAATGTATGCCCCTCCCATTGCAGGAAATATAACATGTAGATCAAATATCACAGGACTACTATTGACACGTGATGGCGGAAAGAATGAAACTAATGATACCACAGAGATATTCAGACCTATAAAAAGAGATATGAAAGATAATTAGAGAAGTGAACTATATAAATATAAAGTAGTAGAAATTAAGCCATTAAGAATAGCACCCACCGAGGCAAAAAGGAGAGTGGTGGAGAGAGAAAAAAGAGCAGTAGGACTAAGAGCTGTGTTCCTTAGGTTCTTAGGAGCAGCAGGAAGCACTATAGGCGCAGCGTCAATAACGCTGACAGTACAAGCCAGACAATTGCTGTCTAGAATAGTGCAACAGCAAAACAATTTGCTGAGGGCTATAGAGGCGCAGCAGCATATGTTGCAACTCACGGTCTAGGGCATTAAGCAACTCCAGGCAAGAGTCCTTGCTATAGAAAGATACCTAAAAGATCAACAGCTCCTAGGGATTTAGGGCTGCTCTAGAAAACTCATCTGCACCACTGCAGTGTCTTAGAACAGTAGTTAGAGTAATAAAACTCTAAGAGATATTTAGAATAACATGACCTAGATGCAGTAGGATAAAGAAATTAGTAATTATACAGAAATAATATATGATCTACTTGAAAAATCGCAGATCCAGCAAGAAAACAATAAAAAAGATTTACTAGCATTGGACAGTTAGAATAATCTGTAGAATTAGTTTAACATATCAAATTAGTTGTGGTATATAAAAATATTTATAATGATAGTAGGAGGCTTGATAAGTTTGAGAATAACTTTTGCTGTGCTTTCTATAGTGAATAGAGTTAGGCAAGGATACTCACCTCTGTCATTGCAGACCCTTACCCAGAACCCAGAAAGACTCGACAGGCTCAGAAGAATCGAAGAAGAAAGTAGAGAGCAAGACAAAGACAAATCCATTCGATTAGTGAGCAGATTCTTAGCACTTGCCTAGGACGACCTGCGGAGCCTATGCCTTTTCAGCTACCACCGATTGAGAGACTTTATATTGATTGCAGCGAGAGCAGTAGAACTTCTAAGACACAGCAGTCTCAGAGGACTACAGAAAGGTTAGGAAATCCTTAAGTATCTAGGAAGTCTTGTGCAATATTGAGGTCTAGAGCTAAAAAAGAGTGCTATTAATTTGTTTGATACCATAGCAATAGCAGTAGCTGAAGGAACAGATAAGATTCTAATATACTTACAAAGAATTTATAAAGCCATCTGCAACATACAAGAATAAGACAAAGATTTGCAGCAGCTTTGCAATAAAATAGGGGGCAAGTGGTCAAAAAGCAGTGTAGTAAGGTGGCCTGCTATAAAAGAAAGAATACAACGCACTGCTCCAGCAGCAGAAGGAGTAAGAGCAGCATCTCAAGACCTAGATAGACACAGGGCACTTACAACCAGCAATACAGCCACCAATAATGCTGCTTGTGCCTGGCTAGAAGCGCAAGAGGAGAGAGACGATGTAAGCTTTCCAGTCAGACCTCAGGTACCTTTAAGGCCAATGACATATAAGGCAGCATTTGATCTCGGCTTCTTTTTAAAAGAAAAGGGGGGACTGGAAGGGTTAATTTACTCTAGGAAAAGGCAAGAGATTCTTGATTTGTAGGTCTATCACACACAAGGCTTCTTCCCTGATTGGCAAAACTACACACCGGGACCAGGGGTCAGATACCCACTGACCTTTGGATGGCCATTCAAGCTAGTGCCAGTCGACCCAAGGGAAATAGAAGAGAACAGCAACAGGGAGAACAACTGCTTGCTACACCCTGAGAACCAGCATAGAATGGAGGATGACCACAGAGAAGTGTTAAGATGGAAGTTTGACAGTCAACTAGCACGCAGACATATGGCCCGTGAGCTACATCCGGAGTAGTACAAAGACTAAGACTGCTGACACAGAAAGGACTTTCCGCTAGGACTTTCCACTGAGGCGTTCCAGGAGGTGTGGTCTAGGCGGAACAAGGAGTAGTCAACCCTCAGACGCTGCATATAAGCAGCTGCTTTTTGCCTGTACTAGGTCTCTCTAAGTAGACCAGATCCGAGCCTAGGAGCTCTCTAGCTATCTAAGGAACCCACTGCTTAAGCCTCAATAAAGCTTGCCTTGAGTGC
```
### Documentation

Doxygen generated documentation can be found at [https://veg.github.io/cawlign/html/index.html](https://veg.github.io/cawlign/html/index.html). A PDF version is available at [https://veg.github.io/cawlign/cawlign-doxygen-docs.pdf](https://veg.github.io/cawlign/cawlign-doxygen-docs.pdf).

This documentation is generated automatically as part of the CI/CD pipeline (see `.github/workflows/doxygen.yml`). 

