---
jupytext:
  formats: ipynb,md:myst
  text_representation:
    extension: .md
    format_name: myst
    format_version: 0.13
    jupytext_version: 1.14.4
kernelspec:
  display_name: Python 3
  name: python3
---

+++ {"id": "xgSYgIQhLJgx"}

# Named axes and easy-to-revise parallelism with `xmap`

This tutorial introduces `jax.xmap` and the named-axis programming model that comes with it. By reading this, you'll learn how to write error-avoiding, self-documenting functions using named axes, then control how they're executed on hardware at any scale, from your laptop CPU to the largest TPU supercomputer.

We start with a toy neural network example.

---
> **xmap is an experimental API. Expect rough edges and changes in the future!**
---

+++ {"id": "7kppYlqDLJg9"}

## From positions to names in a toy neural network

Presentations on JAX often start with a simple neural network prediction function and loss, written in pure NumPy. Here's a simple network with one hidden layer:

```{code-cell} ipython3
:id: 0GlEQAFMLJg9

import os
os.environ["XLA_FLAGS"] = '--xla_force_host_platform_device_count=8' # Use 8 CPU devices
```

```{code-cell} ipython3
:id: mlzPDW6BT3mQ

import jax.numpy as jnp
from jax import lax
from jax.nn import one_hot, relu
from jax.scipy.special import logsumexp

def predict(w1, w2, images):
  hiddens = relu(jnp.dot(images, w1))
  logits = jnp.dot(hiddens, w2)
  return logits - logsumexp(logits, axis=1, keepdims=True)

def loss(w1, w2, images, labels):
  predictions = predict(w1, w2, images)
  targets = one_hot(labels, predictions.shape[-1])
  losses = jnp.sum(targets * predictions, axis=1)
  return -jnp.mean(losses, axis=0)
```

+++ {"id": "E3shL0J4LJg-"}

We can then initialize inputs with the right shapes and compute the loss value:

```{code-cell} ipython3
:id: F3uZUkDtLJg-

w1 = jnp.zeros((784, 512))
w2 = jnp.zeros((512, 10))
images = jnp.zeros((128, 784))
labels = jnp.zeros(128, dtype=jnp.int32)

print(loss(w1, w2, images, labels))
```

+++ {"id": "8V4xLOE2LJg_"}

Here's how we might write the same function using named axes. Don't worry if you can't follow the API details. They are not important now and we will explain everything step-by-step afterwards. This is just to show you what you can do with xmap before you learn them!

```{code-cell} ipython3
:id: 0-cxtgHjLJg_

def named_predict(w1, w2, image):
  hidden = relu(lax.pdot(image, w1, 'inputs'))
  logits = lax.pdot(hidden, w2, 'hidden')
  return logits - logsumexp(logits, 'classes')

def named_loss(w1, w2, images, labels):
  predictions = named_predict(w1, w2, images)
  num_classes = lax.psum(1, 'classes')
  targets = one_hot(labels, num_classes, axis='classes')
  losses = lax.psum(targets * predictions, 'classes')
  return -lax.pmean(losses, 'batch')
```

+++ {"id": "DQFMOqvPLJg_"}

This code is simpler: we don't need to worry about axis order when calling functions like `jnp.dot`, or remember which axis position to reduce over with `logsumexp`, `jnp.sum`, or `jnp.mean`.

But the real win is that names let us use `xmap` to control our function's execution. At its simplest, `xmap` will just vectorize over all named axes, so that the function is executed just like its positional-axis counterpart:

```{code-cell} ipython3
:id: PuVt-LPyLJg_

from jax.experimental.maps import xmap

in_axes = [['inputs', 'hidden', ...],
           ['hidden', 'classes', ...],
           ['batch', 'inputs', ...],
           ['batch', ...]]

loss = xmap(named_loss, in_axes=in_axes, out_axes=[...])
print(loss(w1, w2, images, labels))
```

+++ {"id": "3zpaQh5fLJhA"}

But on a whim we can decide to parallelize over the batch axis:

```{code-cell} ipython3
:id: xnuetq3xLJhA

import jax
import numpy as np
from jax.sharding import Mesh

loss = xmap(named_loss, in_axes=in_axes, out_axes=[...],
            axis_resources={'batch': 'x'})

devices = np.array(jax.local_devices())
with Mesh(devices, ('x',)):
  print(loss(w1, w2, images, labels))
```

+++ {"id": "CeBeCIkOLJhA"}

Or we might want to perform model parallelism over the hidden axis:

```{code-cell} ipython3
:id: WlOGKPcNLJhB

loss = xmap(named_loss, in_axes=in_axes, out_axes=[...],
            axis_resources={'hidden': 'x'})

devices = np.array(jax.local_devices())
with Mesh(devices, ('x',)):
  print(loss(w1, w2, images, labels))
```

+++ {"id": "BMfvN_vaLJhB"}

Or we might want to do both model and batch data parallelism at once:

```{code-cell} ipython3
:id: 4HfECQU0LJhB

loss = xmap(named_loss, in_axes=in_axes, out_axes=[...],
            axis_resources={'batch': 'x', 'hidden': 'y'})

devices = np.array(jax.local_devices()).reshape((4, 2))
with Mesh(devices, ('x', 'y')):
  print(loss(w1, w2, images, labels))
```

+++ {"id": "g94RnX6FLJhB"}

With `xmap`, we can revise our parallelism strategy on a dime, without needing to rewrite our neural network function.

+++ {"id": "mW3Rj3TR2dut"}

## Preliminaries

```{code-cell} ipython3
:id: ZJFCIhOW3fUI

import jax.numpy as jnp
from jax import lax
from functools import partial
import jax
import numpy as np
```

+++ {"id": "ELXdjCCB2ior"}

To better illustrate the new programming model, we make extensive use of custom type annotations in this notebook. The annotations have no effect on how the code evaluates and will be unchecked for now.

```{code-cell} ipython3
:id: Hvkazo9j30Ac

from typing import Any, Callable

class ArrayType:
  def __getitem__(self, idx):
    return Any
f32 = ArrayType()
i32 = ArrayType()
```

+++ {"id": "VetbAa8hT_7I"}

## Tensors with named axes

The NumPy programming model is based around nd-arrays. Each nd-array can be associated with a two-component type:
* the element type (accessible via the `.dtype` attribute)
* shape (a tuple of integers given by `.shape`).

Using our little type annotation language, we will write these types as `dtype[shape_tuple]`.

> For example, a 5x7x4 array of 32-bit floating point numbers will be denoted as `f32[(5, 7, 4)]`.

Here is a small example that shows how the annotations can demonstrate the way shapes propagate through a simple NumPy program:

```{code-cell} ipython3
:id: f75wjZfN3bFa

x: f32[(2, 3)] = np.ones((2, 3), dtype=np.float32)
y: f32[(3, 5)] = np.ones((3, 5), dtype=np.float32)
z: f32[(2, 5)] = x.dot(y)  # matrix multiplication
w: f32[(7, 1, 5)] = np.ones((7, 1, 5), dtype=np.float32)
q: f32[(7, 2, 5)] = z + w  # broadcasting
```

+++ {"id": "VcreZIh242Y0"}

The extension we propose is to add another component of array type: a `named_shape`, mapping axis names (arbitrary hashable objects, with strings being a common choice) to integer sizes. Most importantly, because each axis has a name, their order has no meaning. That is, a named shape of `{'a': 2, 'b': 5}` is indistinguishable from a named shape of `{'b': 5, 'a': 2}`.

> This is not an entirely new idea. Some good examples of where using named axes has been proposed in the past are: [Mesh TensorFlow](https://github.com/tensorflow/mesh), [Tensor Considered Harmful](http://nlp.seas.harvard.edu/NamedTensor) manifesto as well as the [xarray](http://xarray.pydata.org/en/stable/) and [einops](http://einops.rocks/) packages. Keep in mind that many of those are slightly different in that they do assign an order to the named axes, but they are unordered in JAX.

From now on we will allow the type annotations to have two components, the first one still being the value's `.shape`, while the second one will be the `.named_shape`.

```{code-cell} ipython3
:id: mnpgbnDJt3vz

e: f32[(5, 7), {'batch': 20, 'sequence': 30}]
# e.shape == (5, 7)
# e.named_shape == {'batch': 20, 'sequence': 30} == {'sequence': 30, 'batch': 20}
```

+++ {"id": "RWytlNvpGhvu"}

While we don't modify the meaning of `.ndim` (which is always equal to `len(shape)`) and `.size` (equal to the product of `shape`), we do so solely for backward-compatibility reasons. The true rank of an array that has non-empty named axes is `len(shape) + len(named_shape)`. The true number of elements stored in such an array is equal to the product of sizes of all dimensions, both positional and named.

+++ {"id": "uwPDIrykt34L"}

## Introducing and eliminating named axes

But how does one create such arrays, if all top-level JAX operations work in the NumPy model with purely positional axes? While this constraint could be lifted at some point, for the time being the only way to introduce named axes is to use `xmap`.

`xmap` can be thought of as an adapter that takes in arrays with positional axes, makes some of them named (as specified by `in_axes`), and calls the function that it wraps. Once the wrapped function returns arrays, all named axes appearing in those are converted back to positional axes (as specified by `out_axes`).

`in_axes` should have a structure that matches the signature of the `xmap`ped function arguments, except with all places where array arguments would be replaced by an _axis mapping_. There are two ways in which axis mappings can be specified:
* as dictionaries mapping positional axes to axis names (e.g. `{0: 'x', 2: 'y'}`); and
* as lists of axis names terminated by the ellipsis object (e.g. `['a', 'b', ...]`), indicating that a prefix of positional dimensions are to be mapped to given names.

`out_axes` are similar, except that their structure has to match the return signature of the `xmap`ped function (but again, with all arrays replaced by axes mappings).

For each array argument, all positional axes mentioned in its respective `in_axes` axis mapping are converted to named axes. For each array result, all named axes are inserted in the positions indicated by its respective `out_axes`.

```{code-cell} ipython3
:id: CaW3EtPTvwHY

from jax.experimental.maps import xmap

def my_func(x: f32[(5,), {'batch': 20}]) -> f32[(5,), {'batch': 20}]:
  assert x.shape == (5,)
  # assert x.named_shape == {'batch': 20}  # TODO: Implement named_shape
  return x

x: f32[(20, 5)] = jnp.zeros((20, 5), dtype=np.float32)
f = xmap(my_func,
         in_axes={0: 'batch'},   # Name the first axis of the only argument 'batch'
         out_axes={1: 'batch'})  # Place the 'batch' named axis of the output as the second positional axis
y: f32[(5, 20)] = f(x)
assert (y == x.T).all()  # The first dimension was removed from x and then re-inserted as the last dim
```

+++ {"id": "8E7ISmwju0x1"}

While this might seem like a handful at first, if you've seen code that uses `jnp.einsum` you are already familiar with this approach. The `einsum` function interprets an expression such as `nk,km->nm` assigning names (each letter is considered a separate name) to positional axes, performing necessary broadcasts and reductions, and finally putting back the results in positional axes, according to the order given by the right-hand side of the `->` separator. While `einsum` never lets you interact with named axes directly, they do appear naturally in its implementation. `xmap` is a _generalized einsum_ because named axes are now first-class and you get to implement the function that can manipulate them.

Continuing this analogy, `xmap(my_func, ...)` from the above example is equivalent to `jnp.einsum('bx->xb')`. But of course not every `xmap`ped function will have an equivalent `einsum`.

One more similarity with `einsum` is that whenever a name is reused for multiple axes, they do have to have the same size:

```{code-cell} ipython3
:id: eOZOWVsP0CO6

x = jnp.arange(5)
y = jnp.arange(7)
try:
  jnp.einsum('i,i->i', x, y)
except Exception as e:
  print('einsum:', e)
try:
  xmap(lambda x, y: x * y,
       in_axes=(['i', ...], ['i', ...]),
       out_axes=['i', ...])(x, y)
except Exception as e:
  print('xmap:', e)
```

+++ {"id": "OPvXaOZsxtMV"}

## Named axis propagation

We now know how named axes are introduced and eliminated, but what are they good for? How do they propagate throughout the program? Let's explore a few examples.

+++ {"id": "Px1O29LKz-yo"}

### Interactions with positional axes

First rule: named axes never implicitly interact with positional axes. Any function that's written without named axes in mind can always be invoked with inputs that have named dimensions. The result is the same as if `vmap` was applied on a per-named-axis basis.

```{code-cell} ipython3
:id: v9slhhcTyGMD

from jax.scipy.linalg import expm_frechet

# Any other function that does not assume existence of any named axes would do too,
# at least as long as it matches this type signature:
expm_frechet: Callable[[f32[(3, 3)], f32[(3, 3)]], f32[(3, 3)]]
f = partial(expm_frechet, compute_expm=False)

# Each A with each E
batch_A = jnp.ones((5, 3, 3), dtype=np.float32)
batch_E = jnp.ones((5, 3, 3), dtype=np.float32)
batch_AE = xmap(f,
                in_axes=(['b', ...], ['b', ...]),      # Map first axes of both inputs to 'b'
                out_axes=['b', ...])(batch_A, batch_E) # Place 'b' as the first positional axis in the result
for i in range(5):
  np.testing.assert_allclose(batch_AE[i], f(batch_A[i], batch_E[i]))

# All-pairs of As and Es
batch_A = jnp.ones((7, 3, 3), dtype=np.float32)
batch_E = jnp.ones((5, 3, 3), dtype=np.float32)
batch_AE = xmap(f,
                in_axes=(['ba', ...], ['be', ...]),           # Map first axes of inputs to 'ba' and 'be' respectively
                out_axes=['ba', 'be', ...])(batch_A, batch_E) # Prefix all positional dimensions of output with 'ba' and 'be'
for i in range(7):
  for j in range(5):
    np.testing.assert_allclose(batch_AE[i,j], f(batch_A[i], batch_E[j]))
```

+++ {"id": "3Gmgwh-qsAq_"}

### Broadcasting

Secondly, named axes are broadcast _by name_, and every existing NumPy (and almost every JAX) operator implicitly broadcasts the named dimensions. Whenever a standard NumPy function is called with arrays with named axes, the NumPy function determines the positional shape of the result array, while the named shape becomes a union of all named shapes of its inputs. Analyze the following example to understand how the axes propagate:

```{code-cell} ipython3
:id: g-NSgbhkyIBI

def named_broadcasting(
    x: f32[(2, 1, 1), {'a': 2}],
    y: f32[(1, 3, 1), {'b': 3}],
    z: f32[(1, 1, 5), {'c': 5}]) \
      -> f32[(2, 3, 5), {'a': 2, 'b': 3, 'c': 5}]:
  i: f32[(2, 3, 1), {'a': 2, 'b': 3}] = x + y
  j: f32[(1, 3, 5), {'b': 3, 'c': 5}] = y + z
  k: f32[(2, 3, 5), {'a': 2, 'b': 3, 'c': 5}] = i + j
  return k

x = jnp.ones((2, 2, 1, 1), dtype=np.float32)
y = jnp.ones((3, 1, 3, 1), dtype=np.float32)
z = jnp.ones((5, 1, 1, 5), dtype=np.float32)
k = xmap(named_broadcasting,
         in_axes=(['a', ...], ['b', ...], ['c', ...]),
         out_axes=['a', 'b', 'c', ...])(x, y, z)
assert k.shape == (2, 3, 5, 2, 3, 5)
```

+++ {"id": "5DOQimOayFRh"}

To recap, the named shape of the result of an expression such as `i + j` with `i` having a named shape of `{'a': 2, 'b': 3}` and `j` of `{'b': 3, 'c': 5}` is `{'a': 2, 'b': 3, 'c': 5}`. The `'b'` axis is present in both inputs, so no broadcasting is necessary, while `'a'` and `'c'` occur in only one of the two inputs, causing the other one to get broadcast along the axis missing in its named shape.

No shape errors can occur when operating over named axes, because `xmap` enforces that a single name is associated with a single size inside its body.

> While the rule for broadcasting named axes might seem like an arbitrary extension of the NumPy model, it is actually consistent with it.
>
> Broadcasting first looks for pairs of dimensions it considers as equivalent in both operands. For all matched pairs, it asserts that both sizes are equal or one of them is 1. All unpaired dimensions are carried over to the result.
>
> Now, in the positional world the way NumPy broadcasting chooses to form the pairs is by right-aligning the shapes. But our axes are named, so there is a straightforward way of finding equivalent axes: just check their names for equality!

+++ {"id": "upHKB9x-sBTi"}

### Reductions

But named axes are not only good for batching! In fact, our goal is that named axes should be equivalent to positional axes. In particular, every NumPy function that takes in positional axes as arguments should also accept named axes.

> The paragraph above is aspirational and the set of NumPy functions that do accept named axes is relatively limited. At the moment named axes are only supported in:
> * `jnp.sum`, `jnp.max`, `jnp.min`

Reductions are a good example:

```{code-cell} ipython3
:id: o4MUDvTayIDK

def named_broadcast_and_reduce(
    x: f32[(), {'x': 2}],
    y: f32[(5,), {'y': 4}]) \
      -> f32[()]:
  z: f32[(5,), {'x': 2, 'y': 4}] = x + y
  w: f32[()] = jnp.sum(z, axis=(0, 'x', 'y'))
  # We could also reduce in steps:
  # w0 : f32[(), {'x': 2, 'y': 4}] = jnp.sum(z, 0)      # eliminate the positional axis
  # w0x: f32[(), {'y': 4}]         = jnp.sum(w0, 'x')   # eliminate the `x` axis
  # w  : f32[()]                   = jnp.sum(w0x, 'y')  # eliminate the `y` axis
  return w

positional_broadcast_and_reduce: Callable[[f32[(2,)], f32[(5, 4)]], f32[()]]
positional_broadcast_and_reduce = \
  xmap(named_broadcast_and_reduce,
       in_axes=({0: 'x'}, {1: 'y'}),
       out_axes={})
positional_broadcast_and_reduce(jnp.arange(2, dtype=np.float32),
                                jnp.arange(20, dtype=np.float32).reshape((5, 4)))
```

+++ {"id": "2LPCZZIHr84C"}

### `einsum`

Similarly to how we have extended reductions with support for named axes, we've also made it possible to contract over named axes using `jnp.einsum`.

Operands and results still use a convention of one letter per positional axis, but now it is also possible to mention named axes in curly braces. For example, `n{b,k}` implies that a value will have a single positional dimension `n` and named dimensions `b` and `k` (their order doesn't matter). Following the usual einsum semantics, any named axes that appear in inputs, but do not appear in an output will be contracted (summed after all multiplications are performed).

It is acceptable to omit a named dimension from _all arguments and the result_ in which case it will be treated according to the usual broadcasting semantics. However, it is not acceptable to mention a named axis in one argument that has it in its named shape and skip it in another argument that also has it in its named shape. Of course, skipping it in the arguments that don't have it is required.

> NOTE: This invariant is **unchecked** at the moment (it is still work-in-progress). Such axis skipping will result in undefined behavior.

> At the moment `jnp.einsum` with named axes only supports two inputs and a single result.

```{code-cell} ipython3
:id: rQgkof6lyJps

def named_batch_matrix_single_matrix(
    x: f32[(5,), {'b': 20, 'k': 7}],
    y: f32[(), {'k': 7, 'm': 11}]) \
      -> f32[(5,), {'b': 20, 'm': 11}]:
  return jnp.einsum('n{b,k},{k,m}->n{b,m}', x, y)

x = jnp.ones((20, 5, 7))
y = jnp.ones((7, 11))
z = jnp.einsum('bnk,km->bnm', x, y)
zx = xmap(named_batch_matrix_single_matrix,
          in_axes=[{0: 'b', 2: 'k'}, ['k', 'm', ...]],
          out_axes={0: 'b', 2: 'm'})(x, y)
np.testing.assert_allclose(z, zx)
```

+++ {"id": "JuFg25Yro4kZ"}

The example above is admittedly no clearer than using `jnp.einsum` directly. But contractions over named axes are a crucial component of larger applications such as Transformer models and this is only meant to be an exercise to show you how the names propagate.

+++ {"id": "Ydrpm0wlzfp2"}

### Collectives

Finally, all collectives that could have been used with `pmap`ped functions also work with named axes. As we'll show later, `xmap` can be used as a drop-in replacement for `pmap` that makes programming for multi-dimensional hardware meshes much easier.

```{code-cell} ipython3
:id: HPRGocb40v8y

x = jnp.arange(8)
xmap(lambda x: lax.pshuffle(x, 'i', list(reversed(range(8)))),
     in_axes=['i', ...], out_axes=['i', ...])(x)
```

+++ {"id": "ZHoCsWkCEnKt"}

## Parallelism support

While the new programming paradigm can be nice at times, the killer feature of `xmap` is its ability to parallelize code over supercomputer-scale hardware meshes!

> Named axes are the secret sauce that makes all this possible, thanks to the carefully tuned rules that describe their propagation. Good support for partitioning in a purely positional programming model is notoriously difficult. Positional axes are usually disposable and it is hard to keep track of the way axis partitioning propagates through the program. As you'll see below, named axes enable us to define a straightforward correspondence between their names and hardware resources, making it easy to reason about the way different values end up partitioned.

In all the previous examples, we haven't said a word about parallelism and for a good reason. By default `xmap` doesn't perform any parallelization and vectorizes the computation in the same way `vmap` does (i.e. it still executes on a single device). To partition the computation over multiple accelerators we have to introduce one more concept: _resource axes_.

The basic idea is that logical axes (the ones that appear in named shapes) assume that we have abundant hardware and memory, but before the program is to be executed, they have to be placed somewhere. The default (`vmap`-like) evaluation style pays a high memory cost on the default JAX device. By mapping logical axes to (one or more) resource axes through the `axis_resources` argument, we can control how `xmap` evaluates the computation.

```{code-cell} ipython3
:id: NnggOzOD8rl1

x = jnp.ones((2048, 2048))

local_matmul = xmap(jnp.vdot,
                    in_axes=({0: 'left'}, {1: 'right'}),
                    out_axes=['left', 'right', ...])
distr_matmul = xmap(jnp.vdot,
                    in_axes=({0: 'left'}, {1: 'right'}),
                    out_axes=['left', 'right', ...],
                    axis_resources={'left': 'x', 'right': 'y'})
```

+++ {"id": "gNP4TFiAQcwq"}

Both `local_matmul` and `distr_matmul` implement matrix multiplication, but `distr_matmul` will additionally partition the `left` and `right` logical axes over the `x` and `y` resource axes.

+++ {"id": "mjmKyo-QTqKG"}

### But... where do those resource names come from?

Well, it depends, but one good choice is... a hardware mesh!

For our purposes a mesh is an nd-array of devices with named axes. But, because NumPy doesn't support named axes (that's our extension!), the meshes are represented by a pair of an nd-array of JAX device objects (as obtained from `jax.devices()` or `jax.local_devices()`) and a tuple of resource axis names of length matching the rank of the array.

+++ {"id": "x3EXj1TMwZtS"}

<img alt="How real hardware is represented as an abstract mesh" src="data:image/png;base64,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" />

```{code-cell} ipython3
:id: BOYbaJ7QUEbJ

axis_names = ('x', 'y')
mesh_devices = np.array(jax.devices()).reshape((2, 4))
assert len(axis_names) == mesh_devices.ndim
mesh_def = (mesh_devices, axis_names)
mesh_def
```

+++ {"id": "VmPuQpXdU_wm"}

The mesh axis names are exactly the names of resources that named axes can be mapped to. But just creating a mesh definition won't make the resource names visible to `distr_matmul`:

```{code-cell} ipython3
:id: HGVwUOErVJzR

try:
  distr_matmul(x, x)
except Exception as e:
  print(e)
```

+++ {"id": "KHbRwYl0BOr1"}

To introduce the resources in a scope, use the `with Mesh` context manager:

```{code-cell} ipython3
:id: kYdoeaSS9m9f

from jax.sharding import Mesh

local = local_matmul(x, x)  # The local function doesn't require the mesh definition
with Mesh(*mesh_def):  # Makes the mesh axis names available as resources
  distr = distr_matmul(x, x)
np.testing.assert_allclose(local, distr)
```

+++ {"id": "NB_I4gqt8M3v"}

Anyway, the best part of it is that specifying `axis_resources` **never changes program semantics**. You are free to experiment with different ways of partitioning your computation (just change the assignment of resources to named axes!) and even how the physical devices are organized in the mesh (by changing the construction of the NumPy array of devices). None of those things should have any significant influence on the results you get back (up to, for example, floating point inaccuracy), though of course some of them will achieve significantly better performance than the others.

`xmap` doesn't provide any automatic scheduling options at the moment, because the best schedule often has to be somewhat carefully matched to your program. We're considering adding support for that in the future, but it will take time.

> Once you map a logical axis to a mesh dimension, the size of that logical axis has to be divisible by the mesh dimension size.

+++ {"id": "C1N6yqV_PVVv"}

### Is my data replicated? Or partitioned? Where is it?

Named axes also give us a neat way of reasoning about partitioning and replication. A value is partitioned over a mesh axis if and only if it has a named axis that has been mapped to that mesh axis in its shape. Otherwise, it will be replicated over all slices along that axis.

For example, assume that we're in an `xmap` that had `axis_resources={'a': 'x', 'b': 'y'}` specified (i.e. we are running the computation over a 2D mesh with `x` and `y` axes with sizes 2 and 3 respectively). Then:
* An array of type `f32[(5, 5), {}]` is completely replicated over the whole mesh. All devices store a local copy of the value.
* An array of type `f32[(6,), {'a': 8}]` is partitioned over mesh axis `x`, because it has `'a'` in its named shape, and `'a'` is mapped to `x`. It is replicated over mesh axis `y`. To put it differently, all devices in a slice of the mesh with the same `x` coordinate will store a local copy of a chunk of this array. But, mesh slices with different `x` coordinates will store different chunks of the data.
* An array of type `f32[(), {'a': 8, 'c': 7}]` is partitioned just like in the previous case: split over the `x` mesh axis and replicated over the `y` axis. Named dimensions with no resources specified are no different than positional dimensions when considering partitioning, so `'c'` has no influence on it.
* An array of type `f32[(), {'a': 8, 'b': 12}]` is completely partitioned over the whole mesh. Every device holds a distinct chunk of the data.

+++ {"id": "RzIFv8oJwQ2M"}

<img alt="An illustration for the above examples" src="data:image/png;base64,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" />

+++ {"id": "zTqR_meLsl8C"}

This also highlights one restriction: `xmap` won't complain if you specify `axis_resources={'a': 'x', 'b': 'x'}`, but consider how would an array with type `f32[(2, 8), {'a': 4, 'b': 12}]` be partitioned. If the size of the `x` mesh axis is 2, then we only have 2 devices, but we have 4 chunks to place (2 along `'a'` and 2 along `'b'`)! Now we can state it in full: **named axes mapped to the same resources can never both appear in the named shape of a single array**. But they can appear in named shapes of two distinct arrays, such as in this program:

```{code-cell} ipython3
:id: BG5AxRf1ZlBv

def sum_two_args(x: f32[(), {'a': 4}], y: f32[(), {'b': 12}]) -> f32[()]:
  return jnp.sum(x, axis='a') + jnp.sum(y, axis='b')

q = jnp.ones((4,), dtype=np.float32)
u = jnp.ones((12,), dtype=np.float32)
with Mesh(np.array(jax.devices()[:4]), ('x',)):
  v = xmap(sum_two_args,
           in_axes=(['a', ...], ['b', ...]),
           out_axes=[...],
           axis_resources={'a': 'x', 'b': 'x'})(q, u)
  print(v)
```

+++ {"id": "GcMAKNqeZlJr"}

This program is valid, because `jnp.sum` eliminates the axes that cannot co-occur before the values are added.

> While the final release of `xmap` will ensure that you don't accidentally end up doing so, the current implementation _doesn't verify it_. Violating this restriction will result in _undefined behavior_.

+++ {"id": "5QMu7izKSwzu"}

### Why `axis_resources` and not a more direct mapping to hardware?

At this point you might wonder why go through the detour of introducing yet another concept of resource axes in the mix. For as long as you're interested in partitioning your computations over hardware, there is no good reason, but this mental framework is more flexible than that!

For example, there is one additional resource we all deal with: time! Just like a computation can be partitioned over multiple hardware devices, e.g. to lower its memory usage, the same thing can be achieved with a single accelerator that evaluates a chunk of the computation in multiple steps.

So, while hardware meshes are the only source of resource axes in JAX programs at the moment, we are planning to extend the whole system with other sources.

+++ {"id": "_0hswtHXrLXq"}

## Porting positional code to named code

In this section we will go over a few more real examples to show how `xmap` can help you implement and distribute various models.

> **This section is a work in progress**
