/* OOQP                                                               *
 * Authors: E. Michael Gertz, Stephen J. Wright                       *
 * (C) 2001 University of Chicago. See Copyright Notification in OOQP */

#ifndef MA57LINSYS_H
#define MA57LINSYS_H

#include "DoubleLinearSolver.h"
#include "SparseSymMatrixHandle.h"
#include "SparseStorageHandle.h"
#include "OoqpVectorHandle.h"
#include "SparseStorage.h"

//J.Currie Edit to enable 64bit integers in MA57
#ifdef X64MA57
	typedef ptrdiff_t ipfint;
#else
	typedef int ipfint;
#endif

extern "C" {
  void MA57ID( double cntl[],  ipfint icntl[] );

  void MA57AD( ipfint * n,        ipfint * ne,       ipfint irn[],     
		ipfint jcn[],      ipfint * lkeep,    ipfint keep[],
		ipfint iwork[],    ipfint icntl[],    ipfint info[],
		double rinfo[] );

  void MA57BD( ipfint * n,        ipfint * ne,       double a[],
		double fact[],  ipfint * lfact,    ipfint ifact[],
		ipfint * lifact,   ipfint * lkeep,    ipfint keep[],
		ipfint ppos[],     ipfint * icntl,    double cntl[],
		ipfint info[],     double rinfo[] );
  void MA57CD( ipfint * job,      ipfint * n,        double fact[],
		ipfint * lfact,    ipfint ifact[],    ipfint * lifact,
		ipfint * nrhs,     double rhs[],   ipfint * lrhs,
		double w[],     ipfint * lw,       ipfint iw1[],
		ipfint icntl[],    ipfint info[]);
  void MA57DD( ipfint * job,      ipfint * n,        ipfint * ne,
		double a[],     ipfint irn[],      ipfint jcn[],
		double fact[],  ipfint * lfact,    ipfint ifact[],
		ipfint * lifact,   double rhs[],   double x[],
		double resid[], double w[],     ipfint iw[],
		ipfint icntl[],    double cntl[],  ipfint info[],
		double rinfo[] );
  void MA57ED( ipfint * n,        ipfint * ic,       ipfint keep[],
		double fact[],  ipfint * lfact,    double * newfac,
		ipfint * lnew,     ipfint  ifact[],   ipfint * lifact,
		ipfint newifc[],   ipfint * linew,    ipfint * info );
}

/** implements the linear solver class using the HSL MA57 solver
 *
 * @ingroup LinearSolvers 
 */
class Ma57Solver : public DoubleLinearSolver {
private:
  Ma57Solver() {};
protected:
  ipfint     icntl[20];
  ipfint     info[40];
  double  cntl[5];
  double  rinfo[20];

  /** the Threshold Pivoting parameter, stored as U in the ma27dd
   *  common block. Takes values in the range [0,1]. Larger values
   *  enforce greater stability in the factorization as they insist on
   *  larger pivots. Smaller values preserve sparsity at the cost of
   *  using smaller pivots.  */
  double   kThresholdPivoting;

  /** the Threshold Pivoting parameter may need to be increased during
   * the algorithm if poor precision is obtained from the linear
   * solves.  kThresholdPivoting indicates the largest value we are
   * willing to tolerate.  */
  double   kThresholdPivotingMax;

  /** the factor in the range (1,inf) by which kThresholdPivoting is
   * increased when it is found to be inadequate.  */
  double   kThresholdPivotingFactor;

  /** the "Treat As Zero" parameter, stored as pivtol in the common
   * block ma27td. The factorization will not accept a pivot whose
   * absolute value is less than this parameter as a 1x1 pivot or as
   * the off-diagonal in a 2x2 pivot.  */
  double   kTreatAsZero;

  /** precision we demand from the linear system solver. If it isn't
   * attained on the first solve, we use iterative refinement and
   * possibly refactorization with a higher value of
   * kThresholdPivoting. */
  double  kPrecision;

  /** index array for the factorization */
  ipfint     *irowM,    *jcolM, *krowM;

  /** storage for the original matrix */
  double  *M;

  /** dimension of the whole matrix */
  ipfint      n;

  /** number of nonzeros in the matrix */
  ipfint      nnz;

  /** temporary storage */
  ipfint     lkeep, *keep;

  /** temporary storage for the factorization process */
  ipfint     lifact, *ifact, lfact;

  /* storage for the factors */
  double *fact;

  /** amounts by which to increase allocated factorization space when
   * inadequate space is detected. ipessimism is for array "iw",
   * rpessimism is for the array "fact". */
  double  ipessimism, rpessimism;

  /** used to indicate when we need a fresh factorization (when
   * iterative refinement has failed to improve the precision of the
   * computed solution satisfactorily */
  ipfint     freshFactor;

  /** store as a sparse symmetric matrix */
  SparseStorageHandle mStorage;

  /** called the very first time a matrix is factored. Allocates space
   * for the factorization and performs ordering */
  virtual void firstCall();
public:
  /** sets mStorage to refer to the argument sgm */
  Ma57Solver( SparseSymMatrix * sgm );
  
  virtual void diagonalChanged( int idiag, int extent );
  virtual void matrixChanged();
  virtual void solve( OoqpVector& rhs );

  /** set the Treat As Zero parameter in the MA27 data structures to
   *  the current value of kTreatAsZero */
  double setTreatAsZero() { return cntl[1] = kTreatAsZero; }

  /** set the Pivoting Threshold parameter in the MA27 data structures
   *  to the current value of kThresholdPivoting */
  double setThresholdPivoting() { return cntl[0] = kThresholdPivoting; }  

  /** destructor */
  virtual ~Ma57Solver();
};

#endif
