/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*             This file is part of the program and software framework       */
/*                  UG --- Ubquity Generator Framework                       */
/*                                                                           */
/*    Copyright (C) 2010-2013 Konrad-Zuse-Zentrum                            */
/*                            fuer Informationstechnik Berlin                */
/*                                                                           */
/*  UG is distributed under the terms of the ZIB Academic Licence.           */
/*                                                                           */
/*  You should have received a copy of the ZIB Academic License              */
/*  along with UG; see the file COPYING. If not email to scip@zib.de.        */
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**@file    paraParamSet.cpp
 * @brief   Parameter set for UG framework.
 * @author  Yuji Shinano
 *
 *
 *
 */

/*---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2*/


#include <string>
#include <map>
#include <cstdio>
#include <cstring>
#include <fstream>
#include <cfloat>
#include <climits>
#include <cassert>
#include "paraComm.h"
#include "paraParamSet.h"

using namespace UG;

ParaParam *ParaParamSet::paraParams[ParaParamsSize];

ParaParamSet::~ParaParamSet(
      )
{
   for( int i = 0; i <  ParaParamsSize; i++ )
   {
      delete paraParams[i];
   }
}

ParaParamSet::ParaParamSet()
{
   for(int i = 0; i <  ParaParamsSize; i++ )
   {
      paraParams[i] = 0;
   }

  /** bool params */
   paraParams[Quiet] = new ParaParamBool(
         "Quiet",
         "# Quiet is TRUE: output minimul information. The others are suppressed. [Default value: TRUE]",
         true);
   paraParams[TagTrace] = new ParaParamBool(
         "TagTrace",
         "# TagTrace is TRUE: outputs communication tags, FALSE: no outputs communication tags [Default value: FALSE]",
         false);
   paraParams[LogSolvingStatus] = new ParaParamBool(
         "LogSolvingStatus",
         "# LogSolvingStatus is TRUE: outputs solving statues log, FALSE: no outputs solving statues [Default value: FALSE]",
         false);
   paraParams[LogNodesTransfer] = new ParaParamBool(
         "LogNodesTransfer",
         "# LogNodeTransfer is TRUE: outputs nodes transfer log, FALSE: no outputs nodes transfer log [Default value: FALSE]",
         false);
   paraParams[LogSubtreeInfo] = new ParaParamBool(
         "LogSubtreeInfo",
         "# LogSubtreeInfo is TRUE: outputs subtree log, FALSE: no outputs subtree log [Default value: FALSE]",
         false);
   paraParams[OutputTabularSolvingStatus] = new ParaParamBool(
         "OutputTabularSolvingStatus",
         "# OutputTabularSolvingStatus is TRUE: outputs tabular solving statues log, FALSE: no outputs tabular solving statues [Default value: TRUE]",
         true);
   paraParams[DeterministicTabularSolvingStatus] = new ParaParamBool(
         "DeterministicTabularSolvingStatus",
         "# Outputs tabular solving statues log using deterministic time (only valid when Deterministic = TRUE)  [Default value: FALSE]",
         false);
   paraParams[RootNodeSolvabilityCheck] = new ParaParamBool(
         "RootNodeSolvabilityCheck",
         "# Indicate if root node solvability is checked before transfer or not. TRUE: root node solvability is checked, FALSE: no check [Default value: FALSE]",
         false);
   paraParams[UseRootNodeCuts] = new ParaParamBool(
         "UseRootNodeCuts",
         "# Use root node cuts to ParaInstance, that is, instance data for solvers has root node cuts of original problem. [Default value: FALSE]",
         false);
   paraParams[TransferLocalCuts] = new ParaParamBool(
         "TransferLocalCuts",
         "# Transfer local cuts generated by a solver as diffSubproblem info. [Default value: FALSE]",
         false);
   paraParams[TransferConflicts] = new ParaParamBool(
         "TransferConflicts",
         "# Transfer conflicts generated by a solver as diffSubproblem info. [Default value: FALSE]",
         false);
   paraParams[TransferBranchStats] = new ParaParamBool(
         "TransferBranchStats",
         "# Transfer branching stats generated by a solver as diffSubproblem info. [Default value: TRUE]",
         true);
   paraParams[CheckEffectOfRootNodePreprocesses] = new ParaParamBool(
         "CheckEffectOfRootNodePreprocesses",
         "# Check effect of root node preprocesses. [Default value: FALSE]",
         false);
   paraParams[Checkpoint] = new ParaParamBool(
         "Checkpoint",
         "# Whether if checkpoint is set or not. [Default value: FALSE (no checkpoint)]",
         false);
   paraParams[CollectOnce] = new ParaParamBool(
         "CollectOnce",
         "# After ramp-up, whether all open nodes are collected to LC or not. [Default value: TRUE (collect)]",
         true);
   paraParams[ProvingRun] = new ParaParamBool(
         "ProvingRun",
         "# Run is for proving or not. [Default value: FALSE (no proving)]",
         false);
   paraParams[SetAllDefaultsAfterRacing] = new ParaParamBool(
         "SetAllDefaultsAfterRacing",
         "# Set all defaults setting of parameter values after racing. [Default value: TRUE (default parameter set)]",
         true);
   paraParams[DistributeBestPrimalSolution] = new ParaParamBool(
         "DistributeBestPrimalSolution",
         "# The best primal solution is distributed or not. [Default value: TRUE (best primal solution is distributed)]",
         true);
   paraParams[LightWeightRootNodeProcess] = new ParaParamBool(
         "LightWeightRootNodeProcess",
         "# Light weight root node process is applied. [Default value: FALSE ( Not Applied )]",
         false);
   paraParams[RacingStatBranching] = new ParaParamBool(
         "RacingStatBranching",
         "# Racing status branching is applied. [Default value: TRUE ( Applied )]",
         true);
   paraParams[IterativeBreakDown] = new ParaParamBool(
         "IterativeBreakDown",
         "# Iterative break down is applied. [Default value: FALSE ( Not applied )]",
         false);
   paraParams[NoPreprocessingInLC] = new ParaParamBool(
         "NoPreprocessingInLC",
         "# No preprocessing is applied in LoadCoordinator. [Default value: FALSE ( applied )]",
         false);
   paraParams[NoUpperBoundTransferInRacing] = new ParaParamBool(
         "NoUpperBoundTransferInRacing",
         "# No upper bound is transferred to the other solvers in racing. [Default value: FALSE ( transfer )]",
         false);
   paraParams[MergeNodesAtRestart] = new ParaParamBool(
         "MergeNodesAtRestart",
         "# Merge nodes at restart or not. [Default value: TRUE ( merge )]",
         true);
   paraParams[NChangeIntoCollectingModeNSolvers] = new ParaParamBool(
         "NChangeIntoCollectingModeNSolvers",
         "# set NChangeIntoCollectingMode = the number of solvers/2  [Default value: TRUE]",
         true);
   paraParams[Deterministic] = new ParaParamBool(
         "Deterministic",
         "# set deterministic run  [Default value: FALSE]",
         false);
   paraParams[EventWeightedDeterministic] = new ParaParamBool(
         "EventWeightedDeterministic",
         "# set event weighted (each event has its weight of deterministic time) deterministic run. [Default value: FALSE]",
         false);
   paraParams[NoSolverPresolvingAtRoot] = new ParaParamBool(
         "NoSolverPresolvingAtRoot",
         "# set no presolving at root in Solver: distribute memory version should set FALSE  [Default value: FALSE - presolving]",
         false);
   paraParams[NoSolverPresolvingAtRootDefaultSet] = new ParaParamBool(
         "NoSolverPresolvingAtRootDefaultSet",
         "# set no presolving at root and default parameter settings in Solver: distribute memory version should set FALSE  [Default value: FALSE - presolving]",
         false);
   paraParams[NoAggressiveSeparatorInRacing] = new ParaParamBool(
         "NoAggressiveSeparatorInRacing",
         "# set no aggressive separator settings in racing  [Default value: TRUE]",
         true);
   paraParams[StatisticsToStdout] = new ParaParamBool(
         "StatisticsToStdout",
         "# set statistics output to stdout [Default value: FALSE]",
         false);
   paraParams[AllBoundChangesTransfer] = new ParaParamBool(
         "AllBoundChangesTransfer",
         "# Transfer bound changes not only for branching variables, but also for all variables. [Default value: TRUE]",
         true);
   paraParams[NoAllBoundChangesTransferInRacing] = new ParaParamBool(
         "NoAllBoundChangesTransferInRacing",
         "# Do not transfer all bound changes when the stage is in racing, even if  AllBoundChangesTransfer=TRUE  [Default value: FALSE]",
         false);
   paraParams[BreakFirstSubtree] = new ParaParamBool(
         "BreakFirstSubtree",
         "# Break the first subtree  : [Default value: FALSE]",
         false);
   paraParams[InitialNodesGeneration] = new ParaParamBool(
         "InitialNodesGeneration",
         "# Run to generate initial nodes : [Default value: FALSE]",
         false);

   /** int params */
   paraParams[OutputParaParams] = new ParaParamInt(
         "OutputParaParams",
         "# Whether if  outputs ParaParams or not: 0 - no output, 1 - output only not default values, 2 - output only not default value with comments, 3 - output all values, 4 - output all values with comments [Default value: 4]",
         4,
         0,
         4);
   paraParams[RampUpPhaseProcess] = new ParaParamInt(
         "RampUpPhaseProcess",
         "# Ramp-up phase process: 0 - normal process, 1 - racing process, 2 - rebuild tree after racing [Default value: 2]",
         2,
         0,
         2);
   paraParams[NChangeIntoCollectingMode] = new ParaParamInt(
         "NChangeIntoCollectingMode",
         "# Number of nodes in LoadCoordinator which decides to change into collecting mode [Default value: 2][0, INT_MAX]",
         2,
         0,
         INT_MAX);
   paraParams[NodeTransferMode] = new ParaParamInt(
         "NodeTransferMode",
         "# Node transfer mode: 0 - best estimate node transfer, 1 - best bound node transfer [Default value: 1]",
         1,
         0,
         1);
   paraParams[AddDualBoundCons] = new ParaParamInt(
         "AddDualBoundCons",
         "# Adding constraint: objective func >= dualBoundValue (This is not a good idea, because it creates many degenerate solutions) : 0 - no adding, 1 - adding to discarded ParaNodes only, 2 - adding always, 3 - adding at warm start [Default value: 0]",
         0,
         0,
         3);
   paraParams[NotificationSynchronization] = new ParaParamInt(
         "NotificationSynchronization",
         "# Notification synchronization strategy : 0 - always synchronize, 1 - every iteration in collecting, 2 - no synchronize [Default value: 0]",
         0,
         0,
         2);
   paraParams[MinNumberOfCollectingModeSolvers] = new ParaParamInt(
         "MinNumberOfCollectingModeSolvers",
         "# Minimum number of Solvers that can be in collecting mode. [Default value: 1]",
         1,
         1,
         INT_MAX);
   paraParams[MaxNumberOfCollectingModeSolvers] = new ParaParamInt(
         "MaxNumberOfCollectingModeSolvers",
         "# Maximum number of Solvers that can be in collecting mode : -1 - no limit, 0 - half of the number of Solvers, n > 0 - n [Default value: 0]",
         0,
         -1,
         INT_MAX);
   paraParams[SolverOrderInCollectingMode] = new ParaParamInt(
         "SolverOrderInCollectingMode",
         "# The collecting mode request messages sending order  : -1 - no ordering, 0 - ordered by best dual bound value, 1 - ordered by number of nodes left, 2 - choose alternatively the best bound and the number of nodes orders  [Default value: 0]",
         0,
         -1,
         2);
   paraParams[RacingRampUpTerminationCriteria] = new ParaParamInt(
         "RacingRampUpTerminationCriteria",
         "# Racing ramp-up termination criteria : 0 - stop at the number of nodes left reached, 1 - stop at time limit, 2: - stop at the Solver with the best dual bound value has a certain number of nodes, 3 - adaptive(node first), 4 - adaptive (time first):  [Default value: 4]",
         5,
         0,
         7);
   paraParams[StopRacingNumberOfNodesLeft] = new ParaParamInt(
         "StopRacingNumberOfNodesLeft",
         "# The number of nodes left in a solver to stop racing : [1, INT_MAX]:  [Default value: 300]",
         300,
         1,
         INT_MAX);
   paraParams[NumberOfInitialNodes] = new ParaParamInt(
         "NumberOfInitialNodes",
         "# The number of nodes left in all solvers to stop generating initial nodes : [1, INT_MAX]:  [Default value: 300]",
         300,
         1,
         INT_MAX);
   paraParams[MaxNRacingParamSetSeed] = new ParaParamInt(
         "MaxNRacingParamSetSeed",
         "# The maximum number of seeds for racing parameter set : [1, INT_MAX]:  [Default value: 64]",
         64,
         1,
         INT_MAX);
   paraParams[TryNVariablegOrderInRacing] = new ParaParamInt(
         "TryNVariablegOrderInRacing",
         "# The number of trying variable order in racing ramp-up : [1, INT_MAX]:  [Default value: 100]",
         100,
         1,
         INT_MAX);
   paraParams[TryNBranchingOrderInRacing] = new ParaParamInt(
         "TryNBranchingOrderInRacing",
         "# The number of trying branching order in racing ramp-up : [1, INT_MAX]:  [Default value: 100]",
         100,
         1,
         INT_MAX);
   paraParams[NEvaluationSolversToStopRacing] = new ParaParamInt(
          "NEvaluationSolversToStopRacing",
          "# The number of evaluation solvers to stop racing. ( -1 stops at all of the solvers, 0 stop at half of the solvers ) : [-1, INT_MAX]:  [Default value: -1]",
          -1,
          -1,
          INT_MAX);
   paraParams[NMaxCanditatesForCollecting] = new ParaParamInt(
          "NMaxCanditatesForCollecting",
          "# The maximum number of candidates for collecting mode solvers. : [1, INT_MAX]:  [Default value: 10]",
          10,
          1,
          INT_MAX);
   paraParams[NSolverNodesStartBreaking] = new ParaParamInt(
          "NSolverNodesStartBreaking",
          "# The number of nodes left in a solver to start breaking. : [0, INT_MAX]:  [Default value: 0: no breaking]",
          0,
          0,
          INT_MAX);
   paraParams[NStopBreaking] = new ParaParamInt(
          "NStopBreaking",
          "# The number of nodes left in the system to stop breaking. : [0, INT_MAX]:  [Default value: 0: no breaking]",
          0,
          0,
          INT_MAX);
   paraParams[NTransferLimitForBreaking] = new ParaParamInt(
          "NTransferLimitForBreaking",
          "# The number of nodes left in the system to stop breaking. : [1, INT_MAX]:  [Default value: 100]",
          100,
          1,
          INT_MAX);
   paraParams[NStopSolvingMode] = new ParaParamInt(
         "NStopSolvingMode",
         "# Number of nodes solved less than this value can be a candidate of stop solving mode. The value less than 0 means no stop solving. [Default: -1][-1.0, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[NCollectOnce] = new ParaParamInt(
         "NCollectOnce",
         "# Number of nodes collected once to LC. The value -1 means all. 0 means # solvers * 5. [Default: 0][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[AggressivePresolveDepth] = new ParaParamInt(
         "AggressivePresolveDepth",
         "# Depth to try presolving in the other solvers. Offset value will be added. The value -1 means no aggressive presolving. [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[AggressivePresolveStopDepth] = new ParaParamInt(
         "AggressivePresolveStopDepth",
         "# Depth to stop aggressive presolving. This value is only valid for AggressivePresolveDepth > 0. [Default: 8][1, INT_MAX]",
         8,
         1,
         INT_MAX);
   paraParams[BigDualGapSubtreeHandling] = new ParaParamInt(
         "BigDualGapSubtreeHandling",
         "# Big dual gap subtree handling. 0 - throw away, 1 - send back to LC [Default: 0][0, 1]",
         0,
         0,
         1);
   paraParams[InstanceTransferMethod] = new ParaParamInt(
         "InstanceTransferMethod",
         "# Instance data transfer method. 0 - on memory, 1 - by file, 2 - by original data [Default: 0][0, 2]",
         0,
         0,
         2);
   paraParams[KeepNodesDepth] = new ParaParamInt(
         "KeepNodesDepth",
         "# Not transfer and keep nodes depth from sub-MIP root less than this value. [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[UgCplexRunningMode] = new ParaParamInt(
         "UgCplexRunningMode",
         "# Runnign mode of ug[CPLEX,*]. 0 - all generator, 1 - use generator, 2 - no generator. [Default: 1][0, 2]",
         0,
         0,
         2);

   /** longint params */

   /** real params */
   paraParams[MultiplierForCollectingMode] = new ParaParamReal(
         "MultiplierForCollectingMode",
         "# Multiplier mp in collecting mode. When the # of good nodes becomes greater than mp * NChangeIntoCollectingMode, stop collecting [Default value: 2.0][1.0, DBL_MAX]",
         2.0,
         1.0,
         DBL_MAX);
   paraParams[MultiplierToDetermineThresholdValue] = new ParaParamReal(
         "MultiplierToDetermineThresholdValue",
         "# Multiplier mth to determine threshold value. When # of nodes left grater than mth * (meanRootNodeTime/ (meanNodeTime except root)), a Solver starts sending nodes in collecting mode [Default: 1.5][1.0, DBL_MAX]",
         1.5,
         1.0,
         DBL_MAX);
   paraParams[BgapCollectingMode] = new ParaParamReal(
         "BgapCollectingMode",
         "# Bound gap_p to identify good nodes, (bound value of a node - best bound value )/ (best bound value ) < gap_p is treated as good nodes. [Default: 0.4][0.0,1.0]",
         0.4,
         0.0,
         1.0);
   paraParams[MultiplierForBgapCollectingMode] = new ParaParamReal(
         "MultiplierForBgapCollectingMode",
         "# A Solver in collecting mode becomes out of collecting mode, when (the solver's best dual bound value - g.b.d.b.v )/g.b.d.b.v > gap_p* (MultiplierForBgapCollectingMode) [Default:10.0][1.0, 10.0]",
         10.0,
         1.0,
         10.0);
   paraParams[ABgapForSwitchingToBestSolver] = new ParaParamReal(
         "ABgapForSwitchingToBestSolver",
         "# A Solver in collecting mode becomes out of collecting mode, when the absolute dual gap to the best solver is grater than this value and the best solver is not in collecting mode (Don't set 0.0)  [Default:1.0][0.0, DBL_MAX]",
         1.0,
         0.0,
         DBL_MAX);
   paraParams[BgapStopSolvingMode] = new ParaParamReal(
         "BgapStopSolvingMode",
         "# A solver is stopped and all generated nodes except the root throws away, when ( its local best dual bound value - LC.b.d.b.v ) / LC.b.d.b.v > BgapStopSolvingMode. [Default: 0.10][0.0, 10.0]",
         0.03,
         0.0,
         10.0);

   paraParams[NotificationInterval] = new ParaParamReal(
         "NotificationInterval",
         "# An active Solver notifies its solvers status to LoadCoordinator, when time elapsed from its previous notification. [Default: 1.0][0.0, DBL_MAX]",
         1.0,
         0.0,
         DBL_MAX);
   paraParams[TimeLimit] = new ParaParamReal(
         "TimeLimit",
         "# Time limit of computation. -1.0 means no time limit. [Default: -1.0] [-1.0, DBL_MAX]",
         -1.0,
         -1.0,
         DBL_MAX);
   paraParams[CheckpointInterval] = new ParaParamReal(
         "CheckpointInterval",
         "# Interval time for checkpoint. [Default: 3600.0] [10.0, DBL_MAX]",
         3600.0,
         10.0,
         DBL_MAX);
   paraParams[StopRacingTimeLimit] = new ParaParamReal(
         "StopRacingTimeLimit",
         "# The time limit to stop racing. [Default: 36.0] [1.0, DBL_MAX]",
         360.0,
         1.0,
         DBL_MAX);
   paraParams[StopRacingTimeLimitMultiplier] = new ParaParamReal(
         "StopRacingTimeLimitMultiplier",
         "# An extending time limit multiplier in case nodes left less than stop-racing number of nodes left. [Default: 50.0] [1.0, DBL_MAX]",
         50.0,
         1.0,
         DBL_MAX);
   paraParams[StopRacingNumberOfNodesLeftMultiplier] = new ParaParamReal(
         "StopRacingNumberOfNodesLeftMultiplier",
         "# The number of nodes left limit multiplier not to collect once.. [Default: 20.0] [1.0, DBL_MAX]",
         3.0,
         1.0,
         DBL_MAX);
   paraParams[TimeToIncreaseCMS] = new ParaParamReal(
         "TimeToIncreaseCMS",
         "# Time stays in empty node pool. The limit number of collecting mode solver increase, when node pool stays empty more than this time  [Default: 10.0] [1.0, DBL_MAX]",
         10.0,
         1.0,
         DBL_MAX);
   paraParams[TabularSolvingStatusInterval] = new ParaParamReal(
         "TabularSolvingStatusInterval",
         "# Output tabular solving status log evry this interval time  [Default: 5.0] [0.0, DBL_MAX]",
         5.0,
         0.0,      // for debug, this is useful
         DBL_MAX);
   paraParams[RatioToApplyLightWeightRootProcess] = new ParaParamReal(
         "RatioToApplyLightWeightRootProcess",
         "# Idle solvers ratio to apply light weight root node process.\n# When # of idle solver > this value * # of solvers, it is applied.  [Default: 0.5] [0.0, 1.0]",
         0.5,
         0.0,
         1.0);
   paraParams[MultiplierForBreakingTargetBound] = new ParaParamReal(
         "MultiplierForBreakingTargetBound",
         "# Multiplier for a target dual bound for breaking.  [Default: 1.03] [1.0, DBL_MAX]",
         1.03,
         1.0,
         DBL_MAX);
   paraParams[FixedVariablesRatioInMerging] = new ParaParamReal(
         "FixedVariablesRatioInMerging",
         "# Fixed variables ration in merging open nodes.  [Default: 0.9] [0.0, 1.0]",
         0.9,
         0.0,
         1.0);
   paraParams[AllowableRegressionRatioInMerging] = new ParaParamReal(
         "AllowableRegressionRatioInMerging",
         "# Allowable regression ratio of dual bound value in merging.  [Default: 0.0] [0.0, 1.0]",
         0.0,
         0.0,
         1.0);
   paraParams[CountingSolverRatioInRacing] = new ParaParamReal(
         "CountingSolverRatioInRacing",
         "# Ratio for the number of Solvers that count on the racing ones (The solvers have to notify its status to LC).  [Default: 0.5] [0.0, 1.0]",
         0.5,
         0.0,
         1.0);
   paraParams[ProhibitCollectOnceMultiplier] = new ParaParamReal(
         "ProhibitCollectOnceMultiplier",
         "# When racing winner has nodes less than (the number of Solvers)*(this value), CollectOnce should be prohibited.  [Default: 2.0] [0.0, DBL_MAX]",
         2.0,
         0.0,
         DBL_MAX);
   paraParams[GeneratorRatio] = new ParaParamReal(
         "GeneratorRatio",
         "# Generator ratio. When UgCplexRunningMode = 1, at least one solver becomes generator. [Default: 0.5] [0.0, 1.0]",
         0.5,
         0.0,
         1.0);

   /** char params */

   /** string params */
   paraParams[SolverSettingsForInitialPresolving] = new ParaParamString(
         "SolverSettingsForInitialPresolving",
         "# Solver parameter settings file name that is applied at initial presolving in LoadCoordinator. Empty name use default settings. [Default: ]",
         "");
   paraParams[SolverSettingsAtRootNode] = new ParaParamString(
         "SolverSettingsAtRootNode",
         "# Solver parameter settings file name that is applied at root node solving. Empty name use default settings. [Default: ]",
         "");
   paraParams[SolverSettingsExceptRootNode] = new ParaParamString(
         "SolverSettingsExceptRootNode",
         "# Solver parameter settings file name that is applied at all nodes solving except root node. Empty name use default settings. [Default: ]",
         "");
   paraParams[SolverSettingsAtRacing] = new ParaParamString(
         "SolverSettingsAtRacing",
         "# Solver parameter settings file name that is applied at racing stage. Empty name use default settings. [Default: ]",
         "");
   paraParams[TempFilePath] = new ParaParamString(
         "TempFilePath",
         "# Path for temporary files [Default: /tmp/]",
         "/tmp/");
   paraParams[TagTraceFileName] = new ParaParamString(
         "TagTraceFileName",
         "# File name to output tag trace [Default: std::cout]",
         "std::cout");
   paraParams[LogSolvingStatusFilePath] = new ParaParamString(
         "LogSolvingStatusFilePath",
         "# Path name to output solving statuses log [Default: ./]",
         "./");
   paraParams[LogNodesTransferFilePath] = new ParaParamString(
         "LogNodesTransferFilePath",
         "# Path name to output nodes transfer log [Default: ./]",
         "./");
   paraParams[SolutionFilePath] = new ParaParamString(
         "SolutionFilePath",
         "# Path name to output solution [Default: ./]",
         "./");
   paraParams[CheckpointFilePath] = new ParaParamString(
         "CheckpointFilePath",
         "# Path name to checkpoint files [Default: ./]",
         "./");
}

/*********************
 * for bool params  *
 ********************/
bool
ParaParamSet::getBoolParamValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeBool);
   ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool *>(paraParams[param]);
   return paraParamBool->getValue();
}

void
ParaParamSet::setBoolParamValue(
      int param,
      bool value
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeBool);
   ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool *>(paraParams[param]);
   return paraParamBool->setValue(value);
}

bool
ParaParamSet::getBoolParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeBool);
   ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool *>(paraParams[param]);
   return paraParamBool->getDefaultValue();
}

void
ParaParamSet::setBoolParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeBool);
   ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool *>(paraParams[param]);
   return paraParamBool->setDefaultValue();
}

bool
ParaParamSet::isBoolParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeBool);
   ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool *>(paraParams[param]);
   return paraParamBool->isDefaultValue();
}

/********************
 * for int params   *
 *******************/
int
ParaParamSet::getIntParamValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeInt);
   ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt *>(paraParams[param]);
   return paraParamInt->getValue();
}

void
ParaParamSet::setIntParamValue(
      int param,
      int value
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeInt);
   ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt *>(paraParams[param]);
   return paraParamInt->setValue(value);
}

int
ParaParamSet::getIntParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeInt);
   ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt *>(paraParams[param]);
   return paraParamInt->getDefaultValue();
}

void
ParaParamSet::setIntParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeInt);
   ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt *>(paraParams[param]);
   paraParamInt->setDefaultValue();
}

bool
ParaParamSet::isIntParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeInt);
   ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt *>(paraParams[param]);
   return paraParamInt->isDefaultValue();
}

/*********************
 * for longint params  *
 ********************/
long long
ParaParamSet::getLongintParamValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeLongint);
   ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint *>(paraParams[param]);
   return paraParamLongint->getValue();
}

void
ParaParamSet::setLongintParamValue(
      int param,
      long long value
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeLongint);
   ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint *>(paraParams[param]);
   paraParamLongint->setValue(value);
}

long long
ParaParamSet::getLongintParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeLongint);
   ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint *>(paraParams[param]);
   return paraParamLongint->getDefaultValue();
}

void
ParaParamSet::setLongintParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeLongint);
   ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint *>(paraParams[param]);
   paraParamLongint->setDefaultValue();
}

bool
ParaParamSet::isLongintParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeLongint);
   ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint *>(paraParams[param]);
   return paraParamLongint->isDefaultValue();
}

/*********************
 * for real params  *
 ********************/
double
ParaParamSet::getRealParamValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeReal);
   ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal *>(paraParams[param]);
   return paraParamReal->getValue();
}

void
ParaParamSet::setRealParamValue(
      int param,
      double value
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeReal);
   ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal *>(paraParams[param]);
   paraParamReal->setValue(value);
}

double
ParaParamSet::getRealParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeReal);
   ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal *>(paraParams[param]);
   return paraParamReal->getDefaultValue();
}

void
ParaParamSet::setRealParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeReal);
   ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal *>(paraParams[param]);
   paraParamReal->setDefaultValue();
}

bool
ParaParamSet::isRealParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeReal);
   ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal *>(paraParams[param]);
   return paraParamReal->isDefaultValue();
}

/*********************
 * for char params  *
 ********************/
char
ParaParamSet::getCharParamValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeChar);
   ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar *>(paraParams[param]);
   return paraParamChar->getValue();
}

void
ParaParamSet::setCharParamValue(
      int param,
      char value
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeChar);
   ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar *>(paraParams[param]);
   paraParamChar->setValue(value);
}

char
ParaParamSet::getCharParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeChar);
   ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar *>(paraParams[param]);
   return paraParamChar->getDefaultValue();
}

void
ParaParamSet::setCharParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeChar);
   ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar *>(paraParams[param]);
   paraParamChar->setDefaultValue();
}

bool
ParaParamSet::isCharParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeChar);
   ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar *>(paraParams[param]);
   return paraParamChar->isDefaultValue();
}

/**********************
 * for string params  *
 *********************/
const char *
ParaParamSet::getStringParamValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeString);
   ParaParamString *paraParamString = dynamic_cast< ParaParamString *>(paraParams[param]);
   return paraParamString->getValue();
}

void
ParaParamSet::setStringParamValue(
      int param,
      const char *value
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeString);
   ParaParamString *paraParamString = dynamic_cast< ParaParamString *>(paraParams[param]);
   char *str = new char[std::strlen(value)];
   std::strcpy(str, value);
   paraParamString->setValue(value);
}

const char *
ParaParamSet::getStringParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeString);
   ParaParamString *paraParamString = dynamic_cast< ParaParamString *>(paraParams[param]);
   return paraParamString->getDefaultValue();
}

void
ParaParamSet::setStringParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeString);
   ParaParamString *paraParamString = dynamic_cast< ParaParamString *>(paraParams[param]);
   paraParamString->setDefaultValue();
}

bool
ParaParamSet::isStringParamDefaultValue(
      int param
      )
{
   assert(paraParams[param]->getType() == ParaParamTypeString);
   ParaParamString *paraParamString = dynamic_cast< ParaParamString *>(paraParams[param]);
   return paraParamString->isDefaultValue();
}

int
ParaParamSet::paramParaseBool(
      ParaParam *paraParam,
      char *valuestr
      )
{
   ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool * >(paraParam);
   assert(valuestr != NULL);
   if( std::string(valuestr) == std::string("TRUE") )
   {
      paraParamBool->setValue(true);
   }
   else if( std::string(valuestr) == std::string("FALSE") )
   {
      paraParamBool->setValue(false);
   }
   else
   {
      std::cout << "Invalid parameter value <" << valuestr
                <<  "> for ParaParam_Bool parameter <"
                << paraParamBool->getParamName() << ">" << std::endl;
      return -1;
   }
   return 0;
}

int
ParaParamSet::paramParaseInt(
      ParaParam *paraParam,
      char *valuestr
      )
{
   int value;
   ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt * >(paraParam);
   assert(valuestr != NULL);
   if( std::sscanf(valuestr, "%d", &value) == 1 )
   {
      if( value >= paraParamInt->getMinValue() && value <= paraParamInt->getMaxValue() )
      {
         paraParamInt->setValue(value);
      }
      else
      {
         OUTPUT_PARAM_VALUE_ERROR("Int", paraParamInt->getParamName(), value, paraParamInt->getComment() );
      }
   }
   else
   {
      std::cout << "Invalid parameter value <" << valuestr
                << "> for int parameter <"
                << paraParamInt->getParamName() << ">" << std::endl;
      return -1;
   }
   return 0;
}

int
ParaParamSet::paramParaseLongint(
      ParaParam *paraParam,
      char *valuestr
      )
{
   long long value;
   ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint * >(paraParam);
   assert(valuestr != NULL);
   if( std::sscanf(valuestr, "%lld", &value) == 1 )
   {
      if( value >= paraParamLongint->getMinValue() && value <= paraParamLongint->getMaxValue() )
      {
         paraParamLongint->setValue(value);
      }
      else
      {
         OUTPUT_PARAM_VALUE_ERROR("Longint", paraParamLongint->getParamName(), value, paraParamLongint->getComment() );
      }
   }
   else
   {
      std::cout << "Invalid parameter value <" << valuestr
                << "> for longint parameter <"
                << paraParamLongint->getParamName() << ">" << std::endl;
      return -1;
   }
   return 0;
}

int
ParaParamSet::paramParaseReal(
      ParaParam *paraParam,
      char *valuestr
      )
{
   double value;
   ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal * >(paraParam);
   assert(valuestr != NULL);
   if( std::sscanf(valuestr, "%lf", &value) == 1 )
   {
      if( value >= paraParamReal->getMinValue() && value <= paraParamReal->getMaxValue() )
      {
         paraParamReal->setValue(value);
      }
      else
      {
         OUTPUT_PARAM_VALUE_ERROR("Real", paraParamReal->getParamName(), value, paraParamReal->getComment() );
      }
   }
   else
   {
      std::cout << "Invalid parameter value <" << valuestr
                << "> for real parameter <"
                << paraParamReal->getParamName() << ">" << std::endl;
      return -1;
   }
   return 0;
}

int
ParaParamSet::paramParaseChar(
      ParaParam *paraParam,
      char *valuestr
      )
{
   char value;
   ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar * >(paraParam);
   assert(valuestr != NULL);
   if( std::sscanf(valuestr, "%c", &value) == 1 )
   {
      std::string allowedString(paraParamChar->getAllowedValues());
      char cString[2]; cString[0] = value; cString[1] = '\0';
      if( allowedString.find(cString) != std::string::npos )
      {
         paraParamChar->setValue(value);
      }
      else
      {
         OUTPUT_PARAM_VALUE_ERROR("Char", paraParamChar->getParamName(), value, paraParamChar->getComment() );
      }
   }
   else
   {
      std::cout << "Invalid parameter value <" << valuestr
                << "> for char parameter <"
                << paraParamChar->getParamName() << ">" << std::endl;
      return -1;
   }
   return 0;
}

int
ParaParamSet::paramParaseString(
      ParaParam *paraParam,
      char *valuestr
      )
{
   ParaParamString *paraParamString = dynamic_cast< ParaParamString * >(paraParam);
   assert(valuestr != NULL);

   /* check for quotes */
   int len = std::strlen(valuestr);
   if( len <= 1 || valuestr[0] != '"' || valuestr[len-1] != '"' )
   {
      std::cout << "Invalid parameter value <" << valuestr
                << "> for string parameter <"
                << paraParamString->getParamName() << ">" << std::endl;
      return -1;
   }
   /* remove the quotes */
   valuestr[len-1] = '\0';
   valuestr++;
   char *paramValue = new char[strlen(valuestr)+1];
   strcpy(paramValue, valuestr);
   paraParamString->setValue(paramValue);
   return 0;
}

// -1: error
/** the parameterParse routine is almost copy from paramset.c of SCIP code */
int
ParaParamSet::parameterParse(
      char *line,
      std::map<std::string, int> &mapStringToId
      )
{
   char* paramname;
   char* paramvaluestr;
   char* lastquote;
   unsigned int quoted;

   /* find the start of the parameter name */
   while( *line == ' ' || *line == '\t' || *line == '\r' )
      line++;
   if( *line == '\0' || *line == '\n' || *line == '#' )
      return 0;
   paramname = line;

   /* find the end of the parameter name */
   while( *line != ' ' && *line != '\t' && *line != '\r' && *line != '\n' && *line != '#' && *line != '\0' && *line != '=' )
      line++;
   if( *line == '=' )
   {
      *line = '\0';
      line++;
   }
   else
   {
      *line = '\0';
      line++;

      /* search for the '=' char in the line */
      while( *line == ' ' || *line == '\t' || *line == '\r' )
         line++;
      if( *line != '=' )
      {
         std::cout << "Character '=' was expected after the parameter name" << std::endl;
         return -1;
      }
      line++;
   }

   /* find the start of the parameter value string */
   while( *line == ' ' || *line == '\t' || *line == '\r' )
      line++;
   if( *line == '\0' || *line == '\n' || *line == '#' )
   {
      std::cout << "Parameter value is missing" << std::endl;
      return -1;
   }
   paramvaluestr = line;

   /* find the end of the parameter value string */
   quoted = (*paramvaluestr == '"');
   lastquote = NULL;
   while( (quoted || (*line != ' ' && *line != '\t' && *line != '\r' && *line != '\n' && *line != '#')) && *line != '\0' )
   {
      if( *line == '"' )
         lastquote = line;
      line++;
   }
   if( lastquote != NULL )
      line = lastquote+1;
   if( *line == '#' )
      *line = '\0';
   else if( *line != '\0' )
   {
      /* check, if the rest of the line is clean */
      *line = '\0';
      line++;
      while( *line == ' ' || *line == '\t' || *line == '\r' )
         line++;
      if( *line != '\0' && *line != '\n' && *line != '#' )
      {
         std::cout << "Additional characters after parameter value" << std::endl;
         return -1;
      }
   }

   std::map<std::string, int>::iterator pos;
   pos = mapStringToId.find(paramname);
   if( pos == mapStringToId.end() )
   {
      std::cout << "Unknown parameter <" << paramname << ">" << std::endl;
      return -1;
   }
   int paramId = pos->second;
   switch ( paraParams[paramId]->getType() )
   {
   case ParaParamTypeBool:
      return paramParaseBool(paraParams[paramId], paramvaluestr);
   case ParaParamTypeInt:
      return paramParaseInt(paraParams[paramId], paramvaluestr);
   case ParaParamTypeLongint:
      return paramParaseLongint(paraParams[paramId], paramvaluestr);
   case ParaParamTypeReal:
      return paramParaseReal(paraParams[paramId], paramvaluestr);
   case ParaParamTypeChar:
      return paramParaseChar(paraParams[paramId], paramvaluestr);
   case ParaParamTypeString:
      return paramParaseString(paraParams[paramId], paramvaluestr);
   default:
      std::cout << "Unknown parameter type" << std::endl;
      return -1;
   }
}

void
ParaParamSet::read(
      ParaComm *comm,
      const char* filename
      )
{
   const int MaxLineSize = 1024;
   char line[MaxLineSize];

   std::ifstream ifs;
   ifs.open(filename);
   if( !ifs ){
      std::cout << "Cannot open ParaParams read file: file name = " << filename << std::endl;
      exit(1);
   }

   std::map<std::string, int> mapStringToId;
   for( int i = ParaParamsFirst; i < ParaParamsSize; i ++ ){
      assert( paraParams[i] );
      mapStringToId.insert(std::make_pair(paraParams[i]->getParamName(),i));
   }

   int lineNo = 0;
   while( ifs.getline(line, MaxLineSize) )
   {
      lineNo++;
      int retCode = parameterParse(line, mapStringToId);
      if( retCode ){
         ifs.close();
         std::cout << "Input error in file <" << filename << "> line " << lineNo << std::endl;
         exit(1);
      }
   }
   ifs.close();
   if( getBoolParamValue(RacingStatBranching) &&
         ( ( getIntParamValue(RampUpPhaseProcess) != 1 && getIntParamValue(RampUpPhaseProcess) != 2 ) ||
               (!getBoolParamValue(CollectOnce)) ) )
   {
      std::cout << "Input error in file <" << filename
            << "> [RacingStatBranching = TRUE] needs to be [RampUpPhaseProcess = 1 or 2] and [CollectOnce = TRUE]"
            << std::endl;
      exit(1);
   }
   if( getBoolParamValue(Quiet) )
   {
      setBoolParamValue(TagTrace, false);
      setBoolParamValue(LogSolvingStatus, false);
      setBoolParamValue(LogNodesTransfer, false);
   }
   if( getIntParamValue(RampUpPhaseProcess) == 2 )
   {
      setBoolParamValue(CollectOnce, true);
      setIntParamValue(NCollectOnce, -1);    // anyway,  collects all nodes once
   }
}

void
ParaParamSet::write(
      std::ostream *os
      )
{
   bool comments = false;
   bool onlyChanged = false;

   if( this->getIntParamValue(OutputParaParams) == 1 ||
         this->getIntParamValue(OutputParaParams) == 2
         )
   {
      onlyChanged = true;
   }

   if( this->getIntParamValue(OutputParaParams) == 2 ||
         this->getIntParamValue(OutputParaParams) == 4
         )
   {
      comments = true;
   }

   for( int i = 0; i <  ParaParamsSize; i++ )
   {
      assert( paraParams[i] );
      switch ( paraParams[i]->getType() )
      {
      case ParaParamTypeBool:
      {
         ParaParamBool *paraParamBool = dynamic_cast< ParaParamBool * >(paraParams[i]);
         if( onlyChanged )
         {
            if (!paraParamBool->isDefaultValue()){
               if( comments )
               {
                  *os << paraParamBool->getComment() << std::endl;
               }
               *os << paraParamBool->getParamName() << " = " << ( ( paraParamBool->getValue() == true ) ? "TRUE" : "FALSE" ) << std::endl << std::endl;
            }
         } else {
            if( comments )
            {
               *os << paraParamBool->getComment() << std::endl;
            }
            *os << paraParamBool->getParamName() << " = " << ( ( paraParamBool->getValue() == true ) ? "TRUE" : "FALSE" ) << std::endl << std::endl;
         }
         break;
      }
      case ParaParamTypeInt:
      {
         ParaParamInt *paraParamInt = dynamic_cast< ParaParamInt * >(paraParams[i]);
         if( onlyChanged )
         {
            if (!paraParamInt->isDefaultValue()){
               if( comments )
               {
                  *os << paraParamInt->getComment() << std::endl;
               }
               *os << paraParamInt->getParamName() << " = " << paraParamInt->getValue() << std::endl<< std::endl;
            }
         }
         else
         {
            if( comments )
            {
               *os << paraParamInt->getComment() << std::endl;
            }
            *os << paraParamInt->getParamName() << " = " << paraParamInt->getValue() << std::endl << std::endl;
         }
         break;
      }
      case ParaParamTypeLongint:
      {
         ParaParamLongint *paraParamLongint = dynamic_cast< ParaParamLongint * >(paraParams[i]);
         if( onlyChanged )
         {
            if (!paraParamLongint->isDefaultValue()){
               if( comments )
               {
                  *os << paraParamLongint->getComment() << std::endl;
               }
               *os << paraParamLongint->getParamName() << " = " << paraParamLongint->getValue() << std::endl << std::endl;
            }
         }
         else
         {
            if( comments )
            {
               *os << paraParamLongint->getComment() << std::endl;
            }
            *os << paraParamLongint->getParamName() << " = " << paraParamLongint->getValue() << std::endl << std::endl;
         }
         break;
      }
      case ParaParamTypeReal:
      {
         ParaParamReal *paraParamReal = dynamic_cast< ParaParamReal * >(paraParams[i]);
         if( onlyChanged )
         {
            if (!paraParamReal->isDefaultValue()){
               if( comments )
               {
                  *os << paraParamReal->getComment() << std::endl;
               }
               *os << paraParamReal->getParamName() << " = " << paraParamReal->getValue() << std::endl << std::endl;
            }
         }
         else
         {
            if( comments )
            {
               *os << paraParamReal->getComment() << std::endl;
            }
            *os << paraParamReal->getParamName() << " = " << paraParamReal->getValue() << std::endl << std::endl;
         }
         break;
      }
      case ParaParamTypeChar:
      {
         ParaParamChar *paraParamChar = dynamic_cast< ParaParamChar * >(paraParams[i]);
         if( onlyChanged )
         {
            if (!paraParamChar->isDefaultValue()){
               if( comments )
               {
                  *os << paraParamChar->getComment() << std::endl;
               }
               *os << paraParamChar->getParamName() << " = " << paraParamChar->getValue() << std::endl << std::endl;
            }
         }
         else
         {
            if( comments )
            {
               *os << paraParamChar->getComment() << std::endl;
            }
            *os << paraParamChar->getParamName() << " = " << paraParamChar->getValue() << std::endl << std::endl;
         }
         break;
      }
      case ParaParamTypeString:
      {
         ParaParamString *paraParamString = dynamic_cast< ParaParamString * >(paraParams[i]);
         if( onlyChanged )
         {
            if (!paraParamString->isDefaultValue())
            {
               if( comments )
               {
                  *os << paraParamString->getComment() << std::endl;
               }
               *os << paraParamString->getParamName() << " = \"" << paraParamString->getValue() << "\"" << std::endl << std::endl;
            }
         }
         else
         {
            if( comments )
            {
               *os << paraParamString->getComment() << std::endl;
            }
            *os << paraParamString->getParamName() << " = \"" << paraParamString->getValue() << "\"" << std::endl << std::endl;
         }
         break;
      }
      default:
         std::cout << "Unknown parameter type" << std::endl;
         throw "Unknown parameter type";
      }
   }
   (*os).clear();
}
