function [ ] = smoldyn2matlab( n, voxel_size, actual_voxel_size)
%SMOLDYN2MATLAB imports and converts Smoldyn molecule trajectory into
%Matlab format.
%
%   The Smoldyn program (http://www.smoldyn.org/index.html) is used
%   to output trajectories of a large number of molecules. This data is
%   saved as text files, here assumed to be called trajs1.txt ...
%   trajsn.txt. This function converts this txt file to a matlab matrix
%   with the dimensions number of molecules x 3 x number of timesteps, wher
%   the 3 refers to the x,y and z co-ordinates of the molecules.
%   Matlab files are called positions1.mat ... positionsn.mat. The input n
%   is the number of files to be converted, assuming consecutive numbering
%   from 1 onwards. Note that the file size generated by Smoldyn can be
%   very large (many GBs), and if greater than approximately 6GB, Matlab
%   will fail to import the text file. To avoid this problem, create
%   multiple molecule types within Smoldyn and save the trajectories of
%   each as a separate txt file. Only those molecules which finish in the
%   voxel to be analysed are saved, which saves some space.
%
% SMOLDYN2MATLAB(N, voxel_size, actual_voxel_size)
%
% N is the number of trajectory files to be converted.
% voxel_size is the 1D length of the side of the volume simulated in um (assumed to be isotropic)
% actual_voxel_size is the 1D length of the side of the volume to be
% analysed (in um) (assumed to be isotropic)

% Author: Jo Bates <jobates81@gmail.com>
% Copyright © 2014 University of Oxford
%  
% University of Oxford means the Chancellor, Masters and Scholars of
% the University of Oxford, having an administrative office at
% Wellington Square, Oxford OX1 2JD, UK. 
%
% This file is part of Gerardus.
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details. The offer of this
% program under the terms of the License is subject to the License
% being interpreted in accordance with English Law and subject to any
% action against the University of Oxford being under the jurisdiction
% of the English Courts.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

% for each file 1 to n
for i = 1:n
    % import the text file
    filetoimport = sprintf('trajs%d.txt', i);
    trajs = dlmread(filetoimport);
        
    % remove the unneeded columns (referencing timepoints and molecule
    % names
    trajs = trajs(:,[1 4:6]);

    % calculate number of timesteps and number of molecules
    timesteps = trajs(:,1);
    no_of_timesteps = max(timesteps)-1; % final timestep sometimes has
    % problems with missing data so it easier to remove this final one
    no_of_molecules = nnz(timesteps ==1);

    % rearrange matrix
    positions = zeros(no_of_timesteps, no_of_molecules, 3);
    for j = 1:no_of_timesteps
         timestep_data = trajs((no_of_molecules*(j-1)+1):no_of_molecules*j,:);
         positions(j,:,:) = timestep_data(:,2:4);
    end

    % rearrange order of matrix to be (molecules, xyz, timestep)
    positions = permute(positions,[2 3 1]);
    
    % exclude those outside the voxels
    % check for molecules finishing outside of voxel
    keep1 = positions(:,1,end)<= (voxel_size - ((voxel_size-actual_voxel_size)/2));
    keep2 = positions(:,1,end)>= (voxel_size - actual_voxel_size)/2;
    keep3 = positions(:,2,end)<= (voxel_size - ((voxel_size-actual_voxel_size)/2));
    keep4 = positions(:,2,end)>= (voxel_size - actual_voxel_size)/2;
    keep5 = positions(:,3,end)<= (voxel_size - ((voxel_size-actual_voxel_size)/2));
    keep6 = positions(:,3,end)>= (voxel_size - actual_voxel_size)/2;
    % create a vector of 0s (discard) & 1s (keep)
    keep_molecule = keep1.*keep2.*keep3.*keep4.*keep5.*keep6;
    % make 0s into nan
    keep_molecule(keep_molecule==0) = nan;
    positions(:,1,1) = positions(:,1,1).*keep_molecule;
    % remove NaNs
    positions = positions(isfinite(positions(:,1)),:,:);

    % save matlab file as positionsi.mat
    filetosave = sprintf('positions%d.mat', i);
    save(filetosave, 'positions', '-v7.3');
    clearvars -except i
end

end

