package org.beryx.viewreka.extras.editor;

import java.net.URL;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import javafx.fxml.FXML;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.TextField;
import javafx.util.Callback;
import jfxtras.scene.control.CalendarPicker;
import jfxtras.scene.control.CalendarTextField;
import jfxtras.scene.control.LocalDateTimeTextField;

import org.beryx.viewreka.core.DataException;
import org.beryx.viewreka.gui.FxUtil;
import org.beryx.viewreka.gui.editor.ParameterEditor;
import org.beryx.viewreka.parameter.DateConfiguration;
import org.beryx.viewreka.parameter.Parameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DateTimeParameterEditor<T> extends ParameterEditor<T> {
	private static final Logger log = LoggerFactory.getLogger(DateTimeParameterEditor.class);

	public static final String ATTR_SHOW_TIME = "show.time";

	private final DateConfiguration<T> dateConfig ;
	
	@FXML private Label lbPrmName;
	@FXML private CalendarTextField calendarText;

	private String currentValue;

	public DateTimeParameterEditor(Parameter<T> parameter, Map<String, String> attributes) {
		super(parameter, attributes);
		if(!(parameter instanceof DateConfiguration)) {
			throw new DataException("Parameter " + parameter.getName() + " does not implement DateConfiguration, therefore is not compatible with DateTimeParameterEditor.");
		}
		this.dateConfig = (DateConfiguration<T>)parameter;
	}

	@Override
	public <TT> void valueChanged(Parameter<TT> parameter, TT oldValue) {
		throw new UnsupportedOperationException("A DateTimeParameterEditor cannot act as ParameterListener.");
	}
	
	@Override
	public void initialize(URL location, ResourceBundle resources) {
		check("lbPrmName", lbPrmName);		
		check("calendarText", calendarText);
		
		log.debug("Initializing DateTimeParameterEditor(" + parameter + "). Editor attributes: " + attributes);
		
		lbPrmName.setText(parameter.getName());
		
		currentValue = parameter.getValueAsString();

		Locale locale = dateConfig.getLocale();
		calendarText.setLocale(locale);
		calendarText.setDateFormat(dateConfig.getDateFormat());
//		calendarText.setPromptText("Prompt: " + parameter.getName());
//		calendarText.setText("Bla bla");
		
		calendarText.setShowTime(Boolean.valueOf(attributes.get(ATTR_SHOW_TIME)));
		
		log.debug("locale: " + locale + ", datePattern: " + dateConfig.getDatePattern());
		
		calendarText.setAllowNull(parameter.isNullAllowed());

		String sVal = parameter.getValueAsString();
		if(sVal != null && !"null".equals(sVal)) {
			Calendar calendar = Calendar.getInstance(locale);
			try {
				Date date = dateConfig.getDateFormat().parse(sVal);				
				calendar.setTime(date);
				calendarText.setCalendar(calendar);
			} catch(ParseException e) {
				throw new DataException("Parameter " + parameter.getName() + ": cannot parse '" + sVal + "' using the date format '" + dateConfig.getDatePattern() + "'");
			}
		}

		calendarText.calendarProperty().addListener((observable, oldValue, newValue) -> {
			if(newValue == null) {
				parameter.setValue(null);
			} else {
				long millis = newValue.getTimeInMillis();
				parameter.setValue(dateConfig.fromMilliseconds(millis));
			}
			updateControls();
		});
		
		calendarText.setParseErrorCallback(e -> {
			log.debug("\t\t\t###### ParseErrorCallback called: " + e + ", text: " + calendarText.getText());
			return null;
		});
		
	}

//	public void trySaveValue() {
//		if(!isValidChanged()) {
//			log.error("Invalid input!!!");
//		} else {
//			saveValue();
//		}
//	}
	
//	public void saveValue() {
//		currentValue = calendarText.getText();
//		parameter.setValueFromString(currentValue);
//		updateControls();
//	}
//	
//	public void cancelValue() {
//		calendarText.setText(currentValue);
//		updateControls();
//		calendarText.requestFocus();
//	}


	private static final List<String> SUFFIXES = Arrays.asList("changed-valid", "changed-invalid", "unchanged-valid", "unchanged-invalid");
	
	@Override
	public void updateControls() {
//		butCancelValue.setDisable(!isChanged());
//		butSaveValue.setDisable(!isValidChanged());
//		
		log.debug("\t\t\t###### updateControls()");

		String suffix = (isChanged() ? "changed" : "unchanged") + "-" + (isValid() ? "valid" : "invalid");				
		FxUtil.applyStyle("input", SUFFIXES, suffix, calendarText);
	}

	public boolean isChanged() {
		long millis = calendarText.getCalendar().getTimeInMillis();
		T value = dateConfig.fromMilliseconds(millis);
		String sVal = parameter.asString(value);
		
		
		log.debug("\t\t\t###### isChanged(): calendarText.getText() = " + calendarText.getText() + ", sVal: " + sVal + ", currentValue: " + currentValue);
		
		return false;
	}
	
	public boolean isValid() {
		try {
			log.debug("\t\t\t###### isValid()");
			return true;
		} catch(Exception e) {
			return false;
		}
	}
	
	public boolean isValidChanged() {
		log.debug("\t\t\t###### isValidChanged()");
		return isChanged() && isValid();
	}
}
