/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.kafka.timeline;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.kafka.common.utils.LogContext;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;


@Timeout(value = 40)
public class SnapshotRegistryTest {
    @Test
    public void testEmptyRegistry() {
        SnapshotRegistry registry = new SnapshotRegistry(new LogContext());
        assertThrows(RuntimeException.class, () -> registry.getSnapshot(0));
        assertIteratorContains(registry.iterator());
    }

    private static void assertIteratorContains(Iterator<Snapshot> iter,
                                               Snapshot... snapshots) {
        List<Snapshot> expected = new ArrayList<>();
        for (Snapshot snapshot : snapshots) {
            expected.add(snapshot);
        }
        List<Snapshot> actual = new ArrayList<>();
        while (iter.hasNext()) {
            Snapshot snapshot = iter.next();
            actual.add(snapshot);
        }
        assertEquals(expected, actual);
    }

    @Test
    public void testCreateSnapshots() {
        SnapshotRegistry registry = new SnapshotRegistry(new LogContext());
        Snapshot snapshot123 = registry.createSnapshot(123);
        assertEquals(snapshot123, registry.getSnapshot(123));
        assertThrows(RuntimeException.class, () -> registry.getSnapshot(456));
        assertIteratorContains(registry.iterator(), snapshot123);
        assertEquals("Can't create a new snapshot at epoch 1 because there is already " +
            "a snapshot with epoch 123", assertThrows(RuntimeException.class,
                () -> registry.createSnapshot(1)).getMessage());
        Snapshot snapshot456 = registry.createSnapshot(456);
        assertIteratorContains(registry.iterator(), snapshot123, snapshot456);
    }

    @Test
    public void testCreateAndDeleteSnapshots() {
        SnapshotRegistry registry = new SnapshotRegistry(new LogContext());
        Snapshot snapshot123 = registry.createSnapshot(123);
        Snapshot snapshot456 = registry.createSnapshot(456);
        Snapshot snapshot789 = registry.createSnapshot(789);
        registry.deleteSnapshot(snapshot456.epoch());
        assertIteratorContains(registry.iterator(), snapshot123, snapshot789);
    }

    @Test
    public void testDeleteSnapshotUpTo() {
        SnapshotRegistry registry = new SnapshotRegistry(new LogContext());
        registry.createSnapshot(10);
        registry.createSnapshot(12);
        Snapshot snapshot14 = registry.createSnapshot(14);
        registry.deleteSnapshotsUpTo(14);
        assertIteratorContains(registry.iterator(), snapshot14);
    }
}
