# zenrpc: JSON-RPC 2.0 Server Implementation with SMD support

[![Go Report Card](https://goreportcard.com/badge/github.com/vmkteam/zenrpc)](https://goreportcard.com/report/github.com/vmkteam/zenrpc) [![Build Status](https://github.com/vmkteam/zenrpc/actions/workflows/go.yml/badge.svg?branch=master)](https://github.com/vmkteam/zenrpc/actions) [![codecov](https://codecov.io/gh/vmkteam/zenrpc/branch/master/graph/badge.svg)](https://codecov.io/gh/vmkteam/zenrpc) [![Go Reference](https://pkg.go.dev/badge/github.com/vmkteam/zenrpc.svg)](https://pkg.go.dev/github.com/vmkteam/zenrpc)

`zenrpc` is a JSON-RPC 2.0 server library with [Service Mapping Description](https://dojotoolkit.org/reference-guide/1.8/dojox/rpc/smd.html) support. 
It's built on top of `go generate` instead of reflection. 

# How to Use

```Service is struct with RPC methods, service represents RPC namespace.```

  1. Install zenrpc generator `go get github.com/vmkteam/zenrpc/v2/zenrpc`
  1. Import `github.com/vmkteam/zenrpc/v2` into our code with rpc service.
  1. Add trailing comment `//zenrpc` to your service or embed `zenrpc.Service` into your service struct.
  1. Write your funcs almost as usual.
  1. Do not forget run `go generate` or `zenrpc` for magic

### Accepted Method Signatures

    func(Service) Method([args]) (<value>, <error>)
    func(Service) Method([args]) <value>
    func(Service) Method([args]) <error>
    func(Service) Method([args])

- Value could be a pointer
- Error is error or *zenrpc.Error

## Example

```go
package main

import (
	"context"
	"errors"
	"flag"
	"log"
	"math"
	"net/http"
	"os"

	"github.com/vmkteam/zenrpc/v2"
	"github.com/vmkteam/zenrpc/v2/testdata"
)

type ArithService struct{ zenrpc.Service }

// Sum sums two digits and returns error with error code as result and IP from context.
func (as ArithService) Sum(ctx context.Context, a, b int) (bool, *zenrpc.Error) {
	r, _ := zenrpc.RequestFromContext(ctx)

	return true, zenrpc.NewStringError(a+b, r.Host)
}

// Multiply multiples two digits and returns result.
func (as ArithService) Multiply(a, b int) int {
	return a * b
}

type Quotient struct {
	Quo, Rem int
}

func (as ArithService) Divide(a, b int) (quo *Quotient, err error) {
	if b == 0 {
		return nil, errors.New("divide by zero")
	} else if b == 1 {
		return nil, zenrpc.NewError(401, errors.New("we do not serve 1"))
	}

	return &Quotient{
		Quo: a / b,
		Rem: a % b,
	}, nil
}

// Pow returns x**y, the base-x exponential of y. If Exp is not set then default value is 2.
//zenrpc:exp=2
func (as ArithService) Pow(base float64, exp float64) float64 {
	return math.Pow(base, exp)
}

//go:generate zenrpc

func main() {
	addr := flag.String("addr", "localhost:9999", "listen address")
	flag.Parse()

	rpc := zenrpc.NewServer(zenrpc.Options{ExposeSMD: true})
	rpc.Register("arith", testdata.ArithService{})
	rpc.Register("", testdata.ArithService{}) // public
	rpc.Use(zenrpc.Logger(log.New(os.Stderr, "", log.LstdFlags)))

	http.Handle("/", rpc)

	log.Printf("starting arithsrv on %s", *addr)
	log.Fatal(http.ListenAndServe(*addr, nil))
}

```


## Magic comments

All comments are optional.

    Method comments
    //zenrpc:<method parameter>[=<default value>][whitespaces<description>]
    //zenrpc:<error code>[whitespaces<description>]
    //zenrpc:return[whitespaces<description>]
     
    Struct comments
    type MyService struct {} //zenrpc
    
## Need to browse your api and do some test api calls?
We recommend to use [SMDBox](https://github.com/semrush/smdbox). It is Swagger-like JSON RPC API browser, compatible with smd scheme, generated by zenrpc. 

# JSON-RPC 2.0 Supported Features

  * [x] Requests
    * [x] Single requests
    * [x] Batch requests
    * [x] Notifications
  * [x] Parameters
    * [x] Named
    * [x] Position
    * [x] Default values
  * [x] SMD Schema
    * [x] Input
    * [x] Output
    * [x] Codes
    * [ ] Scopes for OAuth

# Server Library Features

 * [x] go generate
 * [ ] Transports
   * [x] HTTP
   * [x] WebSocket
   * [ ] RabbitMQ
 * [x] Server middleware
   * [x] Basic support
   * [x] Metrics
   * [x] Logging
