# Справочник API

### Vuex.Store

``` js
import Vuex from 'vuex'

const store = new Vuex.Store({ ...options })
```

### Опции конструктора Vuex.Store

- **state**

  - тип: `Object`

    Корневой объект состояния хранилища Vuex.

    [Подробнее](state.md)

- **mutations**

  - тип: `{ [type: string]: Function }`

    Регистрирует доступные для хранилища мутации. Обработчики мутаций первым аргументом всегда получают `state` (при использовании модулей это будет локальный state модуля). Вторым аргументом передаётся "нагрузка" (`payload`), если она есть.

    [Подробнее](mutations.md)

- **actions**

  - тип: `{ [type: string]: Function }`

    Регистрирует действия хранилища. В функции-обработчики передаётся объект `context`, со следующими свойствами:

    ``` js
    {
      state,     // то же, что и store.state, или локальный state при использовании модулей
      rootState, // то же, что и store.state, только при использовании модулей
      commit,    // то же, что и store.commit
      dispatch,  // то же, что и store.dispatch
      getters    // то же, что и store.getters
    }
    ```

    [Подробнее](actions.md)

- **getters**

  - тип: `{ [key: string]: Function }`

    Регистрирует геттеры, используемые в хранилище. Геттер-функции при вызове получают следующие аргументы:

    ```
    state,     // при использовании модулей — локальный state модуля
    getters,   // то же, что и store.getters
    rootState  // то же, что и store.state
    ```
    Зарегистрированные геттеры далее доступны в `store.getters`.

    [Подробнее](getters.md)

- **modules**

  - тип: `Object`

    Объект, содержащий подмодули для помещения в хранилище, в формате:

    ``` js
    {
      key: {
        state,
        namespaced?,
        mutations?,
        actions?,
        getters?,
        modules?
      },
      ...
    }
    ```

    Каждый модуль может содержать `state` и `mutations`, как и корневое хранилище. Состояние модуля будет прикреплёно к корневому, по указанному ключу. Мутации и геттеры модуля получают при вызове первым аргументом только локальное состояние, а не корневое. При вызове действий `context.state` аналогичным образом указывает на локальное состояние модуля.

    [Подробнее](modules.md)

- **plugins**

  - тип: `Array<Function>`

    Массив функций-плагинов, которые будут применены к хранилищу. Плагины попросту получают хранилище в качестве единственного аргумента, и могут как отслеживать мутации (для сохранения исходящих данных, логирования, или отладки) или инициировать их (для обработки входящих данных, например вебсокетов или observables).

    [Подробнее](plugins.md)

- **strict**

  - тип: `Boolean`
  - default: `false`

    Заставляет хранилище Vuex использовать strict mode. В strict mode любые изменения состояния, происходящие за пределами обработчиков мутаций, будут выбрасывать ошибки.

    [Подробнее](strict.md)

### Свойства инстанса Vuex.Store

- **state**

  - тип: `Object`

    Корневое состояние. Только для чтения.

- **getters**

  - тип: `Object`

    Зарегистрированные геттеры. Только для чтения.

### Методы инстанса Vuex.Store

- **`commit(type: string, payload?: any) | commit(mutation: Object)`**

  Запускает мутацию. [Подробнее](mutations.md)

- **`dispatch(type: string, payload?: any) | dispatch(action: Object)`**

  Инициирует действие. Возвращает то же значение, что и вызванный обработчик действия, или Promise, если вызывается несколько обработчиков. [Подробнее](actions.md)

- **`replaceState(state: Object)`**

  Позволяет заменить корневое состояние хранилища. Используйте только для гидрации состояния / функционала "машины времени".

- **`watch(getter: Function, cb: Function, options?: Object)`**

  Устанавливает наблюдение за возвращаемым значением геттера, вызывая коллбэк в случае его изменения. Геттер получает единственный параметр состояние хранилища. Возможно указание дополнительного объекта опций, с такими же параметрами как и у метода `vm.$watch` корневой библиотеки Vue.

  Для прекращения наблюдения, необходимо вызвать возвращённую функцию-хэндлер.

- **`subscribe(handler: Function)`**

  Подписывается на мутации хранилища. `handler` вызывается после каждой мутации и получает в качестве параметров дескриптор мутации и состояние после мутации:

  ``` js
  store.subscribe((mutation, state) => {
    console.log(mutation.type)
    console.log(mutation.payload)
  })
  ```

  Чаще всего используется в плагинах. [Подробнее](plugins.md)

- **`registerModule(path: string | Array<string>, module: Module)`**

  Регистрирует динамический модуль. [Подробнее](modules.md#динамическая-регистрация-модулей)

- **`unregisterModule(path: string | Array<string>)`**

  Разрегистрирует динамический модуль. [Подробнее](modules.md#динамическая-регистрация-модулей)

- **`hotUpdate(newOptions: Object)`**

  Осуществляет горячую замену действий и мутаций. [Подробнее](hot-reload.md)

### Вспомогательные функции для связывания с компонентами

- **`mapState(namespace?: string, map: Array<string> | Object): Object`**

  Создаёт проксирующие вычисляемые свойства компонента, возвращающие поддерево state'а хранилища Vuex [Подробнее](state.md#вспомогательная-функция-mapstate)

  Первый аргумент опционально может быть строкой с указанным namespace. [Подробнее](modules.md#подключение-с-помощью-вспомогательных-функций-к-пространству-имён)

- **`mapGetters(namespace?: string, map: Array<string> | Object): Object`**

  Создаёт проксирующие вычисляемые свойства компонента, проксирующие доступ к геттерам. [Подробнее](getters.md#вспомогательная-функция-mapgetters)

  Первый аргумент опционально может быть строкой с указанным namespace. [Подробнее](modules.md#подключение-с-помощью-вспомогательных-функций-к-пространству-имён)

- **`mapActions(namespace?: string, map: Array<string> | Object): Object`**

  Создаёт проксирующие методы компонента, позволяющие диспетчеризировать действия. [Подробнее](actions.md#диспетчеризация-действий-в-компонентах)

  Первый аргумент опционально может быть строкой с указанным namespace. [Подробнее](modules.md#подключение-с-помощью-вспомогательных-функций-к-пространству-имён)

- **`mapMutations(namespace?: string, map: Array<string> | Object): Object`**

  Создаёт проксирующие методы компонента, позволяющие инициировать мутации. [Подробнее](mutations.md#вызов-мутаций-из-компонентов)

  Первый аргумент опционально может быть строкой с указанным namespace. [Подробнее](modules.md#подключение-с-помощью-вспомогательных-функций-к-пространству-имён)
