const { assert, skip, test, module: describe, only } = require('qunit');
const { GPU, CPUKernel } = require('../../src');

describe('features: image array');
function getImages(callback) {
  const imageSources = [
    'data:image/jpeg;base64,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',
    'data:image/jpeg;base64,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',
    'data:image/jpeg;base64,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',
    'data:image/jpeg;base64,/9j/4AAQSkZJRgABAgAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRofHh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJCQwLDBgNDRgyIRwhMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjL/wAARCABbAIoDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREAAgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYkNOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOEhYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwDso4Esrr7RHsZSNrxyLuBx+owO49a0I7TzTuZcBsYwenoMn3oht44pUiuD+7mbYjnsxH3T6Z7fl1xlYrgtp1vCcieK6jgmz2ZXGfzwPzrvxGLV/c37nvQi4LR3emv9dinqK2tgmnsWBAY/Me6iNyST9AaybCQ6l4isYYlVfs4Bk2jhcrl8+vBA+rVjaxrPn+XAmGREIRAfvs5zj/vnaPo9dt4T0Q6VYefOCbudfnLdVGcgfXkk+7H2rSipQp+0n8jbMZrCYV8/xSul+Ov3M6Qtkf0qJzSlqhdhWcUfFEcjVWdqfI4qu710wiSxGamFqQtTC1bJEkm6jNRbqXdTsFyTNAPNR7qN1FguTbhQTUJak30coXHlqTdUZam7qqwmySS7RLV7XV12RkYDn5WHPGcdccHcMjjnb0HLaz4m8tntICJJ5fLBeLncyMNsg92XA9iuPepLfRdeuI2gldkt3Uqy3M5kH5YOfzArc0XwtYaPL9oI+0XZBBmk6jPXA7fz968iGEad5s+5q47BYNOXMpy6JbfN9PlfoUfCfhM2hTUdTTNx96OI8hM9z7+3auy3gZ96g34Hv60xpa62nLc+UxmNq4yq6tZ6/gvJE7SVA8tRNL71A8laRpnK2Pkkquz80xpOetQu9dEYENkxems/FV95z1prSVooE3J99OD1TMlKH96fIFy5vpN9VvMo3+9HIO5OXpPMquZKb5lNQFctb6TeKreZRv8AenyBc6EyZxxjHpSGQVR873pDL71yezNLlxpKheWq5l96ieWrjTE2TtL71C0tQNLULS1tGmS2TtJUTSe9QNLURkraMCWyw0mKYZarmSmb6tQFcsmSgSVVL+9AkquQLlvzKXzKqiSl8z3pcg7lgyc0hkqsXpN9CgJss+ZR5nvVUyUnmGq5BGuZuOtJ5x9ap+ZSebWKpl3LZlPrTGl96rGT3phk96pQAnaWojJzUReomf3rRQE2TNJUZkqJpKiaStFAlk5kppfiq5kpDJV8gix5lAeq2+jzKfIBbD8UGSq3mcUnmUuQZZ8yk8yq++ml+aagBYL0nmVWaSm+YapQIuavmUhkqt5lMMtZchsWjJ70wye9VjJTTJVqAiwZPeo2k96rmSmGSqUBMnZ6jL1C0nFMMlaKBLZMXpu+oC9NL8VagS2WfMo8yqvmUoejkFct+ZRvqrvo8yjkKuWt9NL1X300yUKAXJzJTfMqu0nNM8z3q1AycjTMlNMnvVfJ9aTJ9ajlRvcn8zmkMlV8nPWkJPrT5RNkxkphlqBic9aYSfWrUCHImMlMMlQkn1qNic9atRRm5k5k96aZarkn1ppJ9atQM3ULHm0ol96pkn1pyk+tHIJVC4JPel82quTjrRk0uVFqZZMtMMtQEn1qMk+tNQIlUJ2lpnmmoCTnrTc1ooHPKo2z/9k='
  ];
  const images = [];
  let finished = 0;
  function onload() {
    finished++;
    if (finished === imageSources.length) {
      callback(images);
    }
  }
  for (let i = 0; i < imageSources.length; i++) {
    const image = document.createElement('img');
    image.onload = onload;
    image.src = imageSources[i];
    images.push(image);
  }
}

function imageArrayTest(mode, done) {
  const gpu = new GPU({
    mode
  });
  const imageKernel = gpu.createKernel(function(images) {
    const pixel = images[this.thread.z][this.thread.y][this.thread.x];
    this.color(pixel[0], pixel[1], pixel[2], pixel[3]);
  }, {
    graphical: true,
    output : [138, 91]
  });
  getImages(function(images) {
    imageKernel(images);
    const pixels = imageKernel.getPixels();
    assert.equal(pixels.length, 50232);
    // way too large to test the whole picture, just test the first pixel
    assert.equal(pixels[0], 147);
    assert.equal(pixels[1], 168);
    assert.equal(pixels[2], 251);
    assert.equal(pixels[3], 255);
    gpu.destroy();
    done(imageKernel);
  });
}

(typeof Image !== 'undefined' ? test : skip)('image array auto', t => {
  imageArrayTest(null, t.async());
});

(typeof Image !== 'undefined' ? test : skip)('image array gpu', t => {
  imageArrayTest('gpu', t.async());
});

(GPU.isWebGLSupported ? test : skip)('image array webgl', t => {
  const done = t.async();
  imageArrayTest('webgl', kernel => {
    // They aren't supported, so test that kernel falls back
    assert.equal(kernel.kernel.constructor, CPUKernel);
    done();
  });
});

(GPU.isWebGL2Supported ? test : skip)('image array webgl2', t => {
  imageArrayTest('webgl2', t.async());
});

(typeof Image !== 'undefined' ? test : skip)('image array cpu', t => {
  imageArrayTest('cpu', t.async());
});
