import path from 'node:path'
import fs from 'node:fs'
import os from 'node:os'
import { Bench, type Task } from 'tinybench'

import { tools, type Checker } from '../test/bench/stand/index.ts'
import { suites } from './build-coherence-md.ts'

const __dirname = path.dirname(new URL(import.meta.url).pathname)
const outfile = path.resolve(__dirname, '../../BENCHMARK.md')

interface NamedResult {
  name: string
  result: NonNullable<Task['result']>
}

function systemNote(): string {
  const platform = `${os.type()} ${os.release()} (${os.platform()} ${os.arch()})`
  const cpu = os.cpus()[0]?.model ?? 'Unknown CPU'
  const cores = os.cpus().length
  const ram = `${(os.totalmem() / 1024 / 1024 / 1024).toFixed(1)} GB`
  const node = process.version

  return `**System Info**
  - ${cpu} (${cores} cores), ${ram}
  - ${platform}
  - Node.js ${node}
`
}

function toMarkdown(results: NamedResult[]): string {
  // sort best → worst by ops/sec
  const sorted = results.sort((a, b) => b.result.hz - a.result.hz)

  const headers = ['#', 'Name', 'Ops/sec', 'Avg time (ms)', '± %']
  const rows = sorted.map((r, i) => {
    const { hz, mean, sd } = r.result
    const percent = (sd / mean) * 100

    return [
      (i + 1).toString(),
      r.name,
      hz.toFixed(2),
      mean.toFixed(3),
      `±${percent.toFixed(2)}%`,
    ]
  })

  const table = [
    `| ${headers.join(' | ')} |`,
    `| ${headers.map((_, i) => '---' + (i > 1 ? ':' : '')).join(' | ')} |`,
    ...rows.map(row => `| ${row.join(' | ')} |`)
  ]

  return table.join('\n')
}


let output = `# Performance across libraries
> Autogenerated by \`src/scripts/build-bench-md.ts\`

${systemNote()}
`

for (const {title, method, cases} of suites) {
  const bench = new Bench({ time: 200 })
  const addrs = cases.map(c => c[0])
  const results: NamedResult[] = []

  for (const [libName, api] of Object.entries(tools)) {
    const fn = api[method] as Checker
    if (typeof fn !== 'function') continue
    bench.add(`${libName}`, () => {
      for (const a of addrs)
        fn(a)
    })
  }

  for (const task of bench.tasks) {
    await task.run()
    if (task.result) {
      results.push({ name: task.name, result: task.result })
    }
  }

  output += `## ${title}
${toMarkdown(results)}
`
}

fs.writeFileSync(outfile, output)
