import fs from 'node:fs'
import path from 'node:path'

import { tools, type Checker } from '../test/bench/stand/index.ts'

const __dirname = path.dirname(new URL(import.meta.url).pathname)
const outfile = path.resolve(__dirname, '../../COHERENCE.md')

export const suites: {title: string, method: string, cases: [string, boolean, string][]}[] = [
  {
    title: 'IPv4',
    method: 'isIPv4',
    cases: [
      ['127.0.0.1', true, 'localhost'],
      ['192.168.1.1', true, 'private LAN'],
      ['255.255.255.255', true, 'broadcast'],
      ['0.0.0.0', true, 'unspecified'],
      ['8.8.8.8', true, 'Google DNS'],

      ['256.1.1.1', false, 'octet out of range'],
      ['192.168.1', false, 'not enough octets'],
      ['192.168.1.1.1', false, 'too many octets'],
      ['192.168.01.1', false, 'leading octet zero'],
      ['abc.def.gha.bcd', false, 'non-numeric'],
      ['...', false, 'empty octets'],
      ['1..2.3', false, 'empty middle octet'],
      ['', false, 'empty string'],
    ]
  },
  {
    title: 'IPv6',
    method: 'isIPv6',
    cases: [
      ['::1', true, 'loopback'],
      ['::', true, 'unspecified'],
      ['2001:db8::1', true, 'documentation address'],
      ['fe80::1ff:fe23:4567:890a', true, 'link-local'],
      ['::ffff:192.0.2.128', true, 'IPv4-mapped IPv6'],
      ['2001:0db8:85a3:0000:0000:8a2e:0370:7334', true, 'full form'],
      ['0:0:0:0:0:0:0:1', true, 'expanded loopback'],

      ['2001::85a3::7334', false, 'multiple ::'],
      ['12345::', false, 'hextet too long'],
      ['abcd:efgh::1', false, 'non-hex character'],
      ['1:2:3:4:5:6:7', false, 'too few hextets'],
      ['1:2:3:4:5:6:7:8:9', false, 'too many hextets'],
      [':1:2:3:4:5:6:7', false, 'leading colon without ::'],
      ['1:2:3:4:5:6:7:', false, 'trailing colon without ::'],
      ['1:2:3::4:5:6:7:8', false, 'too many hextets with ::'],
      ['1:2::4:5::8', false, 'multiple compressors ::'],
      [':::', false, 'triple colon not allowed'],
      ['::g', false, 'invalid hex digit after compression'],
      ['2001:db8:85a3:0:0:8a2e:37023:7334', false, 'hextet exceeds 16 bits'],
      ['::ffff:999.0.2.128', false, 'invalid embedded IPv4'],
      ['::ffff:192.0.2', false, 'truncated IPv4 in mapped address'],
      ['2001:db8:::', false, 'extra colon at end'],
      ['', false, 'empty string'],
    ]
  },
  {
    title: 'Private IP',
    method: 'isPrivate',
    cases: [
      ['0.0.0.0', true, 'unspecified'],
      // Loopback
      ['127.0.0.1', true, 'loopback'],
      ['127.0.0.2', true, 'loopback range'],
      ['127.1.1.1', true, 'loopback shorthand'],

      // Private 192.168/16
      ['192.168.0.123', true, 'private LAN'],
      ['192.168.122.123', true, 'private LAN'],
      ['192.162.1.2', false, 'public, outside 192.168/16'],

      // Private 172.16–31/12
      ['172.16.0.5', true, 'private start'],
      ['172.16.123.254', true, 'private inside'],
      ['171.16.0.5', false, 'public, below range'],
      ['172.25.232.15', true, 'private inside'],
      ['172.15.0.5', false, 'public, below range'],
      ['172.32.0.5', false, 'public, above range'],

      // Link-local 169.254/16
      ['169.254.2.3', true, 'link-local'],
      ['169.254.221.9', true, 'link-local'],
      ['168.254.2.3', false, 'public, outside 169.254/16'],

      // Private 10/8
      ['10.0.2.3', true, 'private 10/8'],
      ['10.1.23.45', true, 'private 10/8'],
      ['12.1.2.3', false, 'public, outside 10/8'],

      // Benchmark / test-net
      ['198.18.0.0', true, 'benchmark range'],

      // IPv6 reserved/private
      ['fd12:3456:789a:1::1', true, 'ULA fc00::/7'],
      ['fe80::f2de:f1ff:fe3f:307e', true, 'link-local fe80::/10'],
      ['::ffff:10.100.1.42', true, 'IPv4-mapped private'],
      ['::FFFF:172.16.200.1', true, 'IPv4-mapped private'],
      ['::ffff:192.168.0.1', true, 'IPv4-mapped private'],

      // Public IPv4
      ['165.225.132.33', false, 'public'],

      // Special IPv6
      ['::', true, 'unspecified'],
      ['::1', true, 'loopback'],
      ['fe80::1', true, 'link-local'],

      // Security CVE test cases
      ['0x7f.1', true, 'CVE-2023-42282: hex loopback'],
      ['127.1', true, 'CVE-2024-29415: short loopback'],
      ['2130706433', true, 'CVE-2024-29415: integer loopback'],
      ['::fFFf:127.0.0.1', true, 'CVE-2024-29415: IPv6-mapped loopback'],
      ['::0:ffff:127.0.0.1', true, '↑'],
      ['0::0:ffff:127.0.0.1', true, '↑'],
      ['0:0:0:0:0:ffff:127.0.0.1', true, '↑'],

      // Invalid / malformed
      ['017700000001', false, 'octal 127.0.0.1'], // https://github.com/indutny/node-ip/issues/162
      ['01200034567', false, 'invalid: octal-style'],
      ['012.1.2.3', false, 'invalid: octal-style IPv4'],
      ['000:0:0000::01', true, 'valid: zero-compression'],
      ['::fFFf:127.255.255.256', false, 'invalid: IPv4 overflow'],
    ]
  }
]

let output = `# Coherence across libraries
> Autogenerated by \`src/scripts/build-coherence-md.ts\`
`

for (const {title, method, cases} of suites) {
  const apis = Object.entries(tools)
    .filter(([, methods]) => method in methods)
  const thead = ['Address', ...apis.map(([name, {ref}]) => ref ? `[\`${name}\`](${ref})`: name), 'Comment']
  output += `## ${title}
${thead.join(' | ')}
${thead.map(() => '---').join(' | ')}
`

  for (const [addr, expected, comment] of cases) {
    const results = apis.map(([_, methods]) => {
      const res = (methods[method] as Checker)(addr)
      if (res === null) return 'n/a'
      return res === expected ? '✓' : '❌'
    })
    output += `${['`' + addr + '`', ...results, comment].join(' | ')}\n`
  }
}

fs.writeFileSync(outfile, output)
