﻿# Scally Core




## Contents

- [What is it?](#what-is-it)
- [The breakdown](#the-breakdown)
  - [Base](#base)
    - [Sensible styles](#sensible-styles)
    - [Normalize-esque styles](#normalize-esque-styles)
    - [Other noteworthy parts of base](#other-noteworthy-parts-of-base)
      - [Responsive images](#responsive-images)
      - [Print](#print)
      - [Root](#root)
      - [Forms](#forms)
  - [Functions](#functions)
  - [Mixins](#mixins)
  - [Placeholders](#placeholders)
  - [Settings](#settings)
  - [Debug](#debug)
  - [Normalize](#normalize)
  - [Reset](#reset)
- [Specificity](#specificity)
- [Demo's](#demos)
- [Further reading](#further-reading)




## What is it?

Scally core is the foundation of a project's UI build providing things like:

- Sensible global styles.
- Global settings (Sass variables).
- Handy Sass functions and mixins.
- Global Sass silent placeholder selectors.
- A powerful reset, and Normalize.scss.
- Debug styles.

Without core Scally won't work. It is the only mandatory part of the framework.




## The breakdown

Core is broken down into the following sections:

- [Base](#base)
- [Functions](#functions)
- [Mixins](#mixins)
- [Placeholders](#placeholders)
- [Settings](#settings)

And a few things not sectioned:

- [Debug](#debug)
- [Normalize](#normalize)
- [Reset](#reset)

### [Base](base/)

The base section provides very basic styling that most UI's will need. These styles are applied at the most global level, with most being applied via element selectors e.g.

```
p { ... }
```

```
a { ... }
```

Everything in base needs to be easily overridden. This is really easy to do as everything in base sits right at the bottom in terms of Scally's specificity (CSS' first C; the cascade), [see](#specificity).

#### Sensible styles

Base styles try to be as unopinionated as possible—like most things in Scally—with most applying *sensible* styles that you'd expect to find in a user agent stylesheet, like:

- Applying `cursor: help` to abbreviations (`abbr`) that have a `title` attribute.
- Applying `cursor: pointer` to `summary` and `label` elements.
- Removing the gap between media elements (`img`, `video`, `audio`, etc) and the bottom of their containers.
- Applying a bottom margin to all paragraphs (`p`) of text.

The more opinionated styles found in base sit behind a [toggle](settings/_toggles.scss) meaning you decide if you want to use them or not.

#### Normalize-esque styles

Base also provides [Normalize.css](http://necolas.github.io/normalize.css/)-esque styles like:

- Set the default behavior for touch-based browsing in IE 10 on devices running Windows 8.
- Remove rounded corners from iOS search inputs.
- Remove rounded corners from iOS `input` buttons.
- Remove the top inner shadow from iOS inputs. **[optional]**
- Hide the close button generated by IE 10+ for inputs. **[optional]**

Some of the above are actual Normalize.css overrides.

#### Other noteworthy parts of base

##### [Responsive images](base/_media.scss)

Scally—being a responsive ready framework—sets all image elements (`img`) to be responsive by default:

```
img {
  @if $responsive-images {
    max-width: 100%;
    height: auto;
  }
}
```

However this is optional, sitting behind the `$responsive-images` toggle.

##### [Print](base/_print.scss)

Scally provides some *sensible* global print styles, with most taken from the [HTML5 Boilerplate print styles](https://github.com/h5bp/html5-boilerplate/blob/master/src/css/main.css).

All print styles that aren't defined at this global base level will live in context with their corresponding rule sets.

##### [Root](base/_root.scss)

Root defines styles for the `html` element. It mainly focuses on typography styles e.g. setting font size, font family, line height, etc. Root also defines the global foreground colour which is inherited by all HTML elements, and the background colour of the `html` element itself.

##### [Forms](base/_forms.scss)

Scally provides quite a few base form styles. Most are concerned with applying [*sensible* styles](#sensible-styles) and [Normalize-esque styles](#normalize-esque-styles).

The other two main parts of forms are concerned with providing styles for text inputs (including `textarea`) and `select` lists, and a disabled state for **ALL** HTML form elements. These styles, being fairly oppinionated, are all optional, sitting behind the `$disabled-state` and `$text-input-and-select-styles` toggles.




### [Functions](functions/)

Scally features a bunch of handy Sass functions to keep the framework more readable and DRY, by abstracting any common reusable logic.

The [**Convert `px` to `em-rem`**](functions/_convert-px-to-em-rem.scss) function is the most used and powerful function and does what it says on the tin: *To convert pixels to either the `em` or `rem` units*. There is also a [**Convert `px` to `em-rem`**](mixins/_convert-px-to-em-rem.scss) mixin which is dependant on this function.

**Function**

```
margin-left: to-em(8);
```

**Mixin**

```
@include to-em(margin-left, 8);
```

The [Math helpers](functions/_math-helpers.scss) contain a handful of math helper functions which are used to quickly create size variants of property values, e.g.

```
padding: halve(3.2px);
```




### [Mixins](mixins/)

Scally features quite a few powerful Sass mixins. These mixins underpin many parts of the Scally framework and are **required** for many parts of the framework to work.

The most used and powerful Scally mixins are:

- [Convert `px` to `em-rem`](mixins/_convert-px-to-em-rem.scss)

  *some examples:*
  ```
  @include to-em(line-height, 30);

  @include to-em(width height, 125);

  @include to-rem(box-shadow, (inset 0 0 0 1 #2a9022) (inset 0 0 3 #459966));
  ```
- [Font size](mixins/_font-size.scss)

  *some examples:*
  ```
  @include font-size(18);

  @include font-size(12, 1.5);

  @include font-size(12, none);

  @include font-size(24, inherit);

  @include font-size(24, normal);
  ```
- [Generate at breakpoints](mixins/_generate-at-breakpoints.scss)

  *some examples:*
  ```
  @include generate-at-breakpoints('.u-text-size-small', all) {
     @include font-size($font-size-small);
   }

   @include generate-at-breakpoints('.u-demo{bp} li', palm lap) {
     vertical-align: top;
   }
   ```
- [Media queries](mixins/_media-queries.scss)

  *some examples:*
  ```
  @include respond-to(lap) {
    .foo {background: red;}
  }

  @include respond-to(500) {
    .foo {background: red;}
  }

  @include respond-to(palm, max) {
    .foo {background: red;}
  }

  @include respond-to(500, $axis: height) {
    .foo {background: red;}
  }

  @include respond-range(500, 780) {
    .foo {background: red;}
  }

  @include respond-range(lap) {
    .foo {background: red;}
  }
  ```

All of Scally's mixins except for the [**Target Browsers**](mixins/_target-browsers.scss) mixin use arguments which is a good way to use Sass mixins. Argument-less mixins are typically only used as containers for common CSS rules which isn't very optimal e.g. DRY. Scally negates the need for argument-less mixins through it's [utilities](../utilities/) and in certain cases it's [Sass silent placeholder selectors](#placeholders), and just following OOCSS methodologies.




### [Placeholders](placeholders/)

The Scally core [Sass silent placeholder selectors](http://sass-lang.com/documentation/file.SASS_REFERENCE.html#placeholder_selectors_) contain styles that are extremely global to a UI—and where using a class to apply these styles, like [utilities](../utilities/README.md) do—wouldn't be appropiate.

Right now there is only one core placeholder: [**Bottom spacing**](placeholders/_c-bottom-spacing.scss) which applies the base bottom spacing (and half) in order to try to keep a consistent vertical rhythm.

```
%c-bottom-spacing {@include to-rem(margin-bottom, $spacing-base);}
```

The main application for this placeholder is for paragraphs (`p`) and headings (`h1-h6`), applied in [base](#base). This is how it's applied to **ALL** paragraphs:

```
p {@extend %c-bottom-spacing;}
```

It wouldn't be practical to apply this bottom spacing with say one of the [**Spacing**](../utilities/_u-spacing.scss) utility classes, e.g.

```
<p class="u-s-mb-base"> [...] </p>
```

It's safe to assume that all paragraphs will need this bottom spacing, just as Scally defines [*sensible styles*](#sensible-styles) in [base](#base) we're doing the same here but encapsulating it into a placeholder as these styles need to be applied in various places, e.g. for horizontal rules (`hr`):

```
hr {
  @extend %c-bottom-spacing;
  [...]
}
```

The core placeholders are namespaced with `c-` so they're easily identifiable when used outside of core, e.g. every utility comes with a placeholder version, all of which are namespaced with `u-`.




### [Settings](settings/)

One of Scally's most powerful features is that it is highly configurable, only requiring you to bring in the parts you are using, keeping your CSS light weight and scalable. Scally ignores its own settings in favour of using your own, so you can completely modify how Scally works without ever having to alter the framework itself.

Scally achieves this by using Sass variables and being as unopinionated about design as possible.

The Scally settings section contains the most global settings for the framework, all other settings live in context in their relevant file (Sass partial).

Any settings you find set in Scally that you do not wish to keep, simply
redefine above or below the relevant `@import` in the [master stylesheet](../style.scss). This means that if Scally, for example, sets your `$font-size` at `16px` and you wish it to be `14px`, simply redeclare it **above** the relevant `@import`, like so:

 ```
$font-size: 14;
@import "bower_components/scally/core/settings/typography";
```

If you want to use a Scally setting to override something then you need to define the override **below** the `@import`, like so:

 ```
@import "bower_components/scally/core/settings/colours";
$colour-text-base: $colour-brand;
```

If you try to redeclare above the `@import` your Sass won't compile as you
don't have access to the Scally setting at the point of compilation.




### [Debug](_debug.scss)

`_debug.scss` is used to visually detect any improperly nested or
potentially invalid markup, or any potentially inaccessible code.

The legend:

- Red = definite error
- Yellow = double-check
- None = should be fine

Scally turns off the debug styles by default, to turn on change the `$debug-mode` [toggle](settings/_toggles.scss) to `true`, like so:

```
$debug-mode: true;
```




### [Normalize](_normalize.scss)

`_normalize.scss` is a [third party solution](http://necolas.github.io/normalize.css/). Scally always makes sure it's using the latest version.




### [Reset](_reset.scss)

In addition to `_normalize.css` Scally also provides a powerful reset, resetting things like:

- `margin`, `padding`, and `border`'s to zero, for **ALL** HTML elements.
- `box-sizing` to the more friendly `border-box` value.
- no bullets for unordered (`ul`) and ordered (`ol`) lists .




## Specificity

Scally core is the foundation of the framework therefore it sits right at the bottom when it comes to specificity (CSS' first C; the cascade) as any styles defined in core need to be easily overridden.

That's why core is declared first when pulling in the Scally framework via your [master stylesheet](../style.scss) so they're compiled before everything else.




## Demo's

- [Base](http://codepen.io/team/westfieldlabs/full/zIgBs)
- [Mixins](http://codepen.io/team/westfieldlabs/full/Bcfyz)
- [Settings](http://codepen.io/team/westfieldlabs/full/aLDdb)
- [Debug](http://codepen.io/team/westfieldlabs/full/GHzvE)




## Further reading

*Make sure to read the documentation within each core Sass partial file as it will contain information about the core feature and it's implementation.*

- [Using pure Sass functions to make reusable logic more useful](http://thesassway.com/advanced/pure-sass-functions)
- [Leveraging Sass mixins for cleaner code](http://thesassway.com/intermediate/leveraging-sass-mixins-for-cleaner-code)
- [When to use `@⁠extend`; when to use a mixin](http://csswizardry.com/2014/11/when-to-use-extend-when-to-use-a-mixin/)
- [Normalize.css](http://necolas.github.io/normalize.css/)