﻿#region License

/*
    Copyright [2011] [Jeffrey Cameron]

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

#endregion

using System;
using System.Linq;
using System.Xml.Linq;

using PicklesDoc.Pickles.ObjectModel;

namespace PicklesDoc.Pickles.DocumentationBuilders.HTML
{
    public class HtmlTableFormatter
    {
        private readonly XNamespace xmlns;
        private readonly HtmlImageResultFormatter htmlImageResultFormatter;

        public HtmlTableFormatter(HtmlImageResultFormatter htmlImageResultFormatter)
        {
          this.htmlImageResultFormatter = htmlImageResultFormatter;
          this.xmlns = HtmlNamespace.Xhtml;
        }

        public XElement Format(Table table)
        {
            return Format(table, null, false);
        }
        
        public XElement Format(Table table, ScenarioOutline scenarioOutline, bool includeResults)
        {
            if (table == null) return null;

            var headerCells = table.HeaderRow.ToArray();

            if (includeResults)
            {
                headerCells = headerCells.Concat(new[] { " " }).ToArray();
            }

            return new XElement(this.xmlns + "div",
                                new XAttribute("class", "table_container"),
                                new XElement(this.xmlns + "table",
                                             new XAttribute("class", "datatable"),
                                             new XElement(this.xmlns + "thead",
                                                          new XElement(this.xmlns + "tr",
                                                                       headerCells.Select(
                                                                           cell => new XElement(this.xmlns + "th", cell))
                                                              )
                                                 ),
                                             new XElement(this.xmlns + "tbody",
                                                          table.DataRows.Select(row => this.FormatRow(row, scenarioOutline, includeResults))
                                                 )
                                    ));
        }

      private XElement FormatRow(TableRow row, ScenarioOutline scenarioOutline, bool includeResults)
      {
        var formattedCells = row.Select(
          cell =>
          new XElement(
            this.xmlns + "td",
            cell)).ToList();

        if (includeResults && scenarioOutline != null)
        {
          formattedCells.Add(
            new XElement(this.xmlns + "td", this.htmlImageResultFormatter.Format(scenarioOutline, row.ToArray())));
        }

        var result = new XElement(this.xmlns + "tr");

        foreach (var cell in formattedCells)
        {
          result.Add(cell);
        }

        return result;
      }
    }
}