"use client"
import React, { useState, useEffect } from "react";
import { Progress, Badge } from 'flowbite-react';
import { useAuth } from "@/app/contexts/AuthContext";
import Loader from "@/components/Loader";

import { AgGridReact } from 'ag-grid-react'; // AG Grid Component
import "ag-grid-community/styles/ag-grid.css"; // Mandatory CSS required by the grid
import "ag-grid-community/styles/ag-theme-quartz.css"; // Optional Theme applied to the grid
import { AdminTextStyles } from "@/styles/TextStyles";


export default function Accountability() {
  const { token } = useAuth();
  const [accountability, setAccountability] = useState<any[]>([]);
  const [isLoading, setIsLoading] = useState<boolean>(true);

  const ProgressBar = ({ currentPoints, required }: { currentPoints: number; required: number }) => {
    return (
      <Progress
        progress={currentPoints / required * 100}
        color={currentPoints >= required ? "green" : "purple"} // Change color to green if points are 100 or more
      />
    )
  }

  const BucketMetBadge = ({ bucketMet }: { bucketMet: boolean }) => (
    <div className="flex flex-row items-center mt-3">
      <Badge color={bucketMet ? 'success' : 'failure'}>{bucketMet ? 'YES' : 'NO'}</Badge>
    </div>
  )


  const [colDefs, setColDefs] = useState([
    { field: "name", flex: 2, filter: true },
    { field: "currentPoints", flex: 1, },
    {
      field: "progress", flex: 4, cellRenderer: (params: any) => (
        <div className="mt-4">
          <ProgressBar currentPoints={params.data.currentPoints} required={params.data.required} />
        </div>
      )
    },
    { field: "Chapter Bucket", flex: 1, cellRenderer: (params: any) => <BucketMetBadge bucketMet={params.data.chapterBucketMet} /> },
    { field: "Rush Bucket", flex: 1, cellRenderer: (params: any) => <BucketMetBadge bucketMet={params.data.rushBucketMet} /> },
    { field: "Fundraising Bucket", flex: 1, cellRenderer: (params: any) => <BucketMetBadge bucketMet={params.data.fundraisingBucketMet} /> },
    { field: "Events Bucket", flex: 1, cellRenderer: (params: any) => <BucketMetBadge bucketMet={params.data.eventsBucketMet} /> },
    { field: "Teams Bucket", flex: 1, cellRenderer: (params: any) => <BucketMetBadge bucketMet={params.data.teamsBucketMet} /> },
    { field: "Variable Bucket", flex: 1, cellRenderer: (params: any) => <BucketMetBadge bucketMet={params.data.variableBucketMet} /> },
    // { field: "electric" }
  ]);


  const fetchData = () => {
    setIsLoading(true);
    fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/accountability?page=0&page_size=100`, {
      headers: {
        Authorization: `Bearer ${token}`,
      },
    })
      .then((response) => response.json())
      .then((data) => {
        setAccountability(data);
        setIsLoading(false);
      })
      .catch((error) => console.error("Error fetching data:", error));
  };

  useEffect(() => {
    fetchData();
  }, []);

  if (isLoading) {
    return <Loader />
  }

  return (
    <div className="overflow-x-auto">
      <h1 className={AdminTextStyles.title}>Accountability Tracker</h1>
      <div
        className="ag-theme-quartz" // applying the grid theme
        style={{ height: "80vh" }} // the grid will fill the size of the parent container
      >
        {accountability && accountability.length > 0 ? (
          <AgGridReact
            rowData={accountability}
            columnDefs={colDefs}
            pagination={true}
            paginationPageSize={20}
            paginationPageSizeSelector={[10, 20, 25, 50, 100]}
          />
        ) : (
          <div className="flex flex-col items-center justify-center h-full">
            <div className="text-center">
              <svg
                xmlns="http://www.w3.org/2000/svg"
                viewBox="0 0 1024 768"
                className="mx-auto h-56 w-auto text-black sm:h-64"
              >
                <g fill="none" fill-rule="evenodd">
                  <g fill="#FF5678">
                    <path
                      d="M555.886 337.572c-1.134-1.652-3.154-2.481-4.757-3.572.718 1.646.78 2.144.052 3.714-.562 1.211-.936 2.501-1.477 3.724-.371.837-.964 1.683-1.2 2.541-.277 1.006-.133 2.027-.504 3.021 4.379-1.183 8.27-3.952 12-6.523-.695-.1-2.532-.07-3.118-.578-.48-.415-.578-1.717-.996-2.327M661.291 322.8c1.91 1.52 1.91 1.52 1.976 4.105-1.607-.978-2.14-2.177-1.976-4.106m-92.56 6.786c-.882.072-1.096-2.48-1.291-3.227l.87-.122c.398.428 1.967 3.221.422 3.349m132.775 26.117c1.25-6.927 2.775-13.812 4.804-20.555a160.721 160.721 0 0 1 3.312-9.807c1.044-2.799 3.03-5.836 3.377-8.771-3.16 2.855-5.734 6.43-8.459 9.699-2.604 3.128-5.337 6.196-7.796 9.439-1.001 1.32-1.905 2.837-1.969 4.533-.068 1.86 1 2.997 1.764 4.563.802 1.64 1.342 3.657 1.737 5.429.434 1.937-.248 2.883-.912 4.638-1.422-.7-2.647-1.09-3.685-2.358-1.103-1.347-2.035-2.86-3.077-4.256-2.271-3.04-4.567-6.062-6.826-9.112-4.53-6.12-8.498-12.295-11.896-19.104-3.44-6.893-6.522-13.972-9.876-20.91-.707-1.46-.997-3.061-2.88-3.058-1.393.002-2.575.867-3.884 1.208a4.424 4.424 0 0 1-.99.172c.01.333-.044.655-.164.965-.344.418-.468.25-.947.225-.555-.03-.955-.378-1.56-.337-.841.057-1.658.612-2.449.863v-1.92c-2.511 1.289-8.591 5.604-9.913.713-.115.89.125 2.495-.75 3.066-.578.378-1.31-.016-1.937.064-.782.101-1.48.665-2.206.935-.049-1.316-1.054-1.168-2.03-1.345-1.262-.23-2.367-.93-3.669-.706-1.294.224-2.232.804-3.534.107-.874-.47-1.689-1.126-2.57-1.621-.894-.503-1.052-.852-1.436-1.744-.424-.989-.347-.765-1.356-.485-2.16.6-3-2.004-3.38-3.77-.928-4.318.136-9.034 2.984-12.403 3.664-4.335 7.545-8.587 12.334-11.719 4.426-2.893 9.13-6.472 14.295-7.874-4.124-9.23-10.002-17.827-15.24-26.466-1.5.344-2.4 1.158-4.002.903-1.264-.2-1.207-.103-1.675 1.212-.419 1.175-.648 1.344-1.77 1.704-1.164.374-.238-.352-1.105.565-.387.407-.598.851-1.079 1.187-1.76 1.233-4.063.523-5.974 1.292-1.32.531-.67 1.207-1.346 2.12-.716.967-1.93.463-2.922.307-2.067-.325-2.69.372-3.126 2.317-.324 1.441-.539 2.19-2.086 2.493-.876.172-2.239-.527-2.8.038-.216.216-.125.823-.225 1.102-.18.507-.498.922-.86 1.317-.735.8-1.739 1.46-2.738 1.878-.95.398-1.952.175-2.853.523-1.42.545-1.075 2.054-1.027 3.25-.805-.194-.992-.17-1.473-.776-.22-.278-.1-.744-.533-.862-.374-.101-2.39 1.14-2.676 1.495-.836 1.036.011 2.5-.204 3.707-.211 1.2-1.188 2.105-1.96 2.989-.251-.73-.504-1.458-.756-2.187-.376.488-.9.836-1.061 1.448-.085.317.174.744.03 1.066-.407.904-.813.332-1.572.363-.667.027-2.462.537-2.146 1.581.215.703 1.603.54 1.437 1.496-.2 1.162-1.707.401-2.318.644-.8.315-.483.997-.756 1.69-.48 1.217-1.448 1.326-2.452 1.922-.992.59-1.377 1.305-1.832 2.317-.567 1.264-.508.923-1.73.812-.752-.069-1.146.084-1.444.779-.244.566.172 1.264-.04 1.785-.263.643-1.624 1.372-2.095 2.025-1.277 1.775-1.827 3.966-2.87 5.87a20.488 20.488 0 0 1-1.571 2.428c-.775 1.032-1.257.836-2.302 1.39-1.693.897-1.199 3.267-1.385 4.832-.097.808-.297 1.553-.873 2.163-.344.364-1.563.78-1.322 1.506.29.879 1.265-.06 1.08 1.31-.146 1.091-1.39 1.987-1.666 3.08-.191.752.06 1.468-.135 2.227-.213.827-.861 1.455-1.172 2.234-.314.783.18 1.77-.089 2.365-.313.692-1.46.514-1.768 1.413-.337.982.451 2.527.593 3.53.09.642.43 1.34-.113 1.869-.462.45-1.556.368-1.084 1.483.218.516 1.038.408 1.127 1.187.069.6-.624 1.51-.998 1.915-1 1.08-2.892 1.98-1.53 3.355.235.237.524.125.745.447.26.374.295 1.045.442 1.473.354 1.027.86 1.993 1.489 2.88 2.396 3.376 6.363 4.966 9.002 8.031 3.22-3.204 6.514-6.33 9.545-9.714 2.57-2.871 5.058-5.843 8-8.354 1.52-1.296 3.258-2.342 4.521-3.913 1.207-1.5 1.651-3.387 2.57-5.054 1.804-3.271 4.403-6.019 6.337-9.209 3.791 2.471 2.207 8.78 1.115 12.319-.667 2.164-1.611 4.237-2.72 6.212-.938 1.672-2.258 3.333-2.708 5.23-.368 1.539.17 2.833 1.061 4.085 1.278 1.793 2.773 3.472 4.12 5.22l9.828 12.762c1.187-1.713 3.145-.592 4.721-.262 2.31.483 4.695.31 7.033.532 2.452.231 4.933.639 7.363 1.033 2.176.352.73 1.36 1.783 2.561 2.225-4.012 6.48-1.475 9.955-1.121 2.18.223 4.495.106 6.634.379 2.16.275 4.293 1.012 6.529 1.227 4.215.404 8.026 1.858 12.163 2.613 2.186.4 4.39.681 6.574 1.083.775.142 2.143.214 2.577 1.012.513.948-.53 2.948-.813 3.801-.423 1.274-.72 3.226-1.576 4.287-.663.82-.833.32-1.687.285-.988-.042-.586.17-1.269.934-.842.945-2.025.951-3.197 1.057-.534.048-1.601-.049-2.015.323-.583.522-.079.557-.124 1.357-.077 1.4-2.011 2.44-3.14 2.99-.725.352-1.503.64-2.32.582-.514-.037-1.125-.532-1.641-.326-.732.29-.676 1.672-1.1 2.256-.783 1.081-1.836.279-2.865.826-1.232.654-1.931 2.208-3.102 2.978-2.32 1.526-4.99 1.694-6.145-1.168-.463.958-1.553 2.444-1.144 3.409.363.855 1.51 1.487 2.283 1.907 2.265 1.23 4.485 2.457 6.657 3.858 4.991 3.22 9.79 6.746 15.127 9.4 5.253 2.612 10.774 5.427 16.572 6.57 5.954 1.17 12.04.43 17.572-2.036-.825-1.648-.951-2.71-1.002-4.59a394.968 394.968 0 0 1-.108-5.333c-.043-3.55-.024-7.1.1-10.647a148.045 148.045 0 0 1 2.29-21.15M764.413 350.616c-.536-1.91-1.145-3.9-1.265-5.884-.087-1.432.444-3.56-.41-4.807-.36-.53-1.01-.715-1.22-1.352-.168-.512.09-1.537.021-2.1-.177-1.465-.758-2.832-1.295-4.195-1.217-3.092-1.252-6.337-2.27-9.47-.555-1.717-1.6-3.183-2.21-4.872-.558-1.537-.84-3.16-1.016-4.78-.188-1.727-.185-3.44-.6-5.137-.388-1.584-.87-3.134-1.22-4.728-.834-3.793-1.801-7.507-3.13-11.163-.502-1.381-.867-3.697-2.092-4.625-.89-.674-2.06-.5-3.095-.43-1.177.08-2.143.135-2.926 1.117-1 1.255-1.638 2.62-2.796 3.798-1.18 1.2-2.654 2.235-3.716 3.537-.834 1.023-.356 1.371.54 2.244-1.16.15-2.114-.17-2.958.709-.86.893-1.512 2.376-2.058 3.473-1.23 2.471-1.37 4.694-1.736 7.37l-.98-1.526c-.414 1.126-1.015 2.296-2.208 2.759-1.461.569-1.417.384-1.421 2.022-.008 2.354-1.959 3.835-3.192 5.616-.692 1-1.36 2.138-1.813 3.262-.582 1.438-.213 2.382-.074 3.854.21 2.202-1.914 3.888-3.192 5.45-.4.486-.624.725-.848 1.32-.165.441.064.72-.268 1.19-.6.852-1.856.624-2.338 1.67-.453.978.175 1.964.273 2.932.158 1.546-.774 2.972-1.281 4.382-.553 1.532-.978 3.116-1.182 4.732a17.337 17.337 0 0 0-.131 2.81c.016.434.35 1.524.065 1.918-.368.51-1.174-.074-1.51.492-.691 1.162 1.887 1.533 2.39 1.713-.597.378-1.384.642-1.87 1.176-.642.707-.245 1.127-.51 1.92-.29.864-1.33 1.099-1.654 1.863-.333.784.316.952.654 1.603.826 1.587.084 3.964-.243 5.593-.131.654-.07 2.355-.48 2.861-.26.323-.845.204-1.073.488-.48.594.025.833.22 1.516.249.863.207 1.78-.024 2.643-.219.816-.75 1.569-.899 2.4-.162.911.29 1.22.602 2.067.693 1.886.206 4.219-.03 6.14-.276 2.236.508 4.348-.076 6.418L704 389.8c.198.986.395 1.971.591 2.957l.292 1.457c-.122.631.09.881.635.75 1.07.308 2.719-1.376 3.602-1.874 2.65-1.496 5.51-2.57 8.261-3.864 2.86-1.343 5.252-3.157 7.871-4.87 2.698-1.763 5.544-3.274 8.174-5.14 11.377-8.074 23.711-14.884 35.574-22.229-3.323.376-3.927-4.027-4.587-6.37M778.485 258.764c1.78.518 1.372 2.026.696 3.336-.684-.647-1.892-1.406-2.213-2.33-.397-1.148.493-1.304 1.517-1.006m49.769 44.611c-.095.458-.199 2.327-.55 2.617-.319.262-1.958-.107-2.392-.13-1.375-.072-2.425.207-3.727.528-2.238.554-2.642-.838-2.652-2.922-1.72 3.396-4.586 5.096-8.247 5.75-1.636.294-3.39.644-5.049.715-1.238.053-2.633-.056-3.858-.269-.777-.135-1.484-.613-2.217-.75-1.604-.3-.16.856-1.318 1.163-.297.078-1.42-.407-1.86-.433-.71-.043-1.423.075-2.132.038-1.608-.083-2.133-.978-3.407-1.668-.668-.362-1.124-.31-1.84-.209-.786.11-.189.372-1.107.21-.296-.053-.832-.385-1.074-.494-1.74-.781-3.1-1.476-4.44-2.83-.982-.99-2.22-1.657-3.2-2.64-.352-.353-.544-.786-.856-1.153-.405-.474-.433-.355-.94-.708-.58-.403-1.03-.63-1.383-1.268-.604-1.088-.58-2.592-.675-3.8-.322-4.115.416-7.781 2.371-11.41 2.122-3.94 3.883-8.106 6.368-11.841 2.248-3.379 5.068-6.55 7.836-9.516.782-.838 1.634-1.815 2.582-2.47.617-.423 1.42-.613 2.002-1.077 1.382-1.105.71-3.131.71-4.667.436.772 1.12 1.927 2.165 1.673.687-.167.71-.94 1.207-1.243.383-.233.834-.116 1.259-.16-1.672-8.202-3.824-16.479-6.516-24.411-7.357 7.22-14.72 14.491-21.863 21.926-3.685 3.836-7.088 7.929-10.756 11.78-3.856 4.047-7.796 8.012-11.695 12.017 1.06 1.025 2.274 1.72 2.997 3.042.666 1.214 1.43 2.134 2.345 3.174 2.07 2.349 3.059 5.847 4.338 8.675.253.556.321.723.254 1.226-.04.307-.357.578-.363.952-.008.474.217.79.5 1.15.263.337 1.04.72 1.105 1.16.077.525-.462.876-.641 1.367-.791 2.161 1.806 3.69 2.683 5.386.483.93.706 1.872.086 2.803-.507.761-1.626 1.056-1.032 2.215.37.726 1.405.328 1.252 1.337-.083.54-1.127 1.047-.15 1.64.79.48 1.73-.377 1.361 1.107-.164.663-1.052 1.33-.799 2.073.248.722 1.54.84 2.055 1.405.647.714.94 1.907 1.068 2.835.438 3.164 1.166 6.035 1.892 9.143.486 2.082-.766 2.119-2.616 2.133 1.247 1.748 2.474 3.502 3.805 5.188 1.303 1.65 1.64 4.21 2.25 6.206.734 2.399 1.45 4.805 2.163 7.207.325 1.096.55 2.166.115 3.267-.5 1.268-.492 1.273-.261 2.586 5.77-3.308 11.216-6.987 16.69-10.767 2.624-1.812 5.253-3.619 7.848-5.473 2.043-1.46 4.056-3.043 5.37-5.221 3.05-5.058 7.537-9.024 11.616-13.29 2.074-2.17 4.006-4.865 6.541-6.513 2.452-1.595 5.054-3.006 7.528-4.591l-2.746-3.77"
                    />
                    <path
                      d="M645.674 371.323c2.35-1.12 4.518-2.626 6.804-3.872 1.982-1.082 3.785-1.604 5.489-3.102.713-.628 1.382-.864 2.219-1.278 1.062-.526 1.87-1.459 2.814-2.157-2.408-.937-4.769-2.387-7.43-2.419-1.18-.013-2.268.407-3.45.188-1.2-.222-2.136-.838-3.092-1.547-1.732-1.285-2.5-1.095-4.378-.032-.448.254-1.06.821-1.606.805-.883-.027-.75-.57-1.132-1.176-.348-.55.095-.613-.542-.63-.385-.01-.558.52-.986.603-.813.158-2.347-.636-3.16-.784-1.369-.247-2.42.049-3.774.134-1.405.09-2.773-.236-4.09-.685-1.202-.409-2.357-.985-3.572-1.352-1.582-.477-1.903.142-3.366.48-2.225.51-4.384-2.693-6.422-1 2.989 4.354 7.08 8.029 10.858 11.73 2.033 1.988 4.09 3.936 6.321 5.706 1.005.8 2.032 1.573 3.055 2.35.464.355.93.707 1.393 1.065.83.641.754.833 1.542.464 1.06-.496 1.88-1.546 3.042-2.05 1.147-.497 2.332-.903 3.463-1.441M647.158 275.908c.085 1.959 1.301 3.564-.416 5.222-.05-.972-.101-1.946-.153-2.918-.053-1.013-.563-2.114.569-2.304m-1.915-8.537c-.29.851-.66 1.466-1.471 1.9-1.16.62-2.767-.384-3.575.498-.812.888-.717 2.974-1.777 3.7-.34.232-.715.07-1.118.373-.347.26-.361.878-.714 1.15-.579.446-1.072-.01-1.618.25-.613.292-.713.99-.876 1.601-.16.6-.156 1.375-.596 1.844-.537.571-1.779.976-2.433 1.382-2.356 1.465-4.71 2.932-7.065 4.399.701.71 2.25 1.518 2.43 2.565.122.714-.373.793.22 1.355.269.256 1.22.528 1.544.69 1.334.67 2.688 1.26 4.15 1.577 1.737.378 3.477.2 5.208.614 1.818.434 3.609.77 5.486.727 1.603-.036 3.116-.482 4.703-.62 2.975-.256 6.833.012 9.259-2.085-.241-2.154-.48-3.791-1.28-5.842a438.06 438.06 0 0 0-2.246-5.633c-1.539-3.78-3.105-7.426-5.06-10.998-.4 2.503-1.355-.793-2.298-.818-.694-.018-.7.87-.873 1.37M810 294.557c1.31.325 2.362.49 3.589 1.055.908.417 1.254.638 2.045-.075.43-.386.51-.962 1.062-1.25.545-.286 1.287-.193 1.87-.222 1.478-.079 2.956-.11 4.434-.165-1.7-1.632-3.399-3.266-5.099-4.9-.264.453-.972 2.123-1.492 2.23-.156.032-.642-.49-.904-.51-.28-.023-.481.104-.726.212-.628.279-.909.924-1.61 1.096-.689.17-1.47-.15-2.167-.166-.191.658-.682 1.846-1.002 2.695M807.92 273.36c.296-1.223-.34-2.485-.146-3.728l-.422-.632c-.328.733-.538 1.455-.984 2.135-.37.567-.952 1.043-1.203 1.69-.456 1.171.117 2.136.69 3.175.768-.858 1.782-1.47 2.065-2.64M797.813 276c-1.032.024-.639.505-.779 1.275-.089.486-.328.919-.593 1.328-.53.822-1.803 1.65-1.076 2.762.618.945.652-.235 1.167.23.54.484-3.026 4.123-3.56 4.322-.29.107-.607-.222-.881.01-.247.21.088 2.305.077 2.732-.027 1.121-.101 2.328.81 3.13 1.06.93 2.805.965 4.06 1.494.86.362 1.664.88 2.57 1.124.955.256 1.778-.276 2.6-.153.62.092 1.828 1.097 2.454 1.396.84.402 1.003.464 1.822.152.451-.172 1.332-.505 1.638-.896.461-.593.108-.584.17-1.391.11-1.472 1.65-1.301 2.708-1.717-2.093-.627-3.25-2.875-4.28-4.71-.454-.81-1.377-1.777-1.623-2.683-.306-1.136.54-1.869 1.05-2.817-.694.282-3.414 1.532-2.104.055.407-.459 2.932-1.208 1.904-2.404-.62-.719-1.423.75-2.139.607-.83-.166-1.814-1.556-2.507-2.015-.836-.554-2.47-1.853-3.488-1.83M546.76 318.047c.05-3.504.62-6.99 1.636-10.34 2.088-6.883 6.514-12.915 7.938-20.014.685-3.417.464-6.882 1.122-10.305.692-3.608.664-7.303.794-10.963.13-3.605.261-7.213.215-10.822-.015-1.295-.38-3.094-.051-4.346.275-1.05.695-.657 1.586-1.036-.62-.38-1.89-.57-2.025-1.363-.09-.544.44-1.14.413-1.74-.02-.492-.258-.988-.4-1.448-.782-2.525-1.893-6.411-.454-8.837-.866-.146-1.216-.415-2.017-.049-.866.395-1.726 1.272-2.476 1.868-1.491 1.184-2.974 2.385-4.313 3.743-2.522 2.56-4.593 5.508-6.79 8.34-2.19 2.821-4.456 5.615-6.708 8.386-1.752 2.154-4.53 5.433-6.97 2.532-.732.974-1.198 1.954-2.396 2.426-1.463.577-2.089 1.175-3.144 2.413-.212.25-.845 1.312-1.119 1.412-.555.202-.237-.175-.672-.456-.684-.44-1.009-.74-1.253.479-.11.54.18 1.05-.083 1.587-.245.499-1.008.995-1.47 1.267-1.074.631-2.294.57-3.445.957-1.9.637-3.113 3.52-5.199 2.14-.757-.503-1.357-1.285-2.346-.752-.736.398-.862 1.381-1.923.992-.557-.203-1.012-.933-1.658-1.17-.456-.166-.945-.233-1.424-.29-1.061-.128-2.139-.083-3.167-.413-2.238-.718-3.776-2.703-5.3-4.376-1.137-1.249-.786-2.03-.47-3.617.27-1.35.55-2.697.882-4.032.658-2.65 1.508-5.267 2.782-7.69 1.48-2.82 3.642-5.184 5.617-7.656 4.111-5.14 7.99-10.56 12.45-15.403.838-.91 2.273-1.897 2.12-3.283-.168-1.522-1.895-.782-2.955-.518-2.844.708-5.46 2.084-8.172 3.15-1.5.59-3.509 1.74-5.116.984-1.516-.713-3.917-2.626-3.069-4.378.65-1.345 1.355-2.32 1.7-3.84.309-1.356.498-2.741.782-4.104.67-3.235 1.529-6.428 2.492-9.588 1.837-6.03 4.08-11.926 6.102-17.894-8.295.747-15.854 3.16-23.65 5.947.37.76 1.062 1.847.783 2.736-.343 1.094-1.82.727-2.185 1.48-.433.889.88 1.4.49 2.54-.327.958-1.266 2.065-1.81 2.926 2.55 1.645 1.559-1.773 3.349-2.006.496 2.72-.648 4-1.707 6.288-.942 2.039 1.223 1.844 1.668 3.369-1.246.207-2.1-.366-2.644 1.104-.233.625.117 2.913-.629 2.977-.55.047-.376-.955-.82-1.049-.553-.116-1.97 1.879-2.317 2.252 1.509 0 2.333.16 3.762.733a96.384 96.384 0 0 1-2.823 2.66c.88.364 2.377.446 2.829 1.427.418.912-.217 2.56-.306 3.506.376-.566.925-1.904 1.508-2.268.767-.48.988-.005.94 1.045-.088 1.93-2.606 1.806-2.958 3.345-.35 1.522 1.83.463 1.886 1.975.032.862-.974 1.6-.476 2.512-1.115-.625-1.252-.994-1.857-2.213-.425 1.543-1.345 3.986-.84 5.588.413 1.308 1.276.985.287 2.252-.367.47-.874.838-.887 1.478-.029 1.477 2.083 2.622.443 3.937-.637.51-1.325.567-1.649 1.454-.28.767.007 1.79-.221 2.625-.452 1.658-1.655 2.89-2.858 4.048-1.074 1.036-2.054 2.236-3.229 3.136-.637.49-.872.776-1.304 1.413-.183.272.042.527-.205.78-.29.293-1.026.263-1.415.338-1.282.25-2.84.767-3.267 2.193-1.564-1.184-3.157.255-4.673.724-.756.233-1.448.29-2.162-.107-1.14-.635-1.086-.729-2.451-.53-3.804.548-6.94-3.415-9.059-6.08-1.54-1.941-2.39-4.355-2.791-6.779-.474-2.846-.966-5.672-1.227-8.55-.546-6.032-.34-11.97-.124-18.013-7.03 3.707-13.616 8.536-19.913 13.364-6.278 4.814-11.65 10.838-17.294 16.39 1.435.092 2.82.6 4.26.63.709.014 1.405-.071 2.056-.367.358-.164.685-.373.982-.63.26-.523.624-.613 1.09-.272 1.414-.116 2.72-.409 4.138-.613 1.794-.259 3.702.282 5.464.587 3.295.57 6.79.191 9.973 1.214 2.784.896 5.685 2.177 8.366 3.336 2.58 1.117 4.846 3.558 7.032 5.304 1.26 1.005 2.65 1.435 3.978 2.258 1.284.796 2.259 2.53 3.166 3.704 2.191 2.834 4.637 5.452 6.831 8.271.98 1.26 1.866 2.627 2.422 4.132.286.768.31 1.63.585 2.38.245.67.709 1.057 1.116 1.626 1.345-3.62 2.647-7.254 4.038-10.855.535-1.384 2.926-8.03 4.796-5.48 1.61 2.193.698 6.697.195 9.145-.673 3.28-1.42 6.548-2.154 9.816-1.458 6.484-2.794 13.028-4.648 19.41-.12.414-.36.914-.39 1.343-.053.743-.31.321.199.84.26.264.879.185 1.125.501.45.577.296 2.585.263 3.287-.062 1.333-.807 1.918-1.156 3.105-.303 1.037.214 2.103.224 3.149.025 2.59-1.32 5.409-1.788 7.975-.298 1.627.64 4.205-1.645 4.517.568.958-.104 1.366-.604 2.072-.62.878-.66 1.763-1.053 2.775-.347.892-.884 1.506-.98 2.392-.086.805.272 1.576-.005 2.393-.325.964-.94 1.075-1.752 1.495-.785.406-.555-.068-.828.723-.12.348.04 1.15-.035 1.553-.355 1.897-1.866 3.166-3.127 4.494-.471.498-1.083 1.653-1.661 1.96-.846.448-2.113-.331-2.889.084-.712.38-1.457 3.42-1.72 4.159-.556 1.57-1.084 3.154-1.614 4.733-1.944 5.787-3.856 11.585-5.778 17.379.919-.023 2.453-1.035 3.343-.62 1.02.474.363 1.874.746 2.636.274.546 1.87.463 2.62.504 1.247.068 1.815-.291 2.937-.617 1.94-.563 4.377-.366 6.396-.585 4.47-.486 9.069-.622 13.464-1.603 4.209-.938 8.242-2.498 12.373-3.711-1.928-1.014-2.287-3.514-2.84-5.416-.305-1.053-.419-1.886-1.011-2.841-.686-1.107-1.279-2.23-1.713-3.464-1.686-4.79-1.763-10.729-1.193-15.722 1.166-10.203 4.8-19.756 10.402-28.318 2.68-4.094 5.777-7.87 9.206-11.354 3.5-3.557 7.094-7.45 11.202-10.315 1.86-1.3 4.033-1.756 5.66-3.303 1.25-1.186 3.25-1.449 4.782-2.287 2.138-1.171 3.956-2.092 6.361-2.585 1.771-.364 3.76-1.267 5.605-1.037 1.262.157.79.747 1.532 1.328.98.768.625-.315 1.35-.586.613-.23.999.355 1.51.54.232.084 1.27.069 1.216.036.562.34.468 1.542.788 2.11.345.613.855 1.028 1.07 1.723.288.929.092 1.633-.3 2.423-.43.872-1.08 1.3-.04 2.147-2.597 1.709-.993 5.142-2.948 7.212-.664.703-2.158 1.685-2.136 2.778.025 1.178 2.14 1.644 1.052 2.82-.991 1.07-3.127.564-4.274 1.516-1.155.96-.771 2.612-.875 3.933-.08 1.011-1.064 2.231-.655 3.207.268.64.734.545.754 1.244.018.566-.452 1.319-.606 1.866l-1.299-.567c-.206.762-.56 1.5-.655 2.29-.113.926.265 1.373.459 2.199.194.829-.253 2.268-.335 3.258-.095 1.135-.207 2.325-1.013 3.205-.478.522-1.134.603-1.649 1.007-.48.376-.626.881-.954 1.38.745-.22 1.49-.44 2.237-.658-.302 1.424-.838 2.001-1.788 3.12l-.469-2.466c-.27.863-.76 1.55-.712 2.48.04.795.397 1.558.4 2.378.007 1.598-1.172 2.645-.689 4.182.583 1.858.704 2.834.127 4.737-.505 1.667-1.212 3.327-1.336 5.083-.138 1.981.755 3.687.57 5.7-.085.933-.474 1.591.564 1.694.927.093 1.933-.673 2.734-1.055 3.956-1.884 7.898-3.798 11.844-5.703-.735-1.024-.555-1.056-.095-2.316l1.034-2.849c.73-2.018 1.465-4.034 2.22-6.044 1.402-3.74 3.28-7.437 3.963-11.398.618-3.576.299-7.141.351-10.741M305.303 605.62l-3.917-2.47 2.283-1.743c.825-.629 1.197-1.215 1.684.022.512 1.3-.1 2.843-.05 4.192m-1.66-9.49c1.838 2.149-1.596 2.77-2.92 3.04.214-1.034.979-5.31 2.92-3.04m40.276-7.558c-.595-1.826-1.561-6.872-4.318-6.56-.66.076-1.255.734-2.037.79-.668.047-1.324-.131-1.985-.187-1.449-.122-3.007-.04-4.393.422-1.613.538-2.803 1.303-4.558 1.416-1.253.082-2.57-.023-3.793.332-2.096.606-1.954 2.465-1.66 4.31-.355-.403-.758-.71-1.015-1.183-.29-.534-.291-1.134-.743-1.6-.894-.927-1.923-.488-3.028-.408-1.25.09-2.853-.746-4.053-.277-.533.208-.848.805-1.403 1.053-.493.22-1.036.259-1.563.321-1.18.138-2.388.531-3.456.51-1.515-.032-2.794-.13-4.297.277-1.402.378-2.742.954-4.142 1.338-1.555.426-3.162.468-4.74.777-3.147.619-6.247 2.031-9.464 2.213-1.685.095-3.312.093-4.99.359-.518.081-1.686.055-2.108.345-.47.323-.792 1.665-1.059 2.216-2.884 5.954-4.976 12.29-7.278 18.484-2.393 6.442-4.774 12.89-7.181 19.327-1.132 3.026-2.577 6.045-3.515 9.132-.867 2.856 2.504 4.866 4.296 6.663 4.388 4.398 8.825 8.745 13.808 12.478 5.017 3.76 10.69 6.289 16.4 8.809 5.86 2.586 12.015 4.063 18.345 4.956 3.252.459 6.524.772 9.797 1.035 3.28.265 6.062-.189 9.297-.604-.32-.747-1.362-2.056-1.24-2.886.036-.233.514-.621.627-.895.154-.374.139-.77.148-1.164.045-1.883-.692-3.709 1.722-4.38l.241 3.084c1.167-1.755.43-3.639-.277-5.407l-2.46 2.052c.169-.5 1.542-3.668 1.161-3.967-.378-.296-1.061.495-1.468.113-.33-.311.038-1.047-.066-1.444-.167-.639-.86-1.13-.853-1.83.011-.931 1.206-1.651 1.77-2.251.828-.879 1.166-1.508 1.162-2.71-.005-1.66.773-1.568 1.984-2.676 1.959-1.79 2.08-4.255 1.07-6.613l2.6-.2c-.803-2.022-1.63-4.04-2.417-6.068-.779-2.005.284-3.057.81-4.927 1.188 1.178 2.378 2.355 3.568 3.533.337-3.807 1.346-7.647.869-11.485-.148-1.19-.37-1.119.176-2.008.393-.638.948-1.135 1.293-1.829.677-1.364.88-2.888 1.017-4.385.127-1.391.066-2.92.46-4.27.298-1.018 1.274-1.59 1.509-2.72.407-1.961-2.722-5.383-.902-6.836.928-.74 2.082-.032 2.752-1.242.575-1.038.798-2.47 1.134-3.595.537-1.792.79-3.185.142-4.987-.57-1.586-1.173-3.147-1.696-4.75M372.653 496.758c-.458-1.047-1.173-1.235-2.256-1.444-4.66-.896-9.361-.901-14.064-1.333-1.336-.122-2.667-.311-3.971-.636-1.092-.274-2.123-.882-3.195-1.138-2.136-.51-4.255.904-6.39.801-1.093-.053-2.016-.704-3.057-.977-.853-.224-1.629-.049-2.499.007-1.685.108-3.386.022-5.062-.152a38.422 38.422 0 0 1-6.129-1.124c-1.326-.361-3.335-.747-4.317-1.762-2.238 3.433-4.12 7.027-5.74 10.783-1.578 3.653-3.674 7.108-5.598 10.59-3.918 7.099-6.724 14.72-9.375 22.353 1.461-.67 2.52-1.355 3.618.162 1.035 1.433 2.263.964 3.938 1.235 2.688.434 2.979 3.409 3.568 5.631.868-1.082 1.738-2.164 2.608-3.245.697.776 1.818 1.18 2.387 2.073.364.572.159 1.102.375 1.647.2.507.712.712 1.097 1.07 2.662-3.402 5.161 3 5.482 5.19l1-2.545c.473.513 1.118.951 1.488 1.55.231.373.22 1.04.466 1.35.367.462.608.258 1.135.386.849.205.685.157 1.176.811 1.482 1.976 3.69 3.497 4.94 5.624 1.568 2.673 3.706 5.428 4.649 8.383.54 1.693.776 3.281 1.658 4.856.463.83 1.265 1.54 1.683 2.349.521 1.007-.397 1.422-.293 2.323.102.907 1.608 1.79 2.356 2.071 1.176.443 2.57.304 3.81.34 1.597.046 3.49-.012 5.04-.472 1.351-.402 1.247-1.096 1.617-2.286.439-1.412 1.208-2.765 1.82-4.108.91-2 1.351-3.332-1.374-3.931 1.565-.262 4.121-.722 3.623-2.819-.239-1.006-1.146-1.683-1.007-2.794.14-1.132 1.015-1.514 1.851-2.102 1.58-1.113 2.718-3.191 3.291-4.991.319-1.001.406-1.781.07-2.785-.531-1.583-.81-2.877-.943-4.566l2.163.23c-.158-.64-.182-1.147-.473-1.735-.097-.19-.597-.34-.568-.61.049-.47.684-.46.96-.788.717-.844.925-2.818 1.222-3.842.282-.973.431-2.154.961-3.027.986-1.627 2.204.668 1.958-1.641-.124-1.17-.215-2.904-1.673-3.074-1.735-.202-2.235.482-2.072-1.599.113-1.472-.283-4.632 1.296-5.422.752-.377 1.991-.332 2.83-.402 1.005-.085 1.795.447 2.72.416 1.55-.053 1.46-2.263 1.639-3.386.786-4.996 1.182-10.08.695-15.126-.192-1.992-.316-4.498-1.134-6.369"
                    />
                    <path
                      d="M413.045 299.57c-.102-3.646-.102-3.646 1.921-5.028.536 2.32-.795 3.485-1.92 5.028m36.757 9.305c.436-.869 2.236-.615 2.998-.604-.356.735-1.593 2.525-1.97 3.89-.418-.79-1.455-2.439-1.028-3.286m-4.725 25.824c.79-.626.606-.595.983-1.351.292-.586.223-.893.849-1.283 1.268-.786 1.464-.782 1.79-2.46.284-1.462.167-3.547.808-4.9.46-.97 1.978-1.06 2.005-2.277.05-2.366-4.228-3.572-3.412-5.959.244-.71 1.015-.931 1.342-1.542.2-.373.063-.86.13-1.259.146-.89.392-1.085 1.214-1.319-.09.408-.096.77.045 1.042.54 1.045 2.691-.207 3.18-.928 1.011-1.489.077-3.96-.255-5.512-.452-2.11-.384-3.716-.216-5.832l2.752 3.02c.5-1.523.587-2.343 1.984-3.295 1.296-.883 1.799-1.468 2.63-2.778.316-.498.53-1.05 1.156-1.226.613-.17 1.622.49 2.136.235.32-.16.664-1.81.772-2.105.344-.935.687-1.872 1.029-2.807-1.962-4.954-3.565-10.207-6.497-14.713-2.686-4.128-6.575-7.465-10.505-10.396-1.96-1.463-3.944-2.952-6.004-4.277-2.067-1.332-4.3-1.812-6.505-2.806-2.21-.996-4.751-2.028-7.205-2.152-1.248-.063-2.451.229-3.701.064-1.383-.182-2.711-.65-4.084-.883-.003 7.18-2.617 14.464-5.062 21.144-1.112 3.04-2.707 5.845-3.875 8.862-1.177 3.035-2.333 6.063-3.612 9.058-2.635 6.17-5.04 12.607-8.437 18.41-.3.515-.643.882-.463 1.338.052.13.483.392.595.507.344.353.623.768.86 1.198.557 1.014.815 2.155 1.225 3.228.944 2.455 2.627 4.864 3.928 7.163 1.488 2.627 3.479 4.923 5.67 7 1.031.979 2.556 1.879 3.175 3.175.485 1.016.505 2.21.849 3.272l.36-2.548c.9 1.912 3.222 2.09 4.901 3.042 2.126 1.206 3.975 2.694 6.243 3.669 2.285.983 4.666 1.774 7.129 2.177 1.226.199 2.508.348 3.752.294 1.273-.056 2.187-.672 3.339-1.125-3.617-3.117-2.3-8.524-.237-12.118l1.904 4.19c.418-.994 1.849-1.931 1.212-3.098-.385-.706-.885-.524-.996-1.34-.086-.622.331-1.472.635-2.001.273.74.384 1.417.484 2.202M434.538 357.951c-.813-1.794-4.16-2.21-5.84-2.554-2.776-.57-5.435-2.21-7.93-3.516-2.39-1.251-4.411-2.419-6.346-4.283-.903-.869-1.652-1.055-2.766-1.554-1.07-.479-1.799-1.229-2.572-2.087-1.617-1.793-3.728-3.152-4.955-5.258-1.243-2.139-2.966-3.505-4.493-5.43-1.786-2.254-3.055-4.861-4.609-7.269-2.786 3.001-3.54 7.684-5.23 11.326-2.105 4.538-4.118 9.12-6.095 13.716-3.544 8.245-6.988 16.541-10.074 24.97-.113.31-.72 1.34-.616 1.673.103.325.888.681 1.136.946.652.692 1.062 1.54 1.586 2.331a31.203 31.203 0 0 0 3.242 4.059c2.674 2.854 5.788 5.528 9.193 7.465 1.48.843 3.588 2.512 5.282 2.367 2.179-.186 3.835-.333 6.025.161 2.135.481 3.996.9 6.196.841.858-.023 1.75-.137 2.57.2.588.243 1.006.914 1.67.944.833.037 1.333-.748 2.102-1.033.672-.248 1.392-.285 2.1-.25 1.036.05 3.452.985 4.28.184.897-.863.555-3.358 2.233-3.685 1.245-.244 3.767 2.034 4.785 2.672-.44-1.324-1.952-3.913-1.13-5.302.355-.603.85-.71 1.561-.566.828.168.898.76 1.556 1.129 3.57 2.005 3.253-7.531 3.134-8.991L427.81 381c0 .836.264 2.711-.298 3.052-.463.281-2.149 0-2.734 0 1.241 1.18 1.822 2.19-.05 3.198-1.276.687-2.856-.077-2.99-1.646-.143-1.643 2.422-1.894 1.95-3.517-.63-2.178-2.549 1.51-2.898 2.153-.512-1.89-.191-3.62 1.076-5.114 1.5-1.772 3.21-1.878 5.339-1.318.203-.895.662-1.512.908-2.328.294-.975-.154-.742-.583-1.588-.7-1.379.7-1.857.964-3.042.157-.707-.132-1.326-.477-1.917-.383-.658-1.127-1.112-.026-1.471.817-.267 1.782.516 2.553-.189.835-.764.528-2.36.982-3.31.682-1.437 1.442-.55 2.55-.147.833.303 1.083.274 1.183-.645.05-.46-.012-.918-.364-1.243-.575-.53-1.37-.152-2.019-.33-.881-.24-.924-.796-1.26-1.592 1.535-.136 1.558.353 2.95.941 1.106.468 2.32.399 3.434-.01-.773-.278-1.715-.348-2.406-.828-.878-.61-.678-1.324-1.056-2.158M530.243 305.993c1.519-2.807 2.893-5.681 4.26-8.564.83-1.751 2.622-4.172.494-5.615-1.435-.973-4.058-2.34-5.871-1.605-1.539.624-2.384 4.264-4.502 1.877-.386 1.101-1.7.921-2.487 1.517-.585.443-.532.892-.886 1.46-.367.589-.984.992-1.439 1.506a11.923 11.923 0 0 0-1.372-2.132c-.483.812-.37 1.306-.637 2.125-.153.473-.288.8-.782.983-1.133.424-2.246-.964-2.819-1.729.34 1.194.516 2.583-.3 3.624-.633.808-1.663.965-2.258 1.81-.667.945-.71 1.663-1.907 2.052-.23.075-1.033.28-1.28.275-.264-.005-.393-.46-.504-.457-.365.01-.986.46-1.32.615.745.938.453 1.33-.351 1.941-.975.741-.921.818-1.4 1.954-.228.54-.496 1.03-1.03 1.328-.536.3-1.25.044-1.696.355-.902.63-.8 2.26-1.512 3.057-.958 1.072-2.169 1.814-3.053 3.014-.894 1.21-1.608 2.54-2.292 3.875-1.326 2.59-3.099 5.005-3.278 7.95-.074 1.213.027 2.827-.53 3.951-.565 1.138-1.785 1.598-2.256 2.92-.398 1.12-.646 2.516-.787 3.693-.17 1.422.718 2.498.1 3.876-.59 1.313-.708 2.257-.822 3.708-.107 1.345-1.167 2.259-.516 3.408.633 1.115 1.283 1.933 1.462 3.277.194 1.457-.04 1.987-1.304 2.602.765 1.299 1.602 2.304 2.024 3.772.405 1.406.933 2.629 1.826 3.803 1.768 2.323 4.002 3.548 3.962 6.781l24.07-8.053c-3.095-.582-1.513-6.007-1.312-8.034.37-3.713.77-7.444 1.306-11.138.462-3.178 1.364-6.26 1.857-9.436.256-1.646.5-3.3.931-4.912.375-1.404 1.166-2.72 1.482-4.097.717-3.125 1.16-6.03 2.46-9.018 1.246-2.86 2.787-5.578 4.27-8.32M296.544 582.616c.63-.44 1.224-.355 1.96-.434 1.896-.206 3.537-.625 5.39-1.169 2.216-.65 4.575-1.026 6.851-1.42 2.181-.378 4.393-.288 6.524-.927 2.131-.64 3.839-1.24 6.08-1.327 2.454-.096 3.895-1.239 5.902.81 1.799-2.51 4.399-2.494 7.264-2.89 1.028-.143 2.224-.004 2.453-1.231.214-1.143-.695-2.656-1.191-3.583-2.635-4.922-4.86-10.126-8.367-14.514-3.435-4.302-7.885-9.069-12.74-11.756-2.631-1.456-5.669-2.258-8.457-3.354-3.024-1.188-5.983-2.537-8.967-3.821-1.552 1.662-1.68 3.672-2.456 5.779-.725 1.964-1.968 3.694-2.629 5.684-1.322 3.989-2.451 7.938-4.104 11.82-3.493 8.204-6.883 16.378-10.057 24.717 4.996-.776 9.864-1.264 14.91-1.579.988-.062.922-.306 1.634-.805M561 276.25c1.432.153 3.125.522 4.032 1.75 3.88-5.62 8.543-10.64 13.393-15.43 4.711-4.655 9.496-9.256 14.66-13.413 10.447-8.406 21.982-15.154 33.915-21.203-3.782-4.357-7.503-8.777-11.312-13.107-3.556-4.04-7.745-7.448-11.899-10.847-.334 3.372-.92 6.823-.89 10.214.03 3.417.498 6.818.544 10.234.04 2.776-.05 5.735-.764 8.43-.55 2.086-1.795 5.136-4.074 5.802-3.108.907-8.378-.632-10.703-2.816-2.003-1.882-3.062-4.687-4.653-6.888-1.09-1.505-2.233-2.148-4.067-1.55-2.745.895-5.389 2.524-7.966 3.814-2.914 1.459-5.818 2.938-8.734 4.395.2.143 1.387.752 1.437 1.022.016.084-.664 1.491-.685 1.767-.08 1.093.449 2.14.778 3.151.35 1.077.63 2.198.584 3.34-.049 1.214-.717 2.046-.957 3.158-.253 1.172.612 1.586 1.058 2.608.458 1.046.477 2.176.279 3.285-.162.903-.712 1.837-.705 2.749.005.49.423 1.037.275 1.542-.146.502-.741.786-1.086 1.12 1.573 1.034 1.866.883 1.243 2.571-.449 1.215-.288 2.414-.09 3.66.107.677.806 2.48.334 3.164-.431.627-1.85.137-2.503.114.24 1.263.582 1.131 1.175 2.178.387.682.349 1.567.309 2.327-.106 1.969-1.315 2.187-2.928 2.86M329.565 484.61c2.111.62 4.144 1.081 6.346 1.228 1.905.128 3.873.182 5.645.991.65.297 1.358 1.103 2.065 1.09.846-.013.582-.59 1.174-.965.566-.357.986.043 1.54.194.827.225 1.656.216 2.513.245 2.369.078 4.738.127 7.107.186 2.134.053 4.2.403 6.323.415 2.24.013 4.48 0 6.722 0-1.764-3.39-3.246-6.945-6-9.662-1.396-1.375-2.566-2.914-3.889-4.356-1.312-1.431-2.925-2.452-4.38-3.715-1.53-1.329-2.774-2.714-4.558-3.733-.736-.42-1.568-.975-2.373-1.223-.665-.205-1.322-.117-1.936-.48-.72-.426-1.366-1.307-2.063-1.812-.731-.528-1.506-.989-2.294-1.426-1.573-.874-3.211-1.627-4.737-2.587A309.874 309.874 0 0 0 324 485.541c1.711-1.362 3.528-1.528 5.565-.93M452.325 218.852c.439.52.93 1.04 1.223 1.66.337.714.14 1.403.345 2.085.253.84 1.225 1.216.634 2.272-.4.72-1.361.889-1.638 1.73-.497 1.511 1.23 2.9 1.914 4.09.408.708.69 1.46.53 2.28-.099.502-.536.874-.652 1.318-.198.778-.001-.012.09.78.094.819.428 1.68.49 2.542.065.896-.25 1.859-.083 2.727.363 1.904 1.741 4.54 2.846 6.14.957 1.388 1.747 1.88 3.32 1.259 8.4-3.317 10.096-12.415 11.797-20.187 1.009-4.607 1.578-9.05 1.631-13.76.055-4.93.436-9.863.073-14.788-1.39.399-3.129.541-4.092 1.737-.514.639-.526 1.441-1.017 2.021-.677.797-2.082 1.474-2.956 2.129l-13.206 9.877-1.843 1.38c-.633.472-.806.396-.704 1.083.08.54.957 1.224 1.298 1.625M511.299 246.308c-.79 1.031-2.29 1.265-3.252 2.215-.932.922-1.468 2.091-1.914 3.295-.594 1.597-1.232 2.75-1.273 4.48-.038 1.608-.548 3.083-.791 4.653-.163 1.063-.107 1.39.968 1.702 1.494.433 3.305.395 4.836.238 7.337-.75 13.125-6.433 17.578-11.749 5.105-6.092 9.161-12.918 13.549-19.507-2.102-1.186-4.378-2.072-6.384-3.423-1.887-1.272-3.573-2.87-5.43-4.212-.253.773-.253 1.656-.571 2.404-.362.852-.868.993-1.64 1.477-1.394.873-2.523 2.069-3.416 3.427-.82 1.249-1.225 2.842-2.214 3.946-.925 1.036-2.265 1.448-3.315 2.274-1.544 1.216-2.818 3.39-4.063 4.91-1.015 1.239-1.72 2.634-2.668 3.87M374.46 313.167c.284.771.875 1.852.444 2.67-.787 1.494-1.567-.71-2.341-1.067.182 1.283.322 2.56.654 3.813.222.844.52 1.702.361 2.59-.168.948-.946 1.48-.956 2.443-.009.896.462 1.815.598 2.699-1.474-.174-1.794-.365-2.688.748.366.113 1.275.191 1.565.43.426.347.205.18.301.828.13.863-.033 2.294-.063 3.222-.033 1.073-.647.877-1.112 1.613-.323.512-.241 1.502-.16 2.07.115.784.765 1.646.67 2.433-.115.953-1.016 1.009-1.464 1.72-.586.936-.07 1.622.193 2.584.358 1.316.304 2.693.477 4.037 6.436-12.966 11.477-26.59 18.686-39.153.176-.305.35-.612.523-.92a55.67 55.67 0 0 1-.638-.378c-.096-.63-.07-1.412-.126-2.113-.124-1.517-.542-2.988-.735-4.494-.215-1.684.16-3.3-.004-4.969-.055-.577-.19-1.148-.203-1.73-.02-.88.459-1.524.529-2.3.069-.795-.396-1.852-.4-2.706-.002-.67.07-1.338.166-2 .256-1.773.787-3.493 1.055-5.265.55-3.639.586-7.338 1.208-10.972-.97 1.431-2.067 2.772-3.03 4.21-.96 1.433-1.579 3.011-2.449 4.488-.736 1.25-1.63 2.392-2.581 3.481-.793.909-.782.85-.464 1.977.135.484.323 1.05.245 1.563-.337 2.206-3.633 1.46-3.155 3.862.243 1.213 1.155 2.329.66 3.626-.443 1.163-1.776 1.756-2.282 2.911-.556 1.273.189 2.149.237 3.384.042 1.077-.44 1.915-1.175 2.65-1.01 1.01-.964 1.672-1.223 3.103a11.194 11.194 0 0 1-1.331 3.593c-.64 1.1-1.43 1.432-1.094 2.67.245.895.779 1.772 1.102 2.649M359.134 333.02c1.104 4.23 2.804 8.305 5.138 11.98.396-2.146 1.024-4.21 1.133-6.404.107-2.171.18-4.323.418-6.486.447-4.048 1.2-8.052 2.027-12.034 1.735-8.358 3.382-16.727 5.15-25.076a2039.87 2039.87 0 0 0-7.841 12.52c-1.264 2.041-2.527 4.083-3.786 6.126-1.03 1.67-2.91 3.761-3.198 5.745-.328 2.246-.109 4.884-.033 7.156.074 2.211.435 4.34.992 6.473M396.008 282.95c.045.574.447 1.207.559 1.795.133.698.125 1.417.143 2.125.042 1.71.114 3.42.172 5.13 3.455-4.756 5.305-10.413 8.419-15.374 1.347-2.148 2.835-4.225 3.825-6.569 1.086-2.574 2.089-5.137 3.342-7.641.685-1.368 1.39-2.726 2.067-4.097.324-.657.668-.965.319-1.606-.148-.27-.626-.63-.915-.69-.896-.19-1.417.867-2.128 1.226-.97.49-2.19.324-3.207.667-2.344.789-4.59 2.25-6.885 3.208-.73.305-1.257.425-1.2 1.301.052.778.854.62.028 1.297-.447.366-1.033.196-1.547.145 0 1.68.345 2.698-.475 4.196-.655 1.2-.386 2.116-.464 3.384-.09 1.452-.652 2.955-1.197 4.292-.51 1.255-1.27 2.236-.529 3.544.409.721 1.287 1.249.775 2.073-.458.739-1.194.462-1.102 1.595M386.058 505.442c-2.49-1.477-.133-2.978 0-4.36v4.36Zm-3.806-8.436c-.377.018-1.658.154-1.89.544-.102.173.1 1.779.1 2.022-.007 1.228-.132 2.455-.064 3.683.068 1.257.7 2.416.752 3.623.05 1.134-.668 1.741-1.005 2.752-.446 1.332.148 4.998 1.631 5.37.035-.315.1-2.745.318-2.848.71-.336 1.055 1.5 1.257 2.002 1.966-1.945 2.548-4.614 3.47-7.224 1.062-3.003 2.12-6.006 3.179-9.01-1.301-.129-2.608-.208-3.905-.379-1.258-.166-2.565-.597-3.843-.535ZM347.82 582.211c-1.103.466-.787 1.797-.751 2.969.046 1.6.582 2.901 1.99 3.82.52-1.384 2.385-4.416 1.843-5.862-.437-1.167-2.062-1.358-3.082-.927"
                    />
                    <path
                      d="M561 249c.727-.663 1.853-1.095 1.988-2.175.133-1.068-.874-2.846-1.192-3.825l-.796 6M399.614 454.552c-1.89-.008.154-3.94.457-4.842.596 1.019 2.022 4.853-.457 4.842m15.75-49.73c.745-.445 2.152-.963 2.558-1.805.47-.975-.495-.966-1.342-1.025-1-.07-2.315-.135-3.293.055-1.003.196-1.432.426-2.371-.167-.533-.335-1.038-1.168-1.65-1.29-.636-.126-1.37.546-1.92.801-1.92.887-3.957.313-5.993.41-.769.037-1.407.22-2.12-.111-.912-.424-1.12-.988-2.241-.913-1.817.123-3.208.677-5.07.181-1.155-.307-2.744-1.526-3.987-1.15-1.154.349-1.757 2.164-2.67 2.933-.306-1.583-.473-3.332-1.686-4.52-1.359-1.33-3.56-1.807-5.134-2.9-3.554-2.466-6.128-6.015-8.837-9.321-1.72 2.596-2.943 4.975-4.092 7.89-1.097 2.782-2.163 5.575-3.26 8.357-2.21 5.6-4.567 11.146-6.988 16.66-4.915 11.194-10.433 22.082-15.268 33.319 3.307.08 6.537 1.706 9.386 3.265.745.408.777.569 1.256 1.213.231.31.546 1.07.966 1.174.246.06 1.031-.376 1.42-.32.582.085.99.463 1.43.812a167.954 167.954 0 0 1 8.204 6.871c1.213 1.087 2.036 2.114 2.86 3.515.906 1.545 1.887 2.527 3.269 3.654 2.23 1.818 3.519 5.01 5.078 7.443.895 1.398 1.768 2.817 2.453 4.332.62 1.37.836 3.016 1.963 4.09 2.383 2.265 6.111.964 8.999 1.339 1.136.147 1.41.468 2.432.01.012-.403.165-.547.46-.432.337.086.644.239.92.457.752.409 1.468.902 2.197 1.351l1.276-3.145c-.478-.01-2.205.2-2.474-.521-.226-.605.722-.774 1.079-1.148.396-.416.628-1.769 1.4-1.741.632.023.597.577.588 1.118 1.362.029 1.35-1.158 1.581-2.169.157-.68.377-1.351.554-2.028.117-.456.882-2.325.689-2.717-.486-.977-3.607-.917-1.617-2.107.634-.38 2.122-.26 2.614-.796.391-.43.426-2.19.542-2.779-.928.065-2.044.626-2.613-.2-.822-1.194.378-3.017.985-4.014.784.617 1.274 1.292 2.168.578.96-.765 1.834-2.194 2.191-3.296-1.036-.031-.851.175-1.378-.633-.3-.462-.386-1.263-.535-1.78-.304-1.063-.92-1.873.362-2.157.802-.177 1.002.008 1.235.77.307 1.004.196 2.725.316 3.798 1.255-1.583 2.542-3.287 2.07-5.434-.799.795-.88.58-1.844-.038-.962-.617-1.909-1.257-2.862-1.89 1.206-1.093 3.119-.034 4.307-.932.808-.61.395-2.483.45-3.325.068-1.06.528-2.131.465-3.192-.066-1.147-1.381-1.65-.684-2.702.49-.737 1.625-.626 1.837-1.73.214-1.107-.906-1.774-1.74-2.257 1.234-.401 3.604-.828 2.49-2.48-.527-.78-1.165-.848-1.154-1.956.01-.801.538-1.667.88-2.364.461 1.058 1.142 3.541 2.578 3.457 1.197-.07 2.26-1.145 1.915-2.228-1.061-3.338-4.08-3.581-7.017-4.209.549-2.023 1.997-1.906 3.46-.879.867.607 1.409 1.119 2.512.493 1.15-.652.594-.895.087-1.832-1.035-1.906.398-2.999 1.8-4.11.808-.64 1.2-1.44 1.89-2.135.707-.71 1.851-.867 2.743-1.284-1.068-.522-2.036-1.067-3.225-.66-.79.272-1.368.802-2.27.344-2.019-1.022.603-1.283.872-2.127.152-.48-.366-1.454-.17-2.135.2-.696.633-1.287 1.019-1.887.82-1.275 1.717-2.563 1.228-4.153.988 1.137 1.32 1.387 1.399 2.986.07 1.42.079 2.846.364 4.244.061-2.061 1.953-3.435 2.271-5.418.098-.618-.404-2.133.185-2.463.46-.258 1.401.4 1.888.403-.339-1.883-.395-3.866-1.032-5.685-.623-1.782-2.244-2.466-3.604-3.633"
                    />
                  </g>
                  <g fill="currentColor">
                    <path
                      d="M933.692 204.51c-2.507.524-4.924 1.334-7.45 1.815-2.994.568-5.992 1.09-9.01 1.528-2.025.294-3.858.756-5.189 2.449-2.477 3.153-4.97 8.59-1.559 11.936 1.604 1.572 4.416 1.077 6.477 1.731 2.296.73 4.435 1.539 6.842 1.853 11.221 1.457 22.382 4.17 33.36 6.87 5.01 1.232 10.008 2.936 14.797 4.858 2.452.984 4.848 2.13 7.04 3.613 1.875 1.267 4.014 2.516 5.524 4.218-6.004 1.02-12.745 1.139-18.75.12-3.011-.512-6.042-.369-9.08-.599-3.044-.23-6.088-.446-9.129-.714-2.95-.26-5.881-.587-8.811-1.026-1.085-.162-2.183-.323-3.281-.18-.638.082-1.041.276-1.624.144-.488-.111-1.012-.494-1.502-.658-2.248-.75-4.647-.606-6.978-.74-1.747-.1-3.611-.625-5.36-.238-3.074.678-4.476 4.996-5.18 7.643-.442 1.668-.09 3.741.45 5.39.62 1.89 1.758 1.855 3.282 2.713 2.178 1.227 4.089 2.568 6.438 3.525 1.967.803 3.98 1.5 5.927 2.354 4.287 1.877 8.935 3.506 12.973 5.875 2.19 1.286 4.262 2.57 6.558 3.684a44.463 44.463 0 0 1 5.949 3.457c3.852 2.672 7.204 5.994 10.062 9.703-1.326 0-2.409.172-3.678-.278-1.559-.55-3.016-1.359-4.595-1.857-3.008-.95-5.957-1.845-8.931-2.928-2.971-1.083-6.015-2.156-9.049-3.052-1.555-.46-3.126-.664-4.703-1.001-1.391-.299-2.626-.923-3.94-1.45-6.31-2.534-13.063-3.753-19.59-5.582-6.357-1.78-12.64-3.63-19.13-4.883-3.02-.584-6.047-1.135-9.097-1.554-2.985-.41-5.585.003-8.47.768-3.036.806-5.79 1.552-8.689 2.828-2.072.912-5.143 1.71-6.759 3.284-.966.942-1.597 1.874-2.827 2.562-.935.523-1.95.888-2.954 1.252-2.276.825-4.431 1.91-6.693 2.764-1.424.538-2.9.962-4.424 1.06-1.457.093-1.967.446-3.274 1.084-2.044.994-4.285 1.461-6.44 2.14-2.146.677-4.588 2.466-6.912 1.651-2.258-.793-3.704-3.701-5.05-5.506-3.548-4.745-6.597-9.445-8.57-15.058-2.092-5.953-4.088-11.939-6.113-17.915-1.052-3.108-2.11-6.213-3.183-9.314-.597-1.72-1.833-3.572-.485-5.161 1.564-1.844 2.12-4.748 3.737-6.812 1.607-2.051 3.419-3.937 5.122-5.907.995-1.15 1.966-2.324 2.852-3.561.726-1.014 1.264-2.257 2.07-3.198.732-.855 1.809-1.358 2.612-2.148.873-.859 1.65-1.877 2.451-2.805 2.102-2.436 4.101-5.018 6.343-7.326 1.85-1.906 3.964-3.602 5.865-5.48 4.117-4.064 8.49-7.858 12.757-11.76 4.155-3.8 8.405-7.478 12.449-11.368 3.8-3.657 8.36-6.613 12.62-9.67 4.666-3.35 9.14-6.884 14.165-9.708 4.807-2.703 9.646-3.517 15.012-4.318-.53 1.042-.908 2.145-1.61 3.087-1.039 1.39-2.398 2.63-3.573 3.907a1007.678 1007.678 0 0 1-7.206 7.758c-4.703 5.009-9.629 9.815-14.193 14.952-3.637 4.093-6.464 8.27-7.966 13.589-.792 2.804-1.186 5.812.515 8.389 1.302 1.97 3.442 1.817 5.6 1.624 5.56-.498 11.076-2.39 16.58-3.373 6.118-1.093 12.374-1.361 18.55-2.02 3.1-.33 6.161-1.183 9.261-1.417 1.398-.105 2.778.18 4.175.064 1.39-.116 2.783-.455 4.152-.709 2.879-.534 5.494-.838 8.408-.45 2.897.386 5.666.084 8.564-.154 6.12-.505 12.393.156 18.48.816 5.827.632 11.606 1.334 17.297 2.724-5.96 1.516-11.909 3.07-17.903 4.453-6.119 1.41-12.437 2.36-18.444 4.18-2.614.792-5.3.906-7.962 1.463ZM461.873 330.8c.653-2.372 1.314-4.92 2.144-7.287.191-.547.486-.972.795-1.377-.563.938-.96 1.87-1.34 3.128-.555 1.839-1.066 3.69-1.599 5.535Zm-1.306 2.425Zm-.477 1.174Zm1.654-3.147Zm364.847-22.126c-2.377 1.514-4.69 2.471-6.637 4.57-4.39 4.728-9.118 9.203-13.342 14.076-1.698 1.956-2.555 4.38-4.452 6.202-2.156 2.069-4.786 3.68-7.225 5.385-5.418 3.787-10.782 7.88-16.499 11.202-1.282.745-2.58 1.517-3.902 2.184-.87.438-1.966.674-2.728 1.294-.726.59-1.019 1.467-1.81 2.041-.582.424-1.282.77-1.892 1.157a857.973 857.973 0 0 1-15.14 9.362c-4.963 3-9.927 6.04-14.786 9.207-2.582 1.685-5.021 3.577-7.615 5.238-2.113 1.353-4.32 2.556-6.392 3.972-2.197 1.502-4.248 3.04-6.653 4.217-2.326 1.138-4.745 2.08-7.06 3.24-1.365.682-2.6 1.512-3.9 2.297-.85.512-1.563.438-2.488.754-2.696.923-5.113 2.521-7.863 3.373-11.426 3.54-22.474-1.51-32.433-6.881-5.02-2.708-9.59-6.174-14.399-9.225-2.102-1.333-4.726-2.302-6.65-3.869-.967-.788-1.34-1.618-2.472-2.24-1.025-.56-1.959-1.124-2.883-1.844-4.559-3.548-8.818-7.191-12.887-11.3-2.221-2.244-4.442-4.495-6.608-6.79-1.697-1.8-3.02-3.985-4.586-5.915-3.77-4.639-7.426-9.346-11.06-14.09-1.764-2.303-3.509-4.622-5.278-6.922-1.18-1.534-2.959-3.049-3.364-5.035-.374-1.826.698-3.574 1.58-5.08 1.327-2.269 2.594-4.523 3.484-7.007 1.339-3.736 3.382-10.893-.706-13.573-1.503 2.497-3.404 4.718-5.025 7.134-1.63 2.434-2.333 5.51-4.29 7.678-2.057 2.276-4.796 3.927-6.953 6.146-2.15 2.212-4.117 4.594-6.2 6.87-2.18 2.381-4.495 4.633-6.79 6.903-2.158 2.135-4.622 3.85-7.052 5.665-2.543 1.9-5.163 3.591-7.897 5.202-1.18.696-2.396 1.554-3.69 2.02-1.379.494-2.589.792-3.915 1.478-5.397 2.793-10.84 5.473-16.319 8.097-1.47.705-3.02 1.684-4.6 2.076-.607.15-.954.013-1.622.227-.703.223-1.403.62-2.078.917-2.812 1.242-5.683 2.199-8.596 3.175-6.06 2.03-12.215 3.884-18.209 6.1-5.61 2.077-11.068 4.322-16.85 5.93-5.697 1.585-11.714 1.85-17.579 2.424-1.576.155-3.426-.007-4.94.433-1.36.395-2.112.685-3.593.566-2.62-.21-5.067.288-7.666.687 3.957-7.789 6.012-16.66 8.877-24.91.757-2.182 1.677-4.218 2.307-6.437.561-1.984 1.378-3.888 2.362-5.698 1.2-2.208 1.568-4.763 2.37-7.132-.624 1.862-.962 3.844-1.554 5.71.822-1.568 1.706-3.105 2.49-4.694.82-1.66.81-2.648.82-4.434 2.126.245 2.206-1.941 2.534-3.5.498-2.37 1.024-4.735 1.534-7.102.258-.546.17-1.444.306-2.039.162-.704.432-1.379.672-2.057 1.855-5.256 3.894-10.423 5.24-15.843 1.486-5.976 2.786-12.007 4.122-18.018.71-3.191 1.65-6.443 1.96-9.702.148-1.562.944-6.6-1.268-7.302-1.709-.543-3.54 4.666-3.986 5.813-1.084 2.787-2.108 5.596-3.14 8.403-.923 2.51-1.638 5.045-2.667 7.518-1.04 2.5-1.405 5.104-2.327 7.64-.541 1.49-1.412 2.587-2.008 4.012-.434 1.041-.594 2.171-.95 3.238-.74 2.21-1.885 4.372-2.718 6.572-1.066 2.81-2.379 5.517-3.548 8.285-1.07 2.532-1.68 5.2-2.684 7.738-.966 2.438-2.464 4.615-3.425 7.075-1.056 2.703-2.016 5.442-3.06 8.15-2.154 5.587-4.336 11.131-6.025 16.88-.422 1.439-1.142 2.852-1.493 4.133-.319 1.162-.887 2.082-1.344 3.238-.818 2.073-1.16 4.405-1.575 6.589l.124-.707c-.862.036-1.786.224-2.386-.547-.685-.877-.494-2.579-.622-3.632-1.014.133-.618.59-.507 1.507l.169 1.417.426 3.544c.323-.19 1.003-1.292 1.28-1.323.423-.047.884 1.102 1.08 1.67a4.61 4.61 0 0 1-.268.63c-.268.525-.524.89-.729 1.46-.38 1.063-.668 2.145-1.209 3.147a7.96 7.96 0 0 1-.925 1.386c-.27.316-1.15.84-1.298 1.173-.369.833.194 2.912.141 3.882-.067 1.261-.269 2.611-.82 3.763-.232.483-.653.776-.87 1.193-.306.588-.32.892-.391 1.654-.25 2.715-1.29 5.259-2.23 7.791-1.053 2.844-1.876 5.765-2.834 8.64-.905 2.722-2.064 5.339-3.021 8.04-1.116 3.143-2.11 6.33-3.154 9.496-.781 2.371-1.977 4.478-2.815 6.81-.86 2.387-2.07 4.533-2.488 7.014-.463 2.747-1.91 5.496-2.953 8.067-.872 2.148-2.677 4.431-3.036 6.719-.168 1.074.098 2.077-.256 3.144-.307.926-.894 1.716-1.285 2.602-.537 1.22-.137 2.348-.507 3.594-.321 1.085-1.054 1.847-1.646 2.773-1.043 1.634-.69 3.53-1.762 5.234-1.512 2.405-2.047 5.1-2.858 7.832-.4 1.341-.694 2.723-1.052 4.075-.142.539-.213 1.61-.499 2.107-.357.625-.968.38-1.278.74-.267.312-.326 1.32-.444 1.72-.374 1.27-.91 2.502-1.425 3.722-1.11 2.621-2.384 5.172-3.344 7.853-1.004 2.804-2.012 5.608-3.018 8.412-.478 1.335-.943 2.673-1.46 3.992-.506 1.296-1.037 1.692-1.886 2.683-.836.974-1.182 2.34-1.965 3.373-.736.969-1.232 1.68-1.554 2.886-.606 2.269-1.053 4.548-2 6.716-2.394 5.481-5.378 10.695-7.78 16.183-2.518 5.751-4.908 11.562-7.088 17.45-2.047 5.526-3.355 11.256-5.448 16.763-2.072 5.451-3.852 10.985-5.894 16.452-2.12 5.678-4.25 11.346-6.148 17.101-.685 2.076-1.145 4.14-1.589 6.23.138-.94.271-1.863.43-2.729-2.021 3.381-3.878 7.058-2.991 11.11.552-.78.971-1.663 1.307-2.6-.128.483-.27.955-.467 1.374-.566 1.203-1.098 2.12-1.483 3.453-.851 2.95-1.715 5.899-2.552 8.853-.404 1.423-.781 2.86-1.255 4.263-.421 1.247-1.336 2.329-1.722 3.546-.433 1.362-.285 3.03-.619 4.448-.292 1.247-.72 2.464-1.36 3.577-.497.865-1.208 1.62-1.602 2.544-.48 1.126-.19 2.243-.32 3.428-.165 1.474-.693 2.878-.894 4.345-.194 1.413.465 2.29.716 3.64.263 1.411-.27 2.343-.69 3.632-.359 1.096-.566 2.264-.824 3.386-.111.48-.19 1.985-.53 2.305-.249.233-1.678.27-2.037.333-1.346.236-2.69.496-4.063.496-2.88 0-5.832-.419-8.693-.726-5.792-.621-11.617-1.537-17.17-3.333-5.267-1.704-10.576-4.194-15.463-6.793-4.927-2.62-9.298-6.208-13.416-9.945-2.235-2.029-4.363-4.173-6.492-6.312-1.024-1.028-2.046-2.057-3.083-3.072-.826-.811-1.897-1.584-1.98-2.837-.094-1.412.926-3.03 1.423-4.304.444-1.139.881-2.28 1.313-3.423 1.056-2.793 2.093-5.593 3.132-8.391 2.153-5.796 4.292-11.596 6.447-17.391 2.17-5.83 4.396-11.533 7.017-17.172 2.509-5.4 4.792-10.844 6.877-16.422 1.969-5.267 4.312-10.344 6.545-15.5 1.198-2.77 2.332-5.57 3.25-8.445.768-2.408 1.338-4.68 2.507-6.94.752-1.455 1.365-2.878 1.822-4.454.395-1.363.617-2.53 1.5-3.635 1.863-2.329 2.902-4.959 3.86-7.764 1.867-5.463 4.03-10.986 6.644-16.138 2.647-5.222 5.578-10.232 7.99-15.572 2.428-5.37 5.731-10.2 8.003-15.656 4.686-11.254 11.147-21.616 15.897-32.84 4.678-11.056 10.25-21.678 15.028-32.69a491.488 491.488 0 0 0 7.18-17.389c1.085-2.79 2.16-5.584 3.283-8.358 1.014-2.504 2.768-4.424 3.49-7.01.739-2.647 1.921-4.685 3.175-7.07 1.294-2.459 1.963-5.428 2.985-8.02a665.997 665.997 0 0 1 6.963-16.893c2.458-5.76 5.082-11.44 7.634-17.157.964-2.162 1.504-4.486 2.497-6.633.494-1.068 1.496-1.974 1.938-2.996.206-.474.157-1.134.327-1.642.23-.68.656-1.362.972-2.007 2.56-5.24 5.395-10.28 7.73-15.63 2.386-5.47 4.826-10.934 6.924-16.522 2.073-5.522 4.695-10.729 6.478-16.369.849-2.689 1.696-5.39 2.346-8.135.567-2.393 1.634-5.923 1.093-8.381-.333-1.513-2.091-4.595-3.873-3.2-1.639 1.284-2.625 3.827-3.52 5.634-1.346 2.717-2.774 5.4-3.927 8.208-.92 2.242-1.749 4.404-3.023 6.476-3.07 4.996-5.513 10.227-8.278 15.395-.625 1.17-1.361 2.138-2.212 3.142-.825.974-.591 1.44-.929 2.674-.604 2.21-2.44 3.99-3.371 6.079-1.239 2.78-2.312 5.398-3.862 8.037a79.51 79.51 0 0 0-3.667 6.962c-2.521 5.462-4.944 10.971-7.387 16.47-1.296 2.915-2.554 5.853-3.976 8.71-.625 1.254-1.236 2.527-1.914 3.754-.44.798-1.238 1.449-1.61 2.26-.423.92.042 1.696-.32 2.593-.417 1.041-.872 1.465-1.78.856-2.029-1.356-2.433-4.261-3.521-6.333-2.57-4.896-5.018-9.498-5.978-15.037-.531-3.07-.32-6.197-.3-9.295.02-2.75 1.652-4.617 3.088-6.89 3.371-5.345 6.748-10.685 10.178-15.99 1.51-2.334 2.825-4.788 4.37-7.094 1.493-2.228 3.484-3.874 5.123-5.942.914-1.152 1.583-2.416 2.56-3.526.82-.936 1.605-1.907 2.276-2.957 1.279-1.999 2.152-4.162 3.556-6.1 3.313-4.577 6.684-9.128 10.174-13.571 1.86-2.37 4.052-3.838 6.354-5.726 2.254-1.849 4.271-3.958 6.349-6 4.001-3.934 7.774-8.042 12.224-11.46a214.241 214.241 0 0 1 14.085-9.959c2.376-1.537 5.326-2.696 7.236-4.776.788-.858 1.605-1.66 2.615-2.258.97-.574 2.1-.727 3.042-1.252 2.671-1.487 5.128-3.927 7.557-5.789 2.42-1.851 4.838-3.704 7.257-5.555 1.017-.78 2.935-1.7 3.677-2.704.417-.563.367-1.349.839-1.942 1.068-1.341 3.18-1.467 4.678-2.044 2.878-1.107 5.813-1.703 8.833-2.287 2.662-.515 5.203-1.657 7.759-2.552 5.593-1.96 11.39-3.138 17.288-3.67-2.652 7.822-5.663 15.53-7.691 23.55-.821 3.25-.96 7.043-2.33 10.105-.492 1.103-1.81 2.153-1.151 3.398.488.927 1.86 1.93 2.736 2.499 1.817 1.182 4.06-.129 5.845-.843 3.01-1.203 5.927-2.75 9.153-3.304 3.35-.573 6.164-3.746 9.642-3.694 1.454.022 2.634.717 3.723 1.62 1.121.932 2.39 1.68 3.477 2.662 2.388 2.156 5.167 4.007 8.188 5.202 1.048.414 1.942.2 2.912 1.003.724.599 1.299 1.461 2.268 1.683-4.593 4.905-9.047 9.764-12.935 15.268-1.905 2.696-3.324 5.331-4.366 8.469-.493 1.483-.948 2.616-.002 4.05.523.794 1.358 1.447 2.336 1.296 2.363-.364 4.462-3.555 5.773-5.303 1.52-2.03 3.191-3.91 4.775-5.889 3.29-4.11 6.21-8.424 10.307-11.805 1.22-1.006 2.454-2.148 3.806-2.975 1.078-.66 1.839-.156 3.033-.366 2.99-.526 5.675-2.09 8.347-3.442a633.822 633.822 0 0 1 8.705-4.347c1.526-.744 3.773-2.316 5.565-1.564 2.004.84 3.137 3.977 4.236 5.655 1.055 1.613 2.006 3.015 3.737 3.932 2.84 1.507 8.359 3.336 10.847.437 3.523-4.105 3.165-10.828 2.861-15.862-.187-3.086-.565-6.196-.328-9.288.208-2.72.528-5.433.796-8.15 5.007 4.107 9.738 8.275 13.887 13.265 2.207 2.656 4.495 5.241 6.774 7.836 1.14 1.3 2.167 2.436 2.946 3.966.671 1.318 1.627 2.458 2.624 3.539 2.086 2.258 3.51 5.256 5.1 7.89a476.77 476.77 0 0 1 4.549 7.68c2.913 5.038 5.642 9.996 7.87 15.375 1.158 2.796 2.732 5.335 3.916 8.118 1.171 2.76 2.33 5.528 3.458 8.308.651 1.604 1.433 3.223 1.94 4.877.352 1.15.041 2.605.463 3.673.405 1.024 1.376 1.922 1.805 3.025.434 1.116.69 2.293 1.133 3.406 2.171 5.45 5.148 10.647 7.58 15.99 2.522 5.544 5.04 11.097 8.058 16.396 3.062 5.376 6.922 10.307 10.615 15.258 1.934 2.592 3.935 5.145 5.789 7.793.668.953 1.263 2.047 2.155 2.817.843.728 1.908 1.013 2.888 1.499.541-1.44 1.278-2.384 1.046-3.98-.196-1.359-.702-2.691-1.123-3.989-.813-2.51-2.945-4.754-2.41-7.536.499-2.595 2.622-4.651 4.23-6.611 2.125-2.59 4.26-5.173 6.398-7.753 7.183-8.662 14.811-16.962 22.25-25.424 3.787-4.307 7.5-8.686 11.416-12.877 1.755-1.88 3.793-3.408 5.664-5.16 1.534-1.435 2.544-3.247 3.99-4.742 1.073-1.106 2.347-1.913 3.438-3.031l2.66-2.725a875.565 875.565 0 0 0 5.833-6.01c3.868-4.03 7.481-8.281 11.291-12.363 7.357-7.882 15.14-15.45 22.82-23.011 2.218 6.551 4.052 13.38 5.641 20.11.391 1.657.553 3.38 1.14 4.974.578 1.566 1.334 3.073 1.942 4.63 1.35 3.46 2.383 7.15 4.064 10.448 3.191 6.26 5.997 13.522 10.594 18.911 2.288 2.68 5.421 4.977 8.087 7.281l8.613 7.45c-2.584 1.945-5.147 3.897-7.876 5.637Zm-237.763 41.592c-.237-.554-.464-.851-.733-1.05.2.119.376.306.605.749.493.95.796 2.047 1.148 3.056-.318-.927-.635-1.854-1.02-2.755Zm-69.39 49.636c1.984 2.22 4.085 4.338 6.093 6.536 1.852 2.027 3.338 4.25 5.001 6.426a264.29 264.29 0 0 0 11.223 13.7c3.745 4.28 7.789 8.315 11.364 12.742 3.574 4.427 6.273 9.175 8.95 14.166 1.241 2.313 2.654 4.535 4.33 6.557.784.944 1.837 1.771 2.33 2.907.547 1.258.714 2.778 1.026 4.108 2.785 11.894 5.636 23.778 8.32 35.695 1.302 5.776 2.173 11.63 3.431 17.414 1.274 5.855 3.183 11.542 4.387 17.417 2.44 11.898 4.108 23.954 6.728 35.813 2.553 11.547 6.984 22.523 9.391 34.12.339 1.628.827 3.114.45 4.76-.314 1.37-.555 2.572-.579 3.992-.042 2.443-.277 4.366-1.693 6.45-2.79 4.11-6.687 7.686-11.2 9.738-2.421 1.1-4.724 2.58-7.339 3.2-1.258.296-2.48.1-3.727.285-1.266.19-2.31 1.069-3.474 1.546-4.656 1.914-10.6 1.34-15.543 1.344-5.447.005-10.916-.36-15.945-2.64a27.004 27.004 0 0 1-1.218-.59c-.679-.104-.799-.41-.359-.915.002-.551.315-1.18.34-1.763.051-1.24-.319-2.431-.63-3.615-.737-2.81-1.43-5.632-2.074-8.466-2.665-11.75-3.928-23.681-5.567-35.601-.844-6.148-2.074-12.24-3.133-18.354-1.015-5.869-1.625-11.761-2.393-17.664-.765-5.88-1.582-11.756-2.375-17.632-.432-3.218-1.365-6.378-1.775-9.581-.31-2.427.332-4.718-.218-7.151-1.293-5.727-1.346-11.665-2.805-17.355-1.504-5.863-3.678-11.625-5.702-17.325-1.047-2.952-2.09-5.906-3.138-8.858-.618-1.746-1.326-3.62-3.121-4.456-.996-.463-1.981-.411-2.876-1.102-1.098-.848-2.042-2.003-3.02-2.986-2.104-2.118-4.193-4.25-6.305-6.36-8.217-8.212-16.18-16.821-25.25-24.095-4.215-3.38-8.124-7.193-12.793-9.946-5.004-2.95-9.938-6.014-14.944-8.96a55.41 55.41 0 0 0-7.661-3.76c-2.35-.94-.915-3.187-.237-5.007 1.049-2.814 1.958-5.67 2.835-8.543.663-2.17 1.025-4.98 3.246-6.157 1.453-.77 2.088.127 3.481.562.99.308 2.03.356 3.057.311 2.326-.102 5.005-1.145 7.297-.782 1.535.243 2.298.303 3.85.021 1.382-.25 2.838-.255 4.239-.359 2.587-.193 5.248-.225 7.803-.674 5.594-.984 11.26-2.273 16.625-4.152 5.744-2.012 11.218-4.806 16.722-7.378.37 2.534.733 5.11 1.67 7.51.747 1.92 1.984 4.489 3.817 5.507 4.014 2.23 6.114 6.07 9.089 9.4Zm83.955 241.454c1.743-.948 2.075.383 3.11 1.812 2.826 3.906 5.672 7.798 8.526 11.685 2.777 3.784 5.564 7.564 8.344 11.347 2.607 3.545 5.487 6.928 7.881 10.623 2.46 3.791 4.41 7.954 6.298 12.05 1.604 3.487 3.935 7.117 2.25 11.001-1.564 3.606-6.016 5.129-9.255 6.844-4.092 2.167-7.32 3.7-11.95 2.051-8.811-3.137-17.442-7.213-25.79-11.41-7.65-3.848-13.438-9.533-19.78-15.117-3.33-2.932-6.78-5.63-9.895-8.806-3.238-3.304-6.323-6.757-9.584-10.038-5.073-5.11-10.393-9.627-9.517-17.537 4.84 1.621 9.601 3.657 14.63 4.63 5.298 1.024 10.702 1.65 16.103 1.37 2.328-.122 4.457-.757 6.727-1.208 2.216-.44 4.474.019 6.556-1.07 1.842-.963 2.984-2.106 5.091-2.608 2.144-.51 3.953-1.212 5.775-2.475 1.494-1.036 2.88-2.273 4.48-3.144ZM450.63 431.929c.05.01.097.028.142.056a.754.754 0 0 0-.142-.056Zm-5.641-3.187c.052.033.105.071.16.11-.053-.039-.107-.074-.16-.11ZM341.233 708.115a440.206 440.206 0 0 1 6.696 12.35 238.179 238.179 0 0 1 3.124 6.248c.939 1.934 1.78 3.432 1.892 5.564-4.525-.505-9.12-3.186-12.963-5.555-4.041-2.49-7.465-5.574-10.894-8.826-3.346-3.17-6.76-6.307-10.343-9.21-1.653-1.34-3.38-2.49-5.207-3.574-.558-.33-1.123-.67-1.577-1.143-.556-.58-.757-1.592-1.432-2.03-1.379-.893-3.534.44-4.769 1.044-1.635.8-3.209 1.757-4.58 2.962-1.57 1.382-2.623 3.153-3.975 4.728-1.269 1.477-2.552 2.888-3.693 4.47-2.585 3.58-5.395 7.388-8.688 10.35-1.502 1.35-3.395 2.423-4.683 3.946a30.908 30.908 0 0 1-4.01 3.963c-3.267 2.688-6.1 5.376-8.904 8.546-3.09 3.495-6.605 6.626-9.942 9.88-.512-2.06.403-3.784 1.312-5.561 1.058-2.074 1.807-4.314 2.69-6.466a152.331 152.331 0 0 1 6.448-13.744c2.214-4.164 4.48-8.296 6.626-12.497 1.78-3.485 4.534-7.786 3.684-11.858-.332-1.588-1.196-3.985-2.911-4.516-1.71-.53-3.428.971-4.806 1.787-4.038 2.388-8.22 4.269-12.545 6.073-4.313 1.8-8.663 3.55-12.814 5.705-4.027 2.089-7.827 4.588-11.855 6.669-1.992 1.028-4.101 1.79-6.07 2.865-1.898 1.037-3.734 2.18-5.622 3.232-3.73 2.08-7.616 3.535-11.857 4.145 3.025-3.898 7.005-6.757 10.86-9.76 4.497-3.499 8.966-7.024 13.609-10.33 4.173-2.97 8.418-5.83 12.502-8.925 1.91-1.447 3.795-2.926 5.62-4.477 1.534-1.304 3.183-2.608 3.553-4.708.333-1.89-.53-4.07-2.133-5.164-1.6-1.091-3.635-.843-5.43-.587-5.047.718-9.967 2.26-14.852 3.68-1.124.326-2.275.86-3.4 1.123-1.135.264-2.473.153-3.635.212-2.18.11-4.356.225-6.523.502-2.264.29-4.575.207-6.825.568-2.048.327-3.472 1.436-5.566 1.386-9.298-.225-17.928 4.944-27.127 4.944a2228.587 2228.587 0 0 1 16.498-9.583c2.527-1.454 5.157-2.748 7.6-4.346 2.702-1.767 5.184-3.774 7.975-5.426 9.985-5.91 21.095-10.229 30.203-17.51 2.209-1.766 4.316-3.663 6.448-5.52.482-.42 1.117-1.202 1.691-1.473.914-.428.62-.548 1.423.274.883.904 1.495 2.009 2.227 3.035 2.9 4.066 6.461 7.634 10.482 10.593 4.674 3.438 10.06 5.525 15.297 7.94 5.28 2.433 10.49 5.026 16.021 6.857 5.577 1.847 11.482 2.794 17.305 3.481 6.164.73 11.913.316 18.03-.446.356 4.601.822 8.498 2.991 12.666 2.047 3.933 4.648 7.58 6.824 11.447Zm23.178-350.5c.45.099 1.024.136 1.454.294-.373-.094-.848-.097-1.125-.175-.115-.032-.22-.078-.329-.119Zm1.65 2.656c.177-.438.44-.928.485-1.378-.027.445-.267.94-.484 1.378Zm-52.668 77.857c-6.861 4.082-13.01 9.214-19.655 13.618-3.318 2.198-6.778 4.175-10.326 5.977-1.711.87-3.436 1.733-5.22 2.447-1.548.618-3.036 1.615-4.62 2.084-1.723.51-3.489.202-5.233-.004-1.514-.178-2.986-.082-4.232-1.084-2.25-1.807-3.708-5.42-5.045-7.879-1.912-3.516-4.148-7.183-5.517-10.942-1.276-3.51-2.959-6.72-3.625-10.397 1.179-1.956 1.075-3.992.086-5.987-.455-.918-1.042-1.763-1.58-2.631-.609-.982-.582-1.21-1.676-1.228l3.152 9.79c-1.037-3.21-2.439-6.496-3.152-9.79-.624-2.874-.863-5.661-1.889-8.463-.524-1.424-1.35-2.763-1.76-4.22-.337-1.199-.329-2.466-.489-3.694-.446-3.425-1.326-6.817-1.867-10.235-1.03-6.523-1.691-13.118-2.94-19.604-1.285-6.672-2.94-13.303-4.396-19.941-.555-2.534-1.549-4.705.925-6.433 2.368-1.652 5.002-3.03 7.503-4.468 2.374-1.366 4.735-2.746 7.262-3.808 2.145-.902 4.208-1.762 6.159-3.04 1.214-.795 2.457-.79 3.83-1.233 1.414-.457 2.81-.987 4.21-1.482 2.574-.91 5.143-1.828 7.714-2.745 1.406-.501 4.294-2.343 5.611-.947.683.721.762 2.23.955 3.131.316 1.484.616 2.972.908 4.461 1.128 5.77 2.21 11.56 3.105 17.372.879 5.702 1.379 11.459 2.265 17.158.217 1.396.923 3.12.783 4.525-.102 1.035-.821 1.582-1.53 2.256a67.505 67.505 0 0 0-5.092 5.398c-1.77 2.097-3.462 4.266-5.375 6.238-1.864 1.922-3.484 3.65-4.828 5.994-1.128 1.968-2.636 3.814-2.113 6.17.409 1.83.942 3.652 3.116 2.899 2.515-.87 4.689-3.337 6.653-5.05 1.93-1.682 3.836-3.39 5.803-5.028 2.173-1.812 4.602-3.34 6.703-5.226 1.825-1.638 3.387-3.543 5.119-5.274 4.153-4.156 8.622-7.881 13.04-11.739 2.037-1.779 3.66-3.954 5.506-5.918 1.82-1.934 3.797-3.714 5.708-5.556 4.146-3.995 8.459-8.019 12.321-12.29 1.85-2.046 3.835-3.957 5.699-5.99.769-.839 1.339-1.944 2.07-2.743.707-.773 1.795-1.27 2.587-1.993 3.464-3.16 5.394-7.56 8.81-10.782.909 1.988 1.551 4.017 2.218 6.1.615 1.92 1.078 4.162 2.157 5.88.503.798 1.357 1.378 1.792 2.21.416.799.306 1.747.6 2.591.563 1.61 1.724 3.306 2.59 4.778.967 1.644 2.336 3.14 3.63 4.53.334.36.67.68 1.039.956-.151-.084-.3-.169-.457-.248 1.03 1.753 1.961 2.56 3.637 3.712l-.14.291c-.866 1.874-1.84 3.61-2.521 5.57-1.177 3.387-2.905 6.555-4.017 9.967-.598 1.835-.83 3.731-1.701 5.47-.766 1.53-1.762 2.932-2.511 4.472-1.75 3.603-2.891 7.555-4.27 11.31-1.425 3.879-3.07 7.629-4.831 11.364a232.483 232.483 0 0 0-4.928 11.264c-.697 1.71-1.248 3.62-2.349 5.126-1.19 1.63-2.906 2.052-4.714 2.698-7.162 2.56-14.14 6.403-20.667 10.285ZM227.57 328.122c-4.055-3.431-9.715-2.464-14.695-2.485-5.785-.025-11.543 1.113-17.308 1.5-5.762.388-11.552.664-17.329.544-5.152-.108-10.286-.591-15.335-1.641-2.5-.52-4.944-.773-7.482-.993-1.59-.136-3.06-.619-3.86-2.108-1.054-1.966-.78-4.955-.855-7.12-.074-2.122-.38-4.816.465-6.815.832-1.972 2.683-2.94 4.573-3.664 2.32-.89 4.643-1.772 6.963-2.66 9.4-3.601 19.07-7.722 29.196-8.587 2.812-.24 5.635-.265 8.454-.35 2.421-.072 4.66-.66 7.053-.91 4.995-.521 9.88-.635 14.808-1.68 1.354-.288 2.206.433 3.582.653 1.074.172 2.165.14 3.243.062 2.228-.16 4.356-.434 6.593-.243 2.63.223 5.276.361 7.884.768 2.343.365 4.612.998 6.975 1.266 2.553.29 5.136.4 7.65.964 1.27.283 3.206.557 3.954 1.779.99 1.615.247 4.717.15 6.501-.136 2.513-.29 5.03-.264 7.548.027 2.734.436 5.378.964 8.055.38 1.928 1.195 4.191.76 6.18-.24 1.094-1.039 1.611-2.173 1.435-.785-.122-1.43-.89-2.155-.983-1.409-.181-.04.871-.847 1.775-.647.725-1.719.522-2.58.761-.926.258-1.782.711-2.657 1.097-1.983.873-4.041 1.475-5.994 2.438-4.342 2.14-8.468 4.68-12.6 7.191-.691-2.058-1.185-5.176-2.576-6.877-1.197-1.463-3.13-2.192-4.557-3.4Zm284.537-130.14c.931-2.626 2.157-5.192 2.933-7.867.277-.956.277-2.314 1.256-2.852.813-.448 2.286-.165 3.14-.159 1.58.013 3.16-.005 4.739-.012.79-.003 1.932-.182 2.398.645.48.855-.29 2.423-.306 3.375-.024 1.444.318 2.857.684 4.243 1.563 5.907 3.184 11.743 4.98 17.594-1.79-.805-3.182-2.722-5.226-2.939-1.29-.137-2.282.635-3.471.831-1.017.168-2.03.227-3.041.463-4.544 1.057-8.675 3.413-13.29 4.241.214-1.69.282-3.233.801-4.869.452-1.421.953-2.83 1.439-4.241.971-2.823 1.968-5.638 2.964-8.453Zm14.994-39.632c2.183-.966 4.366-3.768 2.322-6.081-1.102-1.247-3.893-.593-5.334-.296-2.022.417-9.786 5.481-10.741 2.038-.628-2.26.496-5.666 1.523-7.651.65-1.258 1.524-2.486 2.57-3.45 1.12-1.036 1.583-.72 2.962-.567 2.276.254 4.391-.45 5.887 1.77.291.43.35 1.017.96 1.107.47.069.75-.514 1.224-.515.767-.002.853.53 1.152 1.125.783 1.555 1.934 4.611 4.265 3.338 2.356-1.285 2.497-5.45 2.5-7.694.005-3.482.24-6.96.43-10.435.148-2.751.39-5.261 2.395-7.335 1.892-1.955 3.358-4.675 5.657-6.173 2.595-1.692 5.972-1.502 8.946-1.73 3.437-.264 6.964-.642 10.403-.289 3.52.361 6.868.527 10.41.367 3.461-.156 6.917-.371 10.382-.46 3.29-.082 6.728.194 9.985-.16 1.324-.143 3.09-.625 3.965.687.995 1.49.912 4.19 1.191 5.907-3.58-.66-9.423-2.312-10.797 2.748-.416 1.53.872 2.897 2 3.787 1.65 1.3 3.958 1.264 5.96 1.544 1.117.157 2.92.128 3.574 1.25.63 1.077.3 3.087.374 4.271.305 4.924 1.008 9.812 1.52 14.716.489 4.687.858 9.497.15 14.182-.362 2.402-1.346 4.66-1.52 7.08-.109 1.499.125 3.199-1.243 4.19-1.588 1.149-4.321 1.155-6.185 1.565-2.46.542-4.894 1.194-7.314 1.893-3.835 1.109-8.514 1.662-9.897 5.951-1.476 4.578-2.564 9.297-3.991 13.898-.726 2.334-1.738 4.449-2.764 6.662-.397.854-1.02 1.846-.938 2.828.086 1.036.81 2.046 1.21 2.97.875-.456 1.792-.71 2.538-1.382.781-.7 1.405-1.688 2.033-2.527 1.305-1.743 2.233-3.239 4.104-4.372.071.922.29 1.713-.152 2.494-.312.555-1.034 1.12-1.445 1.635-1.151 1.445-2.26 2.928-3.509 4.29-2.486 2.714-5.494 4.95-8.08 7.583-2.593 2.64-5.046 5.738-8.208 7.715-3.047 1.905-6.304 3.48-9.474 5.167 2.786-6.33.858-12.702-2.457-18.394-1.543-2.65-3.42-5.081-5.157-7.602-1.375-1.995-4.032-4.672-3.968-7.236.035-1.364.965-2.541 1.636-3.668.744-1.249 1.32-2.58 1.93-3.9 1.237-2.672 2.85-5.498 3.284-8.44.38-2.575-.202-6.143-1.943-8.205-.7-.827-1.726-.747-2.635-1.309-.893-.553-1.569-1.38-2.302-2.116-.832-.835-1.852-1.412-2.681-2.237-.88-.877-1.498-1.99-2.135-3.047-1.51-2.503-3.057-4.985-4.572-7.487Zm9.576-55.636c2.62.046 4.01 1.524 5.23 3.632-.817 1.283-3.006 5.153-4.812 3.61-.334-.286-.49-.903-.896-1.235-.382-.313-.861-.477-1.304-.674-1-.442-2.133-.87-2.773-1.831-1.508-2.262 2.972-3.527 4.555-3.502Zm51.18 98.181c.47-.127 1.921.825 2.383 1.029.85.377 1.908.746 2.66 1.314.753.569 1.03 1.418 1.546 2.18.156 5.698.515 11.439.472 17.154l-.474-17.159c-.334 1.57-.914 3.012-.912 4.634 0 1.643.25 3.335.404 4.967.29 3.109.637 6.213.955 9.319a84.633 84.633 0 0 1-.277 5.36c-2.273-3.162-4.503-6.363-6.782-9.52-2.038-2.82-4.7-2.984-7.917-3.242l6.312-12.744c.241-.487 1.21-3.178 1.63-3.292Zm371.464 9.659c3.394-.939 6.86-2.243 10.358-2.705 3.31-.436 6.251-1.875 9.447-2.812 2.816-.824 5.666-1.53 8.44-2.494 2.108-.731 4.384-1.454 6.044-3.014 1.719-1.617 2.647-4.13 2.328-6.48-.374-2.745-2.4-3.08-4.71-3.867-11.765-4.006-23.672-6.393-35.951-7.976-6.32-.816-12.711-1.83-19.097-1.7-3.309.07-6.58.252-9.888.04-2.994-.191-6.043.93-8.988.346-3.01-.596-5.822.207-8.856.619-3.224.438-6.483.566-9.733.716-2.953.136-6.09.016-8.962.79-2.809.754-4.984 1.296-7.852.361 2.998-3.316 5.985-6.642 8.994-9.948 6.177-6.785 12.59-14.077 16.327-22.56 1.602-3.637 2.38-8.878-.292-12.263-2.376-3.01-7.566-2.39-10.876-1.896-4.256.633-8.32 2.15-11.982 4.393-3.53 2.162-7.431 3.928-10.643 6.536-3.156 2.562-6.791 4.257-9.982 6.77-3.29 2.593-6.604 5.103-9.932 7.638-7.19 5.476-13.162 12.434-19.968 18.353-3.617 3.146-6.984 6.534-10.218 10.068-1.533 1.675-3.122 3.326-4.432 5.186-1.096 1.556-2.049 3.213-3.204 4.73-1.386 1.814-3.126 3.327-4.586 5.085-1.34 1.618-2.573 3.322-3.762 5.053-1.251 1.822-2.453 3.676-3.695 5.503-.652.96-1.275 2.596-2.585 2.428-1.85-.237-2.874-1.826-4.886-.73-.68.368-1.104.997-1.67 1.496-.795.699-1.707.997-2.632 1.46-1.888.947-3.484 2.353-5.175 3.593-1.793 1.314-3.633 2.398-5.294 3.898-6.978 6.305-13.418 13.06-20.128 19.628-1.622 1.587-3.322 3.094-4.899 4.728-1.254 1.301-2.236 2.846-3.497 4.126-.728.74-1.605 1.038-2.374 1.681-.717.602-1.317 1.55-1.929 2.252-1.473 1.692-2.93 3.398-4.444 5.055-2.965 3.25-6.241 6.078-9.092 9.45-2.911 3.445-6.192 6.595-9.168 9.997-6.262 7.161-12.143 14.474-17.9 22.043-2.856 3.755-6.069 7.243-8.793 11.094-2.589 3.659-4.778 7.58-6.732 11.61-.552 1.137-1.26 4.525-3.29 3.327-.727-.429-1.214-1.692-1.664-2.393-.608-.951-1.213-1.903-1.816-2.856-2.625-4.154-5.434-8.308-7.509-12.771-3.993-8.592-7.966-17.113-11.251-26.008-1.661-4.5-3.557-8.888-5.448-13.295-1.873-4.368-3.293-8.897-5.12-13.28-3.636-8.719-8.414-16.967-13.168-25.112-4.356-7.464-9.025-15.164-15.084-21.384-3.204-3.288-5.723-7.059-9.11-10.21-1.715-1.597-3.487-3.133-5.2-4.733-1.239-1.155-2.293-2.87-3.824-3.647-1.083-.549-2.24-.43-3.41-.382-.908.036-1.915-.23-2.79-.151-.788.073-1.106.758-1.868.91-.778.158-1.292-.203-1.904-.625-3.868-2.669-9.155-3.823-13.57-5.423 2.794-1.73 6.22-1.793 9.34-2.522 2.638-.618 5.102-1.706 7.27-3.34 4.779-3.605 5.575-9.303 6.44-14.879.995-6.407 1.766-13.237.913-19.69a1117.625 1117.625 0 0 0-2.665-18.993c-.46-3.07-1.047-6.066-1.962-9.036-.744-2.42-1.703-4.917-1.617-7.484.042-1.268.085-3.226 1.045-4.144.707-.673 1.747-1.033 2.5-1.696 1.762-1.551 3.179-3.444 4.263-5.522.886-1.696 2.316-4.183 1.47-6.145-.92-2.128-3.845-2.52-5.808-3.028-5.812-1.51-11.681-1.266-17.585-2.097-2.836-.399-5.802-1.226-8.65-1.191-2.806.033-5.608.079-8.417.045-6.12-.073-12.261-.251-18.36.379a103.81 103.81 0 0 1-4.834.388c-1.31.076-2.25.742-3.454.95-.733.126-1.77-.27-2.506-.328a14.315 14.315 0 0 0-3.73.178c-2.594.486-4.607 2.139-7.177 2.656-2.002.403-3.534 1.494-4.557 3.268-2.597 4.495-2.738 10.696-2.965 15.73-.267 5.963.177 11.95-.136 17.904-.076 1.44-.23 2.022-1.754 1.872-2.137-.211-4.01-.783-6.17-.213-4.117 1.087-6.697 4.833-8.303 8.536-1.686 3.882-2.589 8.263-2.303 12.501.25 3.728 1.624 8.55 5.733 9.628 2.205.58 4.417-.34 6.499-.994 1.957-.616 3.741-2.38 4.804.094 2.152 5.013 4.32 9.548 7.761 13.865-3.228-.22-6.45-.532-9.68-.694-2.123-.107-5.131-.7-7.163.127-1.194.487-1.246 1.326-2.099 2.07-.94.82-2.762.47-3.926.434-3.484-.107-6.681-.741-10.207-.375-2.596.269-4.94 1.555-7.518 2.044-3.136.593-6.252 1.837-9.211 3.014-2.784 1.11-5.516 2.297-8.38 3.192-2.922.913-5.843 1.852-8.63 3.132-4.96 2.28-9.171 5.964-13.61 9.093-1.167.823-2.44 1.53-3.53 2.458-.892.76-1.482 1.764-2.343 2.55-1.378 1.26-2.987 1.197-4.48 1.864-1.528.683-3.003 2.052-4.39 2.999-1.253.857-2.508 1.713-3.763 2.57-2.384 1.625-4.741 3.559-7.317 4.859-4.784 2.416-8.538 6.244-12.426 9.885-4.196 3.93-8.825 7.354-12.866 11.442-4.03 4.079-7.713 8.407-10.955 13.136-3.38 4.928-6.232 10.152-9.435 15.188-1.465 2.303-3.32 4.174-5.028 6.282a36.68 36.68 0 0 0-2.396 3.304c-.652 1.016-1.112 2.312-2.084 3.056-1.946 1.486-3.888 2.363-5.457 4.35-1.756 2.227-3.27 4.688-4.759 7.098-2.983 4.84-5.783 9.844-8.44 14.87-.786 1.488-.777 2.577-.91 4.2-.1 1.259-.514 2.46-.747 3.696-.381 2.013-.146 4.128-.667 6.126-.34 1.31-1.182 1.482-2.245 2.22-1 .693-1.903 1.512-2.78 2.352-1.785 1.713-3.52 3.403-5.47 4.934-1.982 1.555-4.035 3.057-5.759 4.908-1.79 1.925-3.365 3.772-5.424 5.444-4.375 3.55-8.504 7.48-12.801 11.141-4.462 3.802-8.53 7.974-12.635 12.15-4.243 4.316-8.733 8.363-13.164 12.483-.348-4.89-.635-9.784-1.042-14.67-.44-5.274-1.627-10.416-2.677-15.596-.938-4.625-1.562-9.575-3.132-14.035-1.077-3.056-3.183-5.567-6.44-6.32-1.63-.377-4.099.395-4.888-1.422-.785-1.806-.86-4.554-1.177-6.494-1.398-8.565 1.38-18.125-2.198-26.32-1.553-3.556-4.44-5.44-7.875-6.964-4.498-1.995-9.467-1.499-14.237-2.488-2.177-.452-3.948-1.642-6.208-1.743-2.281-.102-4.566-.027-6.846.06-2.381.09-4.672-.388-7.037-.38-2.06.006-3.68 1.375-5.701 1.054-2.267-.358-4.116-.916-6.406-.258-2.028.582-3.962.756-6.069.814-2.595.074-5.191-.171-7.785-.046-2.384.114-4.658.838-7.02 1.128-4.995.612-9.809 1.181-14.683 2.524-4.828 1.33-9.413 3.247-14.1 4.985-7.513 2.786-16.134 4.92-18.203 13.775-1.146 4.91-1.375 10.199-.758 15.196.498 4.043 1.415 8.52 3.794 11.921 2.542 3.639 6.759 4.741 10.797 5.995 4.628 1.434 9.363 1.654 14.1 2.513 4.914.889 9.827 1.6 14.837 1.465 4.969-.134 9.829-1.427 14.783-1.848 5.094-.432 10.17-1.03 15.26-1.474 3.424-.3 9.968-1.27 11.684 2.616.542 1.229 1.32 2.697.327 3.796-1.183 1.308-2.228 2.648-3.295 4.103-1.448 1.973-2.53 3.863-2.96 6.297-.424 2.387-.49 4.822-.524 7.24-.138 9.976 2.43 20.088 4.865 29.744 2.532 10.047 5.352 20.04 8.882 29.784 3.363 9.285 5.862 18.95 9.851 27.994 2.003 4.54 4.62 8.568 7.376 12.672 2.318 3.451 4.438 7.627 7.852 10.131 2.863 2.102 6.526 3.135 10.067 2.852 4.245-.338 8.278-2.914 11.995-4.795 8.951-4.532 17.221-10.657 25.251-16.654 3.855-2.878 7.35-6.18 11.454-8.72 4.008-2.484 8.052-4.71 12.567-6.017-3.764 7.933-7.185 15.992-11.186 23.814-3.948 7.721-7.215 15.72-10.768 23.62-1.461 3.249-3.614 6.06-5.053 9.296-1.648 3.71-3.302 7.419-4.944 11.131-3.004 6.782-6.202 13.528-8.918 20.424-.617 1.565-.34 3.457-1.14 4.832-.806 1.387-1.687 2.599-2.3 4.105-1.317 3.241-2.362 6.507-3.862 9.682-3.322 7.029-5.432 14.59-8.612 21.692-3.188 7.12-6.064 14.315-8.906 21.58-1.393 3.561-2.96 7.056-4.26 10.653-1.2 3.318-2.327 6.714-4.285 9.682-4.297 6.516-7.628 13.552-10.863 20.659-1.526 3.352-2.872 6.682-3.102 10.411-.208 3.357 1.872 6.076 3.551 8.778-3.057 1.548-5.873 3.548-8.939 5.108-3.29 1.675-6.724 2.989-10.146 4.36-3.46 1.386-6.913 2.791-10.36 4.213-3.038 1.253-5.847 2.793-8.793 4.227-2.787 1.357-5.328 3.147-7.965 4.762-2.852 1.747-5.747 3.426-8.6 5.173-5.592 3.425-11.035 6.859-16.905 9.804-3.06 1.535-6.137 3.038-9.2 4.568-2.517 1.258-5.282 2.364-7.12 4.59-2.718 3.29-2.481 8.584.099 11.933 1.704 2.21 5.888 2.004 8.431 2.076 1.734.049 3.497-.046 5.2-.395 1.408-.287 2.603-1.037 3.976-1.387 1.555-.395 3.294-.31 4.891-.524a69.451 69.451 0 0 0 4.658-.805c3.347-.683 6.665-1.507 10.007-2.217 3.256-.691 6.52-1.362 9.713-2.311 1.325-.394 2.724-1.15 4.088-1.12 1.482.03 2.778.133 4.27-.133 1.524-.272 3.019-.699 4.554-.904 1.427-.19 2.868.039 4.28-.2 1.592-.27 3.07-.973 4.612-1.427 1.508-.446 2.897-.307 4.448-.307-3.822 2.956-7.819 5.678-11.713 8.537-3.934 2.886-7.605 6.075-11.387 9.149-3.062 2.485-6.89 4.876-8.775 8.452-1.516 2.879-1.798 6.446-1.188 9.614.766 3.974 3.707 4.478 7.282 4.726 4.62.32 8.182-1.637 12.215-3.624 7.898-3.891 15.53-8.385 22.994-13.045 3.693-2.306 7.774-4.044 11.697-5.92 2.075-.992 4.153-1.978 6.232-2.961 1.7-.805 3.924-2.408 5.8-2.224-5.163 10.696-10.567 21.277-15.116 32.259-1.907 4.604-3.425 9.364-3.8 14.36-.263 3.505-.08 10.446 4.298 11.606 1.954.518 3.714-.778 5.028-2.07 1.873-1.843 3.598-3.86 5.388-5.785a174.275 174.275 0 0 1 10.98-10.79c3.872-3.49 7.742-6.898 11.218-10.792 3.313-3.711 7.453-6.523 10.608-10.38 3.24-3.963 6.146-8.186 9.409-12.13 2.147-2.596 3.872-6.661 7.556-7.238.549 3.064 3.191 5.788 5.039 8.292 4.078 5.53 9.003 10.06 14.097 14.63 4.607 4.135 9.559 7.601 15.37 9.841 3.825 1.475 11.824 4.892 14.827.251 1.428-2.206.674-6.512.14-8.927-.741-3.354-2.368-6.496-3.837-9.58-2.987-6.275-6.297-12.298-9.657-18.373-1.687-3.053-3.342-6.126-5.162-9.101-1.58-2.58-3.775-5.068-4.552-8.047-.322-1.225-.52-2.888-.124-4.125.42-1.315.97-1.073 2.148-1.241 2.585-.369 1.806-5.141 1.467-7.002-.301-1.657-.922-3.257-1.517-4.827-.504-1.331-1.626-3.145-1.563-4.586.058-1.364 1.381-2.753 1.917-3.977.623-1.423 1.031-2.911 1.472-4.398 2.019-6.817 4.664-13.401 6.344-20.321 1.672-6.887 3.448-13.624 5.9-20.285 2.542-6.907 5.315-13.745 7.628-20.735 2.245-6.781 4.455-13.547 6.921-20.255 4.948-13.454 10.22-26.809 16.053-39.906 5.755-12.924 12.293-25.503 17.267-38.762 2.393-6.377 4.756-12.71 7.01-19.137a374.847 374.847 0 0 1 7.541-19.708c2.76-6.654 5.036-13.523 7.396-20.326 2.322-6.696 4.309-13.619 7.173-20.1.737.802 1.415 1.378 1.944 2.325.528.943.976 1.827 1.706 2.644 1.312 1.465 2.98 2.49 4.57 3.616.996.704 1.821 1.545 2.96 1.7l.09.012c-.205.012-.41.022-.64.069 1.45 1.006 2.9 2.01 4.353 3.02a13.09 13.09 0 0 0-.383-.817c.471.82.962 1.594 1.98 1.769a2.36 2.36 0 0 1-.245.029c.688.509 3.131 2.941 2.916.318.094.425-.065.746.113 1.172.189.449.627.86.956 1.22.69.75 1.494 1.387 2.318 1.98 1.935 1.39 4.021 2.557 5.9 4.028.543.425 2.387 2.565 3.023 2.564l3.557 3.101c-.453-1.589-.909-3.176-1.364-4.764l.99 3.455c.383 1.339.962 1.824 1.968 2.723 1.765 1.578 3.515 3.177 5.347 4.68 3.342 2.738 7.178 4.862 10.757 7.273 1.947 1.31 3.856 2.656 5.709 4.094 1.52 1.18 3.378 1.597 4.89 2.741 1.901 1.443 3.588 3.322 5.238 5.039 1.48 1.538 3.226 3.089 4.248 4.984 1.157 2.149 1.176 4.95 1.536 7.329.365 2.4.726 4.801 1.076 7.204.721 4.93 1.403 9.866 1.995 14.813.582 4.864 1.296 9.724 1.719 14.604.428 4.938.32 9.882.876 14.816 1.112 9.896 2.373 19.984 4.492 29.72 2.136 9.81 3.504 19.76 5.358 29.632.954 5.077 1.963 10.144 2.958 15.213.489 2.493 1.016 4.98 1.51 7.471.407 2.06.195 4.13.557 6.207.192 1.112.463 2.225.463 3.36.001 1.104-.691 2.07-.73 3.058-.055 1.392 1.22 3.588 1.992 4.697 1.166 1.674 2.78 3.188 4.133 4.724 1.363 1.547 1.911 2.102 1.427 4.192-2.078 8.963 3.246 16.472 8.93 22.809 6.258 6.976 12.517 13.7 19.696 19.739 3.76 3.164 7.3 6.579 10.899 9.922 3.45 3.205 7.172 5.707 11.204 8.151 4.086 2.478 8.417 4.284 12.652 6.461 4.176 2.147 8.234 4.184 12.675 5.748 4.223 1.487 8.5 2.753 12.553 4.643 3.2 1.491 7.26 1.427 10.491.048 2.175-.93 4.042-2.623 5.984-3.947a86.777 86.777 0 0 1 5.571-3.524c3.146-1.813 6.222-3.924 7.565-7.464 1.59-4.192.215-9.226-.694-13.436-.93-4.304-2.85-8.314-5.15-12.044-2.398-3.884-5.38-7.245-8.132-10.865-2.847-3.744-5.383-7.685-8.36-11.34-2.949-3.616-6.038-7.064-9.357-10.34-3.312-3.267-6.205-6.91-9.26-10.412-.736-.843-1.861-1.749-2-2.944-.12-1.041.596-1.793.94-2.71.666-1.778.469-3.732.503-5.59.017-.905.002-1.835.453-2.655.42-.765 1.2-1.07 1.498-1.892.265-.732.168-1.658.315-2.423.187-.97.64-1.677.95-2.598.773-2.303-.186-4.988-.62-7.272-.41-2.157-.797-4.318-1.195-6.476-.86-4.667-1.502-9.345-2.968-13.874-2.98-9.193-5.894-18.338-6.092-28.1-.1-4.889-1.463-9.573-2.593-14.3a2041.085 2041.085 0 0 0-3.652-15.026c-2.369-9.584-5.21-19.028-7.637-28.595-2.419-9.532-4.62-19.127-7.498-28.533-2.908-9.507-6.95-18.432-11.274-27.366-2.08-4.295-4.356-8.003-7.416-11.672-3.173-3.803-6.427-7.54-9.687-11.27-6.575-7.527-13.156-15.048-19.827-22.489-3.322-3.704-6.686-7.237-10.562-10.372-3.453-2.795-8.183-5.148-9.272-9.715-.307-1.29-.514-2.602-.853-3.883-.195-.74-.802-1.858-.707-2.664.19-1.6 3.15-2.211 4.347-2.753 4.143-1.875 8.334-3.732 12.398-5.774 2.245-1.128 4.24-2.503 6.33-3.877 1.758-1.158 3.763-1.507 5.746-2.122 4.404-1.366 8.469-3.313 12.533-5.465 4.195-2.22 8.425-4.44 12.491-6.887 4.129-2.485 7.332-5.54 10.832-8.785 1.565-1.45 2.678-.72 3.787.877 1.35 1.941 2.772 3.833 4.167 5.742.65.889 1.209 1.938 2.13 2.47.037.023.065.035.1.053-.16-.054-.33-.112-.522-.189.338 2.402 2.146 4.043 3.74 5.697-.15-.41-.292-.823-.434-1.236.399 1.144.95 1.92 1.686 2.895 1.464 1.94 3.111 3.648 4.842 5.351.895.88 1.8 1.752 2.705 2.625.36.348.724.697 1.084 1.048.583.564.912.745 1.523 1.02 1.238.559 1.526 1.535 2.594 2.306 1.605 1.16 3.603 1.926 5.265 3.07 1.356.933 2.536 2.83 4.105 3.177.568.125.99.034 1.447.483.602.59-.024.578.924.79.478.108.971-.166 1.453-.013.508.162.957.603 1.385.915.98.715 1.961 1.43 2.94 2.148a350.21 350.21 0 0 1 5.25 3.92c3.458 2.64 6.634 5.628 10.102 8.25 1.794 1.356 3.724 2.347 5.64 3.498 1.596.96 2.941 2.256 4.437 3.353 1.786 1.31 3.871 1.949 5.724 3.133 1.82 1.162 3.57 2.362 5.461 3.41 4.132 2.287 8.33 4.42 12.709 6.19 4.427 1.791 8.956 3.236 13.762 3.462 4.878.231 9.494-1.046 13.992-2.826a124.85 124.85 0 0 0 13.047-6.033c4.103-2.198 7.923-4.842 11.946-7.171a176.198 176.198 0 0 0 11.717-7.432c3.657-2.517 7.134-5.329 11.082-7.385 3.984-2.075 7.517-4.777 11.268-7.227 3.855-2.518 7.684-5.054 11.51-7.62 7.978-5.347 15.964-10.785 23.542-16.694 3.638-2.838 7.227-5.754 10.414-9.095 3.401-3.568 7.458-6.606 11.194-9.815l10.754-10.438a.174.174 0 0 0 .018-.018l-10.772 10.456c1.706-.32 2.815-1.428 4.185-2.49a231.124 231.124 0 0 0 4.289-3.406c2.063-1.68 4.45-3.407 4.21-6.383a48.817 48.817 0 0 1 2.786-2.434c1.57-1.252 3.442-2.016 5.11-3.125 3.425-2.276 6.365-5.174 9.582-7.719 1.416-1.118 2.687-1.965 2.908-3.884.24-2.083-.042-4.34-.056-6.439-.011-1.622-.476-3.886.144-5.421.623-1.539 2.693-2.199 4.033-2.933 3.784-2.076 7.511-4.252 11.298-6.323 3.727-2.038 6.967-3.396 11.23-3.76 1.855-.16 3.636-.439 5.471-.724 1.897-.296 3.664.264 5.522.604 4.213.773 8.477 1.234 12.689 2.017 4.241.788 8.436 1.837 12.625 2.87 4.193 1.033 8.378 2.133 12.635 2.88 2.178.381 4.43.632 6.541 1.312 1.597.513 3.132 1.237 4.774 1.6 3.975.876 8.015.98 11.9 2.308 3.879 1.325 7.836 1.993 11.88 2.601 2.117.32 4.227.66 6.322 1.113 1.264.274 2.388.435 3.67.121 2.656-.65 4.811-2.788 5.66-5.362 2.343-7.1-2.19-14.721-6.804-19.757-4.95-5.401-11.203-9.182-17.144-13.36 2.367-.26 4.72.089 7.086.132 2.027.037 4.011-.053 6.043.058 4.497.246 8.98-1.056 13.483-1.152 6.095-.13 13.616-1.373 15.208-8.32 1.481-6.465-5.377-12.292-10.27-15.226-3.144-1.885-6.327-3.61-9.406-5.605-3.29-2.131-7.355-3.385-10.989-4.819-3.698-1.46-7.422-2.887-11.27-3.904-3.697-.978-7.442-1.502-10.987-2.95 3.225-1.407 6.714-1.717 10.168-2.362 3.44-.643 6.821-1.494 10.192-2.426Z"
                    />
                    <path
                      d="M980.03 463.71c-.36 1.524-1.245 2.273-2.754 2.636-2.489.598-5.218.69-7.757 1-4.64.565-9.438.8-13.99 1.9-4.26 1.028-8.52 2.566-12.367 4.678 1.258-6.015 2.684-11.858 2.156-18.053-.388-4.549-.496-11.402-5.093-14.032-1.064-.61-2.75-.38-3.932-.443-1.538-.082-3.078-.154-4.617-.24-3.044-.167-6.072-.487-9.106-.784a370.604 370.604 0 0 1-17.87-2.185c-.984-.144-2.75-.036-3.408-.889-.486-.627-.254-1.618-.217-2.323a41.527 41.527 0 0 0-.276-7.283c-.662-5.296-2.124-10.507-3.759-15.575-.828-2.562-1.713-5.104-2.547-7.664-.402-1.237-1.337-2.925 0-3.81.834-.554 2.292-.55 3.262-.792 1.155-.287 2.28-.664 3.41-1.035 2.666-.877 5.329-1.766 7.993-2.653 10.122-3.372 20.335-6.52 30.428-9.96 1.412-.48 2.815-.984 4.227-1.465.612-.207 1.674-.742 2.333-.454.77.34.621 1.187 1.174 1.692.483.44 1.577.473 2.199.831 1.579.911 2.568 2.5 3.807 3.782 1.594 1.65 3.376 3.099 4.95 4.779 6.351 6.781 11.051 14.945 15.03 23.294 1.071 2.249 1.768 4.575 2.438 6.968.633 2.264 1.442 4.477 1.802 6.808.838 5.443.869 11.044 1.541 16.522.562 4.59 2.027 10.15.943 14.75m-40.35-12.295c-.013 3.626-.204 7.275-.998 10.824-.818 3.658-2.296 7.161-3.614 10.66-1.244 3.301-2.25 6.705-3.694 9.926-1.423 3.178-3.247 6.075-4.389 9.375-2.342 6.765-6.354 12.7-9.194 19.233-2.9 6.677-6.132 13.053-9.737 19.383-1.714 3.011-3.305 6.342-5.473 9.054a40.49 40.49 0 0 1-3.55 3.88c-1.085 1.04-1.606.246-2.918.482-1.913.343-2.9 2.825-4.137 4.102-2.346 2.418-4.743 4.807-7.312 6.987-2.632 2.231-5.54 4.097-8.24 6.245-2.576 2.051-5.15 3.904-8.095 5.4-5.816 2.954-12.503 6.578-19.213 6.268-3.088-.143-4.742-2.153-6.035-4.836-1.37-2.84-.908-6.245-.678-9.292.275-3.655 1.159-6.879 3.108-9.98 1.92-3.056.027-4.953-1.888-7.5-4.272-5.687-9.965-10.115-15.157-14.92-2.68-2.482-5.338-4.99-7.993-7.499a4762.987 4762.987 0 0 1-4.399-4.163c-1.074-1.018-2.11-1.705-2.497-3.135 2.922-1.575 5.967-2.917 8.66-4.886 5.496-4.02 10.539-8.926 15.72-13.348 2.488-2.125 4.455-4.758 6.454-7.327 1.93-2.48 4.14-4.821 5.806-7.496 3.581-5.754 7.666-11.452 10.505-17.622 1.416-3.075 2.147-6.212 2.888-9.496.446-1.978.96-3.94 1.46-5.904.323-1.266.416-3.448 1.227-4.494 1.493-1.927 6.185 1.185 7.904 1.732 3.235 1.03 6.606 1.518 9.952 2.036 7.097 1.097 14.244 2.035 21.296 3.396 3.694.713 7.397 1.503 11.146 1.876 3.634.362 7.288.174 10.93.147.78-.007 1.732.132 2.256-.584.298-.41.098-.919.578-1.302.625-.499 1.442-.23 2.16-.154 1.458.155 2.92.244 4.376.425.629.078 1.88-.028 2.417.303.566.35.371 1.402.368 2.204m-102.149 97.338c-1.698 3.424-2.68 8.213-2.73 12.006-.028 2.036.24 4.067.658 6.056.363 1.736 1.356 3.733 1.044 5.48-4.308-1.991-8.54-4.134-12.934-5.932-3.796-1.553-6.648-4.257-9.772-6.826-3.355-2.757-6.391-5.831-9.614-8.735-3.46-3.118-7.067-6.065-10.585-9.113-6.263-5.427-11.522-11.526-16.663-18.01-4.52-5.7-11-12.971-10.697-20.683.143-3.58.405-7.65 2.16-10.853 1.458-2.661 4.096-5.645 6.562-7.425 2.371-1.71 4.699 2.589 6.032 4.1 2.871 3.251 5.799 6.45 8.711 9.664 5.54 6.113 10.796 12.532 16.787 18.215 6.04 5.726 12.213 11.327 18.716 16.519 3.255 2.598 6.842 5.02 9.797 7.964 1.085 1.08 3.19 2.271 3.915 3.62.776 1.454-.78 2.73-1.387 3.953m148.21-84.417c-.912-.118-.869.005-.946-.89-.12-1.387.349-3.025.49-4.415.672-6.614.35-13.545-.226-20.16-1.133-12.994-3.81-25.769-10.604-37.04-3.136-5.203-6.682-10.11-11.127-14.28-2.05-1.921-4.05-3.99-6.316-5.659-1.137-.838-2.792-1.261-3.828-2.119-.559-.464-.807-1.409-1.279-2.001-1.885-2.363-4.014-2.973-7.021-2.718-3.12.265-6.39 1.243-9.4 2.078-6.403 1.776-12.545 4.796-18.747 7.179-6.231 2.393-12.392 4.943-18.536 7.547-2.672 1.13-5.896 2.02-8.267 3.727-1.433 1.032-1.854 2.313-1.7 4.03.544 6.062 2.794 12.083 4.506 17.883 1.838 6.228 3.336 12.573 3.805 19.065-6.348-.465-12.657-.948-18.976-1.746-3.16-.4-6.338-.727-9.492-1.16-1.631-.225-2.392-.834-3.703-1.681-1.152-.743-1.745-.114-2.236 1.016-2.556 5.872-3.562 12.278-6.33 18.107-1.242 2.618-2.794 5.062-4.085 7.654-1.406 2.818-3.1 5.591-4.862 8.198-.818 1.21-1.867 2.19-2.774 3.324-.84 1.051-1.443 2.26-2.215 3.36-1.613 2.299-3.567 4.324-5.545 6.306a192.418 192.418 0 0 0-5.806 6.079c-1.865 2.038-4.091 3.692-6.062 5.616-1.855 1.81-3.47 3.781-5.373 5.555-1.833 1.71-2.691 4.083-4.663 5.62-1.063.83-2.243 1.495-3.3 2.33-.875.69-1.788 1.897-3.029 1.666-1.377-.258-2.485-2.11-3.346-3.065a421.271 421.271 0 0 0-3.354-3.671c-4.419-4.784-8.95-9.473-13.407-14.225-2.178-2.324-4.574-4.432-6.697-6.808-1.547-1.732-4.079-2.965-6.245-1.78-2.186 1.195-4.81 2.25-6.58 4.044-1.918 1.944-3.447 4.471-5.009 6.702-1.626 2.323-2.272 5.067-2.698 7.829-.452 2.943-.924 5.859-.7 8.846.425 5.655 1.476 11.966 4.746 16.728 1.558 2.27 3.693 3.974 5.602 5.926 2.068 2.117 3.545 4.665 5.25 7.067 3.42 4.82 7.622 9.135 11.907 13.19 2.044 1.933 4.358 3.563 6.412 5.503 2.193 2.073 4.414 3.942 6.889 5.665 2.531 1.762 4.622 4.057 6.909 6.105 2.356 2.11 4.767 4.077 7.36 5.885 4.974 3.47 10.032 5.729 15.816 7.473 6.37 1.922 12.536 4.437 18.87 6.446 3.036.962 6.04-.369 8.932-1.343 2.948-.993 5.84-2.17 8.555-3.7 5.506-3.104 10.794-6.705 16.21-9.975 5.633-3.4 10.338-7.298 14.641-12.273 4.101-4.744 8.442-9.454 12.177-14.491 7.582-10.231 12.167-22.334 17.794-33.663 2.992-6.022 6.023-11.95 8.511-18.209 1.224-3.078 2.32-6.212 3.569-9.279.529-1.3.966-1.805 2.384-2.026 1.638-.256 3.312-.369 4.957-.565 6.508-.776 12.86-1.928 19.31-3.042 5.986-1.032 12.008-2.1 17.36-5.141 1.125-.64 2.475-1.392 3.088-2.588.279-.546.965-3.314.577-3.757-.227-.26-1.753-.228-2.142-.28M719.239 53.315c2.079-1.094 4.2-2.103 6.312-3.13 4.475-2.175 8.882-4.499 13.408-6.57 2.049-.936 4.189-1.533 6.293-2.32 2.235-.834 4.371-1.937 6.516-2.974 4.509-2.181 9.118-4.061 13.8-5.833 1.434-.543 2.87-1.077 4.309-1.609 1.104-.407 2.447-1.19 3.595-1.325 1.821-.213 3.189 1.644 4.43 2.76 6.696 6.028 11.424 14.963 14.3 23.413 1.57 4.61 2.283 9.371 3.12 14.152.858 4.9 1.77 9.696 1.803 14.678.034 5.065.554 10.059.33 15.136-.083 1.894.258 5.3-.783 6.95-1.058 1.672-3.706 2.14-5.41 2.757-9.317 3.378-18.625 7.015-28.078 9.988a3423.61 3423.61 0 0 0-29.459 9.415c.4-2.79.867-5.578 1.18-8.378.295-2.637.075-5.366.052-8.016-.05-5.823-.24-11.614-1.41-17.337-2.124-10.392-5.557-20.623-11.312-29.594-1.35-2.101-2.957-4.018-4.56-5.926-.573-.683-2.493-2.299-2.425-3.35.091-1.377 2.995-2.363 3.989-2.887m111.806-1.129c2.15.098 3.542.593 3.691 2.895.191 2.934.15 5.906.255 8.847.202 5.625.404 11.251.256 16.88-.26 9.861-1.309 19.708-4.329 29.132-1.46 4.555-3.206 9.033-5.093 13.427-1.83 4.264-4.418 8.098-6.708 12.114-1.128 1.975-2.393 3.58-3.948 5.239-.736.787-1.074 1.206-2.115 1.34-1.056.136-2.385-.137-3.445-.213-4.989-.359-9.975-.854-14.972-1.095-4.953-.238-9.928-.03-14.875-.26-2.324-.108-4.514-.863-6.8-1.231-2.3-.371-4.627-.534-6.942-.795-1.193-.135-1.164.147-1.345-.938-.097-.578.07-1.325.1-1.917.06-1.14.119-2.28.188-3.42.062-1.018-.144-2.806.268-3.77.368-.857 1.156-1.006 1.977-1.23 4.607-1.265 9.058-2.83 13.49-4.617 4.521-1.822 9.097-3.543 13.567-5.491 1.927-.84 4.1-1.608 5.56-3.191 1.197-1.299 1.885-2.968 2.414-4.629 2.912-9.136 1.998-19.665 1.145-29.077-.877-9.674-2.26-19.602-5.803-28.697 11.148 0 22.326.185 33.464.697M706.614 60.55c3.096 1.16 5.047 3.865 6.85 6.471 2.246 3.25 4.365 6.536 6.124 10.082 3.694 7.451 6.133 15.662 7.524 23.85 1.331 7.816 1.99 16.533.46 24.374-.41 2.097-1.48 3.78-2.077 5.766-.402 1.335-.122 3.276.24 4.603.52 1.903 2.314 1.254 3.81.946 2.182-.45 4.365-.895 6.545-1.355 8.218-1.734 16.356-4.751 24.723-5.361-.713 2.225-1.111 4.244-2.55 6.151-1.345 1.784-3.137 3.33-3.911 5.48-.645 1.79-.77 4.297-.113 6.115.69 1.908 2.376 1.935 4.145 2.15 4.974.603 9.869 1.422 14.884 1.696 1.078.058 2.154.18 3.218-.025.857-.165 1.458-.42 2.324-.26 1.147.21 2.158.763 3.362.886 1.133.116 2.275.08 3.41.037 4.2-.157 8.307.28 12.51-.129 1.226-.12 3.527-.814 4.672-.114 1.21.74.262 2.233.864 3.26.794 1.35 3.52.658 4.838.585 1.882-.103 3.55-.436 5.237-1.304 3.41-1.755 6.492-4.091 9-7.005 2.624-3.049 4.527-6.61 6.404-10.146 1.958-3.686 3.92-7.287 5.4-11.203 1.513-4 2.996-8.128 3.853-12.325.856-4.191 1.398-8.448 2.206-12.651.843-4.384.99-8.853 1.214-13.3.23-4.612.332-9.144.043-13.756-.283-4.531-.783-8.975-1.314-13.478-.136-1.15-.066-2.342-.249-3.482-.124-.766-.523-1.493-.567-2.27-.088-1.546 1.33-2.54 1.293-4.156-.086-3.755-4.08-5.153-7.047-6.051-3.858-1.168-8.162-1.872-12.196-1.582-8.68.624-17.552.705-25.784 3.874-4.266-7.312-8.24-15.243-15.221-20.37-.787-.58-1.641-.975-2.151-1.83-.436-.733-.628-1.469-1.21-2.13-1.433-1.624-3.199-1.826-5.204-1.4-4.364.923-8.533 1.935-12.67 3.588-1.837.733-3.786 1.078-5.63 1.795-1.774.688-3.47 1.552-5.175 2.392-3.73 1.84-7.588 3.403-11.315 5.258-7.486 3.726-14.886 7.632-21.777 12.391-3.328 2.3-7.144 4.677-9.966 7.596-1.242 1.285-1.805 2.907-1.598 4.68.215 1.846 1.16 1.127 2.572 1.657M216.13 74.834c.366-4.76.218-9.532.563-14.289.33-4.52 1.242-9.122 2.13-13.565.903-4.518 2.544-8.529 4.389-12.723 1.018-2.314 2.002-4.643 2.989-6.97.602-1.428 1.043-2.907 2.713-3.287 2.165-.494 4.779-.094 7.006-.173 2.25-.08 4.499-.216 6.742-.415 4.876-.433 9.72-1.146 14.588-1.651 4.662-.483 9.272-.766 13.875-1.727 2.37-.494 4.718-1.073 7.067-1.65 1.76-.431 4.2-1.535 5.971-1.116-2.19 4.988-2.872 10.312-2.926 15.71-.02 2.127-.272 2.997-2.487 2.867-2.764-.164-5.539-.442-8.288-.758-5.188-.599-10.335-1.549-15.478-2.44-2.438-.422-4.78-1.023-7.17-1.552-1.699-.374-2.413.299-2.98 1.85-.41 1.122-.542 1.627-1.595 2.26-.86.518-1.707 1.008-2.499 1.633-1.41 1.11-2.624 2.446-3.662 3.91-2.304 3.242-3.781 7.053-5.134 10.769-3.4 9.352-5.112 19.523-5.428 29.455-.16 5.05-.993 10-1.048 15.067-.026 2.234.544 5.108-1.321 6.73-1.824 1.59-4.167 2.73-6.202 4.032-1.784-5.11-2.244-10.53-2.298-15.908-.056-5.36.074-10.714.484-16.059m75.485-27.046c2.228.593 4.467 1.118 6.714 1.639-3.821 9.158-3.687 19.289-3.976 29.05-.128 4.367-.011 8.744.418 13.09.442 4.447 1.441 8.69 2.34 13.05.195.952.665 2.513.038 3.386-.695.967-2.971 1.49-3.958 1.935-3.917 1.766-7.773 3.682-11.77 5.267-4.052 1.607-8.261 2.571-12.223 4.438-3.878 1.827-7.807 3.502-11.845 4.946-3.967 1.42-8.084 2.461-11.913 4.247-3.632 1.693-6.63 4.02-9.716 6.53-1.876-5.827-2.607-11.755-2.946-17.862-.325-5.845-.804-11.93-.207-17.762.271-2.66.272-5.136.219-7.8-.048-2.385.503-4.723.693-7.096.418-5.228.401-10.45 1.109-15.657 1.34-9.85 4-20.276 9.366-28.72.342-.538.676-1.406 1.148-1.804.795-.668.513-.543 1.559-.004.976.5 1.938 1.033 2.945 1.471 2.115.92 4.378 1.325 6.584 1.952 2.444.695 4.661 1.759 7.198 2.152 2.457.381 4.954.427 7.41.814 1.358.214 2.604.663 3.926 1 1.076.275 2.13.202 3.262.32 2.188.232 4.318.845 6.512 1.043 2.467.223 4.664-.276 7.113.375m-80.85 42.522c.757 6.029 2.37 11.935 4.72 17.538.756 1.802 1.608 4.324 3.768 4.822 1.992.459 4.406-1.005 6.283-1.519 1.472 8.75 2.116 17.841 6.02 25.951.493 1.024 1.143 2.932 2.157 3.603 1.036.685 2.295.016 3.328-.377 4.338-1.644 8.819-2.696 13.308-3.834 2.19-.555 4.42-1.048 6.557-1.79 1.774-.617 3.45-1.358 5.31-1.694 3.885-.703 7.495-1.974 11.209-3.285 7.795-2.754 15.632-5.247 22.826-9.382 3.433-1.973 7.034-3.906 10.204-6.29 2.6-1.953 5.509-4.882 4.235-8.43-.385-1.073-1.44-2.849-2.655-3.176-1.14-.31-2.164.723-3.207.96-1.577.361-1.223-.911-1.37-2.153-.115-.979-.236-1.957-.355-2.936-.248-2.055-.493-4.11-.677-6.173-.759-8.54-.402-17.076-.41-25.631-.003-4.283.988-8.038 1.988-12.164.53-2.19 1.024-4.388 1.49-6.59.258-1.225.559-2.57.204-3.809-.795-2.778-4.752-3.546-7.095-4.337-2.014-.68-4.059-1.272-6.144-1.688-1.143-.228-4.196-.05-4.935-1.01-.732-.95.078-4.033.152-5.164.137-2.082.28-4.162.555-6.232.442-3.339.394-6.435-2.27-8.877 1.661-1.324 2.84-4.133.425-5.353-2.779-1.405-6.497-1.608-9.432-.872-3.21.806-6.733.816-10.02 1.169-3.499.373-6.987.828-10.478 1.268-6.005.76-12.122.927-18.054 2.146-2.97.61-5.944 1.073-8.938 1.546-3.101.49-4.97.905-6.827 3.522-3.518 4.963-6.766 10.05-8.742 15.844-1.87 5.489-2.743 11.256-3.263 17.015-.56 6.206-.685 12.395-.615 18.623.07 6.263-.033 12.529.749 18.759M84.613 335.594c.356-.603 1.435-.751 2.003-.979.897-.359 1.746-.808 2.58-1.29 4.18-2.412 8.258-4.876 12.12-7.781.947-.712 3.274-3.327 4.645-2.914.672.204 1.038 1.319 1.33 1.863.509.958 1.038 1.9 1.614 2.82 2.274 3.63 5.108 6.889 8.273 9.773 5.95 5.42 12.728 8.921 20.89 9.34-5.27 7.163-10.308 14.645-16.294 21.242-3.02 3.33-6.333 6.34-9.47 9.553-.535.548-.823.877-1.528 1.192-.683.304-1.333.403-1.897.964-.981.977-1.05 2.431-.83 3.721-5.57.214-11.6-2.28-16.655-4.325-2.966-1.201-5.893-2.538-8.68-4.117-1.893-1.072-4.01-2.27-5.399-3.991 5.108-2.718 10.272-5.302 15.167-8.397 3.342-2.112 9.248-4.579 8.77-9.455-.074-.758-.372-1.69-1.044-2.128-.994-.648-1.22.117-2.124.313-1.468.317-3.201-1.622-4.321-2.456-3.029-2.258-5.987-4.751-7.784-8.138-.537-1.014-2.058-3.634-1.366-4.81m-31.11 43.178c.829-2.7-3.04-4.779-4.749-6.078-2.562-1.95-4.344-4.108-6.068-6.832-6.724-10.627-9.106-23.571-8.555-35.997.153-3.449.544-6.891.624-10.342.041-1.791-.075-3.46.317-5.229.371-1.672.815-3.326 1.225-4.99.82-3.316 1.24-6.714 2.218-9.992.909-3.037 2.147-5.966 3.598-8.783.966-1.88 1.867-4.516 4.393-4.156 3.156.452 6.241 2.077 9.315 2.925 3.404.939 6.846 1.723 10.282 2.531 3.324.78 6.574 1.89 9.92 2.564 1.75.353 3.479.497 5.26.466.777-.013 2.173-.088 2.25.975.083 1.163-1.702 2.426-2.435 3.138-1.035 1.006-1.795 1.953-2.36 3.289-1.349 3.18-2.57 6.5-3.07 9.93-.486 3.337-.402 6.846-.44 10.214-.074 6.792.889 13.297 3.326 19.66 2.208 5.756 5.667 9.726 10.402 13.57-4.947 2.464-9.895 4.93-14.843 7.389-4.458 2.215-8.964 4.127-12.775 7.4-1.569 1.348-3.425 2.904-4.545 4.661-1.4 2.196.57 3.69 2.658 2.81 2.298-.965 4.555-2.07 6.805-3.142 1.463-.698 3.906-2.568 5.614-2.521 1.45.04 2.001 1.895 2.647 2.923 1.034 1.644 2.323 3.109 3.744 4.43 2.986 2.778 6.487 5.101 10.17 6.85 4.062 1.932 8.449 3.251 12.882 4.007 2.419.412 4.877.63 7.329.537 1.84-.069 4.446-.074 6.107-.972 1.43-.774.68-2.351 2.095-3.16 1.49-.85 3.067-1.485 4.507-2.44 3.401-2.258 5.42-5.168 7.416-8.649 4.583-7.993 10.157-15.298 13.586-23.914.802-2.018 1.26-4.398 2.573-6.16.873-1.17 2.823-1.867 3.035-3.403.206-1.502-.398-4.823-1.89-5.517-1.726-.802-4.233-.814-6.106-1.25-8.136-1.9-15.727-4.792-22.212-10.176a50.55 50.55 0 0 1-4.59-4.34c-1.069-1.14-2.189-2.338-1.97-4.019.194-1.494 1.392-3.675-1.053-4.094-1.64-.281-3.895.583-5.232 1.427-4.006 2.53-7.728 5.572-11.553 8.365-3.488 2.547-7.061 4.972-10.606 7.44.414-5.863.223-12.217 1.618-17.94.682-2.795 1.872-5.431 3.401-7.862 1.572-2.495 2.825-5.108 4.199-7.71.764.444 2.014 1.436 2.713.254.435-.734.13-2.666-.034-3.51-1.217-6.258-10.72-6.885-15.62-7.953-7.048-1.535-14.087-3.392-21.22-4.482-3.295-.504-6.562-.604-9.877-.806-1.701-.104-3.684-.482-4.713 1.239-.276.462-.316 1.12-.617 1.488-.276.338-1.08.667-1.459.972-1.098.883-2.046 1.923-2.876 3.058-1.7 2.323-2.865 4.996-3.99 7.629-2.715 6.35-4.257 13.09-5.104 19.925-.868 7.006-1.472 14.25-1.01 21.307.849 12.99 4.379 26.205 12.561 36.587 1.876 2.38 3.92 4.329 6.624 5.747 1.308.686 5.42 3.115 6.159.712M146.724 531.62c-2.536 3.26-5.068 6.488-7.334 9.949-2.28 3.483-4.453 7.034-6.684 10.549-2.151 3.39-4.554 6.682-6.284 10.318-.998 2.097-1.095 4.857-3.836 4.759-3.622-.129-7.053-1.163-10.245-2.857-6.396-3.395-11.72-8.164-15.714-14.213-4.145-6.275-7.362-13.376-9.692-20.515-1.289-3.948-2.322-7.973-3.255-12.018-.475-2.052-.982-4.062-.986-6.18-.003-.932.055-1.863.103-2.793.045-.88.535-2.415-.477-2.475.303-2.53.592-5.058.902-7.585.225-1.827.21-3.875 1.811-5.083 1.532-1.158 3.935-1.197 5.74-1.605a89.516 89.516 0 0 0 6.487-1.728c4.39-1.346 8.593-3.13 12.831-4.886 2.115-.876 4.232-1.769 6.23-2.9 1.649-.935 2.832-2.292 3.95-.16 7.616 14.514 13.18 30.781 25.648 42.043 1.009.91 2.659 2.004 2.807 3.496.143 1.453-1.197 2.847-2.002 3.883m10.026-10.259c-1.758-1.205-3.939-1.406-5.722-2.519-2.044-1.276-3.397-4.153-4.793-6.07-3.332-4.572-6.67-9.173-9.68-13.969-2.73-4.354-5.203-8.867-7.827-13.286-1.344-2.263-2.732-4.469-4.316-6.573-.674-.895-2.384-4.058-3.825-3.94-.744.06-.714.784-1.247 1.017-.275.12-.494-.012-.762.025-.965.129-2.013-.36-3.007-.263-2.456.239-5.115 1.27-7.479 1.983-4.934 1.489-9.64 3.583-14.491 5.303-4.81 1.706-10.18 2.844-14.576 5.498-1.736 1.048-3.696 2.725-1.991 4.777.646.778 1.533.757 2.205 1.36.568.51.315.79.184 1.638l-.227 1.454c-.787 5.094-1.43 10.197-.72 15.341a109.71 109.71 0 0 0 3.263 15.413c4.598 15.911 12.823 34.795 28.985 41.847 3.778 1.647 7.536 2.037 11.522 2.78 1.253.234 2.315.802 3.429 1.378.533.275 1.192.72 1.833.64 1.463-.18 2.666-2.452 3.349-3.527 2.58-4.06 5.508-7.87 8.187-11.863 1.386-2.068 2.699-4.186 3.834-6.402.658-1.282 1.129-3.407 2.937-3.359l1.462 1.952c.196.261.49.551.633.843.36.735.331.19.182 1.06-.192 1.11-1.221 2.367-1.87 3.243-3.804 5.14-7.774 10.156-11.761 15.153-3.917 4.912-7.918 9.788-12.508 14.092-.929.87-1.946 1.933-3.304 1.991-1.254.054-2.628-.603-3.873-.82-2.706-.473-5.485-.581-8.123-1.4-1.372-.424-2.52-1.218-3.94-1.314-1.195-.083-2.01-.281-3.057-.915-2.37-1.431-5.006-2.217-7.403-3.57-4.943-2.788-10.05-6.93-13.674-11.302-1.878-2.266-3.418-4.79-5.261-7.084-1.79-2.23-3.559-4.447-5.125-6.847-3.088-4.732-5.546-9.86-7.414-15.193-.971-2.77-1.455-4.58.96-6.606 2.277-1.91 4.409-3.908 6.306-6.206.922-1.12 5.132-5.228 1.98-5.911-1.707-.371-4.433 2.51-5.716 3.553-2.09 1.701-4.138 3.35-6.555 4.567-2.736 1.378-2.787 3.128-2.747 5.99.082 5.902 2.28 11.653 4.557 17.04a79.189 79.189 0 0 0 8.821 15.66c6.752 9.307 14.303 18.749 24.799 23.954 5.189 2.575 10.794 3.933 16.528 4.575 3.386.38 6.799.512 10.203.428 1.437-.036 3.315.159 4.693-.278 1.118-.355 1.301-1.184 1.705-2.198.959-2.402 3.465-3.706 5.16-5.63 1.81-2.057 3.707-3.994 5.568-6.001 4.032-4.348 6.701-9.465 9.94-14.375 1.793-2.716 3.732-5.374 5.296-8.233 1.265-2.317 2.838-5.362 3.28-7.975.35-2.085-1.328-5.115-3.635-5.23-1.203-.06-2.167.886-3.352 1.008-.356.038-1.57-.206-1.995-.032 1.938-4.708 4.811-9.14 6.99-13.722a152.287 152.287 0 0 0 3.473-7.848c.822-2.016 1.8-3.67-.288-5.102M187.759 148.671c3.808 8.793 9.93 15.802 16.182 22.939 3.308 3.777 6.685 7.207 10.696 10.246 3.734 2.828 7.513 5.589 11.283 8.367-3.783 4.86-8.033 9.306-12.248 13.8-3.894 4.15-7.775 8.394-10.341 13.521-.266-2.454-1.859-3.97-4.006-4.972-2.68-1.248-5.144-2.606-7.472-4.447-4.94-3.906-9.537-8.561-13.744-13.233-8.557-9.503-14.01-21.455-19.956-32.632 9.868-4.529 19.738-9.058 29.606-13.589m-31.628 22.182c1.445 4.062 3.54 7.973 5.456 11.832 1.888 3.803 3.845 7.583 6.05 11.215 4.192 6.903 9.315 13.336 15.849 18.178 3.198 2.37 6.627 4.732 10.185 6.538 1.682.852 3.527 1.5 5.441 1.392 1.172-.067 2.794-1.105 3.703-.18.93.95.622 2.847 2.577 1.927 1.604-.757 3.015-2.367 4.195-3.63 3.004-3.21 6.02-6.441 8.905-9.76 2.795-3.217 5.09-6.882 7.45-10.42 1.303-1.953 2.592-3.915 3.884-5.876.924-1.403 2.24-2.807 2.171-4.584-.132-3.467-4.384-5.818-6.8-7.746-2.961-2.364-5.711-4.792-8.233-7.62-2.683-3.01-6.062-5.53-8.965-8.356-3.153-3.066-6-6.401-8.825-9.762-2.243-2.669-4.466-5.404-6.366-8.325-1.485-2.284-3.16-4.036-6.129-3.613-2.323.538-4.653.92-6.892 1.774-2.251.858-4.418 1.942-6.606 2.95-3.965 1.825-7.613 4.134-11.465 6.163-.873.46-1.738.935-2.53 1.526-.437.325-.88.953-1.345 1.186-.344.172-.58.133-.975.244-1.561.438-2.15 1.309-2.93 2.679-1.254 2.203-1.219 3.387-.071 5.644 1.049 2.063 1.491 4.448 2.266 6.624M312.492 280.115c.975 1.566 2.572.795 3.484-.32 2.771-3.383 5.364-7.043 7.852-10.644 2.524-3.654 4.838-7.355 7.699-10.766a426.158 426.158 0 0 1 8.819-10.152c11.156-12.48 22.605-24.665 34.907-36.003 3.007-2.771 6.1-5.478 8.988-8.376 1.27-1.275 2.817-2.69 2.757-4.65-.042-1.399-.954-3.123-2.174-3.837-2.653-1.55-6.214 2.227-8.185 3.638-1.508 1.079-3.125 1.644-4.708 2.53-1.385.777-2.77 1.964-3.957 3.023-2.668 2.382-5.542 4.406-8.247 6.725-2.746 2.355-5.173 5.058-7.87 7.468-2.884 2.577-5.978 4.794-8.546 7.73-2.532 2.895-4.67 6.055-7.312 8.873-5.625 5.997-10.225 12.774-15.022 19.442-2.31 3.21-4.055 6.726-5.616 10.358a248.486 248.486 0 0 0-2.284 5.451c-.698 1.732-.756 3.5-1.077 5.317.404.08.102 3.569.492 4.193M935.423 575.729c2.113-2.54 3.5-6.201 5.035-9.131 1.645-3.137 3.17-6.04 3.866-9.533.335-1.681.536-3.4.92-5.069.361-1.578.878-2.97.73-4.619-.155-1.72-.956-6.906-3.57-6.333-1.52.333-1.215 1.959-1.224 3.147-.011 1.362-.17 2.718-.57 4.022-.948 3.084-3.202 5.802-4.63 8.695-1.384 2.804-3.023 5.357-5.092 7.706a55.473 55.473 0 0 1-6.553 6.303c-2.052 1.67-4.527 2.933-6.021 5.198-.893 1.357-1.042 1.763-2.684 2.137-1.325.302-2.57.786-3.684 1.577-1.263.896-2.16 1.968-3.639 2.557-1.286.513-2.648.819-3.946 1.299-3.002 1.108-5.623 2.978-8.69 4.026-1.588.543-3.2 1.007-4.806 1.488-1.285.384-2.804.648-3.867 1.516-1.785 1.46-2.785 5.183-1.224 7.122 1.68 2.087 5.623.782 7.81.515.028.142.054.285.084.427 3.682-.74 7.52-1.177 11.1-2.333 1.89-.612 3.59-1.485 5.319-2.45 1.464-.817 2.964-1.106 4.53-1.628 1.4-.465 2.518-1.173 3.393-2.381 1.118-1.55 1.592-1.088 3.318-1.466 2.772-.609 5.162-3.756 7.069-5.658 2.324-2.318 4.929-4.612 7.026-7.134M405.68 128.495c.647-1.024 2.903-1.951 3.94-2.618a94.792 94.792 0 0 0 4.177-2.824c2.475-1.787 4.204-.613 6.4 1.099 4.058 3.16 5.687 7.405 5.383 12.427.01 1.467.405 3.922-.725 5.13-1.08 1.158-3.323 1.044-4.75 1.332-1.868.379-4.689 2.742-6.623 2.05-1.696-.606-1.096-3.632-1.231-4.967-.256-2.493-1.086-4.906-2.618-6.92-.788-1.035-1.732-1.915-2.77-2.702-.793-.603-1.826-.99-1.183-2.007m-7.438 7.545c1.598.278 3.246 1.019 4.74 1.625.991.402 2.29.779 3.09 1.492 1.33 1.184.79 3.051 1.1 4.6.18.906.645 1.686.31 2.627-.39 1.1-1.542 1.288-2.126 2.18-.854 1.307.248 3.154.986 4.269 1.28 1.935 3.027.947 5.001.657 2.378-.35 4.821-.446 7.111-1.23 1.866-.636 3.414-2.113 4.837-.149.53.734.64 1.381 1.685 1.106.869-.229 1.752-1.408 2.349-2.012 3.717-3.764 4.603-8.906 4.664-13.98.178-5.633-1.869-10.926-4.77-15.687-.61-1-1.223-2.156-2.134-2.933-.887-.757-2.042-1.055-2.844-1.939-1.197-1.318-.92-3.987-2.829-4.568-2.01-.614-4.901 1.824-6.423 2.868-2.144 1.469-4.24 3-6.303 4.575-4.42 3.373-9.357 6.603-13.353 10.46-1.355 1.309-1.908 3.62-.549 5.183 1.36 1.563 3.687.547 5.458.856M635.397 539.654c-.858-4.98-2.303-9.845-4.057-14.588-1.136-3.072-3.728-8.358-7.868-8.053-4.558.334-1.503 6.283-1.06 8.648 1.731 9.255 2.732 18.722 4.1 28.039.678 4.616.883 9.374 1.974 13.919a37.027 37.027 0 0 0 2.223 6.459c.594 1.328 1.306 3.055 2.793 3.654 4.515 1.817 3.387-6.123 3.337-8.08-.123-4.792-.235-9.583-.353-14.376l.37-.01c-.686-5.194-.567-10.429-1.46-15.612M98.497 626.81c-2.471.807-4.307 1.428-6.996 1.173-1.885-.18-3.75-.563-5.628-.8-4.843-.607-9.46-1.944-13.905-3.94-4.041-1.812-7.825-3.997-11.161-6.907-1.76-1.534-3.35-3.22-4.914-4.947-1.12-1.238-2.154-2.17-3.893-2.389.451 2.502.979 4.474 2.174 6.748a39.528 39.528 0 0 0 3.289 5.23 38.095 38.095 0 0 0 9.108 8.69c3.435 2.337 7.299 4.323 11.35 5.347 2.26.572 4.575.913 6.864 1.352 2.162.414 4.264.832 6.471.53 2.065-.284 4.242-.356 6.253-.938 1.588-.46 3.102-1.47 4.457-2.392 2.64-1.798-.237-7.81-3.469-6.756M179.143 63.844c1.228 1.646 3.865 1.655 3.69-.757-.158-2.211-.845-4.39-1.56-6.474-.727-2.127-.577-4.438-.62-6.659-.098-5.303.075-10.44 1.072-15.66.956-5.025 1.574-10.11 2.694-15.104 1.098-4.901 3.088-9.535 4.313-14.395.38-1.5.684-4.43-1.513-4.773-1.827-.285-2.904 2.276-3.612 3.486-2.355 4.024-4.894 7.745-6.291 12.23-1.26 4.039-2.211 8.34-2.653 12.547-.462 4.41-.393 8.866-.663 13.289.441 5.14.207 10.828 1.858 15.754.745 2.218 1.875 4.626 3.285 6.516M868.655 84.046c-.723 1.735-1.075 3.584.166 5.214 2.253 2.957 5.842-3.777 6.387-5.657 1.184-4.075 2.668-8.18 3.33-12.389.677-4.292.355-8.907.462-13.258-.359-5.164-1.046-10.314-2.278-15.337-1.09-4.455-2.249-9.047-5.877-12.05-1.51-1.248-2.545-.304-2.758 1.478-.231 1.955.28 3.922 1.17 5.635 2.087 4.017 2.091 8.714 2.558 13.205.929 8.931 1.45 17.835-.889 26.597-.6 2.247-1.379 4.423-2.27 6.562M102.582 183.827c.518 2.027 1.103 4.04 1.85 5.996 1.445 3.779 3.313 7.41 5.62 10.735 3.808 5.488 8.833 10.7 14.287 14.56 1.798 1.273 6.163 4.83 8.5 3.643 3.196-1.624-1.081-5.768-2.38-6.981-2.677-2.505-5.34-5.03-8.048-7.503-2.716-2.481-4.737-5.66-6.88-8.632-2.157-2.994-4.266-6.025-6.22-9.157-1.012-1.62-1.893-3.32-2.898-4.94-.869-1.403-2.157-2.365-3.276-3.548-.932.444-.823.366-1.063 1.211-.103.363-.07 1.1-.064 1.434.018 1.097.302 2.125.572 3.182M332.034 215.273c5.07-4.592 10.634-8.472 15.421-13.364 1.205-1.231 2.137-1.95 1.093-3.633-.697-1.123-1.496-1.692-2.673-.923-2.812 1.84-5.491 4.1-8.201 6.106-5.557 4.113-11.319 8.146-16.335 12.916-2.27 2.16-4.37 4.51-6.132 7.101-1.85 2.719-3.248 5.477-1.18 8.524 6.027-5.545 11.94-11.232 18.007-16.727M949.599 593.032c-.908-.223-2.417.78-3.092 1.225-2.689 1.771-5.472 3.408-8.191 5.135a367.184 367.184 0 0 0-7.442 4.88c-2.053 1.373-3.399 3.062-2.677 5.657.33 1.184.876 2.57 2.25 1.892 1.009-.498 1.879-.982 2.988-1.374 2.79-.988 5.605-1.964 8.211-3.39 4.517-2.467 8.748-6.288 9.354-11.67-.336-.762-.432-2.117-1.401-2.355M652.715 538.733c-.918-1.002-2.044-1.014-2.53.197-.5 1.247.15 3.386.328 4.623.29 2.027.61 4.049.956 6.067.202 1.178.35 4.84 1.897 5.316 2.372.732 2.43-5.04 2.634-6.214-.5-1.705-.974-3.416-1.478-5.119-.442-1.494-.711-3.672-1.807-4.87M253.205 439.187c-.568-.27-1.006-.153-1.126-.829-.1-.561.781-1.146.915-1.627.483-1.734-1.335.09-1.994.637.624 1.546 1.247 3.09 1.876 4.632.708-.28 1.417-.565 2.124-.85-.677-.917-.762-1.471-1.795-1.963M585.023 159.861c1.182.39 2.913-.14 4.118-.363 1.162-.216 2.471-.539 3.58-.971 2.736-1.068 2.503-4.305 1.872-6.82-.852-3.39-2.748-6.635-4.25-9.763a123.613 123.613 0 0 1-2.244-4.918c-.536-1.255-1.243-3.014-2.868-3.026-2.506-.02-1.283 5.883-.885 7.287 1.002 3.539 2.863 6.8 4.417 10.103-1.436.098-2.913.287-4.346.238-1.488-.05-1.894-.004-2.258 1.658-.524 2.405.199 5.699 2.864 6.575M562.443 130.937c2.699-.167 5.388-.45 8.073-.752 1.976-.223 5.424-.262 6.828-1.825.963-1.072.797-3.354-.05-4.465-1-1.312-3.336-.795-4.821-.795-4.01.067-7.899.6-11.485 2.368-1.316.65-4.28 1.83-3.965 3.599.395 2.222 3.768 1.971 5.42 1.87M573.684 162.065c-2.253.565-1.82 4.425-1.125 6.048 1.041 2.43 3.665 3.02 6.075 2.864 1.16-.116 2.826.3 3.694-.67.857-.958.846-2.88.24-3.954-.804-1.42-2.886-2.224-4.295-2.884-1.226-.574-3.176-1.758-4.589-1.404M573.984 141.998c.176-2.21-1.089-7.179-4.039-6.993-1.402.088-2.596 2.095-2.853 3.328-.269 1.3.104 2.614.445 3.858.559 2.052 1.398 5.219 4.127 4.765 2.263-.378 2.325-3.084 2.32-4.958M595.26 144.225c.596.58 1.467 1.012 2.304.631 1.234-.56 1.022-2.04 1.022-3.095l.414-.134c-.7-1.941-1.159-6.006-3.591-6.556-2.387-.54-2.562 2.112-2.336 3.766.239 1.765.843 4.08 2.186 5.388M520.235 144c-1.185.119-3.416.669-3.223 2.156.28 2.167 3.678 2.554 5.404 2.764 2.043.25 4.877.043 2.933-2.519-1.184-1.56-3.164-2.25-5.114-2.401M420.274 403 419 405.49l3.999.51c.057-1.057-2.016-2.368-2.725-3"
                    />
                  </g>
                </g>
              </svg>

              <h1 className="mt-6 text-2xl font-bold tracking-tight text-gray-900 sm:text-4xl">Uh-oh!</h1>

              <p className="mt-4 text-gray-500">{`Something broke. We'll try find a fix soon!`}</p>
            </div>
          </div>
        )}

      </div>
    </div>
  );
}

