#include "Bsp.h"
#include "util.h"
#include <algorithm>
#include "globals.h"
#include "Mdl.h"
#include <string.h>

const int NUM_LUMPS_TO_LOAD = 2;
int lumps_to_load[NUM_LUMPS_TO_LOAD] = {LUMP_ENTITIES, LUMP_TEXTURES};

Bsp::Bsp(std::string mapname)
{
	this->name = mapname;
	valid = false;

	bool exists = true;
	string fname = "maps/" + mapname + ".bsp";
	if (!contentExists(fname, true, full_path)) {
		log("ERROR: " + fname + " not found\n");
		return;
	}

	if (!load_lumps(fname)) {
		log(fname + " is not a valid BSP file\n");
		return;
	}

	valid = true;
	path = fname.substr(0, fname.find_last_of("/\\")+1);
}

Bsp::~Bsp()
{
	if (!valid)
		return;
	 
	for (int i = 0; i < NUM_LUMPS_TO_LOAD; i++)
		if (lumps[lumps_to_load[i]])
			delete [] lumps[lumps_to_load[i]];
	delete [] lumps;

	for (int i = 0; i < ents.size(); i++)
		delete ents[i];
}

set_icase Bsp::get_resources()
{
	set_icase resources;

	load_ents();

	push_unique(server_files, "maps/" + name + ".bsp");

	string bsp_fname = name + ".bsp";

	Entity * worldSpawn = NULL;

	for (int i = 0; i < ents.size(); i++)
	{
		string cname = toLowerCase(ents[i]->keyvalues["classname"]);
		string tname = ents[i]->keyvalues["targetname"];
		string ent_trace = bsp_fname + " --> " +  "\"" + tname + "\" (" + cname + ")";

		if (cname == "game_text") // possible some text is interpreted as a file extension.
			continue;
		if (cname == "worldspawn") // deal with this later. It has weird values too.
		{
			worldSpawn = ents[i];
			continue;
		}
		for (hashmap::iterator it = ents[i]->keyvalues.begin(); it != ents[i]->keyvalues.end(); ++it)
		{
			string key = toLowerCase(it->first);
			string val = it->second;

			// support for weapon_custom
			vector<string> vals = splitString(val, ";");
			if (vals.size() > 1)
			{
				val = val[0];
			}
			else
			{
				val.erase(std::remove(val.begin(), val.end(), ';'), val.end());
				vals.clear();
				vals.push_back(val);
			}

			bool isWeaponCustom = cname == "weapon_custom";
			int iext = val.find_last_of(".");
			if (!((isWeaponCustom && key == "sprite_directory") || key == "customspritedir"))
			{
				// no extension in value - probably not a file path (unless it's a HUD sprite folder name!)
				if (iext == string::npos || iext == val.length()-1)
					continue;
			}

			iext++;

			string ext = toLowerCase(val.substr(iext, val.length() - iext));

			if (key == "m_iszscriptfile") // trigger_script
			{
				string script_path = normalize_path("scripts/maps/" + val, true);

				// add extension if needed
				if (toLowerCase(script_path).find(".as") != script_path.length()-3)
					script_path += ".as";
				
				add_script_resources(script_path, resources, ent_trace);
			}
			else if (ext == "mdl") 
			{
				add_model_resources(normalize_path(val, true), resources, ent_trace);
			}
			else if (ext == "spr")
			{
				string spr = val;
				if (key == "item_icon") // item_inventory
					spr = "sprites/" + spr;
				spr = normalize_path(spr, true);
				trace_missing_file(spr, ent_trace, true);
				push_unique(resources, spr);
			}
			else if (key == "soundlist" && val.length())
			{
				// make sure file has an extension
				int lastdot = val.find_last_of(".");
				int lastslash = val.find_last_of("/\\");
				if (lastdot == string::npos || (lastslash != string::npos && lastdot < lastslash))
					continue;

				string res = normalize_path("sound/" + name + "/" + val, true);
				add_replacement_file_resources(res, resources, ent_trace, false);
			}
			else if ((isWeaponCustom && key == "sprite_directory") || 
					 (cname.find("weapon_") == 0 && key == "customspritedir"))
			{
				// Note: Code duplicated in util.cpp
				string wep_name = isWeaponCustom ? ents[i]->keyvalues["weapon_name"] : cname;
				string hud_file = normalize_path("sprites/" + val + "/" + wep_name + ".txt", true);
				trace_missing_file(hud_file, ent_trace, true);
				push_unique(resources, hud_file);

				string hud_path = hud_file;
				if (contentExists(hud_path, true))
				{
					ifstream file(hud_path);
					if (file.is_open())
					{
						int lineNum = 0;
						bool in_func_body = false;
						string line;
						while (getline(file, line))
						{
							lineNum++;

							// strip comments
							size_t cpos = line.find("//");
							if (cpos != string::npos)
								line = line.substr(0, cpos);

							line = trimSpaces(line);

							line = replaceChar(line, '\t', ' ');
							vector<string> parts = splitString(line, " ");

							if (parts.size() < 3)
								continue;
							
							string spr = "sprites/" + parts[2] + ".spr";
							trace_missing_file(spr, ent_trace + " --> " + hud_file, true);
							push_unique(resources, spr);
						}
					}
					file.close();
				}
			}
			else
			{
				for (int v = 0; v < vals.size(); v++)
				{
					string sval = vals[v];
					string sext = get_ext(sval);
					for (int s = 0; s < NUM_SOUND_EXTS; s++)
					{
						if (sext == g_valid_exts[s])
						{
							string prefix = "sound/";
							bool isSentence = key == "usesentence" || key == "unusesentence" || key == "sentence";
							if (isSentence && sval[0] == '+') // means string is a file path, not a sentence
								sval = sval.substr(1);
							if (!isSentence && ((val[0] == '!') || (val[0] == '*'))) // ! = sentence, * = file stream
								sval = sval.substr(1);

							string snd = normalize_path(prefix + sval, true);
							trace_missing_file(snd, ent_trace, true);
							push_unique(resources, snd);
							break;
						}
					}
				}
			}
		}
	}

	if (worldSpawn) 
	{
		set_icase map_textures = get_textures();

		// find wads (TODO: Only include them if the textures are actually used)
		string wadList = worldSpawn->keyvalues["wad"];
		string trace = bsp_fname + " --> \"Map Properties\" (worldspawn)";
		vector<string> wadlist = splitString(wadList, ";");
		vector<string> needed_wads;
		vector<string> missing_wads;
		for (int i = 0; i < wadlist.size(); i++)
		{
			string wadname = wadlist[i];
			int idir = wadname.find_last_of("\\/");
			if (idir != string::npos && idir < wadname.length()-5) { // need at least 5 chars for a WAD name ("a.wad") 
				wadname = wadname.substr(idir+1);
			}

			string wadpath = wadname;
			set<string>& wadTex = default_wads[toLowerCase(wadname)];
			if (wadTex.size())
			{
				bool wad_is_used = false;
				for (set_icase::iterator iter = map_textures.begin(); iter != map_textures.end();)
				{
					if (wadTex.find(*iter) != wadTex.end())
					{
						wad_is_used = true;
						map_textures.erase(iter++);
					}
					else
						iter++;
				}
				if (wad_is_used)
					needed_wads.push_back(wadname);
				else if (print_skip)
				{
					if (unused_wads++ == 0) log("\n");
					log("Unused WAD: " + wadname + "\n");
				}
			}
			else if (contentExists(wadpath, true))
			{
				bool wad_is_used = false;
				Wad wad(wadpath);
				wad.readInfo();
				if (wad.dirEntries)
				{
					for (int k = 0; k < wad.header.nDir; k++)
					{
						string texName = toLowerCase(wad.dirEntries[k].szName);
						set_icase::iterator iter = map_textures.find(texName);
						if (iter != map_textures.end())
						{
							map_textures.erase(iter);
							wad_is_used = true;
						}
					}
				}
				if (wad_is_used)
					needed_wads.push_back(wadname);
				else if (print_skip)
				{
					if (unused_wads++ == 0) log("\n");
					log("Unused WAD: " + wadname + "\n");
				}
			}
			else
				missing_wads.push_back(wadname);
		}
		for (int i = 0; i < needed_wads.size(); i++)
		{
			trace_missing_file(needed_wads[i], bsp_fname + " --> worldspawn wadlist", true);
			push_unique(resources, needed_wads[i]);
		}
		for (int i = 0; i < missing_wads.size(); i++)
		{
			if (map_textures.size())
			{
				trace_missing_file(missing_wads[i], bsp_fname + " --> worldspawn wadlist", true);
				push_unique(resources, missing_wads[i]);
			}
			else if (print_skip)
			{
				// if all map textures are accounted for in the existing wads, then this missing one must not be used.
				if (unused_wads++ == 0) log("\n");
				log("Unused WAD: " + missing_wads[i] + "\n");
			}
		}
		if (map_textures.size())
			log("ERROR: " + to_string(map_textures.size()) + " missing textures\n");

		// find sky
		string sky = worldSpawn->keyvalues["skyname"];
		if (sky.length()) 
		{
			trace_missing_file("gfx/env/" + sky + "bk.tga", trace, true);
			trace_missing_file("gfx/env/" + sky + "dn.tga", trace, true);
			trace_missing_file("gfx/env/" + sky + "ft.tga", trace, true);
			trace_missing_file("gfx/env/" + sky + "lf.tga", trace, true);
			trace_missing_file("gfx/env/" + sky + "rt.tga", trace, true);
			trace_missing_file("gfx/env/" + sky + "up.tga", trace, true);
			push_unique(resources, "gfx/env/" + sky + "bk.tga");
			push_unique(resources, "gfx/env/" + sky + "dn.tga");
			push_unique(resources, "gfx/env/" + sky + "ft.tga");
			push_unique(resources, "gfx/env/" + sky + "lf.tga");
			push_unique(resources, "gfx/env/" + sky + "rt.tga");
			push_unique(resources, "gfx/env/" + sky + "up.tga");
		}

		// parse replacement files (Note: These can also be set in the map CFG)
		string global_model_list = worldSpawn->keyvalues["globalmodellist"];
		if (global_model_list.length())
		{			
			global_model_list = normalize_path("models/" + name + "/" + global_model_list, true);
			add_replacement_file_resources(global_model_list, resources, trace, true);
		}

		string global_sound_list = worldSpawn->keyvalues["globalsoundlist"];
		if (global_sound_list.length())
		{			
			global_sound_list = normalize_path("sound/" + name + "/" + global_sound_list, true);
			add_replacement_file_resources(global_sound_list, resources, trace, false);
		}

		string forced_player_models = worldSpawn->keyvalues["forcepmodels"];
		if (forced_player_models.length()) 
			add_force_pmodels_resources(forced_player_models, resources, trace);

		string sentences_file = normalize_path(worldSpawn->keyvalues["sentence_file"], true);
		if (sentences_file.length())
			add_sentence_file_resources(sentences_file, resources, trace);

		string materials_file = worldSpawn->keyvalues["materials_file"];
		if (materials_file.length()) 
		{
			materials_file = normalize_path("sound/" + name + "/" + materials_file, true);
			trace_missing_file(materials_file, trace, true);
			push_unique(resources, materials_file);
		}
	}

	return resources;
}

set_icase Bsp::get_textures()
{
	set_icase tex_names;

	byte * textures = lumps[LUMP_TEXTURES];
	int num_textures = ((int*)textures)[0];
	int lump_len = header.lump[LUMP_TEXTURES].nLength;

	for (int i = 0; i < num_textures; i++)
	{
		int offset = ((int*)textures)[i + 1];
		if (offset + sizeof(BSPMIPTEX) > lump_len)
		{
			log("ERROR: Invalid texture lump offset: " + to_string(offset) + "\n");
			continue;
		}
		BSPMIPTEX * t = (BSPMIPTEX*)&textures[offset];
		bool inWad = t->nOffsets[0] == 0;

		if (inWad)
		{
			string name = t->szName;
			tex_names.insert(toLowerCase(name));
			//cout << "GOT WAD TEX: " << name << endl;
		}
	}
	return tex_names;
}

bool Bsp::load_lumps(string fname)
{
	bool valid = true;

	// Read all BSP Data
	ifstream fin(fname, ios::binary | ios::ate);
	int size = fin.tellg();
	fin.seekg(0, fin.beg);

	if (size < sizeof(BSPHEADER) + sizeof(BSPLUMP)*HEADER_LUMPS)
		return false;

	fin.read((char*)&header.nVersion, sizeof(int));
	
	for (int i = 0; i < HEADER_LUMPS; i++)
		fin.read((char*)&header.lump[i], sizeof(BSPLUMP));

	lumps = new byte*[HEADER_LUMPS];
	memset(lumps, 0, sizeof(byte*)*HEADER_LUMPS);
	
	for (int i = 0; i < NUM_LUMPS_TO_LOAD; i++)
	{
		int idx = lumps_to_load[i];
		fin.seekg(header.lump[idx].nOffset);
		if (fin.eof()) {
			log("FAILED TO READ BSP LUMP " + to_string(idx) + "\n");
			valid = false;
		}
		else
		{
			lumps[idx] = new byte[header.lump[idx].nLength];
			fin.read((char*)lumps[idx], header.lump[idx].nLength);
		}
	}	
	
	fin.close();

	return valid;
}

void Bsp::load_ents()
{
	bool verbose = true;
	membuf sbuf((char*)lumps[LUMP_ENTITIES], header.lump[LUMP_ENTITIES].nLength);
	istream in(&sbuf);

	int lineNum = 0;
	int lastBracket = -1;
	Entity* ent = NULL;

	string line = "";
	while (getline(in, line))
	{
		lineNum++;
		if (line.length() < 1 || line[0] == '\n')
			continue;

		if (line[0] == '{')
		{
			if (lastBracket == 0)
			{
				log(name + ".bsp ent data (line " + to_string(lineNum) + "): Unexpected '{'\n");
				continue;
			}
			lastBracket = 0;

			if (ent != NULL)
				delete ent;
			ent = new Entity();
		}
		else if (line[0] == '}')
		{
			if (lastBracket == 1)
				log(name + ".bsp ent data (line " + to_string(lineNum) + "): Unexpected '}'\n");
			lastBracket = 1;

			if (ent == NULL)
				continue;

			ents.push_back(ent);
			ent = NULL;

			// you can end/start an ent on the same line, you know
			if (line.find("{") != string::npos)
			{
				ent = new Entity();
				lastBracket = 0;
			}
		}
		else if (lastBracket == 0 && ent != NULL) // currently defining an entity
		{
			Keyvalue k(line);
			if (k.key.length() && k.value.length())
				ent->addKeyvalue(k);
		}
	}	
	//cout << "got " << ents.size() <<  " entities\n";

	if (ent != NULL)
		delete ent;
}