#!/bin/bash

set -e

# This script runs Jekyll to translate our Markdown files at "doc/" into static web pages.
# The pages will be updated in "docs/" (note the extra s).
# The changes will go live on www.vitess.io when the commit is merged into the "master" branch.
#
# Run the script with "--docker" to run Jekyll within Docker (recommended).

if [[ -n "$(git status --porcelain)" ]]; then
  echo "ERROR: Your working directory is not clean."
  echo
  echo "Please make sure that all pending changes are comitted or stashed away."
  echo "This is a safe guard to avoid that the 'git add -A' in the 'docs/' directory adds files which were not generated by Jekyll."
  exit 1
fi

# Output directory were the static web pages will be written to.
# Note: The directory is relative to $VTTOP.
website_dir="docs"

# compile to html pages
if [[ "$1" == "--docker" ]]; then
  docker run -ti --rm --name=vitess_publish_site -v $VTTOP:/vttop vitess/publish-site bash -c \
    "cd /vttop/vitess.io && \
     bundle install && \
     bundle exec jekyll build --destination ../$website_dir && \
     chown -R $UID /vttop/vitess.io /vttop/$website_dir"
else
  (cd vitess.io && bundle install && bundle exec jekyll build --destination ../$website_dir)
fi

# Change to docs/ directory.
pushd $website_dir >/dev/null

# Restore README.md because it gets deleted by Jekyll.
git checkout README.md

# pre-commit checks
set +e
list=$(find . -name '*.html' ! -path '*/vendor/*' ! -path '*/web/*' | xargs grep -lE '^\s*([\-\*]|\d\.) ')
if [ -n "$list" ]; then
  echo
  echo "ERROR: The following pages appear to contain bulleted lists that weren't properly converted."
  echo "Make sure all bulleted lists have a blank line before them."
  echo
  echo "$list"
  exit 1
fi
set -e

# Commit new version.
git add -A .
git commit -m "publish site `date`"

echo
echo
echo "A commit was automatically created."
echo
echo "Please sanity-check the output: git diff HEAD~"
echo
echo "When you're ready to publish, create a pull request."

popd >/dev/null
