import semver from 'semver';
import type { Configuration as WebpackConfig, RuleSetRule } from 'webpack';
import type { NextConfig } from 'next';
import path from 'path';
import { getNextjsVersion } from '../utils';

export const configureImages = (baseConfig: WebpackConfig, nextConfig: NextConfig): void => {
  configureStaticImageImport(baseConfig, nextConfig);
  configureImageDefaults(baseConfig);
};

const fallbackFilename = 'static/media/[path][name][ext]';

const configureImageDefaults = (baseConfig: WebpackConfig): void => {
  const version = getNextjsVersion();
  const resolve = baseConfig.resolve ?? {};
  resolve.alias = {
    ...resolve.alias,
    'sb-original/next/image': require.resolve('next/image'),
    'next/image': path.resolve(__dirname, './images/next-image'),
  };

  if (semver.satisfies(version, '>=13.0.0')) {
    resolve.alias = {
      ...resolve.alias,
      'sb-original/next/legacy/image': require.resolve('next/legacy/image'),
      'next/legacy/image': path.resolve(__dirname, './images/next-legacy-image'),
    };
  }
};

const configureStaticImageImport = (baseConfig: WebpackConfig, nextConfig: NextConfig): void => {
  const version = getNextjsVersion();
  if (semver.lt(version, '11.0.0')) return;

  const rules = baseConfig.module?.rules;

  const assetRule = rules?.find(
    (rule) =>
      rule && typeof rule !== 'string' && rule.test instanceof RegExp && rule.test.test('test.jpg')
  ) as RuleSetRule;

  if (!assetRule) {
    return;
  }

  assetRule.test = /\.(apng|eot|otf|ttf|woff|woff2|cur|ani|pdf)(\?.*)?$/;

  rules?.push({
    test: /\.(png|jpg|jpeg|gif|webp|avif|ico|bmp|svg)$/i,
    issuer: { not: /\.(css|scss|sass)$/ },
    use: [
      {
        loader: require.resolve('@storybook/nextjs/next-image-loader-stub.js'),
        options: {
          filename: assetRule.generator?.filename ?? fallbackFilename,
          nextConfig,
        },
      },
    ],
  });
  rules?.push({
    test: /\.(png|jpg|jpeg|gif|webp|avif|ico|bmp|svg)$/i,
    issuer: /\.(css|scss|sass)$/,
    type: 'asset/resource',
    generator: {
      filename: assetRule.generator?.filename ?? fallbackFilename,
    },
  });
};
