/* SPDX-License-Identifier: BSD-3-Clause
 *
 * Copyright(c) 2016 Intel Corporation. All rights reserved.
 *
 * Author: Liam Girdwood <liam.r.girdwood@linux.intel.com>
 *         Keyon Jie <yang.jie@linux.intel.com>
 */

/**
 * \file include/sof/lib/alloc.h
 * \brief Memory Allocation API definition
 * \author Liam Girdwood <liam.r.girdwood@linux.intel.com>
 * \author Keyon Jie <yang.jie@linux.intel.com>
 */

#ifndef __SOF_LIB_ALLOC_H__
#define __SOF_LIB_ALLOC_H__

#include <sof/bit.h>
#include <sof/string.h>
#include <sof/trace/trace.h>
#include <user/trace.h>

#include <stddef.h>
#include <stdint.h>

/** \addtogroup alloc_api Memory Allocation API
 *  @{
 */

/**
 * \brief Heap Memory Zones
 *
 * The heap has three different zones from where memory can be allocated :-
 *
 * 1) System Zone. Fixed size heap where alloc always succeeds and is never
 * freed. Used by any init code that will never give up the memory.
 *
 * 2) System Runtime Zone. Heap zone intended for runtime objects allocated
 * by the kernel part of the code.
 *
 * 3) Runtime Zone. Main and larger heap zone where allocs are not guaranteed to
 * succeed. Memory can be freed here.
 *
 * 4) Buffer Zone. Largest heap zone intended for audio buffers.
 *
 * See platform/memory.h for heap size configuration and mappings.
 */
enum mem_zone {
	SOF_MEM_ZONE_SYS = 0,		/**< System zone */
	SOF_MEM_ZONE_SYS_RUNTIME,	/**< System-runtime zone */
	SOF_MEM_ZONE_RUNTIME,		/**< Runtime zone */
	SOF_MEM_ZONE_BUFFER,		/**< Buffer zone */
};

/** \name Heap zone flags
 *  @{
 */

/** \brief Indicates that allocated memory block must be shareable between
 *	DSP cores.
 */
#define SOF_MEM_FLAG_SHARED	BIT(0)

/** \brief Indicates that original content should not be copied by realloc. */
#define SOF_MEM_FLAG_NO_COPY	BIT(1)

/** @} */

/**
 * Allocates memory block.
 * @param zone Zone to allocate memory from, see enum mem_zone.
 * @param flags Flags, see SOF_MEM_FLAG_...
 * @param caps Capabilities, see SOF_MEM_CAPS_...
 * @param bytes Size in bytes.
 * @return Pointer to the allocated memory or NULL if failed.
 *
 * @note Do not use for buffers (SOF_MEM_ZONE_BUFFER zone).
 *       Use rballoc(), rballoc_align() to allocate memory for buffers.
 */
void *rmalloc(enum mem_zone zone, uint32_t flags, uint32_t caps, size_t bytes);

/**
 * Similar to rmalloc(), guarantees that returned block is zeroed.
 *
 * @note Do not use  for buffers (SOF_MEM_ZONE_BUFFER zone).
 *       rballoc(), rballoc_align() to allocate memory for buffers.
 */
void *rzalloc(enum mem_zone zone, uint32_t flags, uint32_t caps, size_t bytes);

/**
 * Allocates memory block from SOF_MEM_ZONE_BUFFER.
 * @param flags Flags, see SOF_MEM_FLAG_...
 * @param caps Capabilities, see SOF_MEM_CAPS_...
 * @param bytes Size in bytes.
 * @param alignment Alignment in bytes.
 * @return Pointer to the allocated memory or NULL if failed.
 */
void *rballoc_align(uint32_t flags, uint32_t caps, size_t bytes,
		    uint32_t alignment);

/**
 * Similar to rballoc_align(), returns buffer aligned to PLATFORM_DCACHE_ALIGN.
 */
static inline void *rballoc(uint32_t flags, uint32_t caps, size_t bytes)
{
	return rballoc_align(flags, caps, bytes, PLATFORM_DCACHE_ALIGN);
}

/**
 * Changes size of the memory block allocated from SOF_MEM_ZONE_BUFFER.
 * @param ptr Address of the block to resize.
 * @param flags Flags, see SOF_MEM_FLAG_...
 * @param caps Capabilities, see SOF_MEM_CAPS_...
 * @param bytes New size in bytes.
 * @param old_bytes Old size in bytes.
 * @param alignment Alignment in bytes.
 * @return Pointer to the resized memory of NULL if failed.
 */
void *rbrealloc_align(void *ptr, uint32_t flags, uint32_t caps, size_t bytes,
		      size_t old_bytes, uint32_t alignment);

/**
 * Similar to rballoc_align(), returns resized buffer aligned to
 * PLATFORM_DCACHE_ALIGN.
 */
static inline void *rbrealloc(void *ptr, uint32_t flags, uint32_t caps,
			      size_t bytes, size_t old_bytes)
{
	return rbrealloc_align(ptr, flags, caps, bytes, old_bytes,
			       PLATFORM_DCACHE_ALIGN);
}

/**
 * Frees the memory block.
 * @param ptr Pointer to the memory block.
 *
 * @note Blocks from SOF_MEM_ZONE_SYS cannot be freed, such a call causes
 *       panic.
 */
void rfree(void *ptr);

/**
 * Allocates memory block from the system heap reserved for the specified core.
 * @param core Core id.
 * @param bytes Size in bytes.
 */
void *rzalloc_core_sys(int core, size_t bytes);

/** \brief Zeroes memory block.
 * @param ptr Pointer to the memory block.
 * @param size Size of the block in bytes.
 */
#define bzero(ptr, size) \
	arch_bzero(ptr, size)

/**
 * Calculates length of the null-terminated string.
 * @param s String.
 * @return Length of the string in bytes.
 */
int rstrlen(const char *s);

/**
 * Compares two strings, see man strcmp.
 * @param s1 First string to compare.
 * @param s2 Second string to compare.
 * @return See man strcmp.
 */
int rstrcmp(const char *s1, const char *s2);

/** @}*/

#endif /* __SOF_LIB_ALLOC_H__ */
