// Copyright 2014 Google Inc. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.rules.cpp;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.devtools.build.lib.actions.Action;
import com.google.devtools.build.lib.actions.Artifact;
import com.google.devtools.build.lib.analysis.AnalysisEnvironment;
import com.google.devtools.build.lib.analysis.AnalysisUtils;
import com.google.devtools.build.lib.analysis.CompilationPrerequisitesProvider;
import com.google.devtools.build.lib.analysis.FileProvider;
import com.google.devtools.build.lib.analysis.FilesToCompileProvider;
import com.google.devtools.build.lib.analysis.LanguageDependentFragment;
import com.google.devtools.build.lib.analysis.RuleConfiguredTarget.Mode;
import com.google.devtools.build.lib.analysis.RuleConfiguredTargetBuilder;
import com.google.devtools.build.lib.analysis.RuleContext;
import com.google.devtools.build.lib.analysis.TempsProvider;
import com.google.devtools.build.lib.analysis.TransitiveInfoCollection;
import com.google.devtools.build.lib.collect.nestedset.NestedSet;
import com.google.devtools.build.lib.collect.nestedset.NestedSetBuilder;
import com.google.devtools.build.lib.collect.nestedset.Order;
import com.google.devtools.build.lib.packages.Type;
import com.google.devtools.build.lib.rules.cpp.CppConfiguration.DynamicMode;
import com.google.devtools.build.lib.rules.cpp.CppConfiguration.HeadersCheckingMode;
import com.google.devtools.build.lib.rules.cpp.LinkerInputs.LibraryToLink;
import com.google.devtools.build.lib.rules.test.InstrumentedFilesCollector;
import com.google.devtools.build.lib.rules.test.InstrumentedFilesCollector.LocalMetadataCollector;
import com.google.devtools.build.lib.rules.test.InstrumentedFilesProvider;
import com.google.devtools.build.lib.rules.test.InstrumentedFilesProviderImpl;
import com.google.devtools.build.lib.shell.ShellUtils;
import com.google.devtools.build.lib.syntax.Label;
import com.google.devtools.build.lib.util.FileType;
import com.google.devtools.build.lib.util.FileTypeSet;
import com.google.devtools.build.lib.util.Pair;
import com.google.devtools.build.lib.vfs.PathFragment;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * Common parts of the implementation of cc rules.
 */
public final class CcCommon {

  private static final String NO_COPTS_ATTRIBUTE = "nocopts";

  private static final FileTypeSet SOURCE_TYPES = FileTypeSet.of(
      CppFileTypes.CPP_SOURCE,
      CppFileTypes.CPP_HEADER,
      CppFileTypes.C_SOURCE,
      CppFileTypes.ASSEMBLER_WITH_C_PREPROCESSOR);

  /**
   * Collects all metadata files generated by C++ compilation actions that output the .o files
   * on the input.
   */
  private static final LocalMetadataCollector CC_METADATA_COLLECTOR =
      new LocalMetadataCollector() {
    @Override
    public void collectMetadataArtifacts(Iterable<Artifact> objectFiles,
        AnalysisEnvironment analysisEnvironment, NestedSetBuilder<Artifact> metadataFilesBuilder) {
      for (Artifact artifact : objectFiles) {
        Action action = analysisEnvironment.getLocalGeneratingAction(artifact);
        if (action instanceof CppCompileAction) {
          addOutputs(metadataFilesBuilder, action, CppFileTypes.COVERAGE_NOTES);
        }
      }
    }
  };

  private final CppSemantics semantics;

  private final boolean initExtraPrerequisites;

  /** Aggregated compilation context of dependencies. */
  private CppCompilationContext cachedContext;

  /** C++ configuration */
  private final CppConfiguration cppConfiguration;

  /** The Artifacts from srcs. */
  private final ImmutableList<Artifact> sources;

  private final ImmutableList<Pair<Artifact, Label>> cAndCppSources;
  private final ImmutableList<? extends TransitiveInfoCollection> activePlugins;

  /** Expanded and tokenized copts attribute.  Set by initCopts(). */
  private final ImmutableList<String> copts;

  /**
   * The expanded linkopts for this rule.
   */
  private final ImmutableList<String> linkopts;

  private final InstrumentedFilesCollector instrumentedFilesCollector;

  private final RuleContext ruleContext;

  private final ImmutableList<PathFragment> additionalIncludes;

  private final ImmutableList<String> additionalCOpts;

  public CcCommon(RuleContext ruleContext, CppSemantics semantics, boolean initExtraPrerequisites) {
    this(ruleContext, semantics, initExtraPrerequisites,
        ImmutableList.<PathFragment>of(), ImmutableList.<String>of());
  }

  public CcCommon(RuleContext ruleContext,
      CppSemantics semantics,
      boolean initExtraPrerequisites,
      ImmutableList<PathFragment> additionalIncludes,
      ImmutableList<String> additionalCOpts) {
    this.ruleContext = ruleContext;
    this.semantics = semantics;
    this.initExtraPrerequisites = initExtraPrerequisites;
    this.additionalIncludes = additionalIncludes;
    this.additionalCOpts = additionalCOpts;
    this.cppConfiguration = ruleContext.getFragment(CppConfiguration.class);
    this.instrumentedFilesCollector =
        new InstrumentedFilesCollector(ruleContext, CppRuleClasses.INSTRUMENTATION_SPEC,
            CC_METADATA_COLLECTOR);
    this.sources = hasAttribute("srcs", Type.LABEL_LIST)
        ? ruleContext.getPrerequisiteArtifacts("srcs", Mode.TARGET).list()
        : ImmutableList.<Artifact>of();

    this.cAndCppSources = collectCAndCppSources();
    activePlugins = collectPluginInfo();
    copts = initCopts();
    linkopts = initLinkopts();
  }

  private CppCompilationContext createCppCompilationContext() {
    if (cachedContext != null) {
      return cachedContext;
    }

    CppCompilationContext.Builder contextBuilder =
        new CppCompilationContext.Builder(ruleContext);

    initOwnIncludeInfo(contextBuilder);
    initGeneratedHeadersAndTransitiveIncludeDirs(contextBuilder);

    semantics.setupCompilationContext(ruleContext, contextBuilder);
    contextBuilder.addDefines(getDefines());
    if (initExtraPrerequisites) {
      // Add libraries mentioned in "srcs" as prerequisites, in order to verify they exist.
      contextBuilder.addCompilationPrerequisites(getSharedLibrariesFromSrcs());
      contextBuilder.addCompilationPrerequisites(getStaticLibrariesFromSrcs());
    }
    CppHelper.addCppModuleMapToContext(ruleContext, contextBuilder);
    cachedContext = contextBuilder.build();
    return cachedContext;
  }

  /**
   * Returns the object files built (and linked) by this target, plus any object
   * files listed in "srcs".
   *
   * @param usePic whether to return .pic.o or .o files
   */
  public Iterable<Artifact> getObjectFiles(
      CcCompilationOutputs compilationOutputs, boolean usePic) {
    return Iterables.concat(
        getObjectFilesFromSrcs(usePic),
        compilationOutputs.getObjectFiles(usePic));
  }

  ImmutableList<Artifact> getTemps(CcCompilationOutputs compilationOutputs) {
    return cppConfiguration.isLipoContextCollector()
        ? ImmutableList.<Artifact>of()
        : compilationOutputs.getTemps();
  }

  /**
   * Returns our own linkopts from the rule attribute. This determines linker
   * options to use when building this target and anything that depends on it.
   */
  public ImmutableList<String> getLinkopts() {
    return linkopts;
  }

  public ImmutableList<String> getCopts() {
    return copts;
  }

  public ImmutableList<? extends TransitiveInfoCollection> getPlugins() {
    return activePlugins;
  }

  private boolean hasAttribute(String name, Type<?> type) {
    return ruleContext.getRule().getRuleClassObject().hasAttr(name, type);
  }

  private ImmutableList<? extends TransitiveInfoCollection> collectPluginInfo() {
    if (hasAttribute("plugins", Type.LABEL_LIST) && hasAttribute(":cc_plugins", Type.LABEL_LIST)) {
      List<TransitiveInfoCollection> plugins = new ArrayList<>();
      plugins.addAll(ruleContext.getPrerequisites("plugins", Mode.HOST));
      plugins.addAll(ruleContext.getPrerequisites(":cc_plugins", Mode.HOST));
      return ImmutableList.copyOf(plugins);
    } else {
      return ImmutableList.of();
    }
  }

  private static NestedSet<Artifact> collectExecutionDynamicLibraryArtifacts(
      RuleContext ruleContext,
      List<LibraryToLink> executionDynamicLibraries) {
    Iterable<Artifact> artifacts = LinkerInputs.toLibraryArtifacts(executionDynamicLibraries);
    if (!Iterables.isEmpty(artifacts)) {
      return NestedSetBuilder.wrap(Order.STABLE_ORDER, artifacts);
    }

    Iterable<CcExecutionDynamicLibrariesProvider> deps = ruleContext
        .getPrerequisites("deps", Mode.TARGET, CcExecutionDynamicLibrariesProvider.class);

    NestedSetBuilder<Artifact> builder = NestedSetBuilder.stableOrder();
    for (CcExecutionDynamicLibrariesProvider dep : deps) {
      builder.addTransitive(dep.getExecutionDynamicLibraryArtifacts());
    }
    return builder.build();
  }

  /**
   * Collects all .dwo artifacts in this target's transitive closure.
   */
  public static DwoArtifactsCollector collectTransitiveDwoArtifacts(
      RuleContext ruleContext,
      CcCompilationOutputs compilationOutputs) {
    ImmutableList.Builder<TransitiveInfoCollection> deps =
        ImmutableList.<TransitiveInfoCollection>builder();

    deps.addAll(ruleContext.getPrerequisites("deps", Mode.TARGET));

    if (ruleContext.getRule().getRuleClassObject().hasAttr("malloc", Type.LABEL)) {
      deps.add(CppHelper.mallocForTarget(ruleContext));
    }
    if (ruleContext.getRule().getRuleClassObject().hasAttr("implementation", Type.LABEL_LIST)) {
      deps.addAll(ruleContext.getPrerequisites("implementation", Mode.TARGET));
    }

    return compilationOutputs == null  // Possible in LIPO collection mode (see initializationHook).
        ? DwoArtifactsCollector.emptyCollector()
        : DwoArtifactsCollector.transitiveCollector(compilationOutputs, deps.build());
  }

  public TransitiveLipoInfoProvider collectTransitiveLipoLabels(CcCompilationOutputs outputs) {
    if (cppConfiguration.getFdoSupport().getFdoRoot() == null
        || !cppConfiguration.isLipoContextCollector()) {
      return TransitiveLipoInfoProvider.EMPTY;
    }

    NestedSetBuilder<IncludeScannable> scannableBuilder = NestedSetBuilder.stableOrder();
    CppHelper.addTransitiveLipoInfoForCommonAttributes(ruleContext, outputs, scannableBuilder);
    if (hasAttribute("implementation", Type.LABEL_LIST)) {
      for (TransitiveLipoInfoProvider impl : AnalysisUtils.getProviders(
          ruleContext.getPrerequisites("implementation", Mode.TARGET),
          TransitiveLipoInfoProvider.class)) {
        scannableBuilder.addTransitive(impl.getTransitiveIncludeScannables());
      }
    }

    return new TransitiveLipoInfoProvider(scannableBuilder.build());
  }

  private NestedSet<LinkerInput> collectTransitiveCcNativeLibraries(
      RuleContext ruleContext,
      List<? extends LinkerInput> dynamicLibraries) {
    NestedSetBuilder<LinkerInput> builder = NestedSetBuilder.linkOrder();
    builder.addAll(dynamicLibraries);
    for (CcNativeLibraryProvider dep :
      ruleContext.getPrerequisites("deps", Mode.TARGET, CcNativeLibraryProvider.class)) {
      builder.addTransitive(dep.getTransitiveCcNativeLibraries());
    }
    return builder.build();
  }

  /**
   * Returns a list of ({@link Artifact}, {@link Label}) pairs. Each pair represents an input
   * source file and the label of the rule that generates it (or the label of the source file
   * itself if it is an input file)
   */
  ImmutableList<Pair<Artifact, Label>> getCAndCppSources() {
    return cAndCppSources;
  }

  private boolean shouldProcessHeaders() {
    boolean crosstoolSupportsHeaderParsing =
        CppHelper.getToolchain(ruleContext).supportsHeaderParsing();
    return crosstoolSupportsHeaderParsing && (
        ruleContext.getFeatures().contains(CppRuleClasses.PREPROCESS_HEADERS)
        || ruleContext.getFeatures().contains(CppRuleClasses.PARSE_HEADERS));
  }

  private ImmutableList<Pair<Artifact, Label>> collectCAndCppSources() {
    Map<Artifact, Label> map = Maps.newLinkedHashMap();
    if (!hasAttribute("srcs", Type.LABEL_LIST)) {
      return ImmutableList.<Pair<Artifact, Label>>of();
    }
    Iterable<FileProvider> providers =
        ruleContext.getPrerequisites("srcs", Mode.TARGET, FileProvider.class);
    // TODO(bazel-team): Move header processing logic down in the stack (to CcLibraryHelper or
    // such).
    boolean processHeaders = shouldProcessHeaders();
    if (processHeaders && hasAttribute("hdrs", Type.LABEL_LIST)) {
      providers = Iterables.concat(providers,
          ruleContext.getPrerequisites("hdrs", Mode.TARGET, FileProvider.class));
    }
    for (FileProvider provider : providers) {
      for (Artifact artifact : FileType.filter(provider.getFilesToBuild(), SOURCE_TYPES)) {
        boolean isHeader = CppFileTypes.CPP_HEADER.matches(artifact.getExecPath());
        if ((isHeader && !processHeaders)
            || CppFileTypes.CPP_TEXTUAL_INCLUDE.matches(artifact.getExecPath())) {
          continue;
        }
        Label oldLabel = map.put(artifact, provider.getLabel());
        // TODO(bazel-team): We currently do not warn for duplicate headers with
        // different labels, as that would require cleaning up the code base
        // without significant benefit; we should eventually make this
        // consistent one way or the other.
        if (!isHeader && oldLabel != null && !oldLabel.equals(provider.getLabel())) {
          ruleContext.attributeError("srcs", String.format(
              "Artifact '%s' is duplicated (through '%s' and '%s')",
              artifact.getExecPathString(), oldLabel, provider.getLabel()));
        }
      }
    }

    ImmutableList.Builder<Pair<Artifact, Label>> result = ImmutableList.builder();
    for (Map.Entry<Artifact, Label> entry : map.entrySet()) {
      result.add(Pair.of(entry.getKey(), entry.getValue()));
    }

    return result.build();
  }

  Iterable<Artifact> getLibrariesFromSrcs() {
    return FileType.filter(sources, CppFileTypes.ARCHIVE, CppFileTypes.PIC_ARCHIVE,
        CppFileTypes.ALWAYS_LINK_LIBRARY, CppFileTypes.ALWAYS_LINK_PIC_LIBRARY,
        CppFileTypes.SHARED_LIBRARY,
        CppFileTypes.VERSIONED_SHARED_LIBRARY);
  }

  Iterable<Artifact> getSharedLibrariesFromSrcs() {
    return getSharedLibrariesFrom(sources);
  }

  static Iterable<Artifact> getSharedLibrariesFrom(Iterable<Artifact> collection) {
    return FileType.filter(collection, CppFileTypes.SHARED_LIBRARY,
        CppFileTypes.VERSIONED_SHARED_LIBRARY);
  }

  Iterable<Artifact> getStaticLibrariesFromSrcs() {
    return FileType.filter(sources, CppFileTypes.ARCHIVE, CppFileTypes.ALWAYS_LINK_LIBRARY);
  }

  Iterable<LibraryToLink> getPicStaticLibrariesFromSrcs() {
    return LinkerInputs.opaqueLibrariesToLink(
        FileType.filter(sources, CppFileTypes.PIC_ARCHIVE,
            CppFileTypes.ALWAYS_LINK_PIC_LIBRARY));
  }

  Iterable<Artifact> getObjectFilesFromSrcs(final boolean usePic) {
    if (usePic) {
      return Iterables.filter(sources, new Predicate<Artifact>() {
        @Override
        public boolean apply(Artifact artifact) {
          String filename = artifact.getExecPathString();

          // For compatibility with existing BUILD files, any ".o" files listed
          // in srcs are assumed to be position-independent code, or
          // at least suitable for inclusion in shared libraries, unless they
          // end with ".nopic.o". (The ".nopic.o" extension is an undocumented
          // feature to give users at least some control over this.) Note that
          // some target platforms do not require shared library code to be PIC.
          return CppFileTypes.PIC_OBJECT_FILE.matches(filename) ||
              (CppFileTypes.OBJECT_FILE.matches(filename) && !filename.endsWith(".nopic.o"));
        }
      });
    } else {
      return FileType.filter(sources, CppFileTypes.OBJECT_FILE);
    }
  }

  /**
   * Initializes and populates our various include related attributes directly
   * accessible to this C++ rule.
   */
  private void initOwnIncludeInfo(CppCompilationContext.Builder contextBuilder) {
    HeadersCheckingMode headersCheckingMode = determineHeadersCheckingMode();

    // Files in the 'hdrs' attr are always available for inclusion in dependent rules.
    List<Artifact> hdrs = getHeaders();
    if (!hdrs.isEmpty()) {
      contextBuilder.addDeclaredIncludeSrcs(hdrs);
      contextBuilder.addPregreppedHeaderMap(
          CppHelper.createExtractInclusions(ruleContext, hdrs));
    }

    PathFragment genfilesFragment = ruleContext.getConfiguration().getGenfilesFragment();

    // Add in the roots for well-formed include names for source files and
    // generated files. It is important that the execRoot (EMPTY_FRAGMENT) comes
    // before the genfilesFragment to preferably pick up source files. Otherwise
    // we might pick up stale generated files.
    contextBuilder.addQuoteIncludeDir(PathFragment.EMPTY_FRAGMENT);
    contextBuilder.addQuoteIncludeDir(genfilesFragment);

    // For strict checking, use only declared headers, otherwise grab this package's dirs,
    // either into the loose dir list, or the warn dir list.
    if (headersCheckingMode != HeadersCheckingMode.STRICT) {
      initLooseIncludeDirs(contextBuilder, headersCheckingMode);
    }

    for (PathFragment systemIncludeDir : getSystemIncludeDirs()) {
      contextBuilder.addSystemIncludeDir(systemIncludeDir);
    }
    for (PathFragment includeDir : getIncludeDirs()) {
      contextBuilder.addIncludeDir(includeDir);
    }

    contextBuilder.addIncludeDirs(additionalIncludes);
  }

  /**
   * Returns the files from headers and does some sanity checks. Note that this method reports
   * warnings to the {@link RuleContext} as a side effect, and so should only be called once for any
   * given rule.
   */
  List<Artifact> getHeaders() {
    if (!hasAttribute("hdrs", Type.LABEL_LIST)) {
      return ImmutableList.of();
    }

    List<Artifact> hdrs = new ArrayList<>();
    for (TransitiveInfoCollection target :
        ruleContext.getPrerequisitesIf("hdrs", Mode.TARGET, FileProvider.class)) {
      FileProvider provider = target.getProvider(FileProvider.class);
      for (Artifact artifact : provider.getFilesToBuild()) {
        if (!CppRuleClasses.DISALLOWED_HDRS_FILES.matches(artifact.getFilename())) {
          hdrs.add(artifact);
        } else {
          ruleContext.attributeWarning("hdrs", "file '" + artifact.getFilename()
              + "' from target '" + target.getLabel() + "' is not allowed in hdrs");
        }
      }
    }
    return hdrs;
  }

  HeadersCheckingMode determineHeadersCheckingMode() {
    HeadersCheckingMode headersCheckingMode = cppConfiguration.getHeadersCheckingMode();

    // Package default overrides command line option.
    if (ruleContext.getRule().getPackage().isDefaultHdrsCheckSet()) {
      String value =
          ruleContext.getRule().getPackage().getDefaultHdrsCheck().toUpperCase(Locale.ENGLISH);
      headersCheckingMode = HeadersCheckingMode.valueOf(value);
    }

    // 'hdrs_check' attribute overrides package default.
    if (hasAttribute("hdrs_check", Type.STRING)
        && ruleContext.getRule().isAttributeValueExplicitlySpecified("hdrs_check")) {
      try {
        String value = ruleContext.attributes().get("hdrs_check", Type.STRING)
            .toUpperCase(Locale.ENGLISH);
        headersCheckingMode = HeadersCheckingMode.valueOf(value);
      } catch (IllegalArgumentException e) {
        ruleContext.attributeError("hdrs_check", "must be one of: 'loose', 'warn' or 'strict'");
      }
    }

    return headersCheckingMode;
  }

  private void initGeneratedHeadersAndTransitiveIncludeDirs(
      CppCompilationContext.Builder contextBuilder) {

    LanguageDependentFragment.Checker.depsSupportsLanguage(ruleContext, CppRuleClasses.LANGUAGE);

    contextBuilder.mergeDependentContexts(
        ruleContext.getPrerequisites("deps", Mode.TARGET, CppCompilationContext.class));
    CppHelper.mergeToolchainDependentContext(ruleContext, contextBuilder);

    if (hasAttribute("malloc", Type.LABEL)) {
      CppCompilationContext malloc = CppHelper.mallocForTarget(ruleContext)
          .getProvider(CppCompilationContext.class);
      if (malloc != null) {
        contextBuilder.mergeDependentContext(malloc);
      }
    }

    initAdditionalTransitiveIncludeDirs("implements", contextBuilder);
    initAdditionalTransitiveIncludeDirs("implementation", contextBuilder);

    // In theory, this belongs in initOwnIncludeInfo above, but is here since
    // this is where the headers in temps are being handled post-traversal.
    Iterable<Artifact> headers = FileType.filter(sources, CppFileTypes.CPP_HEADER);
    contextBuilder.addDeclaredIncludeSrcs(headers);
    contextBuilder.addPregreppedHeaderMap(
        CppHelper.createExtractInclusions(ruleContext, headers));
  }

  private void initAdditionalTransitiveIncludeDirs(
      String attributeName, CppCompilationContext.Builder contextBuilder) {
    if (hasAttribute(attributeName, Type.LABEL_LIST)) {
      for (CppCompilationContext publicLib : ruleContext.getPrerequisites(
          attributeName, Mode.TARGET, CppCompilationContext.class)) {
        contextBuilder.mergeDependentContext(publicLib);
      }
    }
  }

  /**
   * Expand and tokenize the copts and nocopts attributes.
   */
  private ImmutableList<String> initCopts() {
    if (!hasAttribute("copts", Type.STRING_LIST)) {
      return ImmutableList.<String>of();
    }
    // TODO(bazel-team): getAttributeCopts should not tokenize the strings.
    // Make a warning for now.
    List<String> tokens = new ArrayList<>();
    for (String str : ruleContext.attributes().get("copts", Type.STRING_LIST)) {
      tokens.clear();
      try {
        ShellUtils.tokenize(tokens, str);
        if (tokens.size() > 1) {
          ruleContext.attributeWarning("copts",
              "each item in the list should contain only one option");
        }
      } catch (ShellUtils.TokenizationException e) {
        // ignore, the error is reported in the getAttributeCopts call
      }
    }

    Pattern nocopts = getNoCopts(ruleContext);
    if (nocopts != null && nocopts.matcher("-Wno-future-warnings").matches()) {
      ruleContext.attributeWarning("nocopts",
          "Regular expression '" + nocopts.pattern() + "' is too general; for example, it matches "
          + "'-Wno-future-warnings'.  Thus it might *re-enable* compiler warnings we wish to "
          + "disable globally.  To disable all compiler warnings, add '-w' to copts instead");
    }

    return ImmutableList.<String>builder()
        .addAll(additionalCOpts)
        .addAll(getPackageCopts(ruleContext))
        .addAll(CppHelper.getAttributeCopts(ruleContext, "copts"))
        .build();
  }

  private static ImmutableList<String> getPackageCopts(RuleContext ruleContext) {
    List<String> unexpanded = ruleContext.getRule().getPackage().getDefaultCopts();
    return ImmutableList.copyOf(CppHelper.expandMakeVariables(ruleContext, "copts", unexpanded));
  }

  Pattern getNoCopts() {
    return getNoCopts(ruleContext);
  }

  /**
   * Returns nocopts pattern built from the make variable expanded nocopts
   * attribute.
   */
  private static Pattern getNoCopts(RuleContext ruleContext) {
    Pattern nocopts = null;
    if (ruleContext.getRule().isAttrDefined(NO_COPTS_ATTRIBUTE, Type.STRING)) {
      String nocoptsAttr = ruleContext.expandMakeVariables(NO_COPTS_ATTRIBUTE,
          ruleContext.attributes().get(NO_COPTS_ATTRIBUTE, Type.STRING));
      try {
        nocopts = Pattern.compile(nocoptsAttr);
      } catch (PatternSyntaxException e) {
        ruleContext.attributeError(NO_COPTS_ATTRIBUTE,
            "invalid regular expression '" + nocoptsAttr + "': " + e.getMessage());
      }
    }
    return nocopts;
  }

  // TODO(bazel-team): calculating nocopts every time is not very efficient,
  // fix this after the rule migration. The problem is that in some cases we call this after
  // the RCT is created (so RuleContext is not accessible), in some cases during the creation.
  // It would probably make more sense to use TransitiveInfoProviders.
  /**
   * Returns true if the rule context has a nocopts regex that matches the given value, false
   * otherwise.
   */
  static boolean noCoptsMatches(String option, RuleContext ruleContext) {
    Pattern nocopts = getNoCopts(ruleContext);
    return nocopts == null ? false : nocopts.matcher(option).matches();
  }

  private static final String DEFINES_ATTRIBUTE = "defines";

  /**
   * Returns a list of define tokens from "defines" attribute.
   *
   * <p>We tokenize the "defines" attribute, to ensure that the handling of
   * quotes and backslash escapes is consistent Bazel's treatment of the "copts" attribute.
   *
   * <p>But we require that the "defines" attribute consists of a single token.
   */
  public List<String> getDefines() {
    List<String> defines = new ArrayList<>();
    for (String define :
      ruleContext.attributes().get(DEFINES_ATTRIBUTE, Type.STRING_LIST)) {
      List<String> tokens = new ArrayList<>();
      try {
        ShellUtils.tokenize(tokens, ruleContext.expandMakeVariables(DEFINES_ATTRIBUTE, define));
        if (tokens.size() == 1) {
          defines.add(tokens.get(0));
        } else if (tokens.isEmpty()) {
          ruleContext.attributeError(DEFINES_ATTRIBUTE, "empty definition not allowed");
        } else {
          ruleContext.attributeError(DEFINES_ATTRIBUTE,
              "definition contains too many tokens (found " + tokens.size()
              + ", expecting exactly one)");
        }
      } catch (ShellUtils.TokenizationException e) {
        ruleContext.attributeError(DEFINES_ATTRIBUTE, e.getMessage());
      }
    }
    return defines;
  }

  /**
   * Collects our own linkopts from the rule attribute. This determines linker
   * options to use when building this library and anything that depends on it.
   */
  private final ImmutableList<String> initLinkopts() {
    if (!hasAttribute("linkopts", Type.STRING_LIST)) {
      return ImmutableList.<String>of();
    }
    List<String> ourLinkopts = ruleContext.attributes().get("linkopts", Type.STRING_LIST);
    List<String> result = new ArrayList<>();
    if (ourLinkopts != null) {
      boolean allowDashStatic = !cppConfiguration.forceIgnoreDashStatic()
          && (cppConfiguration.getDynamicMode() != DynamicMode.FULLY);
      for (String linkopt : ourLinkopts) {
        if (linkopt.equals("-static") && !allowDashStatic) {
          continue;
        }
        CppHelper.expandAttribute(ruleContext, result, "linkopts", linkopt, true);
      }
    }
    return ImmutableList.copyOf(result);
  }

  /**
   * Adds the package's directory as well as relative directory fragments mentioned in the srcs
   * of this rule to the declared includes. The fragments represent package and child dirs
   * containing files to build. The set will later be used to enforce proper
   * include declaration.
   *
   * @param headersCheckingMode headers check mode
   */
  private void initLooseIncludeDirs(CppCompilationContext.Builder contextBuilder,
      HeadersCheckingMode headersCheckingMode) {
    if (headersCheckingMode == HeadersCheckingMode.WARN) {
      for (PathFragment looseIncludeDir : getLooseIncludeDirs()) {
        contextBuilder.addDeclaredIncludeWarnDir(looseIncludeDir);
      }
    } else if (headersCheckingMode == HeadersCheckingMode.LOOSE) {
      for (PathFragment looseIncludeDir : getLooseIncludeDirs()) {
        contextBuilder.addDeclaredIncludeDir(looseIncludeDir);
      }
    }
  }

  /**
   * Determines a list of loose include directories that are only allowed to be referenced when
   * headers checking is {@link HeadersCheckingMode#LOOSE} or {@link HeadersCheckingMode#WARN}.
   */
  List<PathFragment> getLooseIncludeDirs() {
    List<PathFragment> result = new ArrayList<>();
    // The package directory of the rule contributes includes. Note that this also covers all
    // non-subpackage sub-directories.
    PathFragment rulePackage = ruleContext.getLabel().getPackageFragment();
    result.add(rulePackage);

    // Gather up all the dirs from the rule's srcs as well as any of the srcs outputs.
    if (hasAttribute("srcs", Type.LABEL_LIST)) {
      for (FileProvider src :
          ruleContext.getPrerequisites("srcs", Mode.TARGET, FileProvider.class)) {
        PathFragment packageDir = src.getLabel().getPackageFragment();
        for (Artifact a : src.getFilesToBuild()) {
          result.add(packageDir);
          // Attempt to gather subdirectories that might contain include files.
          result.add(a.getRootRelativePath().getParentDirectory());
        }
      }
    }

    // Add in any 'includes' attribute values as relative path fragments
    if (ruleContext.getRule().isAttributeValueExplicitlySpecified("includes")) {
      PathFragment packageFragment = ruleContext.getLabel().getPackageFragment();
      // For now, anything with an 'includes' needs a blanket declaration
      result.add(packageFragment.getRelative("**"));
    }
    return result;
  }

  List<PathFragment> getSystemIncludeDirs() {
    // Add in any 'includes' attribute values as relative path fragments
    if (!ruleContext.getRule().isAttributeValueExplicitlySpecified("includes")
        || !cppConfiguration.useIsystemForIncludes()) {
      return ImmutableList.of();
    }
    return getIncludeDirsFromIncludesAttribute();
  }

  List<PathFragment> getIncludeDirs() {
    if (!ruleContext.getRule().isAttributeValueExplicitlySpecified("includes")
        || cppConfiguration.useIsystemForIncludes()) {
      return ImmutableList.of();
    }
    return getIncludeDirsFromIncludesAttribute();
  }

  private List<PathFragment> getIncludeDirsFromIncludesAttribute() {
    List<PathFragment> result = new ArrayList<>();
    PathFragment packageFragment = ruleContext.getLabel().getPackageFragment();
    for (String includesAttr : ruleContext.attributes().get("includes", Type.STRING_LIST)) {
      includesAttr = ruleContext.expandMakeVariables("includes", includesAttr);
      if (includesAttr.startsWith("/")) {
        ruleContext.attributeWarning("includes",
            "ignoring invalid absolute path '" + includesAttr + "'");
        continue;
      }
      PathFragment includesPath = packageFragment.getRelative(includesAttr).normalize();
      if (!includesPath.isNormalized()) {
        ruleContext.attributeError("includes",
            "Path references a path above the execution root.");
      }
      result.add(includesPath);
      result.add(ruleContext.getConfiguration().getGenfilesFragment().getRelative(includesPath));
    }
    return result;
  }

  /**
   * Collects compilation prerequisite artifacts.
   */
  static CompilationPrerequisitesProvider collectCompilationPrerequisites(
      RuleContext ruleContext, CppCompilationContext context) {
    // TODO(bazel-team): Use context.getCompilationPrerequisites() instead.
    NestedSetBuilder<Artifact> prerequisites = NestedSetBuilder.stableOrder();
    if (ruleContext.getRule().getRuleClassObject().hasAttr("srcs", Type.LABEL_LIST)) {
      for (FileProvider provider : ruleContext
          .getPrerequisites("srcs", Mode.TARGET, FileProvider.class)) {
        prerequisites.addAll(FileType.filter(provider.getFilesToBuild(), SOURCE_TYPES));
      }
    }
    prerequisites.addTransitive(context.getDeclaredIncludeSrcs());
    return new CompilationPrerequisitesProvider(prerequisites.build());
  }

  /**
   * Creates C++ compile actions for all C/C++ sources. These include the
   * regular .o file creating in addition to the "--save_temps" actions which
   * create .i (preprocessed C), .ii (preprocessed C++) and .s (assembly) files.
   */
  public CcCompilationOutputs createCompileActions(boolean fake) {
    return new CppModel(ruleContext, semantics)
        .addSources(getCAndCppSources())
        .setFake(fake)
        .setSaveTemps(true)
        .setContext(createCppCompilationContext())
        .addCopts(copts)
        .setNoCopts(getNoCopts(ruleContext))
        .addAdditionalIncludes(additionalIncludes)
        .addPluginTargets(activePlugins)
        .setEnableLayeringCheck(ruleContext.getFeatures().contains(CppRuleClasses.LAYERING_CHECK))
        .setCompileHeaderModules(ruleContext.getFeatures().contains(CppRuleClasses.HEADER_MODULES))
        .createCcCompileActions();
  }

  /**
   * Replaces shared library artifact with mangled symlink and creates related
   * symlink action. For artifacts that should retain filename (e.g. libraries
   * with SONAME tag), link is created to the parent directory instead.
   *
   * This action is performed to minimize number of -rpath entries used during
   * linking process (by essentially "collecting" as many shared libraries as
   * possible in the single directory), since we will be paying quadratic price
   * for each additional entry on the -rpath.
   *
   * @param library Shared library artifact that needs to be mangled
   * @param preserveName true if filename should be preserved, false - mangled.
   * @return mangled symlink artifact.
   */
  public LibraryToLink getDynamicLibrarySymlink(Artifact library, boolean preserveName) {
    return SolibSymlinkAction.getDynamicLibrarySymlink(
        ruleContext, library, preserveName, true, ruleContext.getConfiguration());
  }

  /**
   * Returns a link action builder that is preconfigured with the compilation
   * prerequisites and the crosstool inputs.
   *
   * <p>This ensures that we always build the prerequisites. If there are no cc
   * compile actions (which happens if there are no cc sources), they would not
   * otherwise be built.
   */
  public CppLinkAction.Builder newLinkActionBuilder(PathFragment outputPath) {
    return new CppLinkAction.Builder(ruleContext, outputPath)
        .setCrosstoolInputs(CppHelper.getToolchain(ruleContext).getLink())
        .addNonLibraryInputs(createCppCompilationContext().getCompilationPrerequisites());
  }

  /**
   * Returns any linker scripts found in the dependencies of the rule.
   */
  Iterable<Artifact> getLinkerScripts() {
    return FileType.filter(ruleContext.getPrerequisiteArtifacts("deps", Mode.TARGET).list(),
        CppFileTypes.LINKER_SCRIPT);
  }

  ImmutableList<Artifact> getFilesToCompile(CcCompilationOutputs compilationOutputs) {
    return cppConfiguration.isLipoContextCollector()
        ? ImmutableList.<Artifact>of()
        : compilationOutputs.getObjectFiles(CppHelper.usePic(ruleContext, false));
  }

  NestedSet<Artifact> getInstrumentationMetadataFiles(Iterable<Artifact> objectFiles) {
    return cppConfiguration.isLipoContextCollector()
        ? NestedSetBuilder.<Artifact>emptySet(Order.STABLE_ORDER)
        : instrumentedFilesCollector.getInstrumentationMetadataFiles(objectFiles);
  }

  NestedSet<Artifact> getInstrumentedFiles(Iterable<Artifact> objectFiles) {
    return cppConfiguration.isLipoContextCollector()
        ? NestedSetBuilder.<Artifact>emptySet(Order.STABLE_ORDER)
        : instrumentedFilesCollector.getInstrumentedFiles(objectFiles);
  }

  // Function for extracting labels from CppCompilationDependencies
  private static final Function<CppCompilationContext, CppModuleMap> CPP_DEPS_TO_MODULES =
    new Function<CppCompilationContext, CppModuleMap>() {
      @Override
      public CppModuleMap apply(CppCompilationContext dep) {
        return dep.getCppModuleMap();
      }
    };

  // Predicate to match C++ header artifacts
  private static final Predicate<Artifact> HEADER_FILTER =
    new Predicate<Artifact>() {
      @Override
      public boolean apply(Artifact input) {
        return CppFileTypes.CPP_HEADER.matches(input.getFilename());
      }
    };

  /** Create C++ module map artifact for this CT */
  public void createModuleMapAction() {
    CppCompilationContext context = createCppCompilationContext();
    if (context.getCppModuleMap() != null) {
      // Header files from 'srcs' attribute are the private headers
      Iterable<Artifact> srcs = (ruleContext.attributes().getAttributeDefinition("srcs") != null)
          ? ruleContext.getPrerequisiteArtifacts("srcs", Mode.TARGET).list()
          : ImmutableList.<Artifact>of();
      Iterable<Artifact> privateHeaders = Iterables.filter(srcs, HEADER_FILTER);

      // Exposed header files from 'hdrs' attribute are the public headers
      ImmutableList<Artifact> publicHeaders =
          (ruleContext.attributes().getAttributeDefinition("hdrs") != null)
              ? ruleContext.getPrerequisiteArtifacts("hdrs", Mode.TARGET).list()
              : ImmutableList.<Artifact>of();

      // Enumerate list of dependencies
      ImmutableList.Builder<CppModuleMap> depsBuilder = ImmutableList.builder();
      Iterable<CppCompilationContext> deps =
          ruleContext.getPrerequisites("deps", Mode.TARGET, CppCompilationContext.class);
      if (deps != null && deps.iterator().hasNext()) {
        depsBuilder.addAll(
            Iterables.filter(Iterables.transform(deps, CPP_DEPS_TO_MODULES), Predicates.notNull()));
      }
      if (ruleContext.attributes().getAttributeDefinition("implements") != null) {
        Iterable<CppCompilationContext> impl =
            ruleContext.getPrerequisites("implements", Mode.TARGET, CppCompilationContext.class);
        if (impl != null && impl.iterator().hasNext()) {
          depsBuilder.addAll(Iterables.filter(Iterables.transform(impl, CPP_DEPS_TO_MODULES),
              Predicates.notNull()));
        }
      }
      CppCompilationContext stl =
          ruleContext.getPrerequisite(":stl", Mode.TARGET, CppCompilationContext.class);
      if (stl != null) {
        depsBuilder.add(stl.getCppModuleMap());
      }

      CcToolchainProvider toolchain = CppHelper.getToolchain(ruleContext);
      if (toolchain != null) {
        CppModuleMap crosstoolModuleMap = toolchain.getCppCompilationContext().getCppModuleMap();
        if (crosstoolModuleMap != null) {
          depsBuilder.add(crosstoolModuleMap);
        }
      }
      Iterable<CppModuleMap> depsLabels =
          Iterables.filter(depsBuilder.build(), Predicates.<CppModuleMap>notNull());

      ImmutableList<PathFragment> bootstrapHackHeaders = ImmutableList.of();
      CppModuleMapAction action = new CppModuleMapAction(
          ruleContext.getActionOwner(), context.getCppModuleMap(),
          privateHeaders, publicHeaders, depsLabels, bootstrapHackHeaders,
          ruleContext.getFeatures().contains(CppRuleClasses.HEADER_MODULES));

      ruleContext.registerAction(action);
    }
  }

  public void addTransitiveInfoProviders(RuleConfiguredTargetBuilder builder,
      NestedSet<Artifact> filesToBuild,
      CcCompilationOutputs ccCompilationOutputs,
      CcLinkingOutputs linkingOutputs,
      DwoArtifactsCollector dwoArtifacts,
      TransitiveLipoInfoProvider transitiveLipoInfo) {
    addTransitiveInfoProviders(builder, filesToBuild, ccCompilationOutputs,
        createCppCompilationContext(), linkingOutputs, dwoArtifacts, transitiveLipoInfo);
  }

  public void addTransitiveInfoProviders(RuleConfiguredTargetBuilder builder,
      NestedSet<Artifact> filesToBuild,
      CcCompilationOutputs ccCompilationOutputs,
      CppCompilationContext cppCompilationContext,
      CcLinkingOutputs linkingOutputs,
      DwoArtifactsCollector dwoArtifacts,
      TransitiveLipoInfoProvider transitiveLipoInfo) {
    List<Artifact> instrumentedObjectFiles = new ArrayList<>();
    instrumentedObjectFiles.addAll(ccCompilationOutputs.getObjectFiles(false));
    instrumentedObjectFiles.addAll(ccCompilationOutputs.getObjectFiles(true));
    builder
        .setFilesToBuild(filesToBuild)
        .add(CppCompilationContext.class, cppCompilationContext)
        .add(TransitiveLipoInfoProvider.class, transitiveLipoInfo)
        .add(CcExecutionDynamicLibrariesProvider.class,
            new CcExecutionDynamicLibrariesProvider(collectExecutionDynamicLibraryArtifacts(
                ruleContext, linkingOutputs.getExecutionDynamicLibraries())))
        .add(CcNativeLibraryProvider.class, new CcNativeLibraryProvider(
            collectTransitiveCcNativeLibraries(ruleContext, linkingOutputs.getDynamicLibraries())))
        .add(InstrumentedFilesProvider.class, new InstrumentedFilesProviderImpl(
            getInstrumentedFiles(instrumentedObjectFiles),
            getInstrumentationMetadataFiles(instrumentedObjectFiles)))
        .add(FilesToCompileProvider.class, new FilesToCompileProvider(
            getFilesToCompile(ccCompilationOutputs)))
        .add(CompilationPrerequisitesProvider.class,
            collectCompilationPrerequisites(ruleContext, cppCompilationContext))
        .add(TempsProvider.class, new TempsProvider(getTemps(ccCompilationOutputs)))
        .add(CppDebugFileProvider.class, new CppDebugFileProvider(
            dwoArtifacts.getDwoArtifacts(),
            dwoArtifacts.getPicDwoArtifacts()));
  }
}
