//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// test numeric_limits

// infinity()

#include <cuda/std/cassert>
#include <cuda/std/cfloat>
#include <cuda/std/limits>

#include "common.h"

// MSVC has issues with producing INF with divisions by zero.
#if defined(_MSC_VER)
#  include <cmath>
#endif

#include "test_macros.h"

TEST_NV_DIAG_SUPPRESS(221);

template <class T>
__host__ __device__ void test(T expected)
{
  assert(float_eq(cuda::std::numeric_limits<T>::infinity(), expected));
  assert(float_eq(cuda::std::numeric_limits<const T>::infinity(), expected));
  assert(float_eq(cuda::std::numeric_limits<volatile T>::infinity(), expected));
  assert(float_eq(cuda::std::numeric_limits<const volatile T>::infinity(), expected));
}

int main(int, char**)
{
  test<bool>(false);
  test<char>(0);
  test<signed char>(0);
  test<unsigned char>(0);
  test<wchar_t>(0);
#if TEST_STD_VER > 2017 && defined(__cpp_char8_t)
  test<char8_t>(0);
#endif
#ifndef _LIBCUDACXX_HAS_NO_UNICODE_CHARS
  test<char16_t>(0);
  test<char32_t>(0);
#endif // _LIBCUDACXX_HAS_NO_UNICODE_CHARS
  test<short>(0);
  test<unsigned short>(0);
  test<int>(0);
  test<unsigned int>(0);
  test<long>(0);
  test<unsigned long>(0);
  test<long long>(0);
  test<unsigned long long>(0);
#ifndef _LIBCUDACXX_HAS_NO_INT128
  test<__int128_t>(0);
  test<__uint128_t>(0);
#endif
#if !defined(_MSC_VER)
  test<float>(1.f / 0.f);
  test<double>(1. / 0.);
#  ifndef _LIBCUDACXX_HAS_NO_LONG_DOUBLE
  test<long double>(1. / 0.);
#  endif
#  if defined(_CCCL_HAS_NVFP16)
  test<__half>(__double2half(1.0 / 0.0));
#  endif // _CCCL_HAS_NVFP16
#  if defined(_CCCL_HAS_NVBF16)
  test<__nv_bfloat16>(__double2bfloat16(1.0 / 0.0));
#  endif // _CCCL_HAS_NVBF16
#  if _CCCL_HAS_NVFP8()
  test<__nv_fp8_e4m3>(__nv_fp8_e4m3{});
  test<__nv_fp8_e5m2>(make_fp8_e5m2(1.0 / 0.0));
#  endif // _CCCL_HAS_NVFP8()
// MSVC has issues with producing INF with divisions by zero.
#else
  test<float>(INFINITY);
  test<double>(INFINITY);
#  ifndef _LIBCUDACXX_HAS_NO_LONG_DOUBLE
  test<long double>(INFINITY);
#  endif
#  if defined(_CCCL_HAS_NVFP16)
  test<__half>(__double2half(INFINITY));
#  endif // _CCCL_HAS_NVFP16
#  if defined(_CCCL_HAS_NVBF16)
  test<__nv_bfloat16>(__double2bfloat16(INFINITY));
#  endif // _CCCL_HAS_NVBF16
#  if _CCCL_HAS_NVFP8()
  test<__nv_fp8_e4m3>(__nv_fp8_e4m3{});
  test<__nv_fp8_e5m2>(make_fp8_e5m2(INFINITY));
#  endif // _CCCL_HAS_NVFP8()
#endif

  return 0;
}
