#!/bin/bash
#
# This is the Telegraf CircleCI test script. Using this script allows total control
# the environment in which the build and test is run, and matches the official
# build process for InfluxDB.

BUILD_DIR=$HOME/telegraf-build
VERSION=`git describe --always --tags`

# Executes the given statement, and exits if the command returns a non-zero code.
function exit_if_fail {
    command=$@
    echo "Executing '$command'"
    eval $command
    rc=$?
    if [ $rc -ne 0 ]; then
        echo "'$command' returned $rc."
        exit $rc
    fi
}

# Check that go fmt has been run.
function check_go_fmt {
    fmtcount=`git ls-files | grep '.go$' | grep -v Godep | xargs gofmt -l 2>&1 | wc -l`
    if [ $fmtcount -gt 0 ]; then
        echo "run 'go fmt ./...' to format your source code."
        exit 1
    fi
}

# build takes three arguments: GOOS & GOARCH & VERSION
function build {
  echo -n "=> $1-$2: "
  GOOS=$1 GOARCH=$2 godep go build -o telegraf-$1-$2 \
                    -ldflags "-X main.Version=$3" \
                    ./cmd/telegraf/telegraf.go
  du -h telegraf-$1-$2
}

# Set up the build directory, and then GOPATH.
exit_if_fail mkdir $BUILD_DIR
export GOPATH=$BUILD_DIR
# Turning off GOGC speeds up build times
export GOGC=off
export PATH=$GOPATH/bin:$PATH
exit_if_fail mkdir -p $GOPATH/src/github.com/influxdb

# Get golint
go get github.com/golang/lint/golint
# Get godep tool
go get github.com/tools/godep

# Dump some test config to the log.
echo "Test configuration"
echo "========================================"
echo "\$HOME: $HOME"
echo "\$GOPATH: $GOPATH"
echo "\$CIRCLE_BRANCH: $CIRCLE_BRANCH"

# Move the checked-out source to a better location
exit_if_fail mv $HOME/telegraf $GOPATH/src/github.com/influxdb
exit_if_fail cd $GOPATH/src/github.com/influxdb/telegraf

# Verify that go fmt has been run
check_go_fmt

# Build the code
exit_if_fail godep go build -v ./...

# Run the tests
exit_if_fail godep go vet ./...
exit_if_fail godep go test -short ./...

# Build binaries
build "linux" "amd64" $VERSION
build "linux" "386" $VERSION
build "linux" "arm" $VERSION

# Simple Integration Tests
#   check that version was properly set
exit_if_fail "./telegraf-linux-amd64 -version | grep $VERSION"
#   check that one test cpu & mem output work
tmpdir=$(mktemp -d)
./telegraf-linux-amd64 -sample-config > $tmpdir/config.toml
exit_if_fail ./telegraf-linux-amd64 -config $tmpdir/config.toml \
    -test -filter cpu:mem

# Artifact binaries
mv telegraf* $CIRCLE_ARTIFACTS

exit $rc
