-- The 'addon.default_keyboard_layouts' table stores different default keyboard layouts for the addon. 
-- Each layout is represented by a key-value pair where the key is the layout name 
-- and the value is a table describing the positions and sizes of each key.

-- Key size variables are defined for various sizes of keys used across the layouts.
-- These variables are used to position and size the keys dynamically.

-- Button name  (column 1): The name of the button or key, such as 'ESCAPE', 'F1', '1', etc.
-- X position   (column 2): The horizontal position of the key on the layout (measured in pixels).
-- Y position   (column 3): The vertical position of the key on the layout (measured in pixels).
-- Width        (column 4): The width of the key in the layout (measured in pixels).
-- Height       (column 5): The height of the key in the layout (measured in pixels).
-- Icon Width   (column 6): The width of the icon displayed on the key if needed (measured in pixels).
-- Icon Height  (column 7): The height of the icon displayed on the key if needed (measured in pixels).

local name, addon = ...

-- Define key sizes
local u = 60                    -- Standard unit size for keys
local u1_25 = 1.25 * u          -- 125% of standard size
local u1_5 = 1.5 * u            -- 150% of standard size
local u1_75 = 1.75 * u          -- 175% of standard size
local u2 = 2 * u + 2            -- 200% of standard size
local u2_25 = 2.25 * u          -- 225% of standard size
local u2_75 = 2.75 * u          -- 275% of standard size
local u6_25 = 6.25 * u + 8      -- 625% of standard size

addon.default_keyboard_layouts = {

    -- ISO --
    -- QWERTZ --
    ['QWERTZ_100%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },

        -- 2 row

        { '^',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { 'ß',              688,  -68,  u,         u },
        { '´',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'Q',              98,   -130, u,         u },
        { 'W',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Z',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { 'Ü',              718,  -130, u,         u },
        { '+',              780,  -130, u,         u },
        { 'ENTER',          859,  -130, u1_25,     u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { 'Ö',              673,  -192, u,         u },
        { "Ä",              735,  -192, u,         u },
        { '#',              797,  -192, u,         u },

        -- 5 row
        { 'LSHIFT',         6,    -254, u1_25 + 2, u },
        { '<',              85,   -254, u,         u },
        { 'Y',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '.',              643,  -254, u,         u },
        { '-',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u1_25 + 2, u },
        { 'RWIN',           705,  -316, u1_25,     u },
        { 'MENU',           782,  -316, u1_25,     u },
        { 'RCTRL',          859,  -316, u1_25,     u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN',    949,  -6,   u,         u },
        { 'SCROLLLOCK',     1011, -6,   u,         u },
        { 'PAUSE',          1073, -6,   u,         u },

        { 'INSERT',         949,  -68,  u,         u },
        { 'HOME',           1011, -68,  u,         u },
        { 'PAGEUP',         1073, -68,  u,         u },

        { 'DELETE',         949,  -130, u,         u },
        { 'END',            1011, -130, u,         u },
        { 'PAGEDOWN',       1073, -130, u,         u },

        { 'LEFT',           949,  -316, u,         u },
        { 'DOWN',           1011, -316, u,         u },
        { 'UP',             1011, -254, u,         u },
        { 'RIGHT',          1073, -316, u,         u },

        -- 100% -----------------------------------------

        { 'NUMLOCK',        1148, -68,  u,         u },
        { 'NUMPADDIVIDE',   1210, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1272, -68,  u,         u },
        { 'NUMPADMINUS',    1334, -68,  u,         u },

        { 'NUMPAD7',        1148, -130, u,         u },
        { 'NUMPAD8',        1210, -130, u,         u },
        { 'NUMPAD9',        1272, -130, u,         u },
        { 'NUMPADPLUS',     1334, -130, u,         u2 },

        { 'NUMPAD4',        1148, -192, u,         u },
        { 'NUMPAD5',        1210, -192, u,         u },
        { 'NUMPAD6',        1272, -192, u,         u },

        { 'NUMPAD1',        1148, -254, u,         u },
        { 'NUMPAD2',        1210, -254, u,         u },
        { 'NUMPAD3',        1272, -254, u,         u },
        { 'ENTER',          1334, -254, u,         u2 },

        { 'NUMPAD0',        1148, -316, u2,        u },
        { 'NUMPADDECIMAL',  1272, -316, u,         u },
    },
    ['QWERTZ_96%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        { 'F1',             68,   -6,   u,         u },
        { 'F2',             130,  -6,   u,         u },
        { 'F3',             192,  -6,   u,         u },
        { 'F4',             254,  -6,   u,         u },
        { 'F5',             316,  -6,   u,         u },
        { 'F6',             378,  -6,   u,         u },
        { 'F7',             440,  -6,   u,         u },
        { 'F8',             502,  -6,   u,         u },
        { 'F9',             564,  -6,   u,         u },
        { 'F10',            626,  -6,   u,         u },
        { 'F11',            688,  -6,   u,         u },
        { 'F12',            750,  -6,   u,         u },
        { 'DELETE',         812,  -6,   u,         u },
        { 'INSERT',         874,  -6,   u,         u },
        { 'HOME',           936,  -6,   u,         u },
        { 'END',            998,  -6,   u,         u },
        { 'PAGEUP',         1060, -6,   u,         u },
        { 'PAGEDOWN',       1122, -6,   u,         u },


        -- 2 row

        { '^',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { 'ß',              688,  -68,  u,         u },
        { '´',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },
        { 'NUMLOCK',        936,  -68,  u,         u },
        { 'NUMPADDIVIDE',   998,  -68,  u,         u },
        { 'NUMPADMULTIPLY', 1060, -68,  u,         u },
        { 'NUMPADMINUS',    1122, -68,  u,         u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'Q',              98,   -130, u,         u },
        { 'W',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Z',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { 'Ü',              718,  -130, u,         u },
        { '+',              780,  -130, u,         u },
        { 'ENTER',          859,  -130, u1_25,     u2 },
        { 'NUMPAD7',        936,  -130, u,         u },
        { 'NUMPAD8',        998,  -130, u,         u },
        { 'NUMPAD9',        1060, -130, u,         u },
        { 'NUMPADPLUS',     1122, -130, u,         u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { 'Ö',              673,  -192, u,         u },
        { "Ä",              735,  -192, u,         u },
        { '#',              797,  -192, u,         u },
        { 'NUMPAD4',        936,  -192, u,         u },
        { 'NUMPAD5',        998,  -192, u,         u },
        { 'NUMPAD6',        1060, -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u1_25 + 2, u },
        { '<',              85,   -254, u,         u },
        { 'Y',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '.',              643,  -254, u,         u },
        { '-',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, 105,       u },
        { 'UP',             874,  -254, u,         u },
        { 'NUMPAD1',        936,  -254, u,         u },
        { 'NUMPAD2',        998,  -254, u,         u },
        { 'NUMPAD3',        1060, -254, u,         u },
        { 'ENTER',          1122, -254, u,         u2 },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u,         u },
        { 'MENU',           688,  -316, u,         u },
        { 'RCTRL',          750,  -316, u,         u },
        { 'LEFT',           812,  -316, u,         u },
        { 'DOWN',           874,  -316, u,         u },
        { 'RIGHT',          936,  -316, u,         u },
        { 'NUMPAD0',        998,  -316, u,         u },
        { 'NUMPADDECIMAL',  1060, -316, u,         u },
    },
    ['QWERTZ_80%'] = {

        -- 1 row

        { 'ESCAPE',      6,    -6,   u,         u },
        -- 1u gap
        { 'F1',          130,  -6,   u,         u },
        { 'F2',          192,  -6,   u,         u },
        { 'F3',          254,  -6,   u,         u },
        { 'F4',          316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',          409,  -6,   u,         u },
        { 'F6',          471,  -6,   u,         u },
        { 'F7',          533,  -6,   u,         u },
        { 'F8',          595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',          688,  -6,   u,         u },
        { 'F10',         750,  -6,   u,         u },
        { 'F11',         812,  -6,   u,         u },
        { 'F12',         874,  -6,   u,         u },

        -- 2 row

        { '^',           6,    -68,  u,         u },
        { '1',           68,   -68,  u,         u },
        { '2',           130,  -68,  u,         u },
        { '3',           192,  -68,  u,         u },
        { '4',           254,  -68,  u,         u },
        { '5',           316,  -68,  u,         u },
        { '6',           378,  -68,  u,         u },
        { '7',           440,  -68,  u,         u },
        { '8',           502,  -68,  u,         u },
        { '9',           564,  -68,  u,         u },
        { '0',           626,  -68,  u,         u },
        { 'ß',           688,  -68,  u,         u },
        { '´',           750,  -68,  u,         u },
        { 'BACKSPACE',   812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',         6,    -130, u1_5 + 1,  u },
        { 'Q',           98,   -130, u,         u },
        { 'W',           160,  -130, u,         u },
        { 'E',           222,  -130, u,         u },
        { 'R',           284,  -130, u,         u },
        { 'T',           346,  -130, u,         u },
        { 'Z',           408,  -130, u,         u },
        { 'U',           470,  -130, u,         u },
        { 'I',           532,  -130, u,         u },
        { 'O',           594,  -130, u,         u },
        { 'P',           656,  -130, u,         u },
        { 'Ü',           718,  -130, u,         u },
        { '+',           780,  -130, u,         u },
        { 'ENTER',       859,  -130, u1_25,     u2 },

        -- 4 row

        { 'CAPSLOCK',    6,    -192, u1_75 + 2, u },
        { 'A',           115,  -192, u,         u },
        { 'S',           177,  -192, u,         u },
        { 'D',           239,  -192, u,         u },
        { 'F',           301,  -192, u,         u },
        { 'G',           363,  -192, u,         u },
        { 'H',           425,  -192, u,         u },
        { 'J',           487,  -192, u,         u },
        { 'K',           549,  -192, u,         u },
        { 'L',           611,  -192, u,         u },
        { 'Ö',           673,  -192, u,         u },
        { "Ä",           735,  -192, u,         u },
        { '#',           797,  -192, u,         u },

        -- 5 row

        { 'LSHIFT',      6,    -254, u1_25 + 2, u },
        { '<',           85,   -254, u,         u },
        { 'Y',           147,  -254, u,         u },
        { 'X',           209,  -254, u,         u },
        { 'C',           271,  -254, u,         u },
        { 'V',           333,  -254, u,         u },
        { 'B',           395,  -254, u,         u },
        { 'N',           457,  -254, u,         u },
        { 'M',           519,  -254, u,         u },
        { ',',           581,  -254, u,         u },
        { '.',           643,  -254, u,         u },
        { '-',           705,  -254, u,         u },
        { 'RSHIFT',      767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',       6,    -316, u1_25 + 2, u },
        { 'LWIN',        85,   -316, u1_25 + 1, u },
        { 'LALT',        163,  -316, u1_25 + 1, u },
        { 'SPACE',       241,  -316, u6_25,     u },
        { 'RALT',        626,  -316, u1_25 + 2, u },
        { 'RWIN',        705,  -316, u1_25,     u },
        { 'MENU',        782,  -316, u1_25,     u },
        { 'RCTRL',       859,  -316, u1_25,     u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN', 949,  -6,   u,         u },
        { 'SCROLLLOCK',  1011, -6,   u,         u },
        { 'PAUSE',       1073, -6,   u,         u },

        { 'INSERT',      949,  -68,  u,         u },
        { 'HOME',        1011, -68,  u,         u },
        { 'PAGEUP',      1073, -68,  u,         u },

        { 'DELETE',      949,  -130, u,         u },
        { 'END',         1011, -130, u,         u },
        { 'PAGEDOWN',    1073, -130, u,         u },

        { 'LEFT',        949,  -316, u,         u },
        { 'DOWN',        1011, -316, u,         u },
        { 'UP',          1011, -254, u,         u },
        { 'RIGHT',       1073, -316, u,         u },
    },
    ['QWERTZ_75%'] = {

        -- 1 row

        { 'ESCAPE',      6,   -6,   u,         u },
        { 'F1',          68,  -6,   u,         u },
        { 'F2',          130, -6,   u,         u },
        { 'F3',          192, -6,   u,         u },
        { 'F4',          254, -6,   u,         u },
        { 'F5',          316, -6,   u,         u },
        { 'F6',          378, -6,   u,         u },
        { 'F7',          440, -6,   u,         u },
        { 'F8',          502, -6,   u,         u },
        { 'F9',          564, -6,   u,         u },
        { 'F10',         626, -6,   u,         u },
        { 'F11',         688, -6,   u,         u },
        { 'F12',         750, -6,   u,         u },
        { 'PRINTSCREEN', 812, -6,   u,         u },
        { 'PAUSE',       874, -6,   u,         u },
        { 'DELETE',      936, -6,   u,         u },

        -- 2 row

        { '^',           6,   -68,  u,         u },
        { '1',           68,  -68,  u,         u },
        { '2',           130, -68,  u,         u },
        { '3',           192, -68,  u,         u },
        { '4',           254, -68,  u,         u },
        { '5',           316, -68,  u,         u },
        { '6',           378, -68,  u,         u },
        { '7',           440, -68,  u,         u },
        { '8',           502, -68,  u,         u },
        { '9',           564, -68,  u,         u },
        { '0',           626, -68,  u,         u },
        { 'ß',           688, -68,  u,         u },
        { '´',           750, -68,  u,         u },
        { 'BACKSPACE',   812, -68,  u2,        u },
        { 'HOME',        936, -68,  u,         u },

        -- 3 row

        { 'TAB',         6,   -130, u1_5 + 1,  u },
        { 'Q',           98,  -130, u,         u },
        { 'W',           160, -130, u,         u },
        { 'E',           222, -130, u,         u },
        { 'R',           284, -130, u,         u },
        { 'T',           346, -130, u,         u },
        { 'Z',           408, -130, u,         u },
        { 'U',           470, -130, u,         u },
        { 'I',           532, -130, u,         u },
        { 'O',           594, -130, u,         u },
        { 'P',           656, -130, u,         u },
        { 'Ü',           718, -130, u,         u },
        { '+',           780, -130, u,         u },
        { 'ENTER',       859, -130, u1_25,     u2 },
        { 'END',         936, -130, u,         u },

        -- 4 row

        { 'CAPSLOCK',    6,   -192, u1_75 + 2, u },
        { 'A',           115, -192, u,         u },
        { 'S',           177, -192, u,         u },
        { 'D',           239, -192, u,         u },
        { 'F',           301, -192, u,         u },
        { 'G',           363, -192, u,         u },
        { 'H',           425, -192, u,         u },
        { 'J',           487, -192, u,         u },
        { 'K',           549, -192, u,         u },
        { 'L',           611, -192, u,         u },
        { 'Ö',           673, -192, u,         u },
        { "Ä",           735, -192, u,         u },
        { '#',           797, -192, u,         u },
        { 'PAGEUP',      936, -192, u,         u },

        -- 5 row

        { 'LSHIFT',      6,   -254, u1_25 + 2, u },
        { '<',           85,  -254, u,         u },
        { 'Y',           147, -254, u,         u },
        { 'X',           209, -254, u,         u },
        { 'C',           271, -254, u,         u },
        { 'V',           333, -254, u,         u },
        { 'B',           395, -254, u,         u },
        { 'N',           457, -254, u,         u },
        { 'M',           519, -254, u,         u },
        { ',',           581, -254, u,         u },
        { '.',           643, -254, u,         u },
        { '-',           705, -254, u,         u },
        { 'RSHIFT',      767, -254, 105,       u },
        { 'UP',          874, -254, u,         u },
        { 'PAGEDOWN',    936, -254, u,         u },

        -- 6 row

        { 'LCTRL',       6,   -316, u1_25 + 2, u },
        { 'LWIN',        85,  -316, u1_25 + 1, u },
        { 'LALT',        163, -316, u1_25 + 1, u },
        { 'SPACE',       241, -316, u6_25,     u },
        { 'RALT',        626, -316, u,         u },
        { 'MENU',        688, -316, u,         u },
        { 'RCTRL',       750, -316, u,         u },
        { 'LEFT',        812, -316, u,         u },
        { 'DOWN',        874, -316, u,         u },
        { 'RIGHT',       936, -316, u,         u },
    },
    ['QWERTZ_60%'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        { '1',         68,  -6,   u,         u },
        { '2',         130, -6,   u,         u },
        { '3',         192, -6,   u,         u },
        { '4',         254, -6,   u,         u },
        { '5',         316, -6,   u,         u },
        { '6',         378, -6,   u,         u },
        { '7',         440, -6,   u,         u },
        { '8',         502, -6,   u,         u },
        { '9',         564, -6,   u,         u },
        { '0',         626, -6,   u,         u },
        { 'ß',         688, -6,   u,         u },
        { '´',         750, -6,   u,         u },
        { 'BACKSPACE', 812, -6,   u2,        u },

        -- 2 row

        { 'TAB',       6,   -68,  u1_5 + 1,  u },
        { 'Q',         98,  -68,  u,         u },
        { 'W',         160, -68,  u,         u },
        { 'E',         222, -68,  u,         u },
        { 'R',         284, -68,  u,         u },
        { 'T',         346, -68,  u,         u },
        { 'Z',         408, -68,  u,         u },
        { 'U',         470, -68,  u,         u },
        { 'I',         532, -68,  u,         u },
        { 'O',         594, -68,  u,         u },
        { 'P',         656, -68,  u,         u },
        { 'Ü',         718, -68,  u,         u },
        { '+',         780, -68,  u,         u },
        { 'ENTER',     859, -68,  u1_25,     u2 },

        -- 3 row

        { 'CAPSLOCK',  6,   -130, u1_75 + 2, u },
        { 'A',         115, -130, u,         u },
        { 'S',         177, -130, u,         u },
        { 'D',         239, -130, u,         u },
        { 'F',         301, -130, u,         u },
        { 'G',         363, -130, u,         u },
        { 'H',         425, -130, u,         u },
        { 'J',         487, -130, u,         u },
        { 'K',         549, -130, u,         u },
        { 'L',         611, -130, u,         u },
        { 'Ö',         673, -130, u,         u },
        { "Ä",         735, -130, u,         u },
        { '#',         797, -130, u,         u },

        -- 4 row

        { 'LSHIFT',    6,   -192, u1_25 + 2, u },
        { '<',         85,  -192, u,         u },
        { 'Y',         147, -192, u,         u },
        { 'X',         209, -192, u,         u },
        { 'C',         271, -192, u,         u },
        { 'V',         333, -192, u,         u },
        { 'B',         395, -192, u,         u },
        { 'N',         457, -192, u,         u },
        { 'M',         519, -192, u,         u },
        { ',',         581, -192, u,         u },
        { '.',         643, -192, u,         u },
        { '-',         705, -192, u,         u },
        { 'RSHIFT',    767, -192, u2_75 + 2, u },

        -- 5 row

        { 'LCTRL',     6,   -254, u1_25 + 2, u },
        { 'LWIN',      85,  -254, u1_25 + 1, u },
        { 'LALT',      163, -254, u1_25 + 1, u },
        { 'SPACE',     241, -254, u6_25,     u },
        { 'RALT',      626, -254, u1_25 + 2, u },
        { 'RWIN',      705, -254, u1_25,     u },
        { 'MENU',      782, -254, u1_25,     u },
        { 'RCTRL',     859, -254, u1_25,     u },
    },
    ['QWERTZ_1800'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },
        { 'HOME',           956,  -6,   u,         u },
        { 'END',            1018, -6,   u,         u },
        { 'PAGEUP',         1080, -6,   u,         u },
        { 'PAGEDOWN',       1142, -6,   u,         u },


        -- 2 row

        { '^',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { 'ß',              688,  -68,  u,         u },
        { '´',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },
        { 'NUMLOCK',        956,  -68,  u,         u },
        { 'NUMPADDIVIDE',   1018, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1080, -68,  u,         u },
        { 'NUMPADMINUS',    1142, -68,  u,         u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'Q',              98,   -130, u,         u },
        { 'W',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Z',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { 'Ü',              718,  -130, u,         u },
        { '+',              780,  -130, u,         u },
        { 'ENTER',          859,  -130, u1_25,     u2 },
        { 'NUMPAD7',        956,  -130, u,         u },
        { 'NUMPAD8',        1018, -130, u,         u },
        { 'NUMPAD9',        1080, -130, u,         u },
        { 'NUMPADPLUS',     1142, -130, u,         u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { 'Ö',              673,  -192, u,         u },
        { "Ä",              735,  -192, u,         u },
        { '#',              797,  -192, u,         u },
        { 'NUMPAD4',        956,  -192, u,         u },
        { 'NUMPAD5',        1018, -192, u,         u },
        { 'NUMPAD6',        1080, -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u1_25 + 2, u },
        { '<',              85,   -254, u,         u },
        { 'Y',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '.',              643,  -254, u,         u },
        { '-',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, 105,       u },
        { 'UP',             884,  -264, u,         u },
        { 'NUMPAD1',        956,  -254, u,         u },
        { 'NUMPAD2',        1018, -254, u,         u },
        { 'NUMPAD3',        1080, -254, u,         u },
        { 'ENTER',          1142, -254, u,         u2 },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u,         u },
        { 'MENU',           688,  -316, u,         u },
        { 'RCTRL',          750,  -316, u,         u },
        { 'LEFT',           822,  -326, u,         u },
        { 'DOWN',           884,  -326, u,         u },
        { 'RIGHT',          946,  -326, u,         u },
        { 'NUMPAD0',        1018, -316, u,         u },
        { 'NUMPADDECIMAL',  1080, -316, u,         u },
    },
    ['QWERTZ_HALF'] = {

        -- 1 row

        { 'ESCAPE',   6,   -6,   u,         u },
        -- 1u gap
        { 'F1',       130, -6,   u,         u },
        { 'F2',       192, -6,   u,         u },
        { 'F3',       254, -6,   u,         u },
        { 'F4',       316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',       409, -6,   u,         u },

        -- 2 row

        { '^',        6,   -68,  u,         u },
        { '1',        68,  -68,  u,         u },
        { '2',        130, -68,  u,         u },
        { '3',        192, -68,  u,         u },
        { '4',        254, -68,  u,         u },
        { '5',        316, -68,  u,         u },
        { '6',        378, -68,  u,         u },

        -- 3 row

        { 'TAB',      6,   -130, u1_5 + 1,  u },
        { 'Q',        98,  -130, u,         u },
        { 'W',        160, -130, u,         u },
        { 'E',        222, -130, u,         u },
        { 'R',        284, -130, u,         u },
        { 'T',        346, -130, u,         u },
        { 'Z',        408, -130, u,         u },

        -- 4 row

        { 'CAPSLOCK', 6,   -192, u1_75 + 2, u },
        { 'A',        115, -192, u,         u },
        { 'S',        177, -192, u,         u },
        { 'D',        239, -192, u,         u },
        { 'F',        301, -192, u,         u },
        { 'G',        363, -192, u,         u },
        { 'H',        425, -192, u,         u },

        -- 5 row

        { 'LSHIFT',   6,   -254, u1_25 + 2, u },
        { '<',        85,  -254, u,         u },
        { 'Y',        147, -254, u,         u },
        { 'X',        209, -254, u,         u },
        { 'C',        271, -254, u,         u },
        { 'V',        333, -254, u,         u },
        { 'B',        395, -254, u,         u },

        -- 6 row

        { 'LCTRL',    6,   -316, u1_25 + 2, u },
        { 'LWIN',     85,  -316, u1_25 + 1, u },
        { 'LALT',     163, -316, u1_25 + 1, u },
        { 'SPACE',    241, -316, 245,       u },
    },
    ['QWERTZ_PRIMARY'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        -- 1u gap
        { 'F1',        130, -6,   u,         u },
        { 'F2',        192, -6,   u,         u },
        { 'F3',        254, -6,   u,         u },
        { 'F4',        316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',        409, -6,   u,         u },
        { 'F6',        471, -6,   u,         u },
        { 'F7',        533, -6,   u,         u },
        { 'F8',        595, -6,   u,         u },
        -- 0,5u gap
        { 'F9',        688, -6,   u,         u },
        { 'F10',       750, -6,   u,         u },
        { 'F11',       812, -6,   u,         u },
        { 'F12',       874, -6,   u,         u },

        -- 2 row

        { '^',         6,   -68,  u,         u },
        { '1',         68,  -68,  u,         u },
        { '2',         130, -68,  u,         u },
        { '3',         192, -68,  u,         u },
        { '4',         254, -68,  u,         u },
        { '5',         316, -68,  u,         u },
        { '6',         378, -68,  u,         u },
        { '7',         440, -68,  u,         u },
        { '8',         502, -68,  u,         u },
        { '9',         564, -68,  u,         u },
        { '0',         626, -68,  u,         u },
        { 'ß',         688, -68,  u,         u },
        { '´',         750, -68,  u,         u },
        { 'BACKSPACE', 812, -68,  u2,        u },

        -- 3 row

        { 'TAB',       6,   -130, u1_5 + 1,  u },
        { 'Q',         98,  -130, u,         u },
        { 'W',         160, -130, u,         u },
        { 'E',         222, -130, u,         u },
        { 'R',         284, -130, u,         u },
        { 'T',         346, -130, u,         u },
        { 'Z',         408, -130, u,         u },
        { 'U',         470, -130, u,         u },
        { 'I',         532, -130, u,         u },
        { 'O',         594, -130, u,         u },
        { 'P',         656, -130, u,         u },
        { 'Ü',         718, -130, u,         u },
        { '+',         780, -130, u,         u },
        { 'ENTER',     859, -130, u1_25,     u2 },

        -- 4 row

        { 'CAPSLOCK',  6,   -192, u1_75 + 2, u },
        { 'A',         115, -192, u,         u },
        { 'S',         177, -192, u,         u },
        { 'D',         239, -192, u,         u },
        { 'F',         301, -192, u,         u },
        { 'G',         363, -192, u,         u },
        { 'H',         425, -192, u,         u },
        { 'J',         487, -192, u,         u },
        { 'K',         549, -192, u,         u },
        { 'L',         611, -192, u,         u },
        { 'Ö',         673, -192, u,         u },
        { "Ä",         735, -192, u,         u },
        { '#',         797, -192, u,         u },

        -- 5 row

        { 'LSHIFT',    6,   -254, u1_25 + 2, u },
        { '<',         85,  -254, u,         u },
        { 'Y',         147, -254, u,         u },
        { 'X',         209, -254, u,         u },
        { 'C',         271, -254, u,         u },
        { 'V',         333, -254, u,         u },
        { 'B',         395, -254, u,         u },
        { 'N',         457, -254, u,         u },
        { 'M',         519, -254, u,         u },
        { ',',         581, -254, u,         u },
        { '.',         643, -254, u,         u },
        { '-',         705, -254, u,         u },
        { 'RSHIFT',    767, -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',     6,   -316, u1_25 + 2, u },
        { 'LWIN',      85,  -316, u1_25 + 1, u },
        { 'LALT',      163, -316, u1_25 + 1, u },
        { 'SPACE',     241, -316, u6_25,     u },
        { 'RALT',      626, -316, u1_25 + 2, u },
        { 'RWIN',      705, -316, u1_25,     u },
        { 'MENU',      782, -316, u1_25,     u },
        { 'RCTRL',     859, -316, u1_25,     u },
    },
    --
    -- AZERTY --
    ['AZERTY_100%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },

        -- 2 row

        { '²',              6,    -68,  u,         u },
        { '&',              68,   -68,  u,         u },
        { 'é',              130,  -68,  u,         u },
        { '"',              192,  -68,  u,         u },
        { "'",              254,  -68,  u,         u },
        { '(',              316,  -68,  u,         u },
        { '-',              378,  -68,  u,         u },
        { 'è',              440,  -68,  u,         u },
        { '_',              502,  -68,  u,         u },
        { 'ç',              564,  -68,  u,         u },
        { 'à',              626,  -68,  u,         u },
        { ')',              688,  -68,  u,         u },
        { '=',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'A',              98,   -130, u,         u },
        { 'Z',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Y',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { '^',              718,  -130, u,         u },
        { '$',              780,  -130, u,         u },
        { 'ENTER',          859,  -130, u1_25,     u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'Q',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { 'M',              673,  -192, u,         u },
        { 'ù',              735,  -192, u,         u },
        { '*',              797,  -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u1_25 + 2, u },
        { '<',              85,   -254, u,         u },
        { 'W',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { ',',              519,  -254, u,         u },
        { ';',              581,  -254, u,         u },
        { ':',              643,  -254, u,         u },
        { '!',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u1_25 + 2, u },
        { 'RWIN',           705,  -316, u1_25,     u },
        { 'MENU',           782,  -316, u1_25,     u },
        { 'RCTRL',          859,  -316, u1_25,     u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN',    949,  -6,   u,         u },
        { 'SCROLLLOCK',     1011, -6,   u,         u },
        { 'PAUSE',          1073, -6,   u,         u },

        { 'INSERT',         949,  -68,  u,         u },
        { 'HOME',           1011, -68,  u,         u },
        { 'PAGEUP',         1073, -68,  u,         u },

        { 'DELETE',         949,  -130, u,         u },
        { 'END',            1011, -130, u,         u },
        { 'PAGEDOWN',       1073, -130, u,         u },

        { 'LEFT',           949,  -316, u,         u },
        { 'DOWN',           1011, -316, u,         u },
        { 'UP',             1011, -254, u,         u },
        { 'RIGHT',          1073, -316, u,         u },

        -- 100% -----------------------------------------

        { 'NUMLOCK',        1148, -68,  u,         u },
        { 'NUMPADDIVIDE',   1210, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1272, -68,  u,         u },
        { 'NUMPADMINUS',    1334, -68,  u,         u },

        { 'NUMPAD7',        1148, -130, u,         u },
        { 'NUMPAD8',        1210, -130, u,         u },
        { 'NUMPAD9',        1272, -130, u,         u },
        { 'NUMPADPLUS',     1334, -130, u,         u2 },

        { 'NUMPAD4',        1148, -192, u,         u },
        { 'NUMPAD5',        1210, -192, u,         u },
        { 'NUMPAD6',        1272, -192, u,         u },

        { 'NUMPAD1',        1148, -254, u,         u },
        { 'NUMPAD2',        1210, -254, u,         u },
        { 'NUMPAD3',        1272, -254, u,         u },
        { 'ENTER',          1334, -254, u,         u2 },

        { 'NUMPAD0',        1148, -316, u2,        u },
        { 'NUMPADDECIMAL',  1272, -316, u,         u },
    },
    ['AZERTY_96%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        { 'F1',             68,   -6,   u,         u },
        { 'F2',             130,  -6,   u,         u },
        { 'F3',             192,  -6,   u,         u },
        { 'F4',             254,  -6,   u,         u },
        { 'F5',             316,  -6,   u,         u },
        { 'F6',             378,  -6,   u,         u },
        { 'F7',             440,  -6,   u,         u },
        { 'F8',             502,  -6,   u,         u },
        { 'F9',             564,  -6,   u,         u },
        { 'F10',            626,  -6,   u,         u },
        { 'F11',            688,  -6,   u,         u },
        { 'F12',            750,  -6,   u,         u },
        { 'DELETE',         812,  -6,   u,         u },
        { 'INSERT',         874,  -6,   u,         u },
        { 'HOME',           936,  -6,   u,         u },
        { 'END',            998,  -6,   u,         u },
        { 'PAGEUP',         1060, -6,   u,         u },
        { 'PAGEDOWN',       1122, -6,   u,         u },


        -- 2 row

        { '²',              6,    -68,  u,         u },
        { '&',              68,   -68,  u,         u },
        { 'é',              130,  -68,  u,         u },
        { '"',              192,  -68,  u,         u },
        { "'",              254,  -68,  u,         u },
        { '(',              316,  -68,  u,         u },
        { '-',              378,  -68,  u,         u },
        { 'è',              440,  -68,  u,         u },
        { '_',              502,  -68,  u,         u },
        { 'ç',              564,  -68,  u,         u },
        { 'à',              626,  -68,  u,         u },
        { ')',              688,  -68,  u,         u },
        { '=',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },
        { 'NUMLOCK',        936,  -68,  u,         u },
        { 'NUMPADDIVIDE',   998,  -68,  u,         u },
        { 'NUMPADMULTIPLY', 1060, -68,  u,         u },
        { 'NUMPADMINUS',    1122, -68,  u,         u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'A',              98,   -130, u,         u },
        { 'Z',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Y',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { '^',              718,  -130, u,         u },
        { '$',              780,  -130, u,         u },
        { 'ENTER',          859,  -130, u1_25,     u2 },
        { 'NUMPAD7',        936,  -130, u,         u },
        { 'NUMPAD8',        998,  -130, u,         u },
        { 'NUMPAD9',        1060, -130, u,         u },
        { 'NUMPADPLUS',     1122, -130, u,         u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'Q',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { 'M',              673,  -192, u,         u },
        { 'ù',              735,  -192, u,         u },
        { '*',              797,  -192, u,         u },
        { 'NUMPAD4',        936,  -192, u,         u },
        { 'NUMPAD5',        998,  -192, u,         u },
        { 'NUMPAD6',        1060, -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u1_25 + 2, u },
        { '<',              85,   -254, u,         u },
        { 'W',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { ',',              519,  -254, u,         u },
        { ';',              581,  -254, u,         u },
        { ':',              643,  -254, u,         u },
        { '!',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, 105,       u },
        { 'UP',             874,  -254, u,         u },
        { 'NUMPAD1',        936,  -254, u,         u },
        { 'NUMPAD2',        998,  -254, u,         u },
        { 'NUMPAD3',        1060, -254, u,         u },
        { 'ENTER',          1122, -254, u,         u2 },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u,         u },
        { 'MENU',           688,  -316, u,         u },
        { 'RCTRL',          750,  -316, u,         u },
        { 'LEFT',           812,  -316, u,         u },
        { 'DOWN',           874,  -316, u,         u },
        { 'RIGHT',          936,  -316, u,         u },
        { 'NUMPAD0',        998,  -316, u,         u },
        { 'NUMPADDECIMAL',  1060, -316, u,         u },
    },
    ['AZERTY_80%'] = {

        -- 1 row

        { 'ESCAPE',      6,    -6,   u,         u },
        -- 1u gap
        { 'F1',          130,  -6,   u,         u },
        { 'F2',          192,  -6,   u,         u },
        { 'F3',          254,  -6,   u,         u },
        { 'F4',          316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',          409,  -6,   u,         u },
        { 'F6',          471,  -6,   u,         u },
        { 'F7',          533,  -6,   u,         u },
        { 'F8',          595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',          688,  -6,   u,         u },
        { 'F10',         750,  -6,   u,         u },
        { 'F11',         812,  -6,   u,         u },
        { 'F12',         874,  -6,   u,         u },

        -- 2 row

        { '²',           6,    -68,  u,         u },
        { '&',           68,   -68,  u,         u },
        { 'é',           130,  -68,  u,         u },
        { '"',           192,  -68,  u,         u },
        { "'",           254,  -68,  u,         u },
        { '(',           316,  -68,  u,         u },
        { '-',           378,  -68,  u,         u },
        { 'è',           440,  -68,  u,         u },
        { '_',           502,  -68,  u,         u },
        { 'ç',           564,  -68,  u,         u },
        { 'à',           626,  -68,  u,         u },
        { ')',           688,  -68,  u,         u },
        { '=',           750,  -68,  u,         u },
        { 'BACKSPACE',   812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',         6,    -130, u1_5 + 1,  u },
        { 'A',           98,   -130, u,         u },
        { 'Z',           160,  -130, u,         u },
        { 'E',           222,  -130, u,         u },
        { 'R',           284,  -130, u,         u },
        { 'T',           346,  -130, u,         u },
        { 'Y',           408,  -130, u,         u },
        { 'U',           470,  -130, u,         u },
        { 'I',           532,  -130, u,         u },
        { 'O',           594,  -130, u,         u },
        { 'P',           656,  -130, u,         u },
        { '^',           718,  -130, u,         u },
        { '$',           780,  -130, u,         u },
        { 'ENTER',       859,  -130, u1_25,     u2 },

        -- 4 row

        { 'CAPSLOCK',    6,    -192, u1_75 + 2, u },
        { 'Q',           115,  -192, u,         u },
        { 'S',           177,  -192, u,         u },
        { 'D',           239,  -192, u,         u },
        { 'F',           301,  -192, u,         u },
        { 'G',           363,  -192, u,         u },
        { 'H',           425,  -192, u,         u },
        { 'J',           487,  -192, u,         u },
        { 'K',           549,  -192, u,         u },
        { 'L',           611,  -192, u,         u },
        { 'M',           673,  -192, u,         u },
        { 'ù',           735,  -192, u,         u },
        { '*',           797,  -192, u,         u },

        -- 5 row

        { 'LSHIFT',      6,    -254, u1_25 + 2, u },
        { '<',           85,   -254, u,         u },
        { 'W',           147,  -254, u,         u },
        { 'X',           209,  -254, u,         u },
        { 'C',           271,  -254, u,         u },
        { 'V',           333,  -254, u,         u },
        { 'B',           395,  -254, u,         u },
        { 'N',           457,  -254, u,         u },
        { ',',           519,  -254, u,         u },
        { ';',           581,  -254, u,         u },
        { ':',           643,  -254, u,         u },
        { '!',           705,  -254, u,         u },
        { 'RSHIFT',      767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',       6,    -316, u1_25 + 2, u },
        { 'LWIN',        85,   -316, u1_25 + 1, u },
        { 'LALT',        163,  -316, u1_25 + 1, u },
        { 'SPACE',       241,  -316, u6_25,     u },
        { 'RALT',        626,  -316, u1_25 + 2, u },
        { 'RWIN',        705,  -316, u1_25,     u },
        { 'MENU',        782,  -316, u1_25,     u },
        { 'RCTRL',       859,  -316, u1_25,     u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN', 949,  -6,   u,         u },
        { 'SCROLLLOCK',  1011, -6,   u,         u },
        { 'PAUSE',       1073, -6,   u,         u },

        { 'INSERT',      949,  -68,  u,         u },
        { 'HOME',        1011, -68,  u,         u },
        { 'PAGEUP',      1073, -68,  u,         u },

        { 'DELETE',      949,  -130, u,         u },
        { 'END',         1011, -130, u,         u },
        { 'PAGEDOWN',    1073, -130, u,         u },

        { 'LEFT',        949,  -316, u,         u },
        { 'DOWN',        1011, -316, u,         u },
        { 'UP',          1011, -254, u,         u },
        { 'RIGHT',       1073, -316, u,         u },
    },
    ['AZERTY_75%'] = {

        -- 1 row

        { 'ESCAPE',      6,   -6,   u,         u },
        { 'F1',          68,  -6,   u,         u },
        { 'F2',          130, -6,   u,         u },
        { 'F3',          192, -6,   u,         u },
        { 'F4',          254, -6,   u,         u },
        { 'F5',          316, -6,   u,         u },
        { 'F6',          378, -6,   u,         u },
        { 'F7',          440, -6,   u,         u },
        { 'F8',          502, -6,   u,         u },
        { 'F9',          564, -6,   u,         u },
        { 'F10',         626, -6,   u,         u },
        { 'F11',         688, -6,   u,         u },
        { 'F12',         750, -6,   u,         u },
        { 'PRINTSCREEN', 812, -6,   u,         u },
        { 'PAUSE',       874, -6,   u,         u },
        { 'DELETE',      936, -6,   u,         u },

        -- 2 row

        { '²',           6,   -68,  u,         u },
        { '&',           68,  -68,  u,         u },
        { 'é',           130, -68,  u,         u },
        { '"',           192, -68,  u,         u },
        { "'",           254, -68,  u,         u },
        { '(',           316, -68,  u,         u },
        { '-',           378, -68,  u,         u },
        { 'è',           440, -68,  u,         u },
        { '_',           502, -68,  u,         u },
        { 'ç',           564, -68,  u,         u },
        { 'à',           626, -68,  u,         u },
        { ')',           688, -68,  u,         u },
        { '=',           750, -68,  u,         u },
        { 'BACKSPACE',   812, -68,  u2,        u },
        { 'HOME',        936, -68,  u,         u },

        -- 3 row

        { 'TAB',         6,   -130, u1_5 + 1,  u },
        { 'A',           98,  -130, u,         u },
        { 'Z',           160, -130, u,         u },
        { 'E',           222, -130, u,         u },
        { 'R',           284, -130, u,         u },
        { 'T',           346, -130, u,         u },
        { 'Y',           408, -130, u,         u },
        { 'U',           470, -130, u,         u },
        { 'I',           532, -130, u,         u },
        { 'O',           594, -130, u,         u },
        { 'P',           656, -130, u,         u },
        { '^',           718, -130, u,         u },
        { '$',           780, -130, u,         u },
        { 'ENTER',       859, -130, u1_25,     u2 },
        { 'END',         936, -130, u,         u },

        -- 4 row

        { 'CAPSLOCK',    6,   -192, u1_75 + 2, u },
        { 'Q',           115, -192, u,         u },
        { 'S',           177, -192, u,         u },
        { 'D',           239, -192, u,         u },
        { 'F',           301, -192, u,         u },
        { 'G',           363, -192, u,         u },
        { 'H',           425, -192, u,         u },
        { 'J',           487, -192, u,         u },
        { 'K',           549, -192, u,         u },
        { 'L',           611, -192, u,         u },
        { 'M',           673, -192, u,         u },
        { 'ù',           735, -192, u,         u },
        { '*',           797, -192, u,         u },
        { 'PAGEUP',      936, -192, u,         u },

        -- 5 row

        { 'LSHIFT',      6,   -254, u1_25 + 2, u },
        { '<',           85,  -254, u,         u },
        { 'W',           147, -254, u,         u },
        { 'X',           209, -254, u,         u },
        { 'C',           271, -254, u,         u },
        { 'V',           333, -254, u,         u },
        { 'B',           395, -254, u,         u },
        { 'N',           457, -254, u,         u },
        { ',',           519, -254, u,         u },
        { ';',           581, -254, u,         u },
        { ':',           643, -254, u,         u },
        { '!',           705, -254, u,         u },
        { 'RSHIFT',      767, -254, 105,       u },
        { 'UP',          874, -254, u,         u },
        { 'PAGEDOWN',    936, -254, u,         u },

        -- 6 row

        { 'LCTRL',       6,   -316, u1_25 + 2, u },
        { 'LWIN',        85,  -316, u1_25 + 1, u },
        { 'LALT',        163, -316, u1_25 + 1, u },
        { 'SPACE',       241, -316, u6_25,     u },
        { 'RALT',        626, -316, u,         u },
        { 'MENU',        688, -316, u,         u },
        { 'RCTRL',       750, -316, u,         u },
        { 'LEFT',        812, -316, u,         u },
        { 'DOWN',        874, -316, u,         u },
        { 'RIGHT',       936, -316, u,         u },
    },
    ['AZERTY_60%'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        { '&',         68,  -6,   u,         u },
        { 'é',         130, -6,   u,         u },
        { '"',         192, -6,   u,         u },
        { "'",         254, -6,   u,         u },
        { '(',         316, -6,   u,         u },
        { '-',         378, -6,   u,         u },
        { 'è',         440, -6,   u,         u },
        { '_',         502, -6,   u,         u },
        { 'ç',         564, -6,   u,         u },
        { 'à',         626, -6,   u,         u },
        { ')',         688, -6,   u,         u },
        { '=',         750, -6,   u,         u },
        { 'BACKSPACE', 812, -6,   u2,        u },

        -- 2 row

        { 'TAB',       6,   -68,  u1_5 + 1,  u },
        { 'A',         98,  -68,  u,         u },
        { 'Z',         160, -68,  u,         u },
        { 'E',         222, -68,  u,         u },
        { 'R',         284, -68,  u,         u },
        { 'T',         346, -68,  u,         u },
        { 'Y',         408, -68,  u,         u },
        { 'U',         470, -68,  u,         u },
        { 'I',         532, -68,  u,         u },
        { 'O',         594, -68,  u,         u },
        { 'P',         656, -68,  u,         u },
        { '^',         718, -68,  u,         u },
        { '$',         780, -68,  u,         u },
        { 'ENTER',     859, -68,  u1_25,     u2 },

        -- 3 row

        { 'CAPSLOCK',  6,   -130, u1_75 + 2, u },
        { 'Q',         115, -130, u,         u },
        { 'S',         177, -130, u,         u },
        { 'D',         239, -130, u,         u },
        { 'F',         301, -130, u,         u },
        { 'G',         363, -130, u,         u },
        { 'H',         425, -130, u,         u },
        { 'J',         487, -130, u,         u },
        { 'K',         549, -130, u,         u },
        { 'L',         611, -130, u,         u },
        { 'M',         673, -130, u,         u },
        { 'ù',         735, -130, u,         u },
        { '*',         797, -130, u,         u },

        -- 4 row

        { 'LSHIFT',    6,   -192, u1_25 + 2, u },
        { '<',         85,  -192, u,         u },
        { 'W',         147, -192, u,         u },
        { 'X',         209, -192, u,         u },
        { 'C',         271, -192, u,         u },
        { 'V',         333, -192, u,         u },
        { 'B',         395, -192, u,         u },
        { 'N',         457, -192, u,         u },
        { ',',         519, -192, u,         u },
        { ';',         581, -192, u,         u },
        { ':',         643, -192, u,         u },
        { '!',         705, -192, u,         u },
        { 'RSHIFT',    767, -192, u2_75 + 2, u },

        -- 5 row

        { 'LCTRL',     6,   -254, u1_25 + 2, u },
        { 'LWIN',      85,  -254, u1_25 + 1, u },
        { 'LALT',      163, -254, u1_25 + 1, u },
        { 'SPACE',     241, -254, u6_25,     u },
        { 'RALT',      626, -254, u1_25 + 2, u },
        { 'RWIN',      705, -254, u1_25,     u },
        { 'MENU',      782, -254, u1_25,     u },
        { 'RCTRL',     859, -254, u1_25,     u },
    },
    ['AZERTY_1800'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },
        { 'HOME',           956,  -6,   u,         u },
        { 'END',            1018, -6,   u,         u },
        { 'PAGEUP',         1080, -6,   u,         u },
        { 'PAGEDOWN',       1142, -6,   u,         u },


        -- 2 row

        { '²',              6,    -68,  u,         u },
        { '&',              68,   -68,  u,         u },
        { 'é',              130,  -68,  u,         u },
        { '"',              192,  -68,  u,         u },
        { "'",              254,  -68,  u,         u },
        { '(',              316,  -68,  u,         u },
        { '-',              378,  -68,  u,         u },
        { 'è',              440,  -68,  u,         u },
        { '_',              502,  -68,  u,         u },
        { 'ç',              564,  -68,  u,         u },
        { 'à',              626,  -68,  u,         u },
        { ')',              688,  -68,  u,         u },
        { '=',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },
        { 'NUMLOCK',        956,  -68,  u,         u },
        { 'NUMPADDIVIDE',   1018, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1080, -68,  u,         u },
        { 'NUMPADMINUS',    1142, -68,  u,         u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'A',              98,   -130, u,         u },
        { 'Z',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Y',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { '^',              718,  -130, u,         u },
        { '$',              780,  -130, u,         u },
        { 'ENTER',          859,  -130, u1_25,     u2 },
        { 'NUMPAD7',        956,  -130, u,         u },
        { 'NUMPAD8',        1018, -130, u,         u },
        { 'NUMPAD9',        1080, -130, u,         u },
        { 'NUMPADPLUS',     1142, -130, u,         u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'Q',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { 'M',              673,  -192, u,         u },
        { 'ù',              735,  -192, u,         u },
        { '*',              797,  -192, u,         u },
        { 'NUMPAD4',        956,  -192, u,         u },
        { 'NUMPAD5',        1018, -192, u,         u },
        { 'NUMPAD6',        1080, -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u1_25 + 2, u },
        { '<',              85,   -254, u,         u },
        { 'W',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { ',',              519,  -254, u,         u },
        { ';',              581,  -254, u,         u },
        { ':',              643,  -254, u,         u },
        { '!',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, 105,       u },
        { 'UP',             884,  -264, u,         u },
        { 'NUMPAD1',        956,  -254, u,         u },
        { 'NUMPAD2',        1018, -254, u,         u },
        { 'NUMPAD3',        1080, -254, u,         u },
        { 'ENTER',          1142, -254, u,         u2 },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u,         u },
        { 'MENU',           688,  -316, u,         u },
        { 'RCTRL',          750,  -316, u,         u },
        { 'LEFT',           822,  -326, u,         u },
        { 'DOWN',           884,  -326, u,         u },
        { 'RIGHT',          946,  -326, u,         u },
        { 'NUMPAD0',        1018, -316, u,         u },
        { 'NUMPADDECIMAL',  1080, -316, u,         u },
    },
    ['AZERTY_HALF'] = {

        -- 1 row

        { 'ESCAPE',   6,   -6,   u,         u },
        -- 1u gap
        { 'F1',       130, -6,   u,         u },
        { 'F2',       192, -6,   u,         u },
        { 'F3',       254, -6,   u,         u },
        { 'F4',       316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',       409, -6,   u,         u },

        -- 2 row

        { '²',        6,   -68,  u,         u },
        { '&',        68,  -68,  u,         u },
        { 'é',        130, -68,  u,         u },
        { '"',        192, -68,  u,         u },
        { "'",        254, -68,  u,         u },
        { '(',        316, -68,  u,         u },
        { '-',        378, -68,  u,         u },

        -- 3 row

        { 'TAB',      6,   -130, u1_5 + 1,  u },
        { 'A',        98,  -130, u,         u },
        { 'Z',        160, -130, u,         u },
        { 'E',        222, -130, u,         u },
        { 'R',        284, -130, u,         u },
        { 'T',        346, -130, u,         u },
        { 'Y',        408, -130, u,         u },

        -- 4 row

        { 'CAPSLOCK', 6,   -192, u1_75 + 2, u },
        { 'Q',        115, -192, u,         u },
        { 'S',        177, -192, u,         u },
        { 'D',        239, -192, u,         u },
        { 'F',        301, -192, u,         u },
        { 'G',        363, -192, u,         u },
        { 'H',        425, -192, u,         u },

        -- 5 row

        { 'LSHIFT',   6,   -254, u1_25 + 2, u },
        { '<',        85,  -254, u,         u },
        { 'W',        147, -254, u,         u },
        { 'X',        209, -254, u,         u },
        { 'C',        271, -254, u,         u },
        { 'V',        333, -254, u,         u },
        { 'B',        395, -254, u,         u },

        -- 6 row

        { 'LCTRL',    6,   -316, u1_25 + 2, u },
        { 'LWIN',     85,  -316, u1_25 + 1, u },
        { 'LALT',     163, -316, u1_25 + 1, u },
        { 'SPACE',    241, -316, 245,       u },
    },
    ['AZERTY_PRIMARY'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        -- 1u gap
        { 'F1',        130, -6,   u,         u },
        { 'F2',        192, -6,   u,         u },
        { 'F3',        254, -6,   u,         u },
        { 'F4',        316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',        409, -6,   u,         u },
        { 'F6',        471, -6,   u,         u },
        { 'F7',        533, -6,   u,         u },
        { 'F8',        595, -6,   u,         u },
        -- 0,5u gap
        { 'F9',        688, -6,   u,         u },
        { 'F10',       750, -6,   u,         u },
        { 'F11',       812, -6,   u,         u },
        { 'F12',       874, -6,   u,         u },

        -- 2 row

        { '²',         6,   -68,  u,         u },
        { '&',         68,  -68,  u,         u },
        { 'é',         130, -68,  u,         u },
        { '"',         192, -68,  u,         u },
        { "'",         254, -68,  u,         u },
        { '(',         316, -68,  u,         u },
        { '-',         378, -68,  u,         u },
        { 'è',         440, -68,  u,         u },
        { '_',         502, -68,  u,         u },
        { 'ç',         564, -68,  u,         u },
        { 'à',         626, -68,  u,         u },
        { ')',         688, -68,  u,         u },
        { '=',         750, -68,  u,         u },
        { 'BACKSPACE', 812, -68,  u2,        u },

        -- 3 row

        { 'TAB',       6,   -130, u1_5 + 1,  u },
        { 'A',         98,  -130, u,         u },
        { 'Z',         160, -130, u,         u },
        { 'E',         222, -130, u,         u },
        { 'R',         284, -130, u,         u },
        { 'T',         346, -130, u,         u },
        { 'Y',         408, -130, u,         u },
        { 'U',         470, -130, u,         u },
        { 'I',         532, -130, u,         u },
        { 'O',         594, -130, u,         u },
        { 'P',         656, -130, u,         u },
        { '^',         718, -130, u,         u },
        { '$',         780, -130, u,         u },
        { 'ENTER',     859, -130, u1_25,     u2 },

        -- 4 row

        { 'CAPSLOCK',  6,   -192, u1_75 + 2, u },
        { 'Q',         115, -192, u,         u },
        { 'S',         177, -192, u,         u },
        { 'D',         239, -192, u,         u },
        { 'F',         301, -192, u,         u },
        { 'G',         363, -192, u,         u },
        { 'H',         425, -192, u,         u },
        { 'J',         487, -192, u,         u },
        { 'K',         549, -192, u,         u },
        { 'L',         611, -192, u,         u },
        { 'M',         673, -192, u,         u },
        { 'ù',         735, -192, u,         u },
        { '*',         797, -192, u,         u },

        -- 5 row

        { 'LSHIFT',    6,   -254, u1_25 + 2, u },
        { '<',         85,  -254, u,         u },
        { 'W',         147, -254, u,         u },
        { 'X',         209, -254, u,         u },
        { 'C',         271, -254, u,         u },
        { 'V',         333, -254, u,         u },
        { 'B',         395, -254, u,         u },
        { 'N',         457, -254, u,         u },
        { ',',         519, -254, u,         u },
        { ';',         581, -254, u,         u },
        { ':',         643, -254, u,         u },
        { '!',         705, -254, u,         u },
        { 'RSHIFT',    767, -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',     6,   -316, u1_25 + 2, u },
        { 'LWIN',      85,  -316, u1_25 + 1, u },
        { 'LALT',      163, -316, u1_25 + 1, u },
        { 'SPACE',     241, -316, u6_25,     u },
        { 'RALT',      626, -316, u1_25 + 2, u },
        { 'RWIN',      705, -316, u1_25,     u },
        { 'MENU',      782, -316, u1_25,     u },
        { 'RCTRL',     859, -316, u1_25,     u },
    },
    --
    --

    -- ANSI --
    -- QWERTY --
    ['QWERTY_100%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },

        -- 2 row

        { '`',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { '-',              688,  -68,  u,         u },
        { '=',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'Q',              98,   -130, u,         u },
        { 'W',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Y',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { '[',              718,  -130, u,         u },
        { ']',              780,  -130, u,         u },
        { '\\',             842,  -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { ';',              673,  -192, u,         u },
        { ",",              735,  -192, u,         u },
        { 'ENTER',          797,  -192, u2_25 + 2, u },

        -- 5 row
        { 'LSHIFT',         6,    -254, u2_25 + 4, u },
        { 'Z',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '.',              643,  -254, u,         u },
        { '/',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u1_25 + 2, u },
        { 'RWIN',           705,  -316, u1_25,     u },
        { 'MENU',           782,  -316, u1_25,     u },
        { 'RCTRL',          859,  -316, u1_25,     u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN',    949,  -6,   u,         u },
        { 'SCROLLLOCK',     1011, -6,   u,         u },
        { 'PAUSE',          1073, -6,   u,         u },

        { 'INSERT',         949,  -68,  u,         u },
        { 'HOME',           1011, -68,  u,         u },
        { 'PAGEUP',         1073, -68,  u,         u },

        { 'DELETE',         949,  -130, u,         u },
        { 'END',            1011, -130, u,         u },
        { 'PAGEDOWN',       1073, -130, u,         u },

        { 'LEFT',           949,  -316, u,         u },
        { 'DOWN',           1011, -316, u,         u },
        { 'UP',             1011, -254, u,         u },
        { 'RIGHT',          1073, -316, u,         u },

        -- 100% -----------------------------------------

        { 'NUMLOCK',        1148, -68,  u,         u },
        { 'NUMPADDIVIDE',   1210, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1272, -68,  u,         u },
        { 'NUMPADMINUS',    1334, -68,  u,         u },

        { 'NUMPAD7',        1148, -130, u,         u },
        { 'NUMPAD8',        1210, -130, u,         u },
        { 'NUMPAD9',        1272, -130, u,         u },
        { 'NUMPADPLUS',     1334, -130, u,         u2 },

        { 'NUMPAD4',        1148, -192, u,         u },
        { 'NUMPAD5',        1210, -192, u,         u },
        { 'NUMPAD6',        1272, -192, u,         u },

        { 'NUMPAD1',        1148, -254, u,         u },
        { 'NUMPAD2',        1210, -254, u,         u },
        { 'NUMPAD3',        1272, -254, u,         u },
        { 'ENTER',          1334, -254, u,         u2 },

        { 'NUMPAD0',        1148, -316, u2,        u },
        { 'NUMPADDECIMAL',  1272, -316, u,         u },
    },
    ['QWERTY_96%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        { 'F1',             68,   -6,   u,         u },
        { 'F2',             130,  -6,   u,         u },
        { 'F3',             192,  -6,   u,         u },
        { 'F4',             254,  -6,   u,         u },
        { 'F5',             316,  -6,   u,         u },
        { 'F6',             378,  -6,   u,         u },
        { 'F7',             440,  -6,   u,         u },
        { 'F8',             502,  -6,   u,         u },
        { 'F9',             564,  -6,   u,         u },
        { 'F10',            626,  -6,   u,         u },
        { 'F11',            688,  -6,   u,         u },
        { 'F12',            750,  -6,   u,         u },
        { 'DELETE',         812,  -6,   u,         u },
        { 'INSERT',         874,  -6,   u,         u },
        { 'HOME',           936,  -6,   u,         u },
        { 'END',            998,  -6,   u,         u },
        { 'PAGEUP',         1060, -6,   u,         u },
        { 'PAGEDOWN',       1122, -6,   u,         u },


        -- 2 row

        { '`',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { '-',              688,  -68,  u,         u },
        { '=',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },
        { 'NUMLOCK',        936,  -68,  u,         u },
        { 'NUMPADDIVIDE',   998,  -68,  u,         u },
        { 'NUMPADMULTIPLY', 1060, -68,  u,         u },
        { 'NUMPADMINUS',    1122, -68,  u,         u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'Q',              98,   -130, u,         u },
        { 'W',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Y',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { '[',              718,  -130, u,         u },
        { ']',              780,  -130, u,         u },
        { '\\',             842,  -130, u1_5 + 2,  u },
        { 'NUMPAD7',        936,  -130, u,         u },
        { 'NUMPAD8',        998,  -130, u,         u },
        { 'NUMPAD9',        1060, -130, u,         u },
        { 'NUMPADPLUS',     1122, -130, u,         u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { ';',              673,  -192, u,         u },
        { ",",              735,  -192, u,         u },
        { 'ENTER',          797,  -192, u2_25 + 2, u },
        { 'NUMPAD4',        936,  -192, u,         u },
        { 'NUMPAD5',        998,  -192, u,         u },
        { 'NUMPAD6',        1060, -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u2_25 + 4, u },
        { 'Z',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '.',              643,  -254, u,         u },
        { '/',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, 105,       u },
        { 'UP',             874,  -254, u,         u },
        { 'NUMPAD1',        936,  -254, u,         u },
        { 'NUMPAD2',        998,  -254, u,         u },
        { 'NUMPAD3',        1060, -254, u,         u },
        { 'ENTER',          1122, -254, u,         u2 },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u,         u },
        { 'MENU',           688,  -316, u,         u },
        { 'RCTRL',          750,  -316, u,         u },
        { 'LEFT',           812,  -316, u,         u },
        { 'DOWN',           874,  -316, u,         u },
        { 'RIGHT',          936,  -316, u,         u },
        { 'NUMPAD0',        998,  -316, u,         u },
        { 'NUMPADDECIMAL',  1060, -316, u,         u },
    },
    ['QWERTY_80%'] = {

        -- 1 row

        { 'ESCAPE',      6,    -6,   u,         u },
        -- 1u gap
        { 'F1',          130,  -6,   u,         u },
        { 'F2',          192,  -6,   u,         u },
        { 'F3',          254,  -6,   u,         u },
        { 'F4',          316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',          409,  -6,   u,         u },
        { 'F6',          471,  -6,   u,         u },
        { 'F7',          533,  -6,   u,         u },
        { 'F8',          595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',          688,  -6,   u,         u },
        { 'F10',         750,  -6,   u,         u },
        { 'F11',         812,  -6,   u,         u },
        { 'F12',         874,  -6,   u,         u },

        -- 2 row

        { '`',           6,    -68,  u,         u },
        { '1',           68,   -68,  u,         u },
        { '2',           130,  -68,  u,         u },
        { '3',           192,  -68,  u,         u },
        { '4',           254,  -68,  u,         u },
        { '5',           316,  -68,  u,         u },
        { '6',           378,  -68,  u,         u },
        { '7',           440,  -68,  u,         u },
        { '8',           502,  -68,  u,         u },
        { '9',           564,  -68,  u,         u },
        { '0',           626,  -68,  u,         u },
        { '-',           688,  -68,  u,         u },
        { '=',           750,  -68,  u,         u },
        { 'BACKSPACE',   812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',         6,    -130, u1_5 + 1,  u },
        { 'Q',           98,   -130, u,         u },
        { 'W',           160,  -130, u,         u },
        { 'E',           222,  -130, u,         u },
        { 'R',           284,  -130, u,         u },
        { 'T',           346,  -130, u,         u },
        { 'Y',           408,  -130, u,         u },
        { 'U',           470,  -130, u,         u },
        { 'I',           532,  -130, u,         u },
        { 'O',           594,  -130, u,         u },
        { 'P',           656,  -130, u,         u },
        { '[',           718,  -130, u,         u },
        { ']',           780,  -130, u,         u },
        { '\\',          842,  -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',    6,    -192, u1_75 + 2, u },
        { 'A',           115,  -192, u,         u },
        { 'S',           177,  -192, u,         u },
        { 'D',           239,  -192, u,         u },
        { 'F',           301,  -192, u,         u },
        { 'G',           363,  -192, u,         u },
        { 'H',           425,  -192, u,         u },
        { 'J',           487,  -192, u,         u },
        { 'K',           549,  -192, u,         u },
        { 'L',           611,  -192, u,         u },
        { ';',           673,  -192, u,         u },
        { ",",           735,  -192, u,         u },
        { 'ENTER',       797,  -192, u2_25 + 2, u },

        -- 5 row
        { 'LSHIFT',      6,    -254, u2_25 + 4, u },
        { 'Z',           147,  -254, u,         u },
        { 'X',           209,  -254, u,         u },
        { 'C',           271,  -254, u,         u },
        { 'V',           333,  -254, u,         u },
        { 'B',           395,  -254, u,         u },
        { 'N',           457,  -254, u,         u },
        { 'M',           519,  -254, u,         u },
        { ',',           581,  -254, u,         u },
        { '.',           643,  -254, u,         u },
        { '/',           705,  -254, u,         u },
        { 'RSHIFT',      767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',       6,    -316, u1_25 + 2, u },
        { 'LWIN',        85,   -316, u1_25 + 1, u },
        { 'LALT',        163,  -316, u1_25 + 1, u },
        { 'SPACE',       241,  -316, u6_25,     u },
        { 'RALT',        626,  -316, u1_25 + 2, u },
        { 'RWIN',        705,  -316, u1_25,     u },
        { 'MENU',        782,  -316, u1_25,     u },
        { 'RCTRL',       859,  -316, u1_25,     u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN', 949,  -6,   u,         u },
        { 'SCROLLLOCK',  1011, -6,   u,         u },
        { 'PAUSE',       1073, -6,   u,         u },

        { 'INSERT',      949,  -68,  u,         u },
        { 'HOME',        1011, -68,  u,         u },
        { 'PAGEUP',      1073, -68,  u,         u },

        { 'DELETE',      949,  -130, u,         u },
        { 'END',         1011, -130, u,         u },
        { 'PAGEDOWN',    1073, -130, u,         u },

        { 'LEFT',        949,  -316, u,         u },
        { 'DOWN',        1011, -316, u,         u },
        { 'UP',          1011, -254, u,         u },
        { 'RIGHT',       1073, -316, u,         u },
    },
    ['QWERTY_75%'] = {

        -- 1 row

        { 'ESCAPE',      6,   -6,   u,         u },
        { 'F1',          68,  -6,   u,         u },
        { 'F2',          130, -6,   u,         u },
        { 'F3',          192, -6,   u,         u },
        { 'F4',          254, -6,   u,         u },
        { 'F5',          316, -6,   u,         u },
        { 'F6',          378, -6,   u,         u },
        { 'F7',          440, -6,   u,         u },
        { 'F8',          502, -6,   u,         u },
        { 'F9',          564, -6,   u,         u },
        { 'F10',         626, -6,   u,         u },
        { 'F11',         688, -6,   u,         u },
        { 'F12',         750, -6,   u,         u },
        { 'PRINTSCREEN', 812, -6,   u,         u },
        { 'PAUSE',       874, -6,   u,         u },
        { 'DELETE',      936, -6,   u,         u },

        -- 2 row

        { '`',           6,   -68,  u,         u },
        { '1',           68,  -68,  u,         u },
        { '2',           130, -68,  u,         u },
        { '3',           192, -68,  u,         u },
        { '4',           254, -68,  u,         u },
        { '5',           316, -68,  u,         u },
        { '6',           378, -68,  u,         u },
        { '7',           440, -68,  u,         u },
        { '8',           502, -68,  u,         u },
        { '9',           564, -68,  u,         u },
        { '0',           626, -68,  u,         u },
        { '-',           688, -68,  u,         u },
        { '=',           750, -68,  u,         u },
        { 'BACKSPACE',   812, -68,  u2,        u },
        { 'HOME',        936, -68,  u,         u },

        -- 3 row

        { 'TAB',         6,   -130, u1_5 + 1,  u },
        { 'Q',           98,  -130, u,         u },
        { 'W',           160, -130, u,         u },
        { 'E',           222, -130, u,         u },
        { 'R',           284, -130, u,         u },
        { 'T',           346, -130, u,         u },
        { 'Y',           408, -130, u,         u },
        { 'U',           470, -130, u,         u },
        { 'I',           532, -130, u,         u },
        { 'O',           594, -130, u,         u },
        { 'P',           656, -130, u,         u },
        { '[',           718, -130, u,         u },
        { ']',           780, -130, u,         u },
        { '\\',          842, -130, u1_5 + 2,  u },
        { 'END',         936, -130, u,         u },

        -- 4 row

        { 'CAPSLOCK',    6,   -192, u1_75 + 2, u },
        { 'A',           115, -192, u,         u },
        { 'S',           177, -192, u,         u },
        { 'D',           239, -192, u,         u },
        { 'F',           301, -192, u,         u },
        { 'G',           363, -192, u,         u },
        { 'H',           425, -192, u,         u },
        { 'J',           487, -192, u,         u },
        { 'K',           549, -192, u,         u },
        { 'L',           611, -192, u,         u },
        { ';',           673, -192, u,         u },
        { ",",           735, -192, u,         u },
        { 'ENTER',       797, -192, u2_25 + 2, u },
        { 'PAGEUP',      936, -192, u,         u },

        -- 5 row

        { 'LSHIFT',      6,   -254, u2_25 + 4, u },
        { 'Z',           147, -254, u,         u },
        { 'X',           209, -254, u,         u },
        { 'C',           271, -254, u,         u },
        { 'V',           333, -254, u,         u },
        { 'B',           395, -254, u,         u },
        { 'N',           457, -254, u,         u },
        { 'M',           519, -254, u,         u },
        { ',',           581, -254, u,         u },
        { '.',           643, -254, u,         u },
        { '/',           705, -254, u,         u },
        { 'RSHIFT',      767, -254, 105,       u },
        { 'UP',          874, -254, u,         u },
        { 'PAGEDOWN',    936, -254, u,         u },

        -- 6 row

        { 'LCTRL',       6,   -316, u1_25 + 2, u },
        { 'LWIN',        85,  -316, u1_25 + 1, u },
        { 'LALT',        163, -316, u1_25 + 1, u },
        { 'SPACE',       241, -316, u6_25,     u },
        { 'RALT',        626, -316, u,         u },
        { 'MENU',        688, -316, u,         u },
        { 'RCTRL',       750, -316, u,         u },
        { 'LEFT',        812, -316, u,         u },
        { 'DOWN',        874, -316, u,         u },
        { 'RIGHT',       936, -316, u,         u },
    },
    ['QWERTY_60%'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        { '1',         68,  -6,   u,         u },
        { '2',         130, -6,   u,         u },
        { '3',         192, -6,   u,         u },
        { '4',         254, -6,   u,         u },
        { '5',         316, -6,   u,         u },
        { '6',         378, -6,   u,         u },
        { '7',         440, -6,   u,         u },
        { '8',         502, -6,   u,         u },
        { '9',         564, -6,   u,         u },
        { '0',         626, -6,   u,         u },
        { '-',         688, -6,   u,         u },
        { '=',         750, -6,   u,         u },
        { 'BACKSPACE', 812, -6,   u2,        u },

        -- 2 row

        { 'TAB',       6,   -68,  u1_5 + 1,  u },
        { 'Q',         98,  -68,  u,         u },
        { 'W',         160, -68,  u,         u },
        { 'E',         222, -68,  u,         u },
        { 'R',         284, -68,  u,         u },
        { 'T',         346, -68,  u,         u },
        { 'Y',         408, -68,  u,         u },
        { 'U',         470, -68,  u,         u },
        { 'I',         532, -68,  u,         u },
        { 'O',         594, -68,  u,         u },
        { 'P',         656, -68,  u,         u },
        { '[',         718, -68,  u,         u },
        { ']',         780, -68,  u,         u },
        { '\\',        842, -68,  u1_5 + 2,  u },

        -- 3 row

        { 'CAPSLOCK',  6,   -130, u1_75 + 2, u },
        { 'A',         115, -130, u,         u },
        { 'S',         177, -130, u,         u },
        { 'D',         239, -130, u,         u },
        { 'F',         301, -130, u,         u },
        { 'G',         363, -130, u,         u },
        { 'H',         425, -130, u,         u },
        { 'J',         487, -130, u,         u },
        { 'K',         549, -130, u,         u },
        { 'L',         611, -130, u,         u },
        { ';',         673, -130, u,         u },
        { ",",         735, -130, u,         u },
        { 'ENTER',     797, -130, u2_25 + 2, u },

        -- 4 row

        { 'LSHIFT',    6,   -192, u2_25 + 4, u },
        { 'Z',         147, -192, u,         u },
        { 'X',         209, -192, u,         u },
        { 'C',         271, -192, u,         u },
        { 'V',         333, -192, u,         u },
        { 'B',         395, -192, u,         u },
        { 'N',         457, -192, u,         u },
        { 'M',         519, -192, u,         u },
        { ',',         581, -192, u,         u },
        { '.',         643, -192, u,         u },
        { '/',         705, -192, u,         u },
        { 'RSHIFT',    767, -192, u2_75 + 2, u },

        -- 5 row

        { 'LCTRL',     6,   -254, u1_25 + 2, u },
        { 'LWIN',      85,  -254, u1_25 + 1, u },
        { 'LALT',      163, -254, u1_25 + 1, u },
        { 'SPACE',     241, -254, u6_25,     u },
        { 'RALT',      626, -254, u1_25 + 2, u },
        { 'RWIN',      705, -254, u1_25,     u },
        { 'MENU',      782, -254, u1_25,     u },
        { 'RCTRL',     859, -254, u1_25,     u },
    },
    ['QWERTY_1800'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },
        { 'HOME',           956,  -6,   u,         u },
        { 'END',            1018, -6,   u,         u },
        { 'PAGEUP',         1080, -6,   u,         u },
        { 'PAGEDOWN',       1142, -6,   u,         u },


        -- 2 row

        { '`',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { '-',              688,  -68,  u,         u },
        { '=',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },
        { 'NUMLOCK',        956,  -68,  u,         u },
        { 'NUMPADDIVIDE',   1018, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1080, -68,  u,         u },
        { 'NUMPADMINUS',    1142, -68,  u,         u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { 'Q',              98,   -130, u,         u },
        { 'W',              160,  -130, u,         u },
        { 'E',              222,  -130, u,         u },
        { 'R',              284,  -130, u,         u },
        { 'T',              346,  -130, u,         u },
        { 'Y',              408,  -130, u,         u },
        { 'U',              470,  -130, u,         u },
        { 'I',              532,  -130, u,         u },
        { 'O',              594,  -130, u,         u },
        { 'P',              656,  -130, u,         u },
        { '[',              718,  -130, u,         u },
        { ']',              780,  -130, u,         u },
        { '\\',             842,  -130, u1_5 + 2,  u },
        { 'NUMPAD7',        956,  -130, u,         u },
        { 'NUMPAD8',        1018, -130, u,         u },
        { 'NUMPAD9',        1080, -130, u,         u },
        { 'NUMPADPLUS',     1142, -130, u,         u2 },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'S',              177,  -192, u,         u },
        { 'D',              239,  -192, u,         u },
        { 'F',              301,  -192, u,         u },
        { 'G',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'J',              487,  -192, u,         u },
        { 'K',              549,  -192, u,         u },
        { 'L',              611,  -192, u,         u },
        { ';',              673,  -192, u,         u },
        { ",",              735,  -192, u,         u },
        { 'ENTER',          797,  -192, u2_25 + 2, u },
        { 'NUMPAD4',        956,  -192, u,         u },
        { 'NUMPAD5',        1018, -192, u,         u },
        { 'NUMPAD6',        1080, -192, u,         u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u2_25 + 4, u },
        { 'Z',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'C',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'B',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '.',              643,  -254, u,         u },
        { '/',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, 105,       u },
        { 'UP',             884,  -264, u,         u },
        { 'NUMPAD1',        956,  -254, u,         u },
        { 'NUMPAD2',        1018, -254, u,         u },
        { 'NUMPAD3',        1080, -254, u,         u },
        { 'ENTER',          1142, -254, u,         u2 },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_25 + 2, u },
        { 'LWIN',           85,   -316, u1_25 + 1, u },
        { 'LALT',           163,  -316, u1_25 + 1, u },
        { 'SPACE',          241,  -316, u6_25,     u },
        { 'RALT',           626,  -316, u,         u },
        { 'MENU',           688,  -316, u,         u },
        { 'RCTRL',          750,  -316, u,         u },
        { 'LEFT',           822,  -326, u,         u },
        { 'DOWN',           884,  -326, u,         u },
        { 'RIGHT',          946,  -326, u,         u },
        { 'NUMPAD0',        1018, -316, u,         u },
        { 'NUMPADDECIMAL',  1080, -316, u,         u },
    },
    ['QWERTY_HALF'] = {

        -- 1 row

        { 'ESCAPE',   6,   -6,   u,         u },
        -- 1u gap
        { 'F1',       130, -6,   u,         u },
        { 'F2',       192, -6,   u,         u },
        { 'F3',       254, -6,   u,         u },
        { 'F4',       316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',       409, -6,   u,         u },

        -- 2 row

        { '`',        6,   -68,  u,         u },
        { '1',        68,  -68,  u,         u },
        { '2',        130, -68,  u,         u },
        { '3',        192, -68,  u,         u },
        { '4',        254, -68,  u,         u },
        { '5',        316, -68,  u,         u },
        { '6',        378, -68,  u,         u },

        -- 3 row

        { 'TAB',      6,   -130, u1_5 + 1,  u },
        { 'Q',        98,  -130, u,         u },
        { 'W',        160, -130, u,         u },
        { 'E',        222, -130, u,         u },
        { 'R',        284, -130, u,         u },
        { 'T',        346, -130, u,         u },
        { 'Y',        408, -130, u,         u },

        -- 4 row

        { 'CAPSLOCK', 6,   -192, u1_75 + 2, u },
        { 'A',        115, -192, u,         u },
        { 'S',        177, -192, u,         u },
        { 'D',        239, -192, u,         u },
        { 'F',        301, -192, u,         u },
        { 'G',        363, -192, u,         u },
        { 'H',        425, -192, u,         u },

        -- 5 row

        { 'LSHIFT',   6,   -254, u2_25 + 4, u },
        { 'Z',        147, -254, u,         u },
        { 'X',        209, -254, u,         u },
        { 'C',        271, -254, u,         u },
        { 'V',        333, -254, u,         u },
        { 'B',        395, -254, u,         u },

        -- 6 row

        { 'LCTRL',    6,   -316, u1_25 + 2, u },
        { 'LWIN',     85,  -316, u1_25 + 1, u },
        { 'LALT',     163, -316, u1_25 + 1, u },
        { 'SPACE',    241, -316, 245,       u },
    },
    ['QWERTY_PRIMARY'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        -- 1u gap
        { 'F1',        130, -6,   u,         u },
        { 'F2',        192, -6,   u,         u },
        { 'F3',        254, -6,   u,         u },
        { 'F4',        316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',        409, -6,   u,         u },
        { 'F6',        471, -6,   u,         u },
        { 'F7',        533, -6,   u,         u },
        { 'F8',        595, -6,   u,         u },
        -- 0,5u gap
        { 'F9',        688, -6,   u,         u },
        { 'F10',       750, -6,   u,         u },
        { 'F11',       812, -6,   u,         u },
        { 'F12',       874, -6,   u,         u },

        -- 2 row

        { '`',         6,   -68,  u,         u },
        { '1',         68,  -68,  u,         u },
        { '2',         130, -68,  u,         u },
        { '3',         192, -68,  u,         u },
        { '4',         254, -68,  u,         u },
        { '5',         316, -68,  u,         u },
        { '6',         378, -68,  u,         u },
        { '7',         440, -68,  u,         u },
        { '8',         502, -68,  u,         u },
        { '9',         564, -68,  u,         u },
        { '0',         626, -68,  u,         u },
        { '-',         688, -68,  u,         u },
        { '=',         750, -68,  u,         u },
        { 'BACKSPACE', 812, -68,  u2,        u },

        -- 3 row

        { 'TAB',       6,   -130, u1_5 + 1,  u },
        { 'Q',         98,  -130, u,         u },
        { 'W',         160, -130, u,         u },
        { 'E',         222, -130, u,         u },
        { 'R',         284, -130, u,         u },
        { 'T',         346, -130, u,         u },
        { 'Y',         408, -130, u,         u },
        { 'U',         470, -130, u,         u },
        { 'I',         532, -130, u,         u },
        { 'O',         594, -130, u,         u },
        { 'P',         656, -130, u,         u },
        { '[',         718, -130, u,         u },
        { ']',         780, -130, u,         u },
        { '\\',        842, -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',  6,   -192, u1_75 + 2, u },
        { 'A',         115, -192, u,         u },
        { 'S',         177, -192, u,         u },
        { 'D',         239, -192, u,         u },
        { 'F',         301, -192, u,         u },
        { 'G',         363, -192, u,         u },
        { 'H',         425, -192, u,         u },
        { 'J',         487, -192, u,         u },
        { 'K',         549, -192, u,         u },
        { 'L',         611, -192, u,         u },
        { ';',         673, -192, u,         u },
        { ",",         735, -192, u,         u },
        { 'ENTER',     797, -192, u2_25 + 2, u },

        -- 5 row
        { 'LSHIFT',    6,   -254, u2_25 + 4, u },
        { 'Z',         147, -254, u,         u },
        { 'X',         209, -254, u,         u },
        { 'C',         271, -254, u,         u },
        { 'V',         333, -254, u,         u },
        { 'B',         395, -254, u,         u },
        { 'N',         457, -254, u,         u },
        { 'M',         519, -254, u,         u },
        { ',',         581, -254, u,         u },
        { '.',         643, -254, u,         u },
        { '/',         705, -254, u,         u },
        { 'RSHIFT',    767, -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',     6,   -316, u1_25 + 2, u },
        { 'LWIN',      85,  -316, u1_25 + 1, u },
        { 'LALT',      163, -316, u1_25 + 1, u },
        { 'SPACE',     241, -316, u6_25,     u },
        { 'RALT',      626, -316, u1_25 + 2, u },
        { 'RWIN',      705, -316, u1_25,     u },
        { 'MENU',      782, -316, u1_25,     u },
        { 'RCTRL',     859, -316, u1_25,     u },
    },
    --
    --

    -- DVORAK --
    ['DVORAK_100%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },

        -- 2 row

        { '`',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { '5',              316,  -68,  u,         u },
        { '6',              378,  -68,  u,         u },
        { '7',              440,  -68,  u,         u },
        { '8',              502,  -68,  u,         u },
        { '9',              564,  -68,  u,         u },
        { '0',              626,  -68,  u,         u },
        { '[',              688,  -68,  u,         u },
        { ']',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { '`',              98,   -130, u,         u },
        { ',',              160,  -130, u,         u },
        { '.',              222,  -130, u,         u },
        { 'P',              284,  -130, u,         u },
        { 'Y',              346,  -130, u,         u },
        { 'F',              408,  -130, u,         u },
        { 'G',              470,  -130, u,         u },
        { 'C',              532,  -130, u,         u },
        { 'R',              594,  -130, u,         u },
        { 'L',              656,  -130, u,         u },
        { '/',              718,  -130, u,         u },
        { '=',              780,  -130, u,         u },
        { '\\',             842,  -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { 'A',              115,  -192, u,         u },
        { 'O',              177,  -192, u,         u },
        { 'E',              239,  -192, u,         u },
        { 'U',              301,  -192, u,         u },
        { 'I',              363,  -192, u,         u },
        { 'D',              425,  -192, u,         u },
        { 'H',              487,  -192, u,         u },
        { 'T',              549,  -192, u,         u },
        { 'N',              611,  -192, u,         u },
        { 'S',              673,  -192, u,         u },
        { '-',              735,  -192, u,         u },
        { 'ENTER',          797,  -192, u2_25 + 2, u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u2_25 + 4, u },
        { ';',              147,  -254, u,         u },
        { 'Q',              209,  -254, u,         u },
        { 'J',              271,  -254, u,         u },
        { 'K',              333,  -254, u,         u },
        { 'X',              395,  -254, u,         u },
        { 'B',              457,  -254, u,         u },
        { 'M',              519,  -254, u,         u },
        { 'W',              581,  -254, u,         u },
        { 'V',              643,  -254, u,         u },
        { 'Z',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_5 + 1,  u },
        { 'LWIN',           99,   -316, u,         u },
        { 'LALT',           161,  -316, u1_5 + 1,  u },
        { 'SPACE',          254,  -316, 368,       u },
        { 'RALT',           624,  -316, u1_5 + 2,  u },
        { 'RWIN',           718,  -316, u,         u },
        { 'MENU',           780,  -316, u,         u },
        { 'RCTRL',          842,  -316, u1_5 + 2,  u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN',    949,  -6,   u,         u },
        { 'SCROLLLOCK',     1011, -6,   u,         u },
        { 'PAUSE',          1073, -6,   u,         u },

        { 'INSERT',         949,  -68,  u,         u },
        { 'HOME',           1011, -68,  u,         u },
        { 'PAGEUP',         1073, -68,  u,         u },

        { 'DELETE',         949,  -130, u,         u },
        { 'END',            1011, -130, u,         u },
        { 'PAGEDOWN',       1073, -130, u,         u },

        { 'LEFT',           949,  -316, u,         u },
        { 'DOWN',           1011, -316, u,         u },
        { 'UP',             1011, -254, u,         u },
        { 'RIGHT',          1073, -316, u,         u },

        -- 100% -----------------------------------------

        { 'NUMLOCK',        1148, -68,  u,         u },
        { 'NUMPADDIVIDE',   1210, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1272, -68,  u,         u },
        { 'NUMPADMINUS',    1334, -68,  u,         u },

        { 'NUMPAD7',        1148, -130, u,         u },
        { 'NUMPAD8',        1210, -130, u,         u },
        { 'NUMPAD9',        1272, -130, u,         u },
        { 'NUMPADPLUS',     1334, -130, u,         u2 },

        { 'NUMPAD4',        1148, -192, u,         u },
        { 'NUMPAD5',        1210, -192, u,         u },
        { 'NUMPAD6',        1272, -192, u,         u },

        { 'NUMPAD1',        1148, -254, u,         u },
        { 'NUMPAD2',        1210, -254, u,         u },
        { 'NUMPAD3',        1272, -254, u,         u },
        { 'ENTER',          1334, -254, u,         u2 },

        { 'NUMPAD0',        1148, -316, u2,        u },
        { 'NUMPADDECIMAL',  1272, -316, u,         u },
    },
    ['DVORAK_PRIMARY'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        -- 1u gap
        { 'F1',        130, -6,   u,         u },
        { 'F2',        192, -6,   u,         u },
        { 'F3',        254, -6,   u,         u },
        { 'F4',        316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',        409, -6,   u,         u },
        { 'F6',        471, -6,   u,         u },
        { 'F7',        533, -6,   u,         u },
        { 'F8',        595, -6,   u,         u },
        -- 0,5u gap
        { 'F9',        688, -6,   u,         u },
        { 'F10',       750, -6,   u,         u },
        { 'F11',       812, -6,   u,         u },
        { 'F12',       874, -6,   u,         u },

        -- 2 row

        { '`',         6,   -68,  u,         u },
        { '1',         68,  -68,  u,         u },
        { '2',         130, -68,  u,         u },
        { '3',         192, -68,  u,         u },
        { '4',         254, -68,  u,         u },
        { '5',         316, -68,  u,         u },
        { '6',         378, -68,  u,         u },
        { '7',         440, -68,  u,         u },
        { '8',         502, -68,  u,         u },
        { '9',         564, -68,  u,         u },
        { '0',         626, -68,  u,         u },
        { '[',         688, -68,  u,         u },
        { ']',         750, -68,  u,         u },
        { 'BACKSPACE', 812, -68,  u2,        u },

        -- 3 row

        { 'TAB',       6,   -130, u1_5 + 1,  u },
        { '`',         98,  -130, u,         u },
        { ',',         160, -130, u,         u },
        { '.',         222, -130, u,         u },
        { 'P',         284, -130, u,         u },
        { 'Y',         346, -130, u,         u },
        { 'F',         408, -130, u,         u },
        { 'G',         470, -130, u,         u },
        { 'C',         532, -130, u,         u },
        { 'R',         594, -130, u,         u },
        { 'L',         656, -130, u,         u },
        { '/',         718, -130, u,         u },
        { '=',         780, -130, u,         u },
        { '\\',        842, -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',  6,   -192, u1_75 + 2, u },
        { 'A',         115, -192, u,         u },
        { 'O',         177, -192, u,         u },
        { 'E',         239, -192, u,         u },
        { 'U',         301, -192, u,         u },
        { 'I',         363, -192, u,         u },
        { 'D',         425, -192, u,         u },
        { 'H',         487, -192, u,         u },
        { 'T',         549, -192, u,         u },
        { 'N',         611, -192, u,         u },
        { 'S',         673, -192, u,         u },
        { '-',         735, -192, u,         u },
        { 'ENTER',     797, -192, u2_25 + 2, u },

        -- 5 row

        { 'LSHIFT',    6,   -254, u2_25 + 4, u },
        { ';',         147, -254, u,         u },
        { 'Q',         209, -254, u,         u },
        { 'J',         271, -254, u,         u },
        { 'K',         333, -254, u,         u },
        { 'X',         395, -254, u,         u },
        { 'B',         457, -254, u,         u },
        { 'M',         519, -254, u,         u },
        { 'W',         581, -254, u,         u },
        { 'V',         643, -254, u,         u },
        { 'Z',         705, -254, u,         u },
        { 'RSHIFT',    767, -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',     6,   -316, u1_5 + 1,  u },
        { 'LWIN',      99,  -316, u,         u },
        { 'LALT',      161, -316, u1_5 + 1,  u },
        { 'SPACE',     254, -316, 368,       u },
        { 'RALT',      624, -316, u1_5 + 2,  u },
        { 'RWIN',      718, -316, u,         u },
        { 'MENU',      780, -316, u,         u },
        { 'RCTRL',     842, -316, u1_5 + 2,  u },
    },
    ['DVORAK_RIGHT_100%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },

        -- 2 row

        { '`',              6,    -68,  u,         u },
        { '1',              68,   -68,  u,         u },
        { '2',              130,  -68,  u,         u },
        { '3',              192,  -68,  u,         u },
        { '4',              254,  -68,  u,         u },
        { 'J',              316,  -68,  u,         u },
        { 'L',              378,  -68,  u,         u },
        { 'M',              440,  -68,  u,         u },
        { 'F',              502,  -68,  u,         u },
        { 'P',              564,  -68,  u,         u },
        { '/',              626,  -68,  u,         u },
        { '[',              688,  -68,  u,         u },
        { ']',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { '5',              98,   -130, u,         u },
        { '6',              160,  -130, u,         u },
        { 'Q',              222,  -130, u,         u },
        { '.',              284,  -130, u,         u },
        { 'O',              346,  -130, u,         u },
        { 'R',              408,  -130, u,         u },
        { 'S',              470,  -130, u,         u },
        { 'U',              532,  -130, u,         u },
        { 'Y',              594,  -130, u,         u },
        { 'B',              656,  -130, u,         u },
        { ';',              718,  -130, u,         u },
        { '=',              780,  -130, u,         u },
        { '\\',             842,  -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { '&',              115,  -192, u,         u },
        { '*',              177,  -192, u,         u },
        { 'Z',              239,  -192, u,         u },
        { 'A',              301,  -192, u,         u },
        { 'E',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'T',              487,  -192, u,         u },
        { 'D',              549,  -192, u,         u },
        { 'C',              611,  -192, u,         u },
        { 'K',              673,  -192, u,         u },
        { '-',              735,  -192, u,         u },
        { 'ENTER',          797,  -192, u2_25 + 2, u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u2_25 + 4, u },
        { '9',              147,  -254, u,         u },
        { '0',              209,  -254, u,         u },
        { 'X',              271,  -254, u,         u },
        { ',',              333,  -254, u,         u },
        { 'I',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'W',              519,  -254, u,         u },
        { 'V',              581,  -254, u,         u },
        { 'G',              643,  -254, u,         u },
        { ',',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_5 + 1,  u },
        { 'LWIN',           99,   -316, u,         u },
        { 'LALT',           161,  -316, u1_5 + 1,  u },
        { 'SPACE',          254,  -316, 368,       u },
        { 'RALT',           624,  -316, u1_5 + 2,  u },
        { 'RWIN',           718,  -316, u,         u },
        { 'MENU',           780,  -316, u,         u },
        { 'RCTRL',          842,  -316, u1_5 + 2,  u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN',    949,  -6,   u,         u },
        { 'SCROLLLOCK',     1011, -6,   u,         u },
        { 'PAUSE',          1073, -6,   u,         u },

        { 'INSERT',         949,  -68,  u,         u },
        { 'HOME',           1011, -68,  u,         u },
        { 'PAGEUP',         1073, -68,  u,         u },

        { 'DELETE',         949,  -130, u,         u },
        { 'END',            1011, -130, u,         u },
        { 'PAGEDOWN',       1073, -130, u,         u },

        { 'LEFT',           949,  -316, u,         u },
        { 'DOWN',           1011, -316, u,         u },
        { 'UP',             1011, -254, u,         u },
        { 'RIGHT',          1073, -316, u,         u },

        -- 100% -----------------------------------------

        { 'NUMLOCK',        1148, -68,  u,         u },
        { 'NUMPADDIVIDE',   1210, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1272, -68,  u,         u },
        { 'NUMPADMINUS',    1334, -68,  u,         u },

        { 'NUMPAD7',        1148, -130, u,         u },
        { 'NUMPAD8',        1210, -130, u,         u },
        { 'NUMPAD9',        1272, -130, u,         u },
        { 'NUMPADPLUS',     1334, -130, u,         u2 },

        { 'NUMPAD4',        1148, -192, u,         u },
        { 'NUMPAD5',        1210, -192, u,         u },
        { 'NUMPAD6',        1272, -192, u,         u },

        { 'NUMPAD1',        1148, -254, u,         u },
        { 'NUMPAD2',        1210, -254, u,         u },
        { 'NUMPAD3',        1272, -254, u,         u },
        { 'ENTER',          1334, -254, u,         u2 },

        { 'NUMPAD0',        1148, -316, u2,        u },
        { 'NUMPADDECIMAL',  1272, -316, u,         u },
    },
    ['DVORAK_RIGHT_PRIMARY'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        -- 1u gap
        { 'F1',        130, -6,   u,         u },
        { 'F2',        192, -6,   u,         u },
        { 'F3',        254, -6,   u,         u },
        { 'F4',        316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',        409, -6,   u,         u },
        { 'F6',        471, -6,   u,         u },
        { 'F7',        533, -6,   u,         u },
        { 'F8',        595, -6,   u,         u },
        -- 0,5u gap
        { 'F9',        688, -6,   u,         u },
        { 'F10',       750, -6,   u,         u },
        { 'F11',       812, -6,   u,         u },
        { 'F12',       874, -6,   u,         u },

        -- 2 row

        { '`',         6,   -68,  u,         u },
        { '1',         68,  -68,  u,         u },
        { '2',         130, -68,  u,         u },
        { '3',         192, -68,  u,         u },
        { '4',         254, -68,  u,         u },
        { 'J',         316, -68,  u,         u },
        { 'L',         378, -68,  u,         u },
        { 'M',         440, -68,  u,         u },
        { 'F',         502, -68,  u,         u },
        { 'P',         564, -68,  u,         u },
        { '/',         626, -68,  u,         u },
        { '[',         688, -68,  u,         u },
        { ']',         750, -68,  u,         u },
        { 'BACKSPACE', 812, -68,  u2,        u },

        -- 3 row

        { 'TAB',       6,   -130, u1_5 + 1,  u },
        { '5',         98,  -130, u,         u },
        { '6',         160, -130, u,         u },
        { 'Q',         222, -130, u,         u },
        { '.',         284, -130, u,         u },
        { 'O',         346, -130, u,         u },
        { 'R',         408, -130, u,         u },
        { 'S',         470, -130, u,         u },
        { 'U',         532, -130, u,         u },
        { 'Y',         594, -130, u,         u },
        { 'B',         656, -130, u,         u },
        { ';',         718, -130, u,         u },
        { '=',         780, -130, u,         u },
        { '\\',        842, -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',  6,   -192, u1_75 + 2, u },
        { '&',         115, -192, u,         u },
        { '*',         177, -192, u,         u },
        { 'Z',         239, -192, u,         u },
        { 'A',         301, -192, u,         u },
        { 'E',         363, -192, u,         u },
        { 'H',         425, -192, u,         u },
        { 'T',         487, -192, u,         u },
        { 'D',         549, -192, u,         u },
        { 'C',         611, -192, u,         u },
        { 'K',         673, -192, u,         u },
        { '-',         735, -192, u,         u },
        { 'ENTER',     797, -192, u2_25 + 2, u },

        -- 5 row

        { 'LSHIFT',    6,   -254, u2_25 + 4, u },
        { '9',         147, -254, u,         u },
        { '0',         209, -254, u,         u },
        { 'X',         271, -254, u,         u },
        { ',',         333, -254, u,         u },
        { 'I',         395, -254, u,         u },
        { 'N',         457, -254, u,         u },
        { 'W',         519, -254, u,         u },
        { 'V',         581, -254, u,         u },
        { 'G',         643, -254, u,         u },
        { ',',         705, -254, u,         u },
        { 'RSHIFT',    767, -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',     6,   -316, u1_5 + 1,  u },
        { 'LWIN',      99,  -316, u,         u },
        { 'LALT',      161, -316, u1_5 + 1,  u },
        { 'SPACE',     254, -316, 368,       u },
        { 'RALT',      624, -316, u1_5 + 2,  u },
        { 'RWIN',      718, -316, u,         u },
        { 'MENU',      780, -316, u,         u },
        { 'RCTRL',     842, -316, u1_5 + 2,  u },
    },
    ['DVORAK_LEFT_100%'] = {

        -- 1 row

        { 'ESCAPE',         6,    -6,   u,         u },
        -- 1u gap
        { 'F1',             130,  -6,   u,         u },
        { 'F2',             192,  -6,   u,         u },
        { 'F3',             254,  -6,   u,         u },
        { 'F4',             316,  -6,   u,         u },
        -- 0,5u gap
        { 'F5',             409,  -6,   u,         u },
        { 'F6',             471,  -6,   u,         u },
        { 'F7',             533,  -6,   u,         u },
        { 'F8',             595,  -6,   u,         u },
        -- 0,5u gap
        { 'F9',             688,  -6,   u,         u },
        { 'F10',            750,  -6,   u,         u },
        { 'F11',            812,  -6,   u,         u },
        { 'F12',            874,  -6,   u,         u },

        -- 2 row

        { '`',              6,    -68,  u,         u },
        { '[',              68,   -68,  u,         u },
        { ']',              130,  -68,  u,         u },
        { '/',              192,  -68,  u,         u },
        { 'P',              254,  -68,  u,         u },
        { 'F',              316,  -68,  u,         u },
        { 'M',              378,  -68,  u,         u },
        { 'L',              440,  -68,  u,         u },
        { 'J',              502,  -68,  u,         u },
        { '4',              564,  -68,  u,         u },
        { '3',              626,  -68,  u,         u },
        { '2',              688,  -68,  u,         u },
        { '1',              750,  -68,  u,         u },
        { 'BACKSPACE',      812,  -68,  u2,        u },

        -- 3 row

        { 'TAB',            6,    -130, u1_5 + 1,  u },
        { ';',              98,   -130, u,         u },
        { 'Q',              160,  -130, u,         u },
        { 'B',              222,  -130, u,         u },
        { 'Y',              284,  -130, u,         u },
        { 'U',              346,  -130, u,         u },
        { 'R',              408,  -130, u,         u },
        { 'S',              470,  -130, u,         u },
        { 'O',              532,  -130, u,         u },
        { '.',              594,  -130, u,         u },
        { '6',              656,  -130, u,         u },
        { '5',              718,  -130, u,         u },
        { '=',              780,  -130, u,         u },
        { '\\',             842,  -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',       6,    -192, u1_75 + 2, u },
        { '-',              115,  -192, u,         u },
        { 'K',              177,  -192, u,         u },
        { 'C',              239,  -192, u,         u },
        { 'D',              301,  -192, u,         u },
        { 'T',              363,  -192, u,         u },
        { 'H',              425,  -192, u,         u },
        { 'E',              487,  -192, u,         u },
        { 'A',              549,  -192, u,         u },
        { 'Z',              611,  -192, u,         u },
        { '8',              673,  -192, u,         u },
        { '9',              735,  -192, u,         u },
        { 'ENTER',          797,  -192, u2_25 + 2, u },

        -- 5 row

        { 'LSHIFT',         6,    -254, u2_25 + 4, u },
        { ',',              147,  -254, u,         u },
        { 'X',              209,  -254, u,         u },
        { 'G',              271,  -254, u,         u },
        { 'V',              333,  -254, u,         u },
        { 'W',              395,  -254, u,         u },
        { 'N',              457,  -254, u,         u },
        { 'I',              519,  -254, u,         u },
        { ',',              581,  -254, u,         u },
        { '0',              643,  -254, u,         u },
        { '9',              705,  -254, u,         u },
        { 'RSHIFT',         767,  -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',          6,    -316, u1_5 + 1,  u },
        { 'LWIN',           99,   -316, u,         u },
        { 'LALT',           161,  -316, u1_5 + 1,  u },
        { 'SPACE',          254,  -316, 368,       u },
        { 'RALT',           624,  -316, u1_5 + 2,  u },
        { 'RWIN',           718,  -316, u,         u },
        { 'MENU',           780,  -316, u,         u },
        { 'RCTRL',          842,  -316, u1_5 + 2,  u },

        -- 80% ------------------------------------------

        { 'PRINTSCREEN',    949,  -6,   u,         u },
        { 'SCROLLLOCK',     1011, -6,   u,         u },
        { 'PAUSE',          1073, -6,   u,         u },

        { 'INSERT',         949,  -68,  u,         u },
        { 'HOME',           1011, -68,  u,         u },
        { 'PAGEUP',         1073, -68,  u,         u },

        { 'DELETE',         949,  -130, u,         u },
        { 'END',            1011, -130, u,         u },
        { 'PAGEDOWN',       1073, -130, u,         u },

        { 'LEFT',           949,  -316, u,         u },
        { 'DOWN',           1011, -316, u,         u },
        { 'UP',             1011, -254, u,         u },
        { 'RIGHT',          1073, -316, u,         u },

        -- 100% -----------------------------------------

        { 'NUMLOCK',        1148, -68,  u,         u },
        { 'NUMPADDIVIDE',   1210, -68,  u,         u },
        { 'NUMPADMULTIPLY', 1272, -68,  u,         u },
        { 'NUMPADMINUS',    1334, -68,  u,         u },

        { 'NUMPAD7',        1148, -130, u,         u },
        { 'NUMPAD8',        1210, -130, u,         u },
        { 'NUMPAD9',        1272, -130, u,         u },
        { 'NUMPADPLUS',     1334, -130, u,         u2 },

        { 'NUMPAD4',        1148, -192, u,         u },
        { 'NUMPAD5',        1210, -192, u,         u },
        { 'NUMPAD6',        1272, -192, u,         u },

        { 'NUMPAD1',        1148, -254, u,         u },
        { 'NUMPAD2',        1210, -254, u,         u },
        { 'NUMPAD3',        1272, -254, u,         u },
        { 'ENTER',          1334, -254, u,         u2 },

        { 'NUMPAD0',        1148, -316, u2,        u },
        { 'NUMPADDECIMAL',  1272, -316, u,         u },
    },
    ['DVORAK_LEFT_PRIMARY'] = {

        -- 1 row

        { 'ESCAPE',    6,   -6,   u,         u },
        -- 1u gap
        { 'F1',        130, -6,   u,         u },
        { 'F2',        192, -6,   u,         u },
        { 'F3',        254, -6,   u,         u },
        { 'F4',        316, -6,   u,         u },
        -- 0,5u gap
        { 'F5',        409, -6,   u,         u },
        { 'F6',        471, -6,   u,         u },
        { 'F7',        533, -6,   u,         u },
        { 'F8',        595, -6,   u,         u },
        -- 0,5u gap
        { 'F9',        688, -6,   u,         u },
        { 'F10',       750, -6,   u,         u },
        { 'F11',       812, -6,   u,         u },
        { 'F12',       874, -6,   u,         u },

        -- 2 row

        { '`',         6,   -68,  u,         u },
        { '[',         68,  -68,  u,         u },
        { ']',         130, -68,  u,         u },
        { '/',         192, -68,  u,         u },
        { 'P',         254, -68,  u,         u },
        { 'F',         316, -68,  u,         u },
        { 'M',         378, -68,  u,         u },
        { 'L',         440, -68,  u,         u },
        { 'J',         502, -68,  u,         u },
        { '4',         564, -68,  u,         u },
        { '3',         626, -68,  u,         u },
        { '2',         688, -68,  u,         u },
        { '1',         750, -68,  u,         u },
        { 'BACKSPACE', 812, -68,  u2,        u },

        -- 3 row

        { 'TAB',       6,   -130, u1_5 + 1,  u },
        { ';',         98,  -130, u,         u },
        { 'Q',         160, -130, u,         u },
        { 'B',         222, -130, u,         u },
        { 'Y',         284, -130, u,         u },
        { 'U',         346, -130, u,         u },
        { 'R',         408, -130, u,         u },
        { 'S',         470, -130, u,         u },
        { 'O',         532, -130, u,         u },
        { '.',         594, -130, u,         u },
        { '6',         656, -130, u,         u },
        { '5',         718, -130, u,         u },
        { '=',         780, -130, u,         u },
        { '\\',        842, -130, u1_5 + 2,  u },

        -- 4 row

        { 'CAPSLOCK',  6,   -192, u1_75 + 2, u },
        { '-',         115, -192, u,         u },
        { 'K',         177, -192, u,         u },
        { 'C',         239, -192, u,         u },
        { 'D',         301, -192, u,         u },
        { 'T',         363, -192, u,         u },
        { 'H',         425, -192, u,         u },
        { 'E',         487, -192, u,         u },
        { 'A',         549, -192, u,         u },
        { 'Z',         611, -192, u,         u },
        { '8',         673, -192, u,         u },
        { '9',         735, -192, u,         u },
        { 'ENTER',     797, -192, u2_25 + 2, u },

        -- 5 row

        { 'LSHIFT',    6,   -254, u2_25 + 4, u },
        { ',',         147, -254, u,         u },
        { 'X',         209, -254, u,         u },
        { 'G',         271, -254, u,         u },
        { 'V',         333, -254, u,         u },
        { 'W',         395, -254, u,         u },
        { 'N',         457, -254, u,         u },
        { 'I',         519, -254, u,         u },
        { ',',         581, -254, u,         u },
        { '0',         643, -254, u,         u },
        { '9',         705, -254, u,         u },
        { 'RSHIFT',    767, -254, u2_75 + 2, u },

        -- 6 row

        { 'LCTRL',     6,   -316, u1_5 + 1,  u },
        { 'LWIN',      99,  -316, u,         u },
        { 'LALT',      161, -316, u1_5 + 1,  u },
        { 'SPACE',     254, -316, 368,       u },
        { 'RALT',      624, -316, u1_5 + 2,  u },
        { 'RWIN',      718, -316, u,         u },
        { 'MENU',      780, -316, u,         u },
        { 'RCTRL',     842, -316, u1_5 + 2,  u },
    },
    --

    -- RAZER --
    ['Tartarus_v1'] = {

        { 'TAB',        6,   -16,  u, u },
        { 'Q',          68,  -6,   u, u },
        { 'W',          130, -6,   u, u },
        { 'E',          192, -6,   u, u },
        { 'R',          254, -6,   u, u },

        { 'CAPSLOCK',   6,   -78,  u, u },
        { 'A',          68,  -68,  u, u },
        { 'S',          130, -68,  u, u },
        { 'D',          192, -68,  u, u },
        { 'F',          254, -68,  u, u },

        { 'LSHIFT',     6,   -140, u, u },
        { 'Z',          68,  -130, u, u },
        { 'X',          130, -130, u, u },
        { 'C',          192, -130, u, u },
        { 'V',          254, -130, u, u },

        { 'ALT',        316, -192, u, u },

        { 'LEFT',       356, -296, 40, 40, 30, 30 },
        { '',           378, -254, 40, 40, 30, 30 },
        { 'UP',         420, -234, 40, 40, 30, 30 },
        { '',           462, -254, 40, 40, 30, 30 },
        { 'RIGHT',      482, -296, 40, 40, 30, 30 },
        { '',           462, -338, 40, 40, 30, 30 },
        { 'DOWN',       420, -358, 40, 40, 30, 30 },
        { '',           378, -338, 40, 40, 30, 30 },

        { 'SPACE',      505, -380, u, u },

    },
    ['Tartarus_v2'] = {

        { '1',        6,   -16,  u, u },
        { '2',        68,  -6,   u, u },
        { '3',        130, -6,   u, u },
        { '4',        192, -6,   u, u },
        { '5',        254, -6,   u, u },

        { 'TAB',      6,   -78,  u, u },
        { 'Q',        68,  -68,  u, u },
        { 'W',        130, -68,  u, u },
        { 'E',        192, -68,  u, u },
        { 'R',        254, -68,  u, u },

        { 'CAPSLOCK', 6,   -140, u, u },
        { 'A',        68,  -130, u, u },
        { 'S',        130, -130, u, u },
        { 'D',        192, -130, u, u },
        { 'F',        254, -130, u, u },

        { 'LSHIFT',   6,   -202, u, u },
        { 'Z',        68,  -192, u, u },
        { 'X',        130, -192, u, u },
        { 'C',        192, -192, u, u },

        { 'MOUSEWHEELUP',       254, -192, u, u },
        { 'BUTTON3',            254, -254, u, u },
        { 'MOUSEWHEELDOWN',     254, -316, u, u },

{ 'ALT',        316, -254, u, u },
{ 'LEFT',       356, -358, 40, 40, 30, 30 },
{ '',           378, -316, 40, 40, 30, 30 },
{ 'UP',         420, -296, 40, 40, 30, 30 },
{ '',           462, -316, 40, 40, 30, 30 },
{ 'RIGHT',      482, -358, 40, 40, 30, 30 },
{ '',           462, -400, 40, 40, 30, 30 },
{ 'DOWN',       420, -420, 40, 40, 30, 30 },
{ '',           378, -400, 40, 40, 30, 30 },
{ 'SPACE',      505, -442, u, u },

    },
    --

    -- AZERON --
    ['Cyborg'] = {

        { 'I',     130, -6,   u, u },
        { 'M',     192, -6,   u, u },
        { 'UP',    316, -6,   u, u },

        { 'Z',     6,   -68,  u, u },
        { 'X',     68,  -68,  u, u },
        { 'C',     130, -68,  u, u },
        { 'V',     192, -68,  u, u },
        { 'LEFT',  254, -68,  u, u },
        { 'T',     316, -68,  u, u },
        { 'RIGHT', 378, -68,  u, u },

        { 'Q',     6,   -130, u, u },
        { 'G',     68,  -130, u, u },
        { 'R',     130, -130, u, u },
        { 'E',     192, -130, u, u },
        { 'DOWN',  316, -130, u, u },

        { '1',     6,   -192, u, u },
        { '2',     68,  -192, u, u },
        { '3',     130, -192, u, u },
        { '4',     192, -192, u, u },
        { 'F',     254, -192, u, u },

        { 'A',      316, -234, 40,       40,     30, 30 },
        { 'D',      398, -234, 40,       40,     30, 30 },
        { 'W',      357, -192, 40,       40,     30, 30 },
        { 'S',      357, -274, 40,       40,     30, 30 },

        { 'A',     6,   -254, u, u },
        { 'ALT',   68,  -254, u, u },
        { 'SHIFT', 130, -254, u, u },
        { 'CTRL',  192, -254, u, u },

        { '=',     316, -316, u, u },
        { 'ESCAPE',   378, -316, u, u },
    },
    ['Cyborg_II'] = {

        { 'UP',     378, -6,   u, u },

        { 'Z',      68,  -68,  u, u },
        { 'X',      130, -68,  u, u },
        { 'C',      192, -68,  u, u },
        { 'V',      254, -68,  u, u },
        { 'LEFT',   316, -68,  u, u },
        { 'T',      378, -68,  u, u },
        { 'RIGHT',  440, -68,  u, u },

        { 'Q',      68,  -130, u, u },
        { 'G',      130, -130, u, u },
        { 'R',      192, -130, u, u },
        { 'E',      254, -130, u, u },
        { 'DOWN',   378, -130, u, u },

        { '0',      6,   -192, u, u },
        { '1',      68,  -192, u, u },
        { '2',      130, -192, u, u },
        { '3',      192, -192, u, u },
        { '4',      254, -192, u, u },
        { 'F',      316, -192, u, u },
        { 'Z',      502, -192, u, u },

        { 'A',      378, -234, 40,       40,     30, 30 },
        { 'D',      460, -234, 40,       40,     30, 30 },
        { 'W',      419, -192, 40,       40,     30, 30 },
        { 'S',      419, -274, 40,       40,     30, 30 },

        { 'A',      68,  -254, u, u },
        { 'ALT',    130, -254, u, u },
        { 'SHIFT',  192, -254, u, u },
        { 'CTRL',   254, -254, u, u },
        { 'ESCAPE', 502, -254, u, u },

        { 'P',      68,  -316, u, u },
        { 'L',      130, -316, u, u },
        { 'I',      192, -316, u, u },
        { 'M',      254, -316, u, u },
        { '=',      378, -316, u, u },
    },
    --

    -- ZSA --
    ['Moonlander_MK_I'] = {

        -- Row 1 left half

        { '=',         6,    -18,  u,  u },
        { '1',         68,   -14,  u,  u },
        { '2',         130,  -10,  u,  u },
        { '3',         192,  -6,   u,  u },
        { '4',         254,  -10,  u,  u },
        { '5',         316,  -14,  u,  u },
        { 'LEFT',      378,  -18,  u,  u },

        -- Row 1 right half

        { 'RIGHT',     750,  -18,  u,  u },
        { '6',         812,  -14,  u,  u },
        { '7',         874,  -10,  u,  u },
        { '8',         936,  -6,   u,  u },
        { '9',         998,  -10,  u,  u },
        { '0',         1060, -14,  u,  u },
        { '-',         1122, -18,  u,  u },

        -- Row 2 left half

        { 'DELETE',    6,    -80,  u,  u },
        { 'Q',         68,   -76,  u,  u },
        { 'W',         130,  -72,  u,  u },
        { 'E',         192,  -68,  u,  u },
        { 'R',         254,  -72,  u,  u },
        { 'T',         316,  -76,  u,  u },
        { 'Custom',       378,  -80,  u,  u },

        -- Row 2 right half

        { 'Custom',       750,  -80,  u,  u },
        { 'Y',         812,  -76,  u,  u },
        { 'U',         874,  -72,  u,  u },
        { 'I',         936,  -68,  u,  u },
        { 'O',         998,  -72,  u,  u },
        { 'P',         1060, -76,  u,  u },
        { '\\',        1122, -80,  u,  u },

        -- Row 3 left half

        { 'BACKSPACE', 6,    -142, u,  u },
        { 'A',         68,   -138, u,  u },
        { 'S',         130,  -134, u,  u },
        { 'D',         192,  -130, u,  u },
        { 'F',         254,  -134, u,  u },
        { 'G',         316,  -138, u,  u },
        { 'Custom',       378,  -142, u,  u },

        -- Row 3 right half

        { 'Custom',       750,  -142, u,  u },
        { 'H',         812,  -138, u,  u },
        { 'J',         874,  -134, u,  u },
        { 'K',         936,  -130, u,  u },
        { 'L',         998,  -134, u,  u },
        { ';',         1060, -138, u,  u },
        { '\'',        1122, -142, u,  u },

        -- Row 4 left half

        { 'LSHIFT',    6,    -204, u,  u },
        { 'Z',         68,   -200, u,  u },
        { 'X',         130,  -196, u,  u },
        { 'C',         192,  -192, u,  u },
        { 'V',         254,  -196, u,  u },
        { 'B',         316,  -200, u,  u },
        -- { 'Custom',       378, -142, u, u },

        -- Row 4 right half

        { 'N',         812,  -200, u,  u },
        { 'M',         874,  -196, u,  u },
        { ',',         936,  -192, u,  u },
        { '.',         998,  -196, u,  u },
        { '/',         1060, -200, u,  u },
        { 'RSHIFT',    1122, -204, u,  u },
        -- { 'Custom',    1184, -142, u, u },

        -- Row 5 left half

        { '`',         6,    -266, u,  u },
        { 'CAPSLOCK',  68,   -262, u,  u },
        { 'Custom',       130,  -258, u,  u },
        { 'LEFT',      192,  -254, u,  u },
        { 'RIGHT',     254,  -258, u,  u },
        -- { 'B',         316, -200, u, u },
        -- { 'Custom',       378, -142, u, u },

        -- Row 5 right half

        { 'UP',        874,  -258, u,  u },
        { 'DOWN',      936,  -254, u,  u },
        { '[',         998,  -258, u,  u },
        { ']',         1060, -262, u,  u },
        { 'Custom',       1122, -266, u,  u },
        -- { 'B',      1184, -200, u, u },
        -- { 'Custom',    1246, -142, u, u },

        -- -- Left Thumb Area
        { 'LALT',      378,  -266, u2, u },
        { 'LWIN',      378,  -328, u,  u },
        { 'SPACE',     440,  -328, u,  u },
        { 'BACKSPACE', 502,  -328, u,  u },

        -- -- Right Thumb Area
        { 'ESCAPE',    688,  -266, u2, u },
        { 'LALT',      626,  -328, u, u },
        { 'TAB',       688,  -328, u, u },
        { 'ENTER',     750,  -328, u, u },
    },
    --

    -- LOGITECH --
    ['G13'] = {

        -- 1 row

        { 'G1',        6,   -6,   u,         u },
        { 'G2',        68,  -6,   u,         u },
        { 'G3',        130, -6,   u,         u },
        { 'G4',        192, -6,   u,         u },
        { 'G5',        254, -6,   u,         u },
        { 'G6',        316, -6,   u,         u },
        { 'G7',        378, -6,   u,         u },

        -- 2 row

        { 'G8',        6,   -68,   u,        u },
        { 'G9',        68,  -68,   u,        u },
        { 'G10',       130, -68,   u,        u },
        { 'G11',       192, -68,   u,        u },
        { 'G12',       254, -68,   u,        u },
        { 'G13',       316, -68,   u,        u },
        { 'G14',       378, -68,   u,        u },

        -- 3 row

        { 'G15',       68,  -130,  u,        u },
        { 'G16',       130, -130,  u,        u },
        { 'G17',       192, -130,  u,        u },
        { 'G18',       254, -130,  u,        u },
        { 'G19',       316, -130,  u,        u },

        -- 4 row

        { 'G20',       130, -192,  u,        u },
        { 'G21',       192, -192,  u,        u },
        { 'G22',       254, -192,  u,        u },

        -- Thumb

        { '',          316, -254,  u,        u },
        { '',          378, -264, 40,       40,     30, 30 },
        { '',          420, -264, 40,       40,     30, 30 },
        { '',          462, -264, 40,       40,     30, 30 },
        { '',          420, -222, 40,       40,     30, 30 },
        { '',          420, -306, 40,       40,     30, 30 },
        { '',          410, -348,  u,        u },

    },
    --
}