// /*
//  * Copyright 2024 Kelvin Nishikawa
//  *
//  * Licensed under the Apache License, Version 2.0 (the "License");
//  * you may not use this file except in compliance with the License.
//  * You may obtain a copy of the License at
//  *
//  *     http://www.apache.org/licenses/LICENSE-2.0
//  *
//  * Unless required by applicable law or agreed to in writing, software
//  * distributed under the License is distributed on an "AS IS" BASIS,
//  * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  * See the License for the specific language governing permissions and
//  * limitations under the License.
//  */

using Wacs.Core.Attributes;

// ReSharper disable InconsistentNaming
namespace Wacs.Core.OpCodes
{
    /// <summary>
    /// Represents all WebAssembly opcodes for SIMD (FD prefix)
    /// Theoretically, these could be u32, but I'll keep them as bytes so long as they fit.
    /// </summary>
    // https://www.w3.org/TR/2024/WD-wasm-core-2-20240911/#a7-index-of-instructions
    public enum SimdCode : byte
    {
        [OpCode("v128.load")]                     V128Load                  = 0x00,
        [OpCode("v128.load8x8_s")]                V128Load8x8S              = 0x01,
        [OpCode("v128.load8x8_u")]                V128Load8x8U              = 0x02,
        [OpCode("v128.load16x4_s")]               V128Load16x4S             = 0x03,
        [OpCode("v128.load16x4_u")]               V128Load16x4U             = 0x04,
        [OpCode("v128.load32x2_s")]               V128Load32x2S             = 0x05,
        [OpCode("v128.load32x2_u")]               V128Load32x2U             = 0x06,
        [OpCode("v128.load8_splat")]              V128Load8Splat            = 0x07,
        [OpCode("v128.load16_splat")]             V128Load16Splat           = 0x08,
        [OpCode("v128.load32_splat")]             V128Load32Splat           = 0x09,
        [OpCode("v128.load64_splat")]             V128Load64Splat           = 0x0A,
        [OpCode("v128.store")]                    V128Store                 = 0x0B,
        [OpCode("v128.const")]                    V128Const                 = 0x0C,
        [OpCode("i8x16.shuffle")]                 I8x16Shuffle              = 0x0D,
        [OpCode("i8x16.swizzle")]                 I8x16Swizzle              = 0x0E,
        [OpCode("i8x16.splat")]                   I8x16Splat                = 0x0F,
        [OpCode("i16x8.splat")]                   I16x8Splat                = 0x10,
        [OpCode("i32x4.splat")]                   I32x4Splat                = 0x11,
        [OpCode("i64x2.splat")]                   I64x2Splat                = 0x12,
        [OpCode("f32x4.splat")]                   F32x4Splat                = 0x13,
        [OpCode("f64x2.splat")]                   F64x2Splat                = 0x14,
        [OpCode("i8x16.extract_lane_s")]          I8x16ExtractLaneS         = 0x15,
        [OpCode("i8x16.extract_lane_u")]          I8x16ExtractLaneU         = 0x16,
        [OpCode("i8x16.replace_lane")]            I8x16ReplaceLane          = 0x17,
        [OpCode("i16x8.extract_lane_s")]          I16x8ExtractLaneS         = 0x18,
        [OpCode("i16x8.extract_lane_u")]          I16x8ExtractLaneU         = 0x19,
        [OpCode("i16x8.replace_lane")]            I16x8ReplaceLane          = 0x1A,
        [OpCode("i32x4.extract_lane")]            I32x4ExtractLane          = 0x1B,
        [OpCode("i32x4.replace_lane")]            I32x4ReplaceLane          = 0x1C,
        [OpCode("i64x2.extract_lane")]            I64x2ExtractLane          = 0x1D,
        [OpCode("i64x2.replace_lane")]            I64x2ReplaceLane          = 0x1E,
        [OpCode("f32x4.extract_lane")]            F32x4ExtractLane          = 0x1F,
        [OpCode("f32x4.replace_lane")]            F32x4ReplaceLane          = 0x20,
        [OpCode("f64x2.extract_lane")]            F64x2ExtractLane          = 0x21,
        [OpCode("f64x2.replace_lane")]            F64x2ReplaceLane          = 0x22,
        [OpCode("i8x16.eq")]                      I8x16Eq                   = 0x23,
        [OpCode("i8x16.ne")]                      I8x16Ne                   = 0x24,
        [OpCode("i8x16.lt_s")]                    I8x16LtS                  = 0x25,
        [OpCode("i8x16.lt_u")]                    I8x16LtU                  = 0x26,
        [OpCode("i8x16.gt_s")]                    I8x16GtS                  = 0x27,
        [OpCode("i8x16.gt_u")]                    I8x16GtU                  = 0x28,
        [OpCode("i8x16.le_s")]                    I8x16LeS                  = 0x29,
        [OpCode("i8x16.le_u")]                    I8x16LeU                  = 0x2A,
        [OpCode("i8x16.ge_s")]                    I8x16GeS                  = 0x2B,
        [OpCode("i8x16.ge_u")]                    I8x16GeU                  = 0x2C,
        [OpCode("i16x8.eq")]                      I16x8Eq                   = 0x2D,
        [OpCode("i16x8.ne")]                      I16x8Ne                   = 0x2E,
        [OpCode("i16x8.lt_s")]                    I16x8LtS                  = 0x2F,
        [OpCode("i16x8.lt_u")]                    I16x8LtU                  = 0x30,
        [OpCode("i16x8.gt_s")]                    I16x8GtS                  = 0x31,
        [OpCode("i16x8.gt_u")]                    I16x8GtU                  = 0x32,
        [OpCode("i16x8.le_s")]                    I16x8LeS                  = 0x33,
        [OpCode("i16x8.le_u")]                    I16x8LeU                  = 0x34,
        [OpCode("i16x8.ge_s")]                    I16x8GeS                  = 0x35,
        [OpCode("i16x8.ge_u")]                    I16x8GeU                  = 0x36,
        [OpCode("i32x4.eq")]                      I32x4Eq                   = 0x37,
        [OpCode("i32x4.ne")]                      I32x4Ne                   = 0x38,
        [OpCode("i32x4.lt_s")]                    I32x4LtS                  = 0x39,
        [OpCode("i32x4.lt_u")]                    I32x4LtU                  = 0x3A,
        [OpCode("i32x4.gt_s")]                    I32x4GtS                  = 0x3B,
        [OpCode("i32x4.gt_u")]                    I32x4GtU                  = 0x3C,
        [OpCode("i32x4.le_s")]                    I32x4LeS                  = 0x3D,
        [OpCode("i32x4.le_u")]                    I32x4LeU                  = 0x3E,
        [OpCode("i32x4.ge_s")]                    I32x4GeS                  = 0x3F,
        [OpCode("i32x4.ge_u")]                    I32x4GeU                  = 0x40,
        [OpCode("f32x4.eq")]                      F32x4Eq                   = 0x41,
        [OpCode("f32x4.ne")]                      F32x4Ne                   = 0x42,
        [OpCode("f32x4.lt")]                      F32x4Lt                   = 0x43,
        [OpCode("f32x4.gt")]                      F32x4Gt                   = 0x44,
        [OpCode("f32x4.le")]                      F32x4Le                   = 0x45,
        [OpCode("f32x4.ge")]                      F32x4Ge                   = 0x46, 
        [OpCode("f64x2.eq")]                      F64x2Eq                   = 0x47,
        [OpCode("f64x2.ne")]                      F64x2Ne                   = 0x48,
        [OpCode("f64x2.lt")]                      F64x2Lt                   = 0x49,
        [OpCode("f64x2.gt")]                      F64x2Gt                   = 0x4A,
        [OpCode("f64x2.le")]                      F64x2Le                   = 0x4B,
        [OpCode("f64x2.ge")]                      F64x2Ge                   = 0x4C,
        [OpCode("v128.not")]                      V128Not                   = 0x4D,
        [OpCode("v128.and")]                      V128And                   = 0x4E,
        [OpCode("v128.andnot")]                   V128AndNot                = 0x4F,
        [OpCode("v128.or")]                       V128Or                    = 0x50,
        [OpCode("v128.xor")]                      V128Xor                   = 0x51,
        [OpCode("v128.bitselect")]                V128BitSelect             = 0x52,
        [OpCode("v128.any_true")]                 V128AnyTrue               = 0x53,
        [OpCode("v128.load8_lane")]               V128Load8Lane             = 0x54,
        [OpCode("v128.load16_lane")]              V128Load16Lane            = 0x55,
        [OpCode("v128.load32_lane")]              V128Load32Lane            = 0x56,
        [OpCode("v128.load64_lane")]              V128Load64Lane            = 0x57,
        [OpCode("v128.store8_lane")]              V128Store8Lane            = 0x58,
        [OpCode("v128.store16_lane")]             V128Store16Lane           = 0x59,
        [OpCode("v128.store32_lane")]             V128Store32Lane           = 0x5A,
        [OpCode("v128.store64_lane")]             V128Store64Lane           = 0x5B,
        [OpCode("v128.load32_zero")]              V128Load32Zero            = 0x5C,
        [OpCode("v128.load64_zero")]              V128Load64Zero            = 0x5D,
        [OpCode("f32x4.demote_f64x2_zero")]       F32x4DemoteF64x2Zero      = 0x5E,
        [OpCode("f64x2.promote_low_f32x4")]       F64x2PromoteLowF32x4      = 0x5F,
        [OpCode("i8x16.abs")]                     I8x16Abs                  = 0x60,
        [OpCode("i8x16.neg")]                     I8x16Neg                  = 0x61,
        [OpCode("i8x16.popcnt")]                  I8x16Popcnt               = 0x62,
        [OpCode("i8x16.all_true")]                I8x16AllTrue              = 0x63,
        [OpCode("i8x16.bitmask")]                 I8x16Bitmask              = 0x64,
        [OpCode("i8x16.narrow_i16x8_s")]          I8x16NarrowI16x8S         = 0x65,
        [OpCode("i8x16.narrow_i16x8_u")]          I8x16NarrowI16x8U         = 0x66,
        [OpCode("f32x4.ceil")]                    F32x4Ceil                 = 0x67,
        [OpCode("f32x4.floor")]                   F32x4Floor                = 0x68,
        [OpCode("f32x4.trunc")]                   F32x4Trunc                = 0x69,
        [OpCode("f32x4.nearest")]                 F32x4Nearest              = 0x6A,
        [OpCode("i8x16.shl")]                     I8x16Shl                  = 0x6B,
        [OpCode("i8x16.shr_s")]                   I8x16ShrS                 = 0x6C,
        [OpCode("i8x16.shr_u")]                   I8x16ShrU                 = 0x6D,
        [OpCode("i8x16.add")]                     I8x16Add                  = 0x6E,
        [OpCode("i8x16.add_sat_s")]               I8x16AddSatS              = 0x6F,
        [OpCode("i8x16.add_sat_u")]               I8x16AddSatU              = 0x70,
        [OpCode("i8x16.sub")]                     I8x16Sub                  = 0x71,
        [OpCode("i8x16.sub_sat_s")]               I8x16SubSatS              = 0x72,
        [OpCode("i8x16.sub_sat_u")]               I8x16SubSatU              = 0x73,
        [OpCode("f64x2.ceil")]                    F64x2Ceil                 = 0x74,
        [OpCode("f64x2.floor")]                   F64x2Floor                = 0x75,
        [OpCode("i8x16.min_s")]                   I8x16MinS                 = 0x76,
        [OpCode("i8x16.min_u")]                   I8x16MinU                 = 0x77,
        [OpCode("i8x16.max_s")]                   I8x16MaxS                 = 0x78,
        [OpCode("i8x16.max_u")]                   I8x16MaxU                 = 0x79,
        [OpCode("f64x2.trunc")]                   F64x2Trunc                = 0x7A,
        [OpCode("i8x16.avgr_u")]                  I8x16AvgrU                = 0x7B,
        [OpCode("i16x8.extadd_pairwise_i8x16_s")] I16x8ExtAddPairwiseI8x16S = 0x7C,
        [OpCode("i16x8.extadd_pairwise_i8x16_u")] I16x8ExtAddPairwiseI8x16U = 0x7D,
        [OpCode("i32x4.extadd_pairwise_i16x8_s")] I32x4ExtAddPairwiseI16x8S = 0x7E,
        [OpCode("i32x4.extadd_pairwise_i16x8_u")] I32x4ExtAddPairwiseI16x8U = 0x7F,
        [OpCode("i16x8.abs")]                     I16x8Abs                  = 0x80,
        [OpCode("i16x8.neg")]                     I16x8Neg                  = 0x81,
        [OpCode("i16x8.q15mulr_sat_s")]           I16x8Q15MulRSatS          = 0x82,
        [OpCode("i16x8.all_true")]                I16x8AllTrue              = 0x83,
        [OpCode("i16x8.bitmask")]                 I16x8Bitmask              = 0x84,
        [OpCode("i16x8.narrow_i32x4_s")]          I16x8NarrowI32x4S         = 0x85,
        [OpCode("i16x8.narrow_i32x4_u")]          I16x8NarrowI32x4U         = 0x86,
        [OpCode("i16x8.extend_low_i8x16_s")]      I16x8ExtendLowI8x16S      = 0x87,
        [OpCode("i16x8.extend_high_i8x16_s")]     I16x8ExtendHighI8x16S     = 0x88,
        [OpCode("i16x8.extend_low_i8x16_u")]      I16x8ExtendLowI8x16U      = 0x89,
        [OpCode("i16x8.extend_high_i8x16_u")]     I16x8ExtendHighI8x16U     = 0x8A,
        [OpCode("i16x8.shl")]                     I16x8Shl                  = 0x8B,
        [OpCode("i16x8.shr_s")]                   I16x8ShrS                 = 0x8C,
        [OpCode("i16x8.shr_u")]                   I16x8ShrU                 = 0x8D,
        [OpCode("i16x8.add")]                     I16x8Add                  = 0x8E,
        [OpCode("i16x8.add_sat_s")]               I16x8AddSatS              = 0x8F,
        [OpCode("i16x8.add_sat_u")]               I16x8AddSatU              = 0x90,
        [OpCode("i16x8.sub")]                     I16x8Sub                  = 0x91,
        [OpCode("i16x8.sub_sat_s")]               I16x8SubSatS              = 0x92,
        [OpCode("i16x8.sub_sat_u")]               I16x8SubSatU              = 0x93,
        [OpCode("f64x2.nearest")]                 F64x2Nearest              = 0x94,
        [OpCode("i16x8.mul")]                     I16x8Mul                  = 0x95,
        [OpCode("i16x8.min_s")]                   I16x8MinS                 = 0x96,
        [OpCode("i16x8.min_u")]                   I16x8MinU                 = 0x97,
        [OpCode("i16x8.max_s")]                   I16x8MaxS                 = 0x98,
        [OpCode("i16x8.max_u")]                   I16x8MaxU                 = 0x99,
        [OpCode("i16x8.avgr_u")]                  I16x8AvgrU                = 0x9B,
        [OpCode("i16x8.extmul_low_i8x16_s")]      I16x8ExtMulLowI8x16S      = 0x9C,
        [OpCode("i16x8.extmul_high_i8x16_s")]     I16x8ExtMulHighI8x16S     = 0x9D,
        [OpCode("i16x8.extmul_low_i8x16_u")]      I16x8ExtMulLowI8x16U      = 0x9E,
        [OpCode("i16x8.extmul_high_i8x16_u")]     I16x8ExtMulHighI8x16U     = 0x9F,
        [OpCode("i32x4.abs")]                     I32x4Abs                  = 0xA0,
        [OpCode("i32x4.neg")]                     I32x4Neg                  = 0xA1,
        [OpCode("i32x4.all_true")]                I32x4AllTrue              = 0xA3,
        [OpCode("i32x4.bitmask")]                 I32x4Bitmask              = 0xA4,
        [OpCode("i32x4.extend_low_i16x8_s")]      I32x4ExtendLowI16x8S      = 0xA7,
        [OpCode("i32x4.extend_high_i16x8_s")]     I32x4ExtendHighI16x8S     = 0xA8,
        [OpCode("i32x4.extend_low_i16x8_u")]      I32x4ExtendLowI16x8U      = 0xA9,
        [OpCode("i32x4.extend_high_i16x8_u")]     I32x4ExtendHighI16x8U     = 0xAA,
        [OpCode("i32x4.shl")]                     I32x4Shl                  = 0xAB,
        [OpCode("i32x4.shr_s")]                   I32x4ShrS                 = 0xAC,
        [OpCode("i32x4.shr_u")]                   I32x4ShrU                 = 0xAD,
        [OpCode("i32x4.add")]                     I32x4Add                  = 0xAE,
        [OpCode("i32x4.sub")]                     I32x4Sub                  = 0xB1,
        [OpCode("i32x4.mul")]                     I32x4Mul                  = 0xB5,
        [OpCode("i32x4.min_s")]                   I32x4MinS                 = 0xB6,
        [OpCode("i32x4.min_u")]                   I32x4MinU                 = 0xB7,
        [OpCode("i32x4.max_s")]                   I32x4MaxS                 = 0xB8,
        [OpCode("i32x4.max_u")]                   I32x4MaxU                 = 0xB9,
        [OpCode("i32x4.dot_i16x8_s")]             I32x4DotI16x8S            = 0xBA,
        [OpCode("i32x4.extmul_low_i16x8_s")]      I32x4ExtMulLowI16x8S      = 0xBC,
        [OpCode("i32x4.extmul_high_i16x8_s")]     I32x4ExtMulHighI16x8S     = 0xBD,
        [OpCode("i32x4.extmul_low_i16x8_u")]      I32x4ExtMulLowI16x8U      = 0xBE,
        [OpCode("i32x4.extmul_high_i16x8_u")]     I32x4ExtMulHighI16x8U     = 0xBF,
        [OpCode("i64x2.abs")]                     I64x2Abs                  = 0xC0,
        [OpCode("i64x2.neg")]                     I64x2Neg                  = 0xC1,
        [OpCode("i64x2.all_true")]                I64x2AllTrue              = 0xC3,
        [OpCode("i64x2.bitmask")]                 I64x2Bitmask              = 0xC4,
        [OpCode("i64x2.extend_low_i32x4_s")]      I64x2ExtendLowI32x4S      = 0xC7,
        [OpCode("i64x2.extend_high_i32x4_s")]     I64x2ExtendHighI32x4S     = 0xC8,
        [OpCode("i64x2.extend_low_i32x4_u")]      I64x2ExtendLowI32x4U      = 0xC9,
        [OpCode("i64x2.extend_high_i32x4_u")]     I64x2ExtendHighI32x4U     = 0xCA,
        [OpCode("i64x2.shl")]                     I64x2Shl                  = 0xCB,
        [OpCode("i64x2.shr_s")]                   I64x2ShrS                 = 0xCC,
        [OpCode("i64x2.shr_u")]                   I64x2ShrU                 = 0xCD,
        [OpCode("i64x2.add")]                     I64x2Add                  = 0xCE,
        [OpCode("i64x2.sub")]                     I64x2Sub                  = 0xD1,
        [OpCode("i64x2.mul")]                     I64x2Mul                  = 0xD5,
        [OpCode("i64x2.eq")]                      I64x2Eq                   = 0xD6,
        [OpCode("i64x2.ne")]                      I64x2Ne                   = 0xD7,
        [OpCode("i64x2.lt_s")]                    I64x2LtS                  = 0xD8,
        [OpCode("i64x2.gt_s")]                    I64x2GtS                  = 0xD9,
        [OpCode("i64x2.le_s")]                    I64x2LeS                  = 0xDA,
        [OpCode("i64x2.ge_s")]                    I64x2GeS                  = 0xDB,
        [OpCode("i64x2.extmul_low_i32x4_s")]      I64x2ExtMulLowI32x4S      = 0xDC,
        [OpCode("i64x2.extmul_high_i32x4_s")]     I64x2ExtMulHighI32x4S     = 0xDD,
        [OpCode("i64x2.extmul_low_i32x4_u")]      I64x2ExtMulLowI32x4U      = 0xDE,
        [OpCode("i64x2.extmul_high_i32x4_u")]     I64x2ExtMulHighI32x4U     = 0xDF,
        [OpCode("f32x4.abs")]                     F32x4Abs                  = 0xE0,
        [OpCode("f32x4.neg")]                     F32x4Neg                  = 0xE1,
        [OpCode("f32x4.sqrt")]                    F32x4Sqrt                 = 0xE3,
        [OpCode("f32x4.add")]                     F32x4Add                  = 0xE4,
        [OpCode("f32x4.sub")]                     F32x4Sub                  = 0xE5,
        [OpCode("f32x4.mul")]                     F32x4Mul                  = 0xE6,
        [OpCode("f32x4.div")]                     F32x4Div                  = 0xE7,
        [OpCode("f32x4.min")]                     F32x4Min                  = 0xE8,
        [OpCode("f32x4.max")]                     F32x4Max                  = 0xE9,
        [OpCode("f32x4.pmin")]                    F32x4PMin                 = 0xEA,
        [OpCode("f32x4.pmax")]                    F32x4PMax                 = 0xEB,
        [OpCode("f64x2.abs")]                     F64x2Abs                  = 0xEC,
        [OpCode("f64x2.neg")]                     F64x2Neg                  = 0xED,
        [OpCode("f64x2.sqrt")]                    F64x2Sqrt                 = 0xEF,
        [OpCode("f64x2.add")]                     F64x2Add                  = 0xF0,
        [OpCode("f64x2.sub")]                     F64x2Sub                  = 0xF1,
        [OpCode("f64x2.mul")]                     F64x2Mul                  = 0xF2,
        [OpCode("f64x2.div")]                     F64x2Div                  = 0xF3,
        [OpCode("f64x2.min")]                     F64x2Min                  = 0xF4,
        [OpCode("f64x2.max")]                     F64x2Max                  = 0xF5,
        [OpCode("f64x2.pmin")]                    F64x2PMin                 = 0xF6,
        [OpCode("f64x2.pmax")]                    F64x2PMax                 = 0xF7,
        [OpCode("i32x4.trunc_sat_f32x4_s")]       I32x4TruncSatF32x4S       = 0xF8,
        [OpCode("i32x4.trunc_sat_f32x4_u")]       I32x4TruncSatF32x4U       = 0xF9,
        [OpCode("f32x4.convert_i32x4_s")]         F32x4ConvertI32x4S        = 0xFA,
        [OpCode("f32x4.convert_i32x4_u")]         F32x4ConvertI32x4U        = 0xFB,
        [OpCode("i32x4.trunc_sat_f64x2_s_zero")]  I32x4TruncSatF64x2SZero   = 0xFC,
        [OpCode("i32x4.trunc_sat_f64x2_u_zero")]  I32x4TruncSatF64x2UZero   = 0xFD,
        [OpCode("f64x2.convert_low_i32x4_s")]     F64x2ConvertLowI32x4S     = 0xFE,
        [OpCode("f64x2.convert_low_i32x4_u")]     F64x2ConvertLowI32x4U     = 0xFF,
    }

}