/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.cache;

import java.io.Serializable;
import java.util.Iterator;

import org.miaixz.bus.core.cache.provider.CacheObject;
import org.miaixz.bus.core.center.function.SupplierX;

/**
 * Generic cache interface.
 *
 * @param <K> The type of the key.
 * @param <V> The type of the value.
 * @author Kimi Liu
 * @since Java 17+
 */
public interface Cache<K, V> extends Iterable<V>, Serializable {

    /**
     * Returns the cache capacity. A value of {@code 0} indicates no size limit.
     *
     * @return The cache capacity.
     */
    int capacity();

    /**
     * Returns the cache timeout in milliseconds. A value of {@code 0} indicates that no timeout is set.
     *
     * @return The cache timeout.
     */
    long timeout();

    /**
     * Puts an object into the cache with the default timeout.
     *
     * @param key    The key.
     * @param object The object to be cached.
     * @see #put(Object, Object, long)
     */
    void put(K key, V object);

    /**
     * Puts an object into the cache with a specified timeout. If the cache is full, {@link #prune()} will be called to
     * make space for the new object.
     *
     * @param key     The key.
     * @param object  The object to be cached.
     * @param timeout The timeout for the object in milliseconds.
     */
    void put(K key, V object, long timeout);

    /**
     * Gets an object from the cache. Returns {@code null} if the object is not in the cache or has expired.
     *
     * @param key                The key.
     * @param isUpdateLastAccess Whether to update the last access time, effectively resetting the timeout.
     * @return The cached object, or {@code null} if not found or expired.
     */
    V get(K key, boolean isUpdateLastAccess);

    /**
     * Gets an object from the cache. If the object is not present or has expired, it is generated by the provided
     * {@link SupplierX}.
     *
     * @param key                The key.
     * @param isUpdateLastAccess Whether to update the last access time.
     * @param timeout            A custom timeout for the new object if it is generated.
     * @param supplier           A callback to generate the object if it's not in the cache.
     * @return The value object.
     */
    V get(K key, boolean isUpdateLastAccess, final long timeout, SupplierX<V> supplier);

    /**
     * Gets an object from the cache, updating its last access time. This is equivalent to calling
     * {@code get(key, true)}.
     *
     * @param key The key.
     * @return The cached object, or {@code null} if not found or expired.
     * @see #get(Object, boolean)
     */
    default V get(final K key) {
        return get(key, true);
    }

    /**
     * Gets an object from the cache. If the object is not present or has expired, it is generated by the provided
     * {@link SupplierX}. The last access time is updated upon retrieval.
     *
     * @param key      The key.
     * @param supplier A callback to generate the object if it's not in the cache.
     * @return The value object.
     */
    default V get(final K key, final SupplierX<V> supplier) {
        return get(key, true, supplier);
    }

    /**
     * Gets an object from the cache. If the object is not present or has expired, it is generated by the provided
     * {@link SupplierX}.
     *
     * @param key                The key.
     * @param isUpdateLastAccess Whether to update the last access time.
     * @param supplier           A callback to generate the object if it's not in the cache.
     * @return The value object.
     */
    default V get(final K key, final boolean isUpdateLastAccess, final SupplierX<V> supplier) {
        return get(key, isUpdateLastAccess, timeout(), supplier);
    }

    /**
     * Returns an iterator over the cache objects, including keys and values.
     *
     * @return An iterator for {@link CacheObject}.
     */
    Iterator<CacheObject<K, V>> cacheObjIterator();

    /**
     * Removes expired objects from the cache. The pruning strategy depends on the specific implementation.
     *
     * @return The number of objects removed.
     */
    int prune();

    /**
     * Checks if the cache is full. This is only applicable for caches with a size limit.
     *
     * @return {@code true} if the cache is full, otherwise {@code false}.
     */
    boolean isFull();

    /**
     * Removes an object from the cache by its key.
     *
     * @param key The key of the object to remove.
     */
    void remove(K key);

    /**
     * Clears the entire cache, removing all objects.
     */
    void clear();

    /**
     * Returns the number of objects currently in the cache.
     *
     * @return The size of the cache.
     */
    int size();

    /**
     * Checks if the cache is empty.
     *
     * @return {@code true} if the cache is empty, otherwise {@code false}.
     */
    boolean isEmpty();

    /**
     * Checks if the cache contains the specified key.
     *
     * @param key The key to check.
     * @return {@code true} if the cache contains the key, otherwise {@code false}.
     */
    boolean containsKey(K key);

    /**
     * Sets a listener for cache events, such as object removal.
     *
     * @param listener The listener to set.
     * @return This cache instance.
     */
    default Cache<K, V> setListener(final CacheListener<K, V> listener) {
        return this;
    }

}
