/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.center.date;

import java.io.Serial;
import java.io.Serializable;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.Date;

import org.miaixz.bus.core.center.date.culture.en.Units;
import org.miaixz.bus.core.center.date.format.FormatPeriod;
import org.miaixz.bus.core.lang.Assert;

/**
 * Represents the interval between two dates.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class Between implements Serializable {

    /**
     * The serial version UID for serialization.
     */
    @Serial
    private static final long serialVersionUID = 2852233282300L;

    /**
     * The beginning date of the interval.
     */
    private final Date begin;
    /**
     * The ending date of the interval.
     */
    private final Date end;

    /**
     * Constructs a {@code Between} object. The earlier date is set as the beginning time, and the later date as the
     * ending time. The interval always retains a positive absolute value.
     *
     * @param begin The starting date.
     * @param end   The ending date.
     */
    public Between(final Date begin, final Date end) {
        this(begin, end, true);
    }

    /**
     * Constructs a {@code Between} object.
     *
     * @param begin The starting date.
     * @param end   The ending date.
     * @param isAbs If {@code true}, the date interval will only retain a positive absolute value (swapping begin and
     *              end if begin > end).
     */
    public Between(final Date begin, final Date end, final boolean isAbs) {
        Assert.notNull(begin, "Begin date is null !");
        Assert.notNull(end, "End date is null !");

        if (isAbs && begin.after(end)) {
            // If the interval should be positive, and the begin date is after the end date, swap them.
            this.begin = end;
            this.end = begin;
        } else {
            this.begin = begin;
            this.end = end;
        }
    }

    /**
     * Creates a {@code Between} object. The earlier date is set as the beginning time, and the later date as the ending
     * time. The interval always retains a positive absolute value.
     *
     * @param begin The starting date.
     * @param end   The ending date.
     * @return A new {@code Between} instance.
     */
    public static Between of(final Date begin, final Date end) {
        return new Between(begin, end);
    }

    /**
     * Creates a {@code Between} object.
     *
     * @param begin The starting date.
     * @param end   The ending date.
     * @param isAbs If {@code true}, the date interval will only retain a positive absolute value (swapping begin and
     *              end if begin > end).
     * @return A new {@code Between} instance.
     */
    public static Between of(final Date begin, final Date end, final boolean isAbs) {
        return new Between(begin, end, isAbs);
    }

    /**
     * Calculates the duration between two {@link Temporal} objects. If the end time is earlier than the start time, the
     * result will be negative. The result is a {@link Duration} object, from which the difference in various units can
     * be obtained by calling its {@code toXXX} methods.
     *
     * @param startTimeInclude The inclusive start time.
     * @param endTimeExclude   The exclusive end time.
     * @return A {@link Duration} object representing the time difference.
     */
    public static Duration between(final Temporal startTimeInclude, final Temporal endTimeExclude) {
        return Duration.between(startTimeInclude, endTimeExclude);
    }

    /**
     * Calculates the duration between two {@link LocalDateTime} objects. If the end time is earlier than the start
     * time, the result will be negative. The result is a {@link Duration} object, from which the difference in various
     * units can be obtained by calling its {@code toXXX} methods.
     *
     * @param startTimeInclude The inclusive start time.
     * @param endTimeExclude   The exclusive end time.
     * @return A {@link Duration} object representing the time difference.
     * @see Between#between(Temporal, Temporal)
     */
    public static Duration between(final LocalDateTime startTimeInclude, final LocalDateTime endTimeExclude) {
        return between(startTimeInclude, endTimeExclude);
    }

    /**
     * Calculates the difference between two {@link Temporal} objects in a specified {@link ChronoUnit}. If the end time
     * is earlier than the start time, the result will be negative.
     *
     * @param startTimeInclude The inclusive start time.
     * @param endTimeExclude   The exclusive end time.
     * @param unit             The {@link ChronoUnit} to measure the difference in.
     * @return The time difference as a long value in the specified unit.
     */
    public static long between(final Temporal startTimeInclude, final Temporal endTimeExclude, final ChronoUnit unit) {
        return unit.between(startTimeInclude, endTimeExclude);
    }

    /**
     * Calculates the difference between two {@link LocalDateTime} objects in a specified {@link ChronoUnit}. If the end
     * time is earlier than the start time, the result will be negative.
     *
     * @param startTimeInclude The inclusive start time.
     * @param endTimeExclude   The exclusive end time.
     * @param unit             The {@link ChronoUnit} to measure the difference in.
     * @return The time difference as a long value in the specified unit.
     * @see Between#between(Temporal, Temporal, ChronoUnit)
     */
    public static long between(final LocalDateTime startTimeInclude, final LocalDateTime endTimeExclude,
            final ChronoUnit unit) {
        return between(startTimeInclude, endTimeExclude, unit);
    }

    /**
     * Calculates the period between two {@link LocalDate} objects. If the end date is earlier than the start date, the
     * result will be negative. For example, the period between February 1st, 2011, and August 11th, 2021, would show a
     * difference of 10 days and 6 months.
     *
     * @param startTimeInclude The inclusive start date.
     * @param endTimeExclude   The exclusive end date.
     * @return A {@link Period} object representing the date difference.
     */
    public static Period between(final LocalDate startTimeInclude, final LocalDate endTimeExclude) {
        return Period.between(startTimeInclude, endTimeExclude);
    }

    /**
     * Calculates the duration between the two dates held by this object in the specified {@link Units}.
     *
     * @param unit The unit of time to return the difference in (e.g., {@link Units#DAY}, {@link Units#HOUR}).
     * @return The duration difference in the specified unit.
     */
    public long between(final Units unit) {
        final long diff = end.getTime() - begin.getTime();
        return diff / unit.getMillis();
    }

    /**
     * Calculates the number of months between the two dates. If {@code isReset} is {@code false} and the day of the
     * beginning date is greater than the day of the ending date, the month count will be decremented by one (indicating
     * less than a full month).
     *
     * @param isReset Whether to reset the time of the dates to the beginning of the day (resetting day, hour, minute,
     *                second).
     * @return The number of months between the dates.
     */
    public long betweenMonth(final boolean isReset) {
        final java.util.Calendar beginCal = Calendar.calendar(begin);
        final java.util.Calendar endCal = Calendar.calendar(end);

        final int betweenYear = endCal.get(java.util.Calendar.YEAR) - beginCal.get(java.util.Calendar.YEAR);
        final int betweenMonthOfYear = endCal.get(java.util.Calendar.MONTH) - beginCal.get(java.util.Calendar.MONTH);

        final int result = betweenYear * 12 + betweenMonthOfYear;
        if (!isReset) {
            endCal.set(java.util.Calendar.YEAR, beginCal.get(java.util.Calendar.YEAR));
            endCal.set(java.util.Calendar.MONTH, beginCal.get(java.util.Calendar.MONTH));
            final long between = endCal.getTimeInMillis() - beginCal.getTimeInMillis();
            if (between < 0) {
                return result - 1;
            }
        }
        return result;
    }

    /**
     * Calculates the number of years between the two dates. If {@code isReset} is {@code false} and the month of the
     * beginning date is greater than the month of the ending date, the year count will be decremented by one
     * (indicating less than a full year).
     *
     * @param isReset Whether to reset the time of the dates to the beginning of the year (resetting month, day, hour,
     *                minute, second).
     * @return The number of years between the dates.
     */
    public long betweenYear(final boolean isReset) {
        final java.util.Calendar beginCal = Calendar.calendar(begin);
        final java.util.Calendar endCal = Calendar.calendar(end);

        final int result = endCal.get(java.util.Calendar.YEAR) - beginCal.get(java.util.Calendar.YEAR);
        if (isReset) {
            return result;
        }
        final int beginMonthBase0 = beginCal.get(java.util.Calendar.MONTH);
        final int endMonthBase0 = endCal.get(java.util.Calendar.MONTH);
        if (beginMonthBase0 < endMonthBase0) {
            return result;
        } else if (beginMonthBase0 > endMonthBase0) {
            return result - 1;
        } else if (java.util.Calendar.FEBRUARY == beginMonthBase0 && Calendar.isLastDayOfMonth(beginCal)
                && Calendar.isLastDayOfMonth(endCal)) {
            // Handle February leap year case.
            // If both dates are on the last day of February, treat months as equal. Both are set to the 1st.
            beginCal.set(java.util.Calendar.DAY_OF_MONTH, 1);
            endCal.set(java.util.Calendar.DAY_OF_MONTH, 1);
        }

        endCal.set(java.util.Calendar.YEAR, beginCal.get(java.util.Calendar.YEAR));
        final long between = endCal.getTimeInMillis() - beginCal.getTimeInMillis();
        return between < 0 ? result - 1 : result;
    }

    /**
     * Retrieves the beginning date.
     *
     * @return The beginning date.
     */
    public Date getBegin() {
        return begin;
    }

    /**
     * Retrieves the ending date.
     *
     * @return The ending date.
     */
    public Date getEnd() {
        return end;
    }

    /**
     * Formats and outputs the time difference as a string.
     *
     * @param unit  The date unit to format the difference in.
     * @param level The level of detail for formatting (e.g., {@link FormatPeriod.Level#MILLISECOND}).
     * @return The formatted time difference string.
     */
    public String toString(final Units unit, final FormatPeriod.Level level) {
        return FormatPeriod.of(between(unit), level).format();
    }

    /**
     * Formats and outputs the time difference as a string, using milliseconds as the default unit.
     *
     * @param level The level of detail for formatting.
     * @return The formatted time difference string.
     */
    public String toString(final FormatPeriod.Level level) {
        return toString(Units.MS, level);
    }

    /**
     * Returns a string representation of the time difference, formatted with millisecond level detail.
     *
     * @return A string representation of the object.
     */
    @Override
    public String toString() {
        return toString(FormatPeriod.Level.MILLISECOND);
    }

}
