/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.center.iterator;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * An abstract iterator with computational properties. By extending this abstract iterator and implementing the
 * {@link #computeNext()} method to calculate the next node, iteration can be completed. When {@link #hasNext()} is
 * called, the node generated by {@link #computeNext()} is cached until {@link #next()} is called to retrieve it. When
 * there are no more nodes, {@code null} must be returned to indicate the end of iteration.
 *
 * @param <T> the type of elements returned by this iterator
 * @author Kimi Liu
 * @since Java 17+
 */
public abstract class ComputeIterator<T> implements Iterator<T> {

    /**
     * The next element to be returned by the iterator.
     */
    private T next;
    /**
     * A flag indicating if the iterator has been fully read.
     */
    private boolean finished;

    /**
     * Computes the next element. Implement this method to define how the next element is generated. When
     * {@link #hasNext()} is called, the element produced by this method is cached until {@link #next()} is called to
     * retrieve it. Return {@code null} to indicate that there are no more elements to iterate.
     *
     * @return the next element, or {@code null} if the iteration has no more elements
     */
    protected abstract T computeNext();

    /**
     * Returns {@code true} if the iteration has more elements. (In other words, returns {@code true} if {@link #next}
     * would return an element rather than throwing an exception.)
     *
     * @return {@code true} if the iteration has more elements
     */
    @Override
    public boolean hasNext() {
        if (null != next) {
            // The user has read a node but not used it yet.
            return true;
        } else if (finished) {
            // Iteration has finished.
            return false;
        }

        final T result = computeNext();
        if (null == result) {
            // No more new nodes, end iteration.
            this.finished = true;
            return false;
        } else {
            this.next = result;
            return true;
        }

    }

    /**
     * Returns the next element in the iteration.
     *
     * @return the next element in the iteration
     * @throws NoSuchElementException if the iteration has no more elements
     */
    @Override
    public T next() {
        if (!hasNext()) {
            throw new NoSuchElementException("No more lines");
        }

        final T result = this.next;
        // Clear the cache, indicating this node has been read, and compute a new node next time.
        this.next = null;
        return result;
    }

    /**
     * Manually finishes the iterator, typically used for closing operations. After calling this method,
     * {@link #hasNext()} will return {@code false} and {@link #next()} will throw {@link NoSuchElementException}.
     */
    public void finish() {
        this.finished = true;
        this.next = null;
    }

}
